
package jp.riken.brain.ni.samuraigraph.base;

import java.awt.Dialog;
import java.awt.Frame;
import java.awt.GraphicsConfiguration;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.ArrayList;

import javax.swing.JDialog;
import javax.swing.text.JTextComponent;


/**
 * The base class of dialog.
 */
public abstract class SGDialog extends JDialog
	implements ActionListener, WindowListener, ComponentListener,
		SGIConstants
{

	/**
	 * Text for the OK button.
	 */
	public static final String OK_BUTTON_TEXT = "OK";

	/**
	 * Text for the Cancel button.
	 */
	public static final String CANCEL_BUTTON_TEXT = "Cancel";

	/**
	 * Text for the Yes button.
	 */
	public static final String YES_BUTTON_TEXT = "Yes";

	/**
	 * Text for the No button.
	 */
	public static final String NO_BUTTON_TEXT = "No";

	/**
	 * Text for the Previous button.
	 */
	public static final String PREVIOUS_BUTTON_TEXT = "<Prev";

	/**
	 * Text for the Next button.
	 */
	public static final String NEXT_BUTTON_TEXT = "Next>";


	// A list of action listeners.
	private ArrayList mActionListenerList = new ArrayList();


	// Closed option of the dialog.
	private int mCloseOption = -1;


	/**
	 * Creates a modeless dialog.
	 */
	public SGDialog()
	{
		super();
		this.initialize();
	}


	/**
	 * Creates a modal dialog with given owner.
	 * @param owner - the owner of this dialog
	 */
	public SGDialog( Dialog owner )
	{
		super( owner );
		this.initialize();
	}


	/**
	 * Creates a modal or modeless dialog with given owner.
	 * @param owner - the owner of this dialog
	 * @param modal - true for a modal dialog
	 */
	public SGDialog( Dialog owner, boolean modal )
	{
		super( owner, modal );
		this.initialize();
	}


	/**
	 * Creates a modal dialog with given owner and the title.
	 * @param owner - the owner of this dialog
	 * @param title - the title of this dialog
	 */
	public SGDialog( Dialog owner, String title )
	{
		super( owner, title );
		this.initialize();
	}


	/**
	 * Creates a dialog with given owner and title.
	 * @param owner - the owner of this dialog
	 * @param title - the title of this dialog
	 * @param modal - true for a modal dialog
	 */
	public SGDialog( Dialog owner, String title, boolean modal )
	{
		super( owner, title, modal );
		this.initialize();
	}


	/**
	 * Creates a dialog with given owner, title and GraphicsConfiguration.
	 * @param owner - the owner of this dialog
	 * @param modal - true for a modal dialog
	 * @param title - the title of this dialog
	 * @param gc - graphics configuration
	 */
	public SGDialog( Dialog owner, String title, boolean modal, GraphicsConfiguration gc )
	{
		super( owner, title, modal, gc );
		this.initialize();
	}



	/**
	 * Creates a modal dialog with given owner.
	 * @param owner - the owner of this dialog
	 */
	public SGDialog( Frame owner )
	{
		super( owner );
		this.initialize();
	}


	/**
	 * Creates a modal or modeless dialog with given owner.
	 * @param owner - the owner of this dialog
	 * @param modal - true for a modal dialog
	 */
	public SGDialog( Frame owner, boolean modal )
	{
		super( owner, modal );
		this.initialize();
	}


	/**
	 * Creates a modal dialog with given owner and the title.
	 * @param owner - the owner of this dialog
	 * @param title - the title of this dialog
	 */
	public SGDialog( Frame owner, String title )
	{
		super( owner, title );
		this.initialize();
	}


	/**
	 * Creates a dialog with given owner and title.
	 * @param owner - the owner of this dialog
	 * @param title - the title of this dialog
	 * @param modal - true for a modal dialog
	 */
	public SGDialog( Frame owner, String title, boolean modal )
	{
		super( owner, title, modal );
		this.initialize();
	}


	/**
	 * Creates a dialog with given owner, title and GraphicsConfiguration.
	 * @param owner - the owner of this dialog
	 * @param modal - true for a modal dialog
	 * @param title - the title of this dialog
	 * @param gc - graphics configuration
	 */
	public SGDialog( Frame owner, String title, boolean modal, GraphicsConfiguration gc )
	{
		super( owner, title, modal, gc );
		this.initialize();
	}


	// initialization method
	private void initialize()
	{
		// set unresizable
		this.setResizable(false);

		// add as a listner
		this.addComponentListener(this);
		this.addWindowListener(this);
	}



	public void componentShown(final ComponentEvent e){}

	public void componentHidden(final ComponentEvent e){}

	public void componentMoved(final ComponentEvent e)
	{
		Object owner = this.getOwner();
		if( owner instanceof Window )
		{
			Window wnd = (Window)owner;
			wnd.repaint();
		}
	}

	public void componentResized(final ComponentEvent e){}

	public void windowActivated(final WindowEvent e){}

	public void windowDeactivated(final WindowEvent e){}

	public void windowIconified(final WindowEvent e){}

	public void windowDeiconified(final WindowEvent e){}

	public void windowOpened(final WindowEvent e){}

	public void windowClosed(final WindowEvent e){}


	/**
	 * Overridden to set the canceled-option to the closed option.
	 */
	public void windowClosing(final WindowEvent e)
	{
		this.mCloseOption = CANCEL_OPTION;
	}


	/**
	 * Add an action listener.
	 */
	public void addActionListener( final ActionListener l )
	{
		ArrayList list = this.mActionListenerList;
		for( int ii=0; ii<list.size(); ii++ )
		{
			final ActionListener el = (ActionListener)list.get(ii);
			if( el.equals(l) )
			{
				return;
			}
		}
		list.add(l);
	}


	/**
	 * Remove an action listener.
	 */
	public void removeActionListener( ActionListener l )
	{
		ArrayList list = this.mActionListenerList;
		for( int ii=list.size()-1; ii>=0; ii-- )
		{
			final ActionListener el = (ActionListener)list.get(ii);
			if( el.equals(l) )
			{
				list.remove(l);
			}
		}
	}


	/**
	 * Notify the given command to the action listeners.
	 * @param command - the command to notify to the listeners
	 */
	protected void notifyToListener( final String command )
	{
		ArrayList list = this.mActionListenerList;
		for( int ii=0; ii<list.size(); ii++ )
		{
			final ActionListener el = (ActionListener)list.get(ii);
			el.actionPerformed( this.getActionEvent( command ) );
		}
	}


	// Creates an ActionEvent object with given command.
	private ActionEvent getActionEvent( final String command )
	{
		return new ActionEvent( this, 0, command );
	}


	/**
	 * Set the location of this dialog as the center of this dialog
	 * is equal to the center of the given window.
	 * @param wnd - a window to which the location of the center of this dialog is set
	 */
	public void setCenter( final Window wnd )
	{
		SGUtility.setCenter( this, wnd );
	}
	
	
	
	/**
	 * Parse the text string in the given text component, and returns
	 * a Number object if it is parsed as number.
	 * @param com - text component
	 * @return Number object if parsing succeeds, otherwise null
	 */
	protected Number getNumber( final JTextComponent com )
	{
		if( com==null )
		{
			return null;
		}
		String str = com.getText();
		Number num = SGUtilityText.getDouble(str);
		return num;
	}


	/**
	 * Returns the closed option.
	 * @return the closed option
	 */
	public int getCloseOption()
	{
		return this.mCloseOption;
	}
	
	
	/**
	 * Set the closed option.
	 * @param num - set as the closed option
	 */
	protected void setCloseOption( final int num )
	{
		this.mCloseOption = num;
	}

}


