
package jp.riken.brain.ni.samuraigraph.data;

import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import jp.riken.brain.ni.samuraigraph.base.SGData;
import jp.riken.brain.ni.samuraigraph.base.SGUtilityText;


/**
 * Data which contains date values.
 */
public abstract class SGDateData extends SGData
	implements Cloneable
{

	// array of date
	/**
	 * @uml.property  name="mDateArray"
	 * @uml.associationEnd  multiplicity="(0 -1)"
	 */
	protected InnerDate[] mDateArray = null;


	/**
	 * The default constructor.
	 *
	 */
	public SGDateData()
	{
		super();
	}


	/**
	 * 
	 */
	public SGDateData(
			final String[] titleArray,
			final String[] dateArray ) throws ParseException
	{
		super();

		final DateFormat df = DateFormat.getDateInstance();
		final Calendar cal = Calendar.getInstance();
		final int cYear = Calendar.YEAR;
		final int cMonth = Calendar.MONTH;
		final int cDate = Calendar.DATE;

		final int num = dateArray.length;
		final InnerDate[] array = new InnerDate[num];
		for( int ii=0; ii<num; ii++ )
		{
			array[ii] = new InnerDate( dateArray[ii] );
		}
		this.setTitle( titleArray );
		this.mDateArray = array;		
	}


	/**
	 * 
	 */
	public SGDateData(
		final String[] titleArray,
		final int[] yearArray, 
		final int[] monthArray,
		final int[] dateArray )
	{
		super();

		if( yearArray==null | monthArray==null | dateArray==null )
		{
			throw new IllegalArgumentException("yearArray==null | monthArray==null | dateArray==null");
		}

		if( yearArray.length!=monthArray.length | monthArray.length!=dateArray.length )
		{
			throw new IllegalArgumentException("yearArray.length!=monthArray.length | monthArray.length!=dateArray.length");
		}

		final int num = yearArray.length;
		InnerDate[] array = new InnerDate[num];
		for( int ii=0; ii<num; ii++ )
		{
			array[ii] = new InnerDate( yearArray[ii], monthArray[ii], dateArray[ii] );
		}
		this.setTitle( titleArray );
		this.mDateArray = array;		
	}


	/**
	 * Set data.
	 * @param data - data set to this object
	 */
	public boolean setData( final SGData data )
	{
		if( !(data instanceof SGDateData) )
		{
			throw new IllegalArgumentException("!(data instanceof SGDateData)");
		}

		SGDateData dData = (SGDateData)data;
		this.setTitle( dData.getTitleArray() );
		this.mDateArray = (InnerDate[])dData.mDateArray.clone();

		return true;
	}


	/**
	 * Clone this data object.
	 * @return copy of this data object
	 */
	public Object clone()
	{
		try
		{
			return super.clone();
		}
		catch( CloneNotSupportedException ex )
		{
			// this shouldn't happen, since we are Cloneable
			throw new InternalError();
		}
	}


	/**
	 * Calling the clone method.
	 * @return copy of this data object
	 */
	public Object copy()
	{
		return this.clone();
	}


	/**
	 * 
	 */
	public void dispose()
	{
		super.dispose();
		this.mDateArray = null;
	}


	/**
	 * An inner class for date. 
	 *
	 */
	protected static class InnerDate
	{
		private final String DEFAULT_DATE_FORMAT = "####/##/##";
		protected String mDateString;
		protected int year;
		protected int month;
		protected int date;

		protected InnerDate( final int Year, final int Month, final int Date )
		{
			super();
			this.year = Year;
			this.month = Month;
			this.date = Date;
			final DecimalFormat decimalFormat = new DecimalFormat(this.DEFAULT_DATE_FORMAT);
			this.mDateString = decimalFormat.format(this.year*10000 + this.month*100+ this.date);
		}
		
		protected InnerDate( final String str ) throws ParseException
		{
			super();
			final Calendar cal = Calendar.getInstance();
			final int cYear = Calendar.YEAR;
			final int cMonth = Calendar.MONTH;
			final int cDate = Calendar.DATE;
			this.mDateString = str;
			final Date d = SGUtilityText.getDate( str );
			if( d==null )
			{
				throw new ParseException("Parse error",0);
			}
			cal.setTime(d);
			this.year = cal.get( cYear );
			this.month = cal.get( cMonth );
			this.date = cal.get( cDate );
		}
	}
	

}

