package jp.riken.brain.ni.samuraigraph.application;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Image;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.prefs.Preferences;

import javax.print.attribute.standard.MediaSize;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.WindowConstants;

import jp.riken.brain.ni.samuraigraph.base.SGData;
import jp.riken.brain.ni.samuraigraph.base.SGDialog;
import jp.riken.brain.ni.samuraigraph.base.SGDrawingWindow;
import jp.riken.brain.ni.samuraigraph.base.SGExtensionFileFilter;
import jp.riken.brain.ni.samuraigraph.base.SGFigure;
import jp.riken.brain.ni.samuraigraph.base.SGFileChooser;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementAxisBreak;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementAxis;
import jp.riken.brain.ni.samuraigraph.base.SGIConstants;
import jp.riken.brain.ni.samuraigraph.base.SGIDrawingElementConstants;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElement;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementGraph;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementLegend;
import jp.riken.brain.ni.samuraigraph.base.SGIProgressControl;
import jp.riken.brain.ni.samuraigraph.base.SGIPropertyFileConstants;
import jp.riken.brain.ni.samuraigraph.base.SGIRootObjectConstants;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementGraphSXY;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementShape;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementSignificantDifference;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementString;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementTimingLine;
import jp.riken.brain.ni.samuraigraph.base.SGIFigureElementGraphVXY;
import jp.riken.brain.ni.samuraigraph.base.SGUtility;
import jp.riken.brain.ni.samuraigraph.base.SGUtilityText;
import jp.riken.brain.ni.samuraigraph.data.SGDataTypeConstants;
import jp.riken.brain.ni.samuraigraph.data.SGSXYData;
import jp.riken.brain.ni.samuraigraph.data.SGSXYDateData;
import jp.riken.brain.ni.samuraigraph.data.SGSXYMultipleData;
import jp.riken.brain.ni.samuraigraph.data.SGSXYSamplingData;
import jp.riken.brain.ni.samuraigraph.data.SGVXYData;
import jp.riken.brain.ni.samuraigraph.figure.SGDrawingElementArrow;
import jp.riken.brain.ni.samuraigraph.figure.SGDrawingElementErrorBar;
import jp.riken.brain.ni.samuraigraph.figure.SGDrawingElementLine;
import jp.riken.brain.ni.samuraigraph.figure.SGDrawingElementSymbol;
import jp.riken.brain.ni.samuraigraph.figure.SGIFigureTypeConstants;
import jp.riken.brain.ni.samuraigraph.figure.SGSXYFigure;
import jp.riken.brain.ni.samuraigraph.figure.SGVXYFigure;
import jp.riken.brain.ni.samuraigraph.figure.SGXYFigure;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import foxtrot.Task;
import foxtrot.Worker;


/**
 * The main thread.
 */
class SGMainFunctions
	implements ActionListener,
		SGIConstants,
		SGIUpgradeConstants, SGIApplicationCommandConstants,
		SGIApplicationConstants, SGIPropertyFileConstants,
		SGIPreferencesConstants, SGIApplicationTextConstants,
		SGIRootObjectConstants, SGIImageConstants, SGIArchiveFileConstants
{

    /**
     * Application Properties
     */
    private SGApplicationProperties mAppProp = null;
    
	/**
	 * Data set manager
	 */
	private DataSetManager  mDataSetManager;
	
	
	/**
	 * Proxy manager.
	 */
	private SGProxyManager mProxyManager;


	/**
	 * Upgrade manager.
	 */
	private SGUpgradeManager mUpgradeManager;


	/**
	 * Data creator.
	 */
	private SGDataCreator mDataCreator;

	
	/**
	 * Window manager
	 */
	private WindowManager mWindowManager = null; 
	
	/**
	 * Clip Board manager.
	 */
	private ClipBoardManager mClipBoardManager = null;

	/**
	 * Property manager
	 */
	private PropertyManager mPropertyManager = null;
	
	/**
	 * Property file manager.
	 */
	private PropertyFileManager mPropertyFileManager = null;
	
	/**
	 * Figure creator.
	 */
	private SGFigureCreator mFigureCreator = null;
	
	// The active figure.
	private SGFigure mActiveFigure = null;

	// initializer threadinstatnce
	private Initializer mInit = null;
	
	// Long Task thread
	private Thread mLongTask = null;
	
	/**
	 * Create a thread object.
	 *
	 */
	public SGMainFunctions( final SGApplicationProperties prop )
	{
		super();
		this.mAppProp = prop;
		this.mInit = new Initializer();
		this.mInit.start();
	}


	/**
	 * Create a thread object.
	 *
	 */
	public SGMainFunctions( String path, final int type, final SGApplicationProperties prop )
	{
		super();
		this.mAppProp = prop;
		this.mInit = new Initializer( path, type );
		this.mInit.start();
	}

	/**
	 * SGMainFunctions :: Initializer class
	 * @author okumura
	 */
	private class Initializer extends Thread
	{
	    private String mStartupFilePath = null;
	    private int mStartupFileType = FILETYPE_NONE;
	    
	    private Initializer()
	    {
	    }
	    
	    private Initializer( final String path, final int type )
	    {
	    	this.mStartupFilePath = path;
	    	this.mStartupFileType = type;
	    }

	    public void run()
	    {
			// create and show the splash window
			final SGSplashWindow sw = this.createSplashWindow();

			// remove temporary files used in upgrade
			this.removeTemporaryFiles();

			if( sw==null )
			{
			    exitApplication(1);
			}
			sw.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			sw.setProgressValue(0.0f);
			sw.setVisible(true);
//			sw.show();

			// create instances in attributes
			SGMainFunctions.this.mDataCreator = new SGDataCreator();
			sw.setProgressValue(0.15f);
			SGMainFunctions.this.mPropertyFileManager = new PropertyFileManager();
			SGMainFunctions.this.mDataSetManager = new DataSetManager( sw, 0.15f, 0.50f );
			sw.setProgressValue(0.50f);
			SGMainFunctions.this.mProxyManager = new SGProxyManager();
			sw.setProgressValue(0.60f);
			SGMainFunctions.this.mUpgradeManager = 
				new SGUpgradeManager( SGMainFunctions.this.mProxyManager, SGMainFunctions.this.mAppProp );
			SGMainFunctions.this.mWindowManager = new WindowManager( sw, 0.60f, 0.80f );
			sw.setProgressValue(0.80f);
			SGMainFunctions.this.mPropertyManager = new PropertyManager();
			SGMainFunctions.this.mClipBoardManager = new ClipBoardManager( sw, 0.80f, 1.0f );
			
			// create the figure element
			SGMainFunctions.this.mFigureCreator = new SGFigureCreator( true );
			if(!SGMainFunctions.this.mFigureCreator.validateClasses())
			{
			    exitApplication(1);
			}

			sw.setProgressValue(1.0f);

			SGDrawingWindow wnd = SGMainFunctions.this.mWindowManager.createNewWindow();
			SGMainFunctions.this.mWindowManager.setActiveWindow( wnd );
			wnd.setVisible(true);

			// hide the splash window
			sw.setVisible(false);
//			sw.hide();
			sw.setCursor(Cursor.getDefaultCursor());

			if( this.mStartupFileType == FILETYPE_PROPERTY )
			{
				File f = new File( this.mStartupFilePath );
				if( SGMainFunctions.this.mPropertyFileManager.showMultiDataFileChooserDialog(
					f, new ArrayList(), wnd ) == false )
				{
				   exitApplication(1);
				}
				
				final int ret = SGMainFunctions.this.mPropertyFileManager.mMultiDataFileChooserWizardDialogDD.getCloseOption();
				if( ret==SGIConstants.CANCEL_OPTION )
				{
					// exit this application
				    exitApplication(0);
				}
			}
			else if( this.mStartupFileType == FILETYPE_DATASET )
			{
				// disable window
				wnd.setWaitCursor(true);

				File f = new File( this.mStartupFilePath );
				if( !SGMainFunctions.this.mDataSetManager.loadDataSet( wnd, f ) )
				{
					// exit this application
					SGMainFunctions.this.exitApplication(0);
				}
				// enable window
				wnd.setWaitCursor(false);
			}
			
			// update the preferences
			this.updatePreferences();

			// decide whether to upgrade
			SGMainFunctions.this.mUpgradeManager.upgradeWithCheckDate(wnd);

	    }

	    /**
	     * create splash window
	     * @return
	     */
		private SGSplashWindow createSplashWindow()
		{
			// set the image and the text
			Class inClass = getClass();
			String name = RESOURCES_DIRNAME + "Splash.png";
			URL url = inClass.getResource(name);
			Image image = Toolkit.getDefaultToolkit().getImage( url );
			if( image==null )
			{
				return null;
			}
			SGSplashWindow sw = new SGSplashWindow( image, SGMainFunctions.this.mAppProp.getVersionString() );
			return sw;
		}


		/**
		 * update registry for update check cycle 
		 */
		private void updatePreferences()
		{
			Preferences pref = Preferences.userNodeForPackage( this.getClass() );

			// update the upgrade cycle at the first installation
			final int cycle = pref.getInt( PREF_KEY_UPGRADE_CYCLE, -1 );
			if( cycle==-1 )
			{
				pref.putInt( PREF_KEY_UPGRADE_CYCLE, DEFAULT_UPGRADE_CYCLE );
			}

			// update the date
			final String date = pref.get( PREF_KEY_DATE, null );
			pref.putLong( PREF_KEY_DATE, System.currentTimeMillis() );

			// remove old keys
			pref.remove( PREF_KEY_MAJOR_VERSION_NUMBER );
			pref.remove( PREF_KEY_MINOR_VERSION_NUMBER );
			pref.remove( PREF_KEY_MICRO_VERSION_NUMBER );
		}

        /**
         * Remove temporary files used in upgrade.
         */
        private void removeTemporaryFiles()
        {
        	File temp = new File( TMP_DIR + FILE_SEPARATOR + HELPER_TEMP_DIR_NAME );
        	try
        	{
        		temp = temp.getCanonicalFile();
        	}
        	catch( IOException ex )
        	{
        		return;
        	}
        	if( temp.exists() )
        	{
        		SGApplicationUtility.deleteRecursively( temp );
        	}
        }
	}
	
	public boolean waitInit()
	{
	    try {
	        this.mInit.join();
	    } catch (InterruptedException e)
	    {
	        return false;
	    }
        this.mInit = null;
	    return true;
	}

	
	/**
	 * SGMainFunctions :: DataSetManager class
	 * @author  okumura
	 */
	private class DataSetManager
	{
		private static final String DATASET_TEMPDIR_NAME      = "SamuraiGraphArcvhive";
		private static final String DATASET_PROPERTY_FILENAME = "property.sgp";
		private static final String MSG_SAVE_DATASET_FAILED   = "Failed save dataset";
		private static final String MSG_LOAD_DATASET_FAILED   = "Failed load dataset";

		/**
		 * Data set file extractor.
		 */
		private SGArchiveFileExtractor mArchiveFileExtractor = null;
		/**
		 * Data set file extractor.
		 */
		private SGArchiveFileCreator mArchiveFileCreator = null;

	    private DataSetManager( final SGSplashWindow sw, final float minprog, final float maxprog )
	    {
	        final float step = (maxprog - minprog) / 3;
	        sw.setProgressValue( minprog + step );
	        this.mArchiveFileExtractor = new SGArchiveFileExtractor();
	        sw.setProgressValue( minprog + step * 2.0f );
	        this.mArchiveFileCreator = new SGArchiveFileCreator();
	    }
	    
	    
	    public boolean loadDataSetFromDialog( final SGDrawingWindow wnd )
	    {
	        return this.loadDataSetFromEventDispatchThread( wnd, null );
	    }
	    
	    //
		private boolean loadDataSet( final SGDrawingWindow wnd, final File archiveFile )
		{
			// create temporary directory
			final File datasetTempDir = this._createTemporaryDirectory( wnd );
			if( datasetTempDir==null )
			{
				return false;
			}

			boolean result;
		    int errcode = -1;
			// extract
			if( archiveFile == null )
			{
			    // show dialog
				errcode = this.mArchiveFileExtractor.extract( wnd, datasetTempDir.getPath() );
			} else {
			    // directly extract
				errcode = this.mArchiveFileExtractor.extract( wnd, datasetTempDir.getPath(), archiveFile );
			}
			final ArrayList flist = this.mArchiveFileExtractor.getExtractedFileList();
			if(errcode != OK_OPTION){
				this.mArchiveFileExtractor.deleteExtractedFiles();
				datasetTempDir.delete();
				return false;
			}

			// get property file name
			final String pfname = datasetTempDir.getAbsolutePath() + FILE_SEPARATOR + DATASET_PROPERTY_FILENAME;
			if(!flist.contains(new File(pfname))){
				this.mArchiveFileExtractor.deleteExtractedFiles();
				datasetTempDir.delete();
				return false;
			}

			// get data file name list
			final ArrayList dfnameList = new ArrayList();
			this._sortDataList( datasetTempDir, flist, dfnameList );

			// load
			result = _loadDataSet(wnd, pfname, dfnameList);
			this.mArchiveFileExtractor.deleteExtractedFiles();
			datasetTempDir.delete();
			
			File f = this.mArchiveFileExtractor.getSelectedFile();
			if( f!=null )
			{
				updateCurrentFile( f, FILETYPE_DATASET );
			}

			return result;		
		}

		/**
		 * 
		 * @return
		 */
		private boolean loadDataSetFromEventDispatchThread( final SGDrawingWindow wnd, final File archiveFile )
		{
			// disable window
			wnd.setWaitCursor(true);

			Boolean result = Boolean.FALSE;
			
			try
			{
				result = (Boolean)Worker.post( new Task()
				{
					public Object run() throws Exception
					{
						if ( !loadDataSet( wnd, archiveFile ) )
							return Boolean.FALSE;
						return Boolean.TRUE;
					}
				});
			
			} catch(Exception ex)
			{
				result = Boolean.FALSE;
			}

			// enable window again
			wnd.setWaitCursor(false);
			
			return result.booleanValue();
		}
		
		
		// create the temporary directory to extract an archive file
		private File _createTemporaryDirectory( Component owner )
		{
			String failed = MSG_LOAD_DATASET_FAILED;

			// create temporary directory
			File datasetTempDir = new File( TMP_DIR + FILE_SEPARATOR + DATASET_TEMPDIR_NAME + System.currentTimeMillis() );
			if( datasetTempDir.mkdir() == false ){
				JOptionPane.showMessageDialog(owner, failed);
				return null;
			}
			
			return datasetTempDir;
		}


		// sort the list of data files
		private void _sortDataList( File datasetTempDir, ArrayList flist, ArrayList dfnameList )
		{
			int jj=0;
			for(int ii=0;;ii++){
				for(jj=0;;jj++){
					String fname = datasetTempDir.getAbsolutePath() + FILE_SEPARATOR + "id" + ii + "-" + jj + ".csv";
					if(!flist.contains(new File(fname))) {
						break;
					}
					dfnameList.add(fname);
				}
				if(jj==0){
					break;
				}
			}
		}
		
		private boolean _loadDataSet( final SGDrawingWindow wnd , final String pfname, final ArrayList dfnameList)
		{
			final File pfile = new File(pfname);
			if( pfile.exists()==false )
			{
				return false;
			}
			final String path = SGUtility.getCanonicalPath( pfile.getPath() );
			if( path==null )
			{
				return false;
			}

			// check validity of the file
			URL url = null;
			try
			{
				url = pfile.toURL();
			}
			catch( MalformedURLException ex )
			{
				return false;
			}

			// create a Document object
			Document doc = SGUtilityText.getDocument( url );
			if( doc==null )
			{
				return false;
			}

			//
			Element elWnd = SGMainFunctions.this.mPropertyFileManager.getWindowElement( doc );
			NodeList nListFigure = elWnd.getElementsByTagName( SGFigure.TAG_NAME_FIGURE );
			final int figureNum = nListFigure.getLength();
			int cnt = 0;
			final int[] dataNumArray = new int[figureNum];
			for( int ii=0; ii<figureNum; ii++ )
			{
				Node node = nListFigure.item(ii);
				if( ( node instanceof Element ) == false )
				{
					return false;
				}
				Element elFigure = (Element)node;
				NodeList nListData = elFigure.getElementsByTagName( SGIFigureElementGraph.TAG_NAME_DATA );
				dataNumArray[ii] = nListData.getLength();
				cnt += dataNumArray[ii];
			}
			if( cnt != dfnameList.size() )
			{
				return false;
			}
			
			// create a data map
			Map dataMap = new TreeMap();
			cnt = 0;
			for( int ii=0; ii<figureNum; ii++ )
			{
				final int dataNum = dataNumArray[ii];
				ArrayList dataList = new ArrayList();
				for( int jj=0; jj<dataNum; jj++ )
				{
					dataList.add( dfnameList.get(cnt+jj) );
				}
				dataMap.put( new Integer(ii), dataList );
				cnt+=dataNum;
			}

			// set properties
			wnd.clearUndoBuffer();
			wnd.removeAllFigures();
			wnd.updateClientRect();
			
			if( SGMainFunctions.this.mPropertyFileManager.setPropertyFile( wnd, doc, dataMap ) == false )
			{
				return false;
			}
			return true;	
		}		

	

		//
		public int saveDataSetFromDialog( final SGDrawingWindow wnd )
		{
			final String failed = MSG_SAVE_DATASET_FAILED;
			final String dir = getCurrentFileDirectory();
			final String name = SGMainFunctions.this.mCurrentArchiveFileName;
			
			ArrayList flist;
			
			// create temporary directory
			File datasetTempDir = new File( TMP_DIR + FILE_SEPARATOR + DATASET_TEMPDIR_NAME + System.currentTimeMillis() );
			if( datasetTempDir.mkdir() == false ){
				JOptionPane.showMessageDialog(wnd, failed);
				return -1;
			}

			flist = _dumpDataSet( wnd, datasetTempDir );
			if ( flist == null ){
				JOptionPane.showMessageDialog(wnd, failed);
				datasetTempDir.delete();
				return -1;
			}
			
			// set the selected file name
			this.mArchiveFileCreator.setFile( dir, name );

			// create an archive file
			int ret = this.mArchiveFileCreator.create( wnd, datasetTempDir.getPath() );
			
			// delete temporary files
			for(int kk=0; kk<flist.size(); kk++){
				File f = (File)flist.get(kk);
				f.delete();
			}	
			datasetTempDir.delete();
			
			if( ret!=OK_OPTION )
			{
				return ret;
			}

			File f = this.mArchiveFileCreator.getSelectedFile();
			if( f!=null )
			{
				updateCurrentFile( f, FILETYPE_DATASET );
			}

			return OK_OPTION;
		}
		
		//
		public int saveDataSet( final SGDrawingWindow wnd, final String path )
		{
			final String failed = MSG_SAVE_DATASET_FAILED;
			ArrayList flist;
			
			// create temporary directory
			File datasetTempDir = new File( TMP_DIR + FILE_SEPARATOR + DATASET_TEMPDIR_NAME + System.currentTimeMillis() );
			if( datasetTempDir.mkdir() == false ){
				JOptionPane.showMessageDialog(wnd, failed);
				return -1;
			}

			flist = _dumpDataSet( wnd, datasetTempDir );
			if ( flist == null ){
				JOptionPane.showMessageDialog(wnd, failed);
				datasetTempDir.delete();
				return -1;
			}
			
			
			// create an archive file
			int ret = this.mArchiveFileCreator.create( wnd, datasetTempDir.getPath(), path);
			
			// delete temporary files
			for(int kk=0; kk<flist.size(); kk++){
				File f = (File)flist.get(kk);
				f.delete();
			}
			datasetTempDir.delete();
			
			if( ret!=OK_OPTION )
			{
				return ret;
			}

			File f = this.mArchiveFileCreator.getSelectedFile();
			if( f!=null )
			{
				updateCurrentFile( f, FILETYPE_DATASET );
			}

			return OK_OPTION;
		}
		
		
		private ArrayList _dumpDataSet( final SGDrawingWindow wnd, final File datasetDir )
		{
			ArrayList flist = new ArrayList();
			
			// output property file
			final String pfname = datasetDir.getPath() + FILE_SEPARATOR + DATASET_PROPERTY_FILENAME;
			int ret = SGMainFunctions.this.mPropertyFileManager.saveProperties(wnd, pfname);
			if( ret!=OK_OPTION )
			{
				return null;
			}
			flist.add(new File(pfname));
			
			// output data files
			ArrayList figures = wnd.getVisibleFigureList();
			for(int ii=0; ii< figures.size(); ii++){
				SGFigure figure = (SGFigure)figures.get(ii);
				List dataList;
				dataList = figure.getVisibleDataList();
				for(int jj=0; jj<dataList.size(); jj++){
					SGData data = (SGData)dataList.get(jj);
					String fname = datasetDir.getPath() + FILE_SEPARATOR + "id" + ii + "-" + jj + ".csv";
					File file = new File(fname);
					if( !data.saveData(file) )
					{
						for(int kk=0; kk<flist.size(); kk++){
							File f = (File)flist.get(kk);
							f.delete();
						}	
						return null;
					}
					flist.add(file);				
				}
			}
			return flist;
			
		}
		
	}
	
	/**
	 * SGMainFunctions :: WindowManager class
	 * @author  okumura
	 */
	private class WindowManager 
	implements ActionListener, DropTargetListener, PropertyChangeListener, WindowListener
	{
		/**
		 * Image file creator.
		 */
		private SGImageExportManager mImageFileCreator = null;

		/**
		 * The list of windows.
		 */
		private TreeMap mWndMap = null;

		// The active window.
		private SGDrawingWindow mActiveWindow = null;
		

	    private WindowManager( final SGSplashWindow sw, final float minprog, final float maxprog )
	    {
	        final float step = (maxprog - minprog) / 3;
	        sw.setProgressValue( minprog + step );
	        this.mWndMap = new TreeMap();
	        sw.setProgressValue( minprog + step*2 );	        
			this.mImageFileCreator = new SGImageExportManager();
	    }
	    
		/**
		 * 
		 */
		public static final String WINDOW_TITLE_PREFIX = "Samurai Graph - Window : ";


		/**
		 * Create a window with given ID number.
		 * @param id ID number
		 * @return a created window
		 */
		public SGDrawingWindow createWindow( final int id )
		{
			
			// create an instance
			SGDrawingWindow wnd = new SGDrawingWindow();

			// set ID number
			wnd.setID( id );

			// put into the map
			this.mWndMap.put( new Integer(id), wnd );

			// set the close operation
			wnd.setDefaultCloseOperation( WindowConstants.DO_NOTHING_ON_CLOSE );

			// set window title
			String title = WINDOW_TITLE_PREFIX + wnd.getID();
			wnd.setTitle(title);


			// add this class as an ActionListener of the window
			wnd.addActionListener(this);

			// add this class as a WindowListener of the window
			wnd.addWindowListener(this);

			// add this class as a PropertyChangeListener of the window
			wnd.addPropertyChangeListener(this);


			// create and set the drop target
			DropTarget target = new DropTarget(
				wnd,
				DnDConstants.ACTION_COPY_OR_MOVE,
				this,
				true
			);
			wnd.setDropTarget(target);


			//
			wnd.setImageFileCreator( this.mImageFileCreator );


			// set the tool bar pattern
			Preferences pref = Preferences.userNodeForPackage( this.getClass() );
			final String pattern = pref.get( PREF_KEY_TOOL_BAR_PATTERN, null );
			String[] array;
			if( pattern!=null )
			{
				// tokenize a string
				ArrayList strList = new ArrayList();
				// pattern is not read from data file.
				if( SGUtilityText.tokenize( pattern, strList, false ) )
				{
					ArrayList keyList = new ArrayList();
					for( int ii=0; ii<strList.size(); ii++ )
					{
						String token = (String)strList.get(ii);
						if( token!=null )
						{
							keyList.add( token );
						}
					}

					array = new String[keyList.size()];
					for( int ii=0; ii<keyList.size(); ii++ )
					{
						array[ii] = (String)keyList.get(ii);
					}
				}
				else
				{
					array = SGIRootObjectConstants.TOOLBAR_MENUCMD_ARRAY;
				}
			}
			else
			{
//				array = SGIRootObjectConstants.TOOLBAR_MENUCMD_ARRAY;

				array = new String[]{
					MENUBARCMD_VISIBLE_FILE,
					MENUBARCMD_VISIBLE_INSERT,
					MENUBARCMD_VISIBLE_ZOOM
				};

				// set to preferences
				updateToolBarPatternInPreferences( array );
			}


			// set to the window
			wnd.setToolBarPattern( array );


			// set the location of the window
			{
				wnd.setLocation(100,100);
			}

			// initialize
			if( wnd.init() == false )
			{
				return null;
			}

			// set the menus enabled
			this.setPasteMenuEnabled();	// paste

			return wnd;
		}

		/**
		 * 
		 */
		public SGDrawingWindow getWindow( final int wndID )
		{
			return (SGDrawingWindow)this.mWndMap.get( new Integer(wndID) );
		}

		/**
		 * get active window
		 */
		public SGDrawingWindow getActiveWindow()
		{
		    return this.mActiveWindow;
		}
		
		/**
		 * Create a new window.
		 * @return a created window
		 */
		private SGDrawingWindow createWindow()
		{
			final int id = this.getCurrentWindowID();
			SGDrawingWindow wnd = this.createWindow(id);
			return wnd;
		}

		/**
		 * Returns the ID number for a window to be created.
		 * @return ID number
		 */
		private int getCurrentWindowID()
		{
			int id;
			if( this.mWndMap.size()==0 )
			{
				id = 1;
			}
			else
			{
				id = ((Integer)this.mWndMap.lastKey()).intValue() + 1;
			}
			return id;
		}

		/**
		 * 
		 * @return
		 */
		public int getWindowNumber()
		{
			return this.mWndMap.size();
		}

		/**
		 * Create a new window.
		 * @return a created window
		 */
		public SGDrawingWindow createNewWindow()
		{
			SGDrawingWindow wnd = this.createWindow();
			if( wnd==null )
			{
				return null;
			}
			wnd.initPropertiesHistory();
			return wnd;
		}

		/**
		 * Close the window with confirmation.
		 * @param id - ID number of the window to close
		 */
		public void closeWindow( final int id )
		{
			SGDrawingWindow wnd = this.getWindow(id);
			if( wnd==null )
			{
				return;
			}
			this.closeWindow(wnd);
		}
		
		/**
		 * Close the window with confirmation.
		 * @param wnd - a window to be closed
		 */
		public int closeWindow( final SGDrawingWindow wnd )
		{
			if( wnd.needsConfirmationBeforeDiscard() )
			{
				final int ret = confirmBeforeClosing( wnd );
				if( ret==JOptionPane.YES_OPTION )
				{
					this.removeWindow(wnd);
				}
				else if( ret==JOptionPane.NO_OPTION | ret==JOptionPane.CLOSED_OPTION )
				{
					// canceled and there is nothing to do
					return CANCEL_OPTION;
				}
				else if( ret==JOptionPane.CANCEL_OPTION )
				{
					// save the properties
					final int retSave = SGMainFunctions.this.mPropertyFileManager.savePropertiesByDialog( wnd );
					if( retSave == OK_OPTION )
					{
						this.removeWindow(wnd);
					}

					return retSave;
				}
			}
			else
			{
				this.removeWindow(wnd);
			}

			return OK_OPTION;
		}

		public void closeAllWindow()
		{
		    final ArrayList wndList = new ArrayList( this.mWndMap.values() );
		    for( int ii=wndList.size()-1; ii>=0; ii-- )
		    {
		        SGDrawingWindow wnd = (SGDrawingWindow)wndList.get(ii);
		        final int ret = this.closeWindow(wnd);
		        if( ret==CANCEL_OPTION )
		        {
		            break;
		        }
		    }
		}

		/**
		 * 
		 *
		 */
		public void setActiveWindow( final SGDrawingWindow wnd )
		{
			this.mActiveWindow = wnd;
		}

		/**
		 * 
		 *
		 */
		private void setPasteMenuEnabled()
		{
		    final boolean b = SGMainFunctions.this.mClipBoardManager.getPasteMenuStatus();
			ArrayList wList = new ArrayList( this.mWndMap.values() );
			for( int ii=0; ii<wList.size(); ii++ )
			{
				SGDrawingWindow wnd = (SGDrawingWindow)wList.get(ii);
				wnd.setPasteMenuEnabled(b);
			}
		}

		// Remove the window from window map.
		private boolean removeWindow( final SGDrawingWindow wnd )
		{
			final int id = wnd.getID();
			this.mWndMap.remove( new Integer(id) );
			wnd.removeWindowListener( this );
			wnd.removeActionListener( this );
			wnd.setVisible(false);
			wnd.dispose();

			// run the garbage collector
			System.gc();

			// if all windows are closed, exit the application
			if( this.mWndMap.size() == 0 )
			{
				exitApplication(0);
			}

			return true;
		}

		//
		// Event Listeners..
		//
		
		// ActionListener
		/**
		 * The listener interface for receiving action events. 
		 * 
		 * @param e the action event
		 */
		public void actionPerformed( final ActionEvent e )
		{

			Object source = e.getSource();
			String command = e.getActionCommand();

			SGDrawingWindow wnd = (SGDrawingWindow)source;

			if( command.equals( MENUBARCMD_CREATE_NEW_WINDOW ) )
			{
				SGDrawingWindow w = this.createNewWindow();
				if( w==null )
				{
					return;
				}
				w.setVisible(true);
			}
			else if( command.equals( MENUBARCMD_CLOSE_WINDOW ) )
			{
				this.closeWindow( wnd );
			}
			else if( command.equals( MENUBARCMD_EXIT ) )
			{
			    this.closeAllWindow();
				// exit();
			}
			else if( command.equals( MENUBARCMD_DRAW_GRAPH ) )
			{
			    drawGraphFromDialog(wnd);
			}
			else if( command.equals( MENUBARCMD_SAVE_PROPERTY ) )
			{
				final int ret = SGMainFunctions.this.mPropertyFileManager.savePropertiesByDialog( wnd );
				wnd.setSaved( ret == OK_OPTION );
			}
			else if( command.equals( MENUBARCMD_LOAD_PROPERTY ) )
			{
			    if( wnd.needsConfirmationBeforeDiscard() )
			    {
			        final int ret = beforeDiscard(wnd);
			        if( ret==CANCEL_OPTION )
			        {
			            return;
			        }
			    }
			    SGMainFunctions.this.mPropertyFileManager.loadPropertyFromDialog( wnd );
			}
			else if( command.equals( MENUBARCMD_SAVE_DATASET ) )
			{
				final int ret = SGMainFunctions.this.mDataSetManager.saveDataSetFromDialog( wnd );
				wnd.setSaved( ret == OK_OPTION );
			}
			else if( command.equals( MENUBARCMD_LOAD_DATASET ) )
			{
				if( wnd.needsConfirmationBeforeDiscard() )
				{
					final int ret = beforeDiscard(wnd);
					if( ret==CANCEL_OPTION )
					{
						return;
					}
				}
				final boolean result = SGMainFunctions.this.mDataSetManager.loadDataSetFromDialog( wnd );
				wnd.setSaved( result );
			}
			else if( command.equals( MENUBARCMD_LOAD_BACKGROUND_IMAGE ) )
			{
				JFileChooser chooser = new SGFileChooser();
				
				// set the file extension filter
				SGExtensionFileFilter ff = new SGExtensionFileFilter();
				ff.setDescription( IMAGE_FILE_DESCRIPTION );

				String[] extensions = DRAWABLE_IMAGE_EXTENSIONS;
				for( int ii=0; ii<extensions.length; ii++ )
				{
					ff.addExtension( extensions[ii] );
				}
				chooser.setFileFilter(ff);

				// get current path
				String path = getCurrentFileDirectory();

				// set default directory name and file name of the image file
				String fileName = path;
				if( SGMainFunctions.this.mCurrentImageFileName!=null )
				{
					fileName += FILE_SEPARATOR + SGMainFunctions.this.mCurrentImageFileName;
				}
				chooser.setSelectedFile( new File( fileName ) );

				// show open dialog
				final int ret = chooser.showOpenDialog(wnd);

				File file = null;
				if( ret == JFileChooser.APPROVE_OPTION )
				{
					file = chooser.getSelectedFile();
				}

				// set the image to the window
				if( file!=null )
				{
					updateCurrentFile( file, FILETYPE_IMAGE );
					if( wnd.setImage( file ) == false )
					{
						return;
					}
				}
			}
			else if( command.equals( MENUBARCMD_CUT ) )
			{
				SGMainFunctions.this.mClipBoardManager.cutAndCopy( wnd, false );
			}
			else if( command.equals( MENUBARCMD_COPY ) )
			{
				SGMainFunctions.this.mClipBoardManager.cutAndCopy( wnd, true );
			}
			else if( command.equals( MENUBARCMD_PASTE ) )
			{
				SGMainFunctions.this.mClipBoardManager.pasteToWindow( wnd );
			}
			else if( command.equals( MENUBARCMD_DUPLICATE ) )
			{
				SGMainFunctions.this.duplicateFocusedFigures(wnd);
			}
			else if( command.equals( MENUBARCMD_EXPORT_AS_IMAGE ) )
			{
				this.mImageFileCreator.setBaseDirectory( getCurrentFileDirectory() );
				wnd.exportAsImage( false );
				SGMainFunctions.this.setCurrentFileDirectory( this.mImageFileCreator.getBaseDirectory() );
			}
			else if( command.equals( MENUBARCMD_PRINT ) )
			{
				wnd.printImage( false );
			}
			else if( command.equals( MENUBARCMD_UPGRADE ) )
			{
				SGMainFunctions.this.mUpgradeManager.showUpgradeDialog( wnd );
			}
			else if( command.equals( MENUBARCMD_CHANGE_LOG ) )
			{
				SGMainFunctions.this.showChangeLogDialog(wnd);
			}
			else if( command.equals( MENUBARCMD_PROXY ) )
			{
				SGMainFunctions.this.mProxyManager.showProxySettingDialog( wnd );
			}
			else if( command.equals( MENUBARCMD_ABOUT ) )
			{
				SGMainFunctions.this.showAboutDialog(wnd);
			}
		}
		
		// DropTargetListener
		/**
	     * Called while a drag operation is ongoing, when the mouse pointer enters
	     * the operable part of the drop site for the <code>DropTarget</code>
	     * registered with this listener. 
	     * 
	     * @param dtde the <code>DropTargetDragEvent</code> 
	     */
		public void dragEnter( final DropTargetDragEvent dtde )
		{
			dtde.acceptDrag(DnDConstants.ACTION_COPY_OR_MOVE);
		}

		/**
	     * Called when a drag operation is ongoing, while the mouse pointer is still
	     * over the operable part of the drop site for the <code>DropTarget</code>
	     * registered with this listener.
	     * 
	     * @param dtde the <code>DropTargetDragEvent</code> 
	     */
		public void dragOver( final DropTargetDragEvent dtde )
		{
		}
		
	    /**
	     * Called if the user has modified the current drop gesture.
	     * 
	     * @param dtde the <code>DropTargetDragEvent</code>
	     */
		public void dropActionChanged( final DropTargetDragEvent dtde )
		{
		}

		/**
	     * Called while a drag operation is ongoing, when the mouse pointer has
	     * exited the operable part of the drop site for the <code>DropTarget</code>
	     * registered with this listener.
	     * 
	     * @param dte the <code>DropTargetEvent</code> 
	     */
		public void dragExit( final DropTargetEvent dte )
		{
		}

		/**
	     * Called when the drag operation has terminated with a drop on
	     * the operable part of the drop site for the <code>DropTarget</code>
	     * registered with this listener.  
	     * 
	     * @param dtde the <code>DropTargetDropEvent</code> 
	     */
		public void drop( final DropTargetDropEvent dtde )
		{
			DropTarget tg = (DropTarget)dtde.getSource();
			Component com = tg.getComponent();
			SGDrawingWindow wnd = (SGDrawingWindow)com;
			dtde.acceptDrop( DnDConstants.ACTION_COPY_OR_MOVE );
			try
			{
				if( ( dtde.getDropAction() & DnDConstants.ACTION_COPY_OR_MOVE ) != 0 )
				{
					Transferable trans = dtde.getTransferable();
					ArrayList list = null;
					if ( dtde.isDataFlavorSupported( DataFlavor.javaFileListFlavor ) )
					{
					    List lst = (List)trans.getTransferData( DataFlavor.javaFileListFlavor );
					    list = new ArrayList( lst );
					}
					else if ( dtde.isDataFlavorSupported( DataFlavor.stringFlavor ) )
					{
					    String str;
					    String ss = (String)trans.getTransferData( DataFlavor.stringFlavor );
					    StringTokenizer st = new StringTokenizer( ss, "\n" );
					    list = new ArrayList();
					    while(st.hasMoreTokens()){
					        str = st.nextToken();
					        File file = new File( new URI(str.trim()) );
					        list.add( file );
					    }
					}
					// set attributes
					if( list != null )
					{
					    DropEventHandler th = new DropEventHandler( dtde, list );
					}
				}
			}
			catch ( Exception ex )
			{

			}
			finally
			{
				dtde.dropComplete( false );
			}

		}

		// PropertyChangeListener
		/**
	     * This method gets called when a bound property is changed.
	     * @param e A PropertyChangeEvent object describing the event source 
	     *   	and the property that has changed.
	     */
		public void propertyChange( PropertyChangeEvent e )
		{
			Object source = e.getSource();
			String name = e.getPropertyName();

			if( source instanceof SGDrawingWindow )
			{
				if( SGIRootObjectConstants.PROPERTY_NAME_TOOL_BAR.equals( name ) )
				{
					SGDrawingWindow wnd = (SGDrawingWindow)source;
					updateToolBarPatternInPreferences( wnd.getToolBarPattern() );
				}
			}
		}

	    // WindowListener
	    /**
	     * Invoked the first time a window is made visible.
	     */
		public void windowOpened( final WindowEvent e )
		{
		}

	   	/**
	     * Invoked when the user attempts to close the window
	     * from the window's system menu.  If the program does not 
	     * explicitly hide or dispose the window while processing 
	     * this event, the window close operation will be cancelled.
	     */
		public void windowClosing( final WindowEvent e )
		{
			Object source = e.getSource();
			final SGDrawingWindow wnd = (SGDrawingWindow)e.getSource();
			this.closeWindow( wnd );
		}

		/**
	     * Invoked when a window has been closed as the result
	     * of calling dispose on the window.
	     */
		public void windowClosed( final WindowEvent e )
		{
		}

	    /**
	     * Invoked when a window is changed from a normal to a
	     * minimized state. For many platforms, a minimized window 
	     * is displayed as the icon specified in the window's 
	     * iconImage property.
	     * @see java.awt.Frame#setIconImage
	     */
		public void windowIconified( final WindowEvent e )
		{	
		}
		
	    /**
	     * Invoked when a window is changed from a minimized
	     * to a normal state.
	     */
		public void windowDeiconified( final WindowEvent e )
		{	
		}
		
		/**
	     * Invoked when the Window is set to be the active Window. Only a Frame or
	     * a Dialog can be the active Window. The native windowing system may
	     * denote the active Window or its children with special decorations, such
	     * as a highlighted title bar. The active Window is always either the
	     * focused Window, or the first Frame or Dialog that is an owner of the
	     * focused Window.
	     */
		public void windowActivated( final WindowEvent e )
		{
		}

		/**
	     * Invoked when a Window is no longer the active Window. Only a Frame or a
	     * Dialog can be the active Window. The native windowing system may denote
	     * the active Window or its children with special decorations, such as a
	     * highlighted title bar. The active Window is always either the focused
	     * Window, or the first Frame or Dialog that is an owner of the focused
	     * Window.
	     */
		public void windowDeactivated( final WindowEvent e )
		{
		}

	}
	
	/**
	 * SGMainFunctions :: ClipBoardManager class
	 * @author  okumura
	 */
	private class ClipBoardManager
	{
        /**
         * A private class to store information of a figure.
         */
        private class WindowInfo
        {
        	/**
        	 * A DOM tree object of figures.
        	 */	
        	private Document mDocument;
        
        	/**
        	 * The map of the list of data objects.
        	 */		
        	private Map mDataListMap = new TreeMap();
        
        }
 
		/**
		 * The list of copied objects.
		 */
		private ArrayList mCopiedObjectsBuffer = null;
		
		/**
		 * The list of copied data objects.
		 */
		private ArrayList mCopiedDataObjectBuffer = null;
		
		/**
		 * The list of copied data names.
		 */
		private ArrayList mCopiedDataNameBuffer = null;
		
		/**
		 * The list of copied data properties.
		 */
		private ArrayList mCopiedDataPropertiesBuffer = null;

		/**
		 * The list of copied data propaties for legend.
		 */
		private ArrayList mCopiedDataPropertiesForLegendBuffer = null;
		
		/**
		 * Inforamation of the focused figures.
		 */
		private WindowInfo mWindowInfo;

		
	    private ClipBoardManager(final SGSplashWindow sw, final float minprog, final float maxprog )
	    {
	        final float step = (maxprog - minprog) / 6;
	        sw.setProgressValue( minprog + step );
	        this.mCopiedObjectsBuffer = new ArrayList();
	        sw.setProgressValue( minprog + step*2 );
	        this.mCopiedDataObjectBuffer = new ArrayList();
	        sw.setProgressValue( minprog + step*3 );
	        this.mCopiedDataNameBuffer = new ArrayList();
	        sw.setProgressValue( minprog + step*4 );
	        this.mCopiedDataPropertiesBuffer = new ArrayList();
	        sw.setProgressValue( minprog + step*5 );
	        this.mCopiedDataPropertiesForLegendBuffer = new ArrayList();
	    }
	    
		/**
		 * Cut or copy the objects.
		 * @param wnd
		 * @param isCopy a flag to set cut/copy
		 * @return true:succeeded, false:failed
		 */
		private boolean cutAndCopy( final SGDrawingWindow wnd, final boolean isCopy )
		{
			// initialize the lists
			this.mCopiedObjectsBuffer.clear();
			this.mCopiedDataObjectBuffer.clear();
			this.mCopiedDataNameBuffer.clear();
			this.mCopiedDataPropertiesBuffer.clear();

			// get copied objects list and add them to the buffer of the attribute
			ArrayList cList = wnd.getCopiedObjectsList();
			this.mCopiedObjectsBuffer.addAll( cList );

			ArrayList dList = wnd.getCopiedObjectsDataList();
			this.mCopiedDataObjectBuffer.addAll( dList );

			ArrayList nList = wnd.getCopiedDataNameList();
			this.mCopiedDataNameBuffer.addAll( nList );

			ArrayList pList = wnd.getCopiedDataPropertiesMapList();
			this.mCopiedDataPropertiesBuffer.addAll( pList );


			WindowInfo info = new WindowInfo();

			if( isCopy )
			{
				if( this.copyFigures( wnd, info ) == false )
				{
					return false;
				}
			}
			else
			{
				if( this.cutFigures( wnd, info ) == false )
				{
					return false;
				}
			}

			// set to the attribute
			this.mWindowInfo = info;

			// set the paste menu enabled
			SGMainFunctions.this.mWindowManager.setPasteMenuEnabled();

			return true;
		}

		/**
		 * 
		 * @return
		 */
		private boolean cutFigures(
			SGDrawingWindow wnd,
			WindowInfo info )
		{
			if( this.copyFigures( wnd, info ) == false )
			{
				return false;
			}
			
			if( info.mDataListMap.size()!=0 )
			{
				wnd.hideSelectedObjects();
			}

			return true;
		}


		/**
		 * 
		 * @return
		 */
		private boolean copyFigures( SGDrawingWindow wnd, WindowInfo info )
		{
			DOMImplementation domImpl = SGApplicationUtility.getDOMImplementation();
			if( domImpl==null )
			{
				return false;
			}

			// create a Document object
			Document document = domImpl.createDocument( "", TAG_NAME_FOCUSED_FIGURES, null );

			// create a DOM tree
			if( wnd.createDOMTree( document, SGIRootObjectConstants.FOCUSED_FIGURES_FOR_DUPLICATION ) == false )
			{
				return false;
			}

			// create a map of figure ID and data list
			Map dListMap = new TreeMap();
			ArrayList fList = wnd.getCopiedFiguresList();
			for( int ii=0; ii<fList.size(); ii++ )
			{
				SGFigure figure = (SGFigure)fList.get(ii);
				ArrayList dList = new ArrayList( figure.getVisibleDataList() );
				dList = (ArrayList)dList.clone();

				String key = Integer.toString( figure.getID() );
				dListMap.put( key, dList );
			}


			// set to the attribute
			info.mDocument = document;
			info.mDataListMap = dListMap;

			return true;
		}

		

		/**
		 * Paste objects to the window.
		 * @param wnd - the target object
		 * @return
		 */
		private boolean pasteToWindow( SGDrawingWindow wnd )
		{
			// paste the copied objects to figures
			wnd.pasteToFigures(
				this.mCopiedObjectsBuffer,
				this.mCopiedDataObjectBuffer,
				this.mCopiedDataNameBuffer,
				this.mCopiedDataPropertiesBuffer );


			// paste the copied figure objects
			WindowInfo info = this.mWindowInfo;
			if( info==null )
			{
				return false;
			}
			Map dListMap = info.mDataListMap;
		
			// get root element
			Element root = info.mDocument.getDocumentElement();

			// get the node of window
			NodeList wList = root.getElementsByTagName( SGIRootObjectConstants.TAG_NAME_WINDOW );
			if( wList.getLength()==0 )
			{
				return false;
			}
			Element elWnd = (Element)wList.item(0);

			final int before = wnd.getFigureList().size();

			// create figures from a DOM tree
			final int ret = createFiguresFromPropertyFile( elWnd, wnd, dListMap );
			if( ret!=SGIConstants.SUCCESSFUL_COMPLETION )
			{
				return false;
			}

			final int after = wnd.getFigureList().size();
		
			wnd.setChanged( before!=after );

			return true;
		}

		private boolean getPasteMenuStatus()
		{
			final boolean b1 = (this.mCopiedObjectsBuffer.size()!=0);
			final boolean b2 = (this.mCopiedDataObjectBuffer.size()!=0);

			boolean b3 = false;
			if( this.mWindowInfo!=null )
			{
				Document doc = this.mWindowInfo.mDocument;
				NodeList nodeList = doc.getElementsByTagName( SGFigure.TAG_NAME_FIGURE );
				final int len = nodeList.getLength();
				b3 = (len!=0);
			}

			final boolean b = ( b1 | b2 | b3 );

			return b;
		}


	}
		
	/**
	 * SGMainFunctions :: PropertyManager class
	 * @author okumura
	 */
	private class PropertyManager
	{

	    private PropertyManager()
	    {
	    }
	    
		//
		private boolean contains( final String[] keys, final String str )
		{
			ArrayList list = new ArrayList( Arrays.asList( keys ) );
			return list.contains(str);
		}

		// returns an integer value for axis location from a string
		private int getAxisLocation( final String location )
		{
			final String str = location.toLowerCase();
			int loc = -1;
			if( SGIFigureElementAxis.AXIS_BOTTOM.toLowerCase().equals(str) )
			{
				loc = SGIFigureElementAxis.AXIS_HORIZONTAL_1;
			}
			else if( SGIFigureElementAxis.AXIS_TOP.toLowerCase().equals(str) )
			{
				loc = SGIFigureElementAxis.AXIS_HORIZONTAL_2;
			}
			else if( SGIFigureElementAxis.AXIS_LEFT.toLowerCase().equals(str) )
			{
				loc = SGIFigureElementAxis.AXIS_PERPENDICULAR_1;
			}
			else if( SGIFigureElementAxis.AXIS_RIGHT.toLowerCase().equals(str) )
			{
				loc = SGIFigureElementAxis.AXIS_PERPENDICULAR_2;
			}

			return loc;
		}

		// returns an integer value for line type from a string
		private int getLineType( final String value )
		{
			Number n = SGUtilityText.getInteger( value );
			if( n!=null )
			{
				final int type = n.intValue();
				if( SGDrawingElementLine.getLineTypeName( type ) !=null )
				{
					return type;
				}
				return -1;
			}
			final String str = value.toLowerCase();
			int type = -1;
			final Integer num = SGDrawingElementLine.getLineTypeFromName(str);
			if( num!=null )
			{
			    type = num.intValue();
			}
			return type;
		}

		// returns an integer value for symbol type from a string
		private int getSymbolType( final String value )
		{
			Number n = SGUtilityText.getInteger( value );
			if( n!=null )
			{
				final int type = n.intValue();
				if( SGDrawingElementSymbol.getSymbolTypeName( type ) !=null )
				{
					return type;
				}
				return -1;
			}
			final String str = value.toLowerCase();
			int type = -1;
			final Integer num = SGDrawingElementSymbol.getSymbolTypeFromName(str);
			if( num!=null )
			{
			    type = num.intValue();
			}
			return type;
		}

		// returns an integer value for font style from a string
		private int getFontStyle( final String value )
		{
			Number n = SGUtilityText.getInteger( value );
			if( n!=null )
			{
				final int type = n.intValue();
				if( SGUtilityText.getFontStyleName(type)!=null )
				{
					return type;
				}
				return -1;
			}
			final String str = value.toLowerCase();
			final int style = SGUtilityText.getFontStyle(str);
			return style;
		}

		// returns an integer value for scale type from a string
		private int getScaleType( final String value )
		{
			Number n = SGUtilityText.getInteger( value );
			if( n!=null )
			{
				final int type = n.intValue();
				if( SGUtilityText.getScaleTypeName(type)!=null )
				{
					return type;
				}
				return -1;
			}
			final String str = value.toLowerCase();
			final int style = SGUtilityText.getScaleType(str);
			return style;
		}

		// returns an integer value for error bar style from a string
		private int getErrorBarStyle( final String value )
		{
			Number n = SGUtilityText.getInteger( value );
			if( n!=null )
			{
				final int type = n.intValue();
				if( SGDrawingElementErrorBar.getErrorBarStyleName(type)!=null )
				{
					return type;
				}
				return -1;
			}
			final String str = value.toLowerCase();
			int style = -1;
			final Integer num = SGDrawingElementErrorBar.getErrorBarStyleFromName(str);
			if( num!=null )
			{
			    style = num.intValue();
			}
			return style;
		}

		// set the properties of a window
		private int setWindowProperties(
			final String command, final SGDrawingWindow wnd, final ArrayList argsList )
		{
			final String[] keys = {
				COM_PAPER_WIDTH,
				COM_PAPER_HEIGHT,
				COM_PAPER_SIZE,
				COM_WINDOW_GRID_INTERVAL,
				COM_WINDOW_GRID_LINE_WIDTH,
				COM_WINDOW_GRID_VISIBLE,
				COM_WINDOW_GRID_COLOR,
				COM_WINDOW_BACKGROUND_COLOR,
				COM_IMAGE_X,
				COM_IMAGE_Y,
				COM_IMAGE_SCALING_FACTOR,
				COM_CUT_FIGURE,
				COM_COPY_FIGURE,
				COM_PASTE_FIGURE,
				COM_DELETE_FIGURE,
				COM_DUPLICATE_FIGURE,
				COM_MOVE_TO_FRONT_FIGURE,
				COM_MOVE_TO_BACK_FIGURE
			};
			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}

			if( wnd==null )
			{
				return STATUS_FAILED;
			}

			final int argn = argsList.size();

			if( command.equals( COM_PAPER_WIDTH ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.setPaperWidthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_PAPER_HEIGHT ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.setPaperHeightDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_PAPER_SIZE ) )
			{
				if( argn==1 )
				{
					String str1 = (String)argsList.get(0);
					MediaSize size = SGUtilityText.getMediaSize(str1);
					if( size==null )
					{
						return STATUS_FAILED;
					}
					if( wnd.setPaperSizeDirectly( size ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argn==2 )
				{
					String str1 = (String)argsList.get(0);
					StringBuffer unit1 = new StringBuffer();
					Number num1 = SGUtilityText.getNumber( str1, unit1 );
					if( num1==null )
					{
						return STATUS_FAILED;
					}
					String str2 = (String)argsList.get(1);
					StringBuffer unit2 = new StringBuffer();
					Number num2 = SGUtilityText.getNumber( str2, unit2 );
					if( num2==null )
					{
						return STATUS_FAILED;
					}
					if( wnd.setPaperSizeDirectly(
						num1.floatValue(), unit1.toString(), num2.floatValue(), unit2.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
			}
			else if( command.equals( COM_WINDOW_GRID_INTERVAL ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.setGridLineIntervalDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_WINDOW_GRID_LINE_WIDTH ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.setGridLineWidthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_WINDOW_GRID_VISIBLE ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				String s = SGUtilityText.getBooleanString( str );
				if( s==null )
				{
					return STATUS_FAILED;
				}
				Boolean b = Boolean.valueOf(s);
				if( wnd.setGridLineVisibleDirectly( b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_WINDOW_GRID_COLOR ) )
			{
				if( argn==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor(str);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( wnd.setGridLineColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argn==3 )
				{
					String str0 = (String)argsList.get(0);
					String str1 = (String)argsList.get(1);
					String str2 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str0,str1,str2);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( wnd.setGridLineColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_WINDOW_BACKGROUND_COLOR ) )
			{
				if( argn==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor(str);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( wnd.setPaperColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argn==3 )
				{
					String str0 = (String)argsList.get(0);
					String str1 = (String)argsList.get(1);
					String str2 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str0,str1,str2);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( wnd.setPaperColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_IMAGE_X ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.setImageLocationXDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_IMAGE_Y ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.setImageLocationYDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_IMAGE_SCALING_FACTOR ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				Number num = SGUtilityText.getFloat(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.setImageScalingFactorDirectly( num.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
//			else if( command.equals( COM_CUT_FIGURE ) )
//			{
//				if( argn!=1 )
//				{
//					return FAILED;
//				}
//				String str = (String)argsList.get(0);
//				final Integer num = SGUtilityText.getInteger(str);
//				if( num==null )
//				{
//					return FAILED;
//				}
//				if( wnd.cutOrCopyFigure( num.intValue(), false ) == false )
//				{
//					return FAILED;
//				}
//			}
//			else if( command.equals( COM_COPY_FIGURE ) )
//			{
//				if( argn!=1 )
//				{
//					return FAILED;
//				}
//				String str = (String)argsList.get(0);
//				final Integer num = SGUtilityText.getInteger(str);
//				if( num==null )
//				{
//					return FAILED;
//				}
//				if( wnd.cutOrCopyFigure( num.intValue(), true ) == false )
//				{
//					return FAILED;
//				}
//			}
//			else if( command.equals( COM_PASTE_FIGURE ) )
//			{
//				if( argn!=1 )
//				{
//					return FAILED;
//				}
//				String str = (String)argsList.get(0);
//				final Integer num = SGUtilityText.getInteger(str);
//				if( num==null )
//				{
//					return FAILED;
//				}
//				if( wnd.hideFigure( num.intValue() ) == false )
//				{
//					return FAILED;
//				}
//			}
			else if( command.equals( COM_DELETE_FIGURE ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final Integer num = SGUtilityText.getInteger(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.hideFigure( num.intValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_DUPLICATE_FIGURE ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final Integer num = SGUtilityText.getInteger(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.hideFigure( num.intValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_MOVE_TO_FRONT_FIGURE ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final Integer num = SGUtilityText.getInteger(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.moveFigure( num.intValue(), true ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_MOVE_TO_BACK_FIGURE ) )
			{
				if( argn!=1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final Integer num = SGUtilityText.getInteger(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( wnd.moveFigure( num.intValue(), false ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of a figure
		private int setFigureProperties(
			final String command, final SGFigure fig, final ArrayList argsList )
		{
			if( fig==null )
			{
				return STATUS_FAILED;
			}

			final String[] keys = {
				COM_FIGURE_X,
				COM_FIGURE_Y,
				COM_FIGURE_WIDTH,
				COM_FIGURE_HEIGHT,
				COM_FIGURE_SPACE_TO_SCALE,
				COM_FIGURE_SPACE_TO_TITLE,
				COM_FIGURE_BACKGROUND_COLOR,
				COM_FIGURE_TRANSPARENT,

				// grid properties
				COM_FIGURE_GRID_AXIS_X,
				COM_FIGURE_GRID_AXIS_Y,
				COM_FIGURE_GRID_VISIBLE,
				COM_FIGURE_GRID_AUTO,
				COM_FIGURE_GRID_STEP_X,
				COM_FIGURE_GRID_STEP_Y,
				COM_FIGURE_GRID_BASE_X,
				COM_FIGURE_GRID_BASE_Y,
				COM_FIGURE_GRID_LINE_WIDTH,
				COM_FIGURE_GRID_LINE_TYPE,
				COM_FIGURE_GRID_LINE_COLOR
			};
			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}

			final int argn = argsList.size();
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			String str = (String)argsList.get(0);

			if( command.equals( COM_FIGURE_X ) )
			{
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( fig.setFigureXDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_FIGURE_Y ) )
			{
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( fig.setFigureYDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_FIGURE_WIDTH ) )
			{
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( fig.setFigureWidthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_FIGURE_HEIGHT ) )
			{
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( fig.setFigureHeightDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_FIGURE_SPACE_TO_SCALE ) )
			{
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( fig.setSpaceAxisLineAndNumberDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_FIGURE_SPACE_TO_TITLE ) )
			{
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( fig.setSpaceNumberAndTitleDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_FIGURE_BACKGROUND_COLOR ) )
			{
				if( argn==1 )
				{
					final Color cl = SGUtilityText.getColor(str);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( fig.setBackgroundColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argn==3 )
				{
					String str0 = str;
					String str1 = (String)argsList.get(1);
					String str2 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str0,str1,str2);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( fig.setBackgroundColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_FIGURE_TRANSPARENT ) )
			{
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str ) );
				if( fig.setTransparentDirectly( b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}

			if( fig instanceof SGXYFigure )
			{
				SGXYFigure figxy = (SGXYFigure)fig;

				if( command.equals( COM_FIGURE_GRID_AXIS_X ) )
				{
					final int value = this.getAxisLocation( str );
					if( value==-1 )
					{
						return STATUS_FAILED;
					}
					if( figxy.setXAxisLocationDirectly( value ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_AXIS_Y ) )
				{
					final int value = this.getAxisLocation( str );
					if( value==-1 )
					{
						return STATUS_FAILED;
					}
					if( figxy.setYAxisLocationDirectly( value ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_VISIBLE ) )
				{
					Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str ) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridVisibleDirectly( b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					} 
				}
				else if( command.equals( COM_FIGURE_GRID_AUTO ) )
				{
					Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str ) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setAutoCalculateRangeDirectly( b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					} 
				}
				else if( command.equals( COM_FIGURE_GRID_STEP_X ) )
				{
					Number num = SGUtilityText.getDouble( str );
					if( num==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridStepValueXDirectly( num.doubleValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_STEP_Y ) )
				{
					Number num = SGUtilityText.getDouble( str );
					if( num==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridStepValueYDirectly( num.doubleValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_BASE_X ) )
				{
					Number num = SGUtilityText.getDouble( str );
					if( num==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridBaselineValueXDirectly( num.doubleValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_BASE_Y ) )
				{
					Number num = SGUtilityText.getDouble( str );
					if( num==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridBaselineValueYDirectly( num.doubleValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_LINE_WIDTH ) )
				{
					StringBuffer unit = new StringBuffer();
					Number num = SGUtilityText.getNumber( str, unit );
					if( num==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridLineWidthDirectly( num.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_LINE_TYPE ) )
				{
					final int num = this.getLineType(str);
					if( num==-1 )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridLineTypeDirectly( num ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_FIGURE_GRID_LINE_COLOR ) )
				{
					Color cl = null;
					if( argn==1 )
					{
						cl = SGUtilityText.getColor(str);
					}
					else if( argn==3 )
					{
						String str0 = (String)argsList.get(0);
						String str1 = (String)argsList.get(1);
						String str2 = (String)argsList.get(2);
						cl = SGUtilityText.getColor(str0,str1,str2);
					}
					else
					{
						return STATUS_FAILED;
					}
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( figxy.setGridLineColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}

			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of Data objects
		private int setDataProperties(
			final String command, final SGIFigureElementGraph el, final ArrayList argsList )
		{
			final String[] keys = {
				// common
				COM_DATA_AXIS_X,
				COM_DATA_AXIS_Y,
				COM_DATA_VISIBLE_IN_LEGEND,
				COM_DATA_NAME,

				// line
				COM_DATA_LINE_VISIBLE,
				COM_DATA_LINE_WIDTH,
				COM_DATA_LINE_TYPE,
				COM_DATA_LINE_COLOR,

				// symbol
				COM_DATA_SYMBOL_VISIBLE,
				COM_DATA_SYMBOL_TYPE,
				COM_DATA_SYMBOL_BODY_SIZE,
				COM_DATA_SYMBOL_BODY_COLOR,
				COM_DATA_SYMBOL_LINE_WIDTH,
				COM_DATA_SYMBOL_LINE_COLOR,

				// bar
				COM_DATA_BAR_VISIBLE,
				COM_DATA_BAR_BASELINE,
				COM_DATA_BAR_BODY_WIDTH,
				COM_DATA_BAR_BODY_COLOR,
				COM_DATA_BAR_LINE_WIDTH,
				COM_DATA_BAR_LINE_COLOR,

				// error bar
				COM_DATA_ERROR_BAR_VISIBLE,
				COM_DATA_ERROR_BAR_SYMBOL_TYPE,
				COM_DATA_ERROR_BAR_SYMBOL_SIZE,
				COM_DATA_ERROR_BAR_COLOR,
				COM_DATA_ERROR_BAR_LINE_WIDTH,
				COM_DATA_ERROR_BAR_STYLE,

				// tick label
				COM_DATA_TICK_LABEL_VISIBLE,
				COM_DATA_TICK_LABEL_FONT_NAME,
				COM_DATA_TICK_LABEL_FONT_STYLE,
				COM_DATA_TICK_LABEL_FONT_SIZE,
				COM_DATA_TICK_LABEL_FONT_COLOR,
				COM_DATA_TICK_LABEL_HORIZONTAL,

				// arrow
				COM_DATA_ARROW_MAGNITUDE_PER_CM,
				COM_DATA_ARROW_DIRECTION_INVARIANT,
				COM_DATA_ARROW_LINE_WIDTH,
				COM_DATA_ARROW_LINE_TYPE,
				COM_DATA_ARROW_HEAD_SIZE,
				COM_DATA_ARROW_COLOR,
				COM_DATA_ARROW_START_TYPE,
				COM_DATA_ARROW_END_TYPE,
				COM_DATA_ARROW_OPEN_ANGLE,
				COM_DATA_ARROW_CLOSE_ANGLE,

				//
				COM_DELETE_DATA,
				COM_MOVE_TO_FRONT_DATA,
				COM_MOVE_TO_BACK_DATA
			};

			if( this.contains( keys, command) == false )
			{
				return STATUS_NOT_FOUND;
			}

			final int argn = argsList.size();
			if( argsList.size() < 1 )
			{
				return STATUS_FAILED;
			}

			String str1 = (String)argsList.get(0);
			final Number id = SGUtilityText.getInteger(str1);
			if( id==null )
			{
				return STATUS_FAILED;
			}

			if( argn == 1 )
			{
				if( command.equals( COM_DELETE_DATA ) )
				{
					if( el.hideChildObject( id.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_FRONT_DATA ) )
				{
					if( el.moveChildObject( id.intValue(), true ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_BACK_DATA ) )
				{
					if( el.moveChildObject( id.intValue(), false ) == false )
					{
						return STATUS_FAILED;
					}
				}

				return STATUS_SUCCEEDED;
			}


			String str2 = (String)argsList.get(1);

			if( command.equals( COM_DATA_AXIS_X ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisXDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_DATA_AXIS_Y ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisYDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_DATA_VISIBLE_IN_LEGEND ) )
			{
				final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setVisibleInLegendDirectly( id.intValue(), b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_DATA_NAME ) )
			{
				if( el.setNameDirectly( id.intValue(), str2 ) == false )
				{
					return STATUS_FAILED;
				}
			}


			if( el instanceof SGIFigureElementGraphSXY )
			{
				SGIFigureElementGraphSXY elsxy = (SGIFigureElementGraphSXY)el;

				if( command.equals( COM_DATA_LINE_VISIBLE ) )
				{
					final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setLineVisibleDirectly( id.intValue(), b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_LINE_WIDTH ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_LINE_TYPE ) )
				{
					final int num = this.getLineType(str2);
					if( num==-1 )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setLineTypeDirectly( id.intValue(), num ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_LINE_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setLineColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_SYMBOL_VISIBLE ) )
				{
					final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setSymbolVisibleDirectly( id.intValue(), b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_SYMBOL_TYPE ) )
				{
					final int num = this.getSymbolType(str2);
					if( num==-1 )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setSymbolTypeDirectly( id.intValue(), num ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_SYMBOL_BODY_SIZE ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setSymbolSizeDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_SYMBOL_LINE_WIDTH ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setSymbolLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_SYMBOL_BODY_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setSymbolInnerColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_SYMBOL_LINE_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setSymbolLineColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_BAR_VISIBLE ) )
				{
					final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setBarVisibleDirectly( id.intValue(), b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_BAR_BASELINE ) )
				{
					final Number value = SGUtilityText.getDouble(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setBarBaselineDirectly( id.intValue(), value.floatValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_BAR_BODY_WIDTH ) )
				{
					final Number value = SGUtilityText.getDouble(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setBarWidthDirectly( id.intValue(), value.floatValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_BAR_LINE_WIDTH ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setBarEdgeLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_BAR_BODY_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setBarInnerColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_BAR_LINE_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setBarLineColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ERROR_BAR_VISIBLE ) )
				{
					final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setErrorBarVisibleDirectly( id.intValue(), b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ERROR_BAR_SYMBOL_TYPE ) )
				{
					final Number value = SGDrawingElementErrorBar.getHeadTypeFromName(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setErrorBarSymbolTypeDirectly( id.intValue(), value.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ERROR_BAR_SYMBOL_SIZE ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setErrorBarSymbolSizeDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ERROR_BAR_LINE_WIDTH ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setErrorBarLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ERROR_BAR_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setErrorBarColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ERROR_BAR_STYLE ) )
				{
					final int value = this.getErrorBarStyle(str2);
					if( value==-1 )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setErrorBarStyleDirectly( id.intValue(), value ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_TICK_LABEL_VISIBLE ) )
				{
					final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setTickLabelVisibleDirectly( id.intValue(), b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_TICK_LABEL_FONT_NAME ) )
				{
					if( elsxy.setTickLabelFontNameDirectly( id.intValue(), str2 ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_TICK_LABEL_FONT_STYLE ) )
				{
					final int value = this.getFontStyle(str2);
					if( value==-1 )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setTickLabelFontStyleDirectly( id.intValue(), value ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_TICK_LABEL_FONT_SIZE ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setTickLabelFontSizeDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_TICK_LABEL_FONT_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setTickLabelFontColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_TICK_LABEL_HORIZONTAL ) )
				{
					final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( elsxy.setTickLabelAngleDirectly( id.intValue(), b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
			}
			else if( el instanceof SGIFigureElementGraphVXY )
			{
				SGIFigureElementGraphVXY elvxy = (SGIFigureElementGraphVXY)el;

				if( command.equals( COM_DATA_ARROW_MAGNITUDE_PER_CM ) )
				{
					Number value = SGUtilityText.getFloat(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setMagnitudePerCMDirectly( id.intValue(), value.floatValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_DIRECTION_INVARIANT ) )
				{
					Boolean b = SGUtilityText.getBoolean(str2);
					if( b==null )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setDirectionInvariantDirectly( id.intValue(), b.booleanValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_LINE_WIDTH ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_LINE_TYPE ) )
				{
					final int num = this.getLineType(str2);
					if( num==-1 )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setLineTypeDirectly( id.intValue(), num ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_HEAD_SIZE ) )
				{
					StringBuffer unit = new StringBuffer();
					Number value = SGUtilityText.getNumber( str2, unit );
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setHeadSizeDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_COLOR ) )
				{
					Color cl = null;
					if( argn==2 )
					{
						cl = SGUtilityText.getColor(str2);
					}
					else if( argn==4 )
					{
						String str3 = (String)argsList.get(2);
						String str4 = (String)argsList.get(3);
						cl = SGUtilityText.getColor(str2,str3,str4);
					}
					else
					{
						return STATUS_FAILED;
					}

					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setColorDirectly( id.intValue(), cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_START_TYPE ) )
				{
					final Number value = SGDrawingElementArrow.getSymbolTypeFromName(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setStartTypeDirectly( id.intValue(), value.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_END_TYPE ) )
				{
					final Number value = SGDrawingElementArrow.getSymbolTypeFromName(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					if( elvxy.setEndTypeDirectly( id.intValue(), value.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_OPEN_ANGLE ) )
				{
					final Number value = SGUtilityText.getDouble(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					final float angle = value.floatValue()*SGIConstants.RADIAN_DEGREE_RATIO;
					if( elvxy.setOpenAngleDirectly( id.intValue(), angle ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_DATA_ARROW_CLOSE_ANGLE ) )
				{
					final Number value = SGUtilityText.getDouble(str2);
					if( value==null )
					{
						return STATUS_FAILED;
					}
					final float angle = value.floatValue()*SGIConstants.RADIAN_DEGREE_RATIO;
					if( elvxy.setCloseAngleDirectly( id.intValue(), angle ) == false )
					{
						return STATUS_FAILED;
					}
				}

			}

			return STATUS_SUCCEEDED;
		}

		
		// set the properties of SGIFigureElement
		private int setFigureElementProperties(
			final String command, final SGFigure fig, final ArrayList argsList )
		{
			if( fig==null )
			{
				return STATUS_NOT_FOUND;
			}

			int ret;

			// set Data properties
			SGIFigureElementGraph gElement = fig.getGraphElement();
			if( gElement!=null )
			{
				ret = this.setDataProperties( command, gElement, argsList );
				if( ret!= STATUS_NOT_FOUND )
				{
					return ret;
				}
			}

			// set Axis properties
			SGIFigureElementAxis aElement	= fig.getAxisElement();
			if( aElement!=null )
			{
				ret = this.setAxisProperties( command, aElement, argsList );
				if( ret!= STATUS_NOT_FOUND )
				{
					return ret;
				}
			}

			// set Legend properties
			SGIFigureElementLegend lElement = fig.getLegendElement();
			if( lElement!=null )
			{
				ret = this.setLegendProperties( command, lElement, argsList );
				if( ret!= STATUS_NOT_FOUND )
				{
					return ret;
				}
			}
			
			if ( fig instanceof SGXYFigure )
			{
				SGXYFigure xyfig = (SGXYFigure)fig;
				// set Grid properties TODO:
//				SGIGridElement grElement = fig.getGridElement();
//				if( lElement!=null )
//				{
//					ret = this.setGridProperties( command, grElement, argsList );
//					if( ret!= STATUS_NOT_FOUND )
//					{
//						return ret;
//					}
//				}
				// set Label properties
				SGIFigureElementString sElement
				= (SGIFigureElementString)xyfig.getSymbolElement( SGIDrawingElementConstants.SYMBOL_ELEMENT_TYPE_STRING );
				if( sElement!=null )
				{
					ret = this.setLabelProperties( command, sElement, argsList );
					if( ret!= STATUS_NOT_FOUND )
					{
						return ret;
					}
				}
				
				// set Significant Difference Symbol properties
				SGIFigureElementSignificantDifference sdElement
				= (SGIFigureElementSignificantDifference)xyfig.getSymbolElement( SGIDrawingElementConstants.SYMBOL_ELEMENT_TYPE_SIGDIFF );
				if( sdElement!=null )
				{
					ret = this.setSigDiffProperties( command, sdElement, argsList );
					if( ret!= STATUS_NOT_FOUND )
					{
						return ret;
					}
				}
				
				// set Axis Break Symbol properties
				SGIFigureElementAxisBreak abElement
				= (SGIFigureElementAxisBreak)xyfig.getSymbolElement( SGIDrawingElementConstants.SYMBOL_ELEMENT_TYPE_AXISBREAK);
				if( abElement!=null )
				{
					ret = this.setAxisBreakProperties( command, abElement, argsList );
					if( ret!= STATUS_NOT_FOUND )
					{
						return ret;
					}
				}
				
				// set Timing Line properties
				SGIFigureElementTimingLine tlElement
				= (SGIFigureElementTimingLine)xyfig.getSymbolElement( SGIDrawingElementConstants.SYMBOL_ELEMENT_TYPE_TIMINGLINE );
				if( tlElement!=null )
				{
					ret = this.setTimingLineProperties( command, tlElement, argsList );
					if( ret!= STATUS_NOT_FOUND )
					{
						return ret;
					}
				}
				
				// set Shape properties
				SGIFigureElementShape shElement
				= (SGIFigureElementShape)xyfig.getSymbolElement( SGIDrawingElementConstants.SYMBOL_ELEMENT_TYPE_SHAPE );
				if( shElement!=null )
				{
					ret = this.setShapeProperties( command, shElement, argsList );
					if( ret!= STATUS_NOT_FOUND )
					{
						return ret;
					}
				}
			}
			return STATUS_NOT_FOUND;
		}

		// set the properties of Axis
		private int setAxisProperties(
			final String command, final SGIFigureElementAxis el, final ArrayList argsList )
		{

			final String[] keys = {
				// text
				COM_AXIS_FONT,
				COM_AXIS_FONT_NAME,
				COM_AXIS_FONT_STYLE,
				COM_AXIS_FONT_SIZE,
				COM_AXIS_FONT_COLOR,
		
				// frame
				COM_AXIS_FRAME,
				COM_AXIS_FRAME_VISIBLE,
				COM_AXIS_FRAME_LINE_WIDTH,
				COM_AXIS_FRAME_COLOR,
		
				// lines
				COM_AXIS_LINE_WIDTH,
				COM_TICK_MARK_WIDTH,
				COM_TICK_MARK_LENGTH,
				COM_AXIS_LINE_COLOR,
		
				// each axis
				COM_AXIS_VISIBLE,
				COM_AXIS_TITLE_VISIBLE,
				COM_AXIS_TITLE_LABEL,
				COM_AXIS_SCALE_MIN,
				COM_AXIS_SCALE_MAX,
				COM_AXIS_SCALE_TYPE,
				COM_AXIS_TICK_MARK_VISIBLE,
				COM_AXIS_TICK_MARK_AUTO,
				COM_AXIS_TICK_MARK_STEP,
				COM_AXIS_TICK_MARK_BASE,
				COM_AXIS_TICK_MARK_INNER,
				COM_AXIS_FORMAT_VISIBLE,
				COM_AXIS_FORMAT_INTEGER,
				COM_AXIS_FORMAT_EXPONENT_VISIBLE,
				COM_AXIS_FORMAT_EXPONTNE_VALUE,
			};

			if( this.contains( keys, command) == false )
			{
				return STATUS_NOT_FOUND;
			}

			final int argn = argsList.size();

			if( command.equals( COM_AXIS_FONT_NAME ) )
			{
				String str = (String)argsList.get(0);
				el.setFontNameDirectly( str );
			}
			else if( command.equals( COM_AXIS_FONT_STYLE ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final int style = this.getFontStyle( str );
				if( style==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setFontStyleDirectly( style ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FONT_SIZE ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFontSizeDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FONT_COLOR ) )
			{
				if( argsList.size()==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor(str);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFontColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argsList.size()==3 )
				{
					String str1 = (String)argsList.get(0);
					String str2 = (String)argsList.get(1);
					String str3 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str1,str2,str3);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFontColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FRAME_VISIBLE ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFrameVisibleDirectly( b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FRAME_LINE_WIDTH ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFrameLineWidthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FRAME_COLOR ) )
			{
				if( argsList.size()==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor(str);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFrameLineColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argsList.size()==3 )
				{
					String str1 = (String)argsList.get(0);
					String str2 = (String)argsList.get(1);
					String str3 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str1,str2,str3);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFrameLineColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_LINE_WIDTH ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisLineWidthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_TICK_MARK_WIDTH ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTickMarkWidthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_TICK_MARK_LENGTH ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTickMarkLengthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_LINE_COLOR ) )
			{
				if( argsList.size()==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor(str);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setLineColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argsList.size()==3 )
				{
					String str1 = (String)argsList.get(0);
					String str2 = (String)argsList.get(1);
					String str3 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str1,str2,str3);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setLineColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_VISIBLE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisVisibleDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_TITLE_VISIBLE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTitleVisibleDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_TITLE_LABEL ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				if( el.setTitleDirectly( str1, str2 ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_SCALE_MIN ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Number num = SGUtilityText.getDouble(str2);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setMinValueDirectly( str1, num.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_SCALE_MAX ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Number num = SGUtilityText.getDouble(str2);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setMaxValueDirectly( str1, num.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_SCALE_TYPE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				final int type = this.getScaleType( str2 );
				if( type==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setScaleTypeDirectly( str1, type ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_TICK_MARK_VISIBLE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTickMarkVisibleDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_TICK_MARK_AUTO ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTickMarkAutoDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_TICK_MARK_STEP ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Number num = SGUtilityText.getDouble(str2);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTickMarkStepDirectly( str1, num.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_TICK_MARK_BASE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Number num = SGUtilityText.getDouble(str2);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTickMarkBaseDirectly( str1, num.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_TICK_MARK_INNER ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setTickMarkInnerDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FORMAT_VISIBLE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFormatVisibleDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FORMAT_INTEGER ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFormatIntegerDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FORMAT_EXPONENT_VISIBLE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str2) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFormatExponentVisibleDirectly( str1, b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_AXIS_FORMAT_EXPONTNE_VALUE ) )
			{
				if( argsList.size()<2 )
				{
					return STATUS_FAILED;
				}
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				Number num = SGUtilityText.getInteger( str2 );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFormatExponentValueDirectly( str1, num.intValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of Label
		private int setLabelProperties(
			final String command, final SGIFigureElementString el, final ArrayList argsList )
		{
			final String[] keys =
			{
				COM_LABEL_AXIS_X,
				COM_LABEL_AXIS_Y,
				COM_LABEL_TEXT,
				COM_LABEL_LOCATION_X,
				COM_LABEL_LOCATION_Y,
				COM_LABEL_FONT_NAME,
				COM_LABEL_FONT_STYLE,
				COM_LABEL_FONT_SIZE,
				COM_LABEL_FONT_COLOR,
				COM_LABEL_ANGLE,
				COM_DELETE_LABEL,
				COM_MOVE_TO_FRONT_LABEL,
				COM_MOVE_TO_BACK_LABEL
			};

			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}

			final int argn = argsList.size();
			if( argsList.size() < 1 )
			{
				return STATUS_FAILED;
			}

			String str1 = (String)argsList.get(0);
			final Number id = SGUtilityText.getInteger(str1);
			if( id==null )
			{
				return STATUS_FAILED;
			}

			if( argn == 1 )
			{
				if( command.equals( COM_DELETE_LABEL ) )
				{
					if( el.hideChildObject( id.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_FRONT_LABEL ) )
				{
					if( el.moveChildObject( id.intValue(), true ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_BACK_LABEL ) )
				{
					if( el.moveChildObject( id.intValue(), false ) == false )
					{
						return STATUS_FAILED;
					}
				}

				return STATUS_SUCCEEDED;
			}


			String str2 = (String)argsList.get(1);

			if( command.equals( COM_LABEL_AXIS_X ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisXDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_AXIS_Y ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisYDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_TEXT ) )
			{
				if( el.setTextDirectly( id.intValue(), str2 ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_LOCATION_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationXDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_LOCATION_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationYDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_FONT_NAME ) )
			{
				if( el.setFontNameDirectly( id.intValue(), str2 ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_FONT_STYLE ) )
			{
				final int value = this.getFontStyle(str2);
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setFontStyleDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_FONT_SIZE ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFontSizeDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_FONT_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFontColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_ANGLE ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				final float angle = value.floatValue()*SGIConstants.RADIAN_DEGREE_RATIO;
				if( el.setAngleDirectly( id.intValue(), angle ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of Significant Differece Symbol
		private int setSigDiffProperties(
			final String command, final SGIFigureElementSignificantDifference el, final ArrayList argsList )
		{
			final String[] keys = {
				COM_SIGDIFF_AXIS_X,
				COM_SIGDIFF_AXIS_Y,
				COM_SIGDIFF_LOCATION_HORIZONTAL_Y,
				COM_SIGDIFF_LOCATION_LEFT_X,
				COM_SIGDIFF_LOCATION_LEFT_Y,
				COM_SIGDIFF_LOCATION_RIGHT_X,
				COM_SIGDIFF_LOCATION_RIGHT_Y,
				COM_SIGDIFF_LINE_VISIBLE,
				COM_SIGDIFF_SPACE,
				COM_SIGDIFF_LINE_WIDTH,
				COM_SIGDIFF_TEXT,
				COM_SIGDIFF_FONT_NAME,
				COM_SIGDIFF_FONT_STYLE,
				COM_SIGDIFF_FONT_SIZE,
				COM_SIGDIFF_COLOR,
				COM_DELETE_SIGDIFF,
				COM_MOVE_TO_FRONT_SIGDIFF,
				COM_MOVE_TO_BACK_SIGDIFF
			};
			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}

			final int argn = argsList.size();
			if( argsList.size() < 1 )
			{
				return STATUS_FAILED;
			}

			String str1 = (String)argsList.get(0);
			final Number id = SGUtilityText.getInteger(str1);
			if( id==null )
			{
				return STATUS_FAILED;
			}

			if( argn == 1 )
			{
				if( command.equals( COM_DELETE_SIGDIFF ) )
				{
					if( el.hideChildObject( id.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_FRONT_SIGDIFF ) )
				{
					if( el.moveChildObject( id.intValue(), true ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_BACK_SIGDIFF ) )
				{
					if( el.moveChildObject( id.intValue(), false ) == false )
					{
						return STATUS_FAILED;
					}
				}

				return STATUS_SUCCEEDED;
			}


			String str2 = (String)argsList.get(1);

			if( command.equals( COM_SIGDIFF_AXIS_X ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisXDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_AXIS_Y ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisYDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LABEL_TEXT ) )
			{
				if( el.setTextDirectly( id.intValue(), str2 ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_LOCATION_HORIZONTAL_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationHorizontalYDirectly( id.intValue(), value.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_LOCATION_LEFT_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationLeftXDirectly( id.intValue(), value.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_LOCATION_LEFT_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationLeftYDirectly( id.intValue(), value.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_LOCATION_RIGHT_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationRightXDirectly( id.intValue(), value.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_LOCATION_RIGHT_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationRightYDirectly( id.intValue(), value.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_FONT_NAME ) )
			{
				if( el.setFontNameDirectly( id.intValue(), str2 ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_FONT_STYLE ) )
			{
				final int value = this.getFontStyle(str2);
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setFontStyleDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_FONT_SIZE ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFontSizeDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFontColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_LINE_VISIBLE ) )
			{
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLineVisibleDirectly( id.intValue(), b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_SPACE ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setSpaceDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_LINE_WIDTH ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SIGDIFF_TEXT ) )
			{
				if( el.setTextDirectly( id.intValue(), str2 ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of Axis Break Symbol
		private int setAxisBreakProperties(
			final String command, final SGIFigureElementAxisBreak el, final ArrayList argsList )
		{

			final String[] keys = {
				COM_BREAK_AXIS_X,
				COM_BREAK_AXIS_Y,
				COM_BREAK_LOCATION_X,
				COM_BREAK_LOCATION_Y,
				COM_BREAK_LENGTH,
				COM_BREAK_INTERVAL,
				COM_BREAK_DISTORTION,
				COM_BREAK_ANGLE,
				COM_BREAK_LINE_WIDTH,
				COM_BREAK_ALIGNMENT,
				COM_BREAK_LINE_COLOR,
				COM_BREAK_INNER_COLOR,
				COM_DELETE_BREAK,
				COM_MOVE_TO_FRONT_BREAK,
				COM_MOVE_TO_BACK_BREAK
			};
			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}

			final int argn = argsList.size();
			if( argsList.size() < 1 )
			{
				return STATUS_FAILED;
			}

			String str1 = (String)argsList.get(0);
			final Number id = SGUtilityText.getInteger(str1);
			if( id==null )
			{
				return STATUS_FAILED;
			}

			if( argn == 1 )
			{
				if( command.equals( COM_DELETE_BREAK ) )
				{
					if( el.hideChildObject( id.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_FRONT_BREAK ) )
				{
					if( el.moveChildObject( id.intValue(), true ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_BACK_BREAK ) )
				{
					if( el.moveChildObject( id.intValue(), false ) == false )
					{
						return STATUS_FAILED;
					}
				}

				return STATUS_SUCCEEDED;
			}


			String str2 = (String)argsList.get(1);

			if( command.equals( COM_BREAK_AXIS_X ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisXDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_AXIS_Y ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisYDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_LOCATION_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationXDirectly( id.intValue(), value.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_LOCATION_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLocationYDirectly( id.intValue(), value.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_LENGTH ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLengthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_INTERVAL ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setIntervalDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_DISTORTION ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setDistortionDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_ANGLE ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				final float angle = value.floatValue()*SGIConstants.RADIAN_DEGREE_RATIO;
				if( el.setAngleDirectly( id.intValue(), angle ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_LINE_WIDTH ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_ALIGNMENT ) )
			{
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setAlignmentDirectly( id.intValue(), b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_LINE_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLineColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_BREAK_INNER_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setInnerColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of Timing Line
		private int setTimingLineProperties(
			final String command, final SGIFigureElementTimingLine el, final ArrayList argsList )
		{
			final String[] keys = {
				COM_TIMING_LINE_AXIS,
				COM_TIMING_LINE_WIDTH,
				COM_TIMING_LINE_TYPE,
				COM_TIMING_LINE_COLOR,
				COM_DELETE_TIMING_LINE,
				COM_MOVE_TO_FRONT_TIMING_LINE,
				COM_MOVE_TO_BACK_TIMING_LINE
			};
			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}
			
			final int argn = argsList.size();
			if( argsList.size() < 1 )
			{
				return STATUS_FAILED;
			}

			String str1 = (String)argsList.get(0);
			final Number id = SGUtilityText.getInteger(str1);
			if( id==null )
			{
				return STATUS_FAILED;
			}

			if( argn == 1 )
			{
				if( command.equals( COM_DELETE_TIMING_LINE ) )
				{
					if( el.hideChildObject( id.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_FRONT_TIMING_LINE ) )
				{
					if( el.moveChildObject( id.intValue(), true ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_BACK_TIMING_LINE ) )
				{
					if( el.moveChildObject( id.intValue(), false ) == false )
					{
						return STATUS_FAILED;
					}
				}

				return STATUS_SUCCEEDED;
			}


			String str2 = (String)argsList.get(1);

			if( command.equals( COM_TIMING_LINE_AXIS ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_TIMING_LINE_WIDTH ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_TIMING_LINE_TYPE ) )
			{
				final int num = this.getLineType(str2);
				if( num==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setLineTypeDirectly( id.intValue(), num ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_TIMING_LINE_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of Legend
		private int setLegendProperties(
			final String command, final SGIFigureElementLegend el, final ArrayList argsList )
		{
			final String[] keys = {
				COM_LEGEND_AXIS_X,
				COM_LEGEND_AXIS_Y,
				COM_LEGEND_LOCATION_X,
				COM_LEGEND_LOCATION_Y,
				COM_LEGEND_VISIBLE,
				COM_LEGEND_FONT_NAME,
				COM_LEGEND_FONT_STYLE,
				COM_LEGEND_FONT_SIZE,
				COM_LEGEND_FONT_COLOR,
				COM_LEGEND_FRAME_VISIBLE,
				COM_LEGEND_FRAME_LINE_WIDTH,
				COM_LEGEND_FRAME_COLOR,
				COM_LEGEND_BACKGROUND_TRANSPARENT,
				COM_LEGEND_BACKGROUND_COLOR,
				COM_LEGEND_SYMBOL_SPAN
			};
			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}
			
			final int argn = argsList.size();

			if( command.equals( COM_LEGEND_AXIS_X ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final int loc = this.getAxisLocation( str );
				if( loc==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisXDirectly(loc) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_AXIS_Y ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final int loc = this.getAxisLocation( str );
				if( loc==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisYDirectly(loc) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_LOCATION_X ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				Number num = SGUtilityText.getDouble(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setXValueDirectly( num.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_LOCATION_Y ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				Number num = SGUtilityText.getDouble(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setYValueDirectly( num.doubleValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_VISIBLE ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setLegendVisibleDirectly( b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_FONT_NAME ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				if( el.setFontNameDirectly(str) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_FONT_STYLE ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				int style = this.getFontStyle( str );
				if( style==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setFontStyleDirectly( style ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_FONT_SIZE ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFontSizeDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_FONT_COLOR ) )
			{
				if( argsList.size()==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor( str );
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFontColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argsList.size()==3 )
				{
					String str1 = (String)argsList.get(0);
					String str2 = (String)argsList.get(1);
					String str3 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str1,str2,str3);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFontColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_FRAME_VISIBLE ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				final Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFrameVisibleDirectly( b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_FRAME_LINE_WIDTH ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setFrameLineWidthDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_FRAME_COLOR ) )
			{
				if( argsList.size()==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor( str );
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFrameColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argsList.size()==3 )
				{
					String str1 = (String)argsList.get(0);
					String str2 = (String)argsList.get(1);
					String str3 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor( str1, str2, str3 );
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setFrameColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_BACKGROUND_TRANSPARENT ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str ) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setBackgroundTransparentDirectly( b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_BACKGROUND_COLOR ) )
			{
				if( argsList.size()==1 )
				{
					String str = (String)argsList.get(0);
					final Color cl = SGUtilityText.getColor( str );
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setBackgroundColorDirectly( cl ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( argsList.size()==3 )
				{
					String str1 = (String)argsList.get(0);
					String str2 = (String)argsList.get(1);
					String str3 = (String)argsList.get(2);
					final Color cl = SGUtilityText.getColor(str1,str2,str3);
					if( cl==null )
					{
						return STATUS_FAILED;
					}
					if( el.setBackgroundColorDirectly(cl) == false )
					{
						return STATUS_FAILED;
					}
				}
				else
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_LEGEND_SYMBOL_SPAN ) )
			{
				if( argsList.size()<1 )
				{
					return STATUS_FAILED;
				}
				String str = (String)argsList.get(0);
				StringBuffer unit = new StringBuffer();
				Number num = SGUtilityText.getNumber( str, unit );
				if( num==null )
				{
					return STATUS_FAILED;
				}
				if( el.setSymbolSpanDirectly( num.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

		// set the properties of Shape objects
		private int setShapeProperties(
			final String command, final SGIFigureElementShape el, final ArrayList argsList )
		{
			final String[] keys = {
				// shape properties
				COM_SHAPE_AXIS_X,
				COM_SHAPE_AXIS_Y,

				// rectangular shape
				COM_RECTANGLE_LEFT_X,
				COM_RECTANGLE_RIGHT_X,
				COM_RECTANGLE_TOP_Y,
				COM_RECTANGLE_BOTTOM_Y,
				COM_RECTANGLE_EDGE_LINE_WIDTH,
				COM_RECTANGLE_EDGE_LINE_TYPE,
				COM_RECTANGLE_EDGE_LINE_COLOR,
				COM_RECTANGLE_BACKGROUND_TRANSPARENT,
				COM_RECTANGLE_BACKGROUND_COLOR,

				// arrow
				COM_ARROW_START_X,
				COM_ARROW_START_Y,
				COM_ARROW_END_X,
				COM_ARROW_END_Y,
				COM_ARROW_LINE_WIDTH,
				COM_ARROW_LINE_TYPE,
				COM_ARROW_HEAD_SIZE,
				COM_ARROW_COLOR,
				COM_ARROW_START_TYPE,
				COM_ARROW_END_TYPE,
				COM_ARROW_OPEN_ANGLE,
				COM_ARROW_CLOSE_ANGLE,
				
				//
				COM_DELETE_SHAPE,
				COM_MOVE_TO_FRONT_SHAPE,
				COM_MOVE_TO_BACK_SHAPE
			};
			if( this.contains( keys, command ) == false )
			{
				return STATUS_NOT_FOUND;
			}

			final int argn = argsList.size();
			if( argsList.size() < 1 )
			{
				return STATUS_FAILED;
			}

			String str1 = (String)argsList.get(0);
			final Number id = SGUtilityText.getInteger(str1);
			if( id==null )
			{
				return STATUS_FAILED;
			}

			if( argn == 1 )
			{
				if( command.equals( COM_DELETE_SHAPE ) )
				{
					if( el.hideChildObject( id.intValue() ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_FRONT_SHAPE ) )
				{
					if( el.moveChildObject( id.intValue(), true ) == false )
					{
						return STATUS_FAILED;
					}
				}
				else if( command.equals( COM_MOVE_TO_BACK_SHAPE ) )
				{
					if( el.moveChildObject( id.intValue(), false ) == false )
					{
						return STATUS_FAILED;
					}
				}

				return STATUS_SUCCEEDED;
			}


			String str2 = (String)argsList.get(1);

			if( command.equals( COM_SHAPE_AXIS_X ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisXDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_SHAPE_AXIS_Y ) )
			{
				final int value = this.getAxisLocation( str2 );
				if( value==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setAxisYDirectly( id.intValue(), value ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_LEFT_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleLeftXValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_RIGHT_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleRightXValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_TOP_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleTopYValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_BOTTOM_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleBottomYValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_EDGE_LINE_WIDTH ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleEdgeLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_EDGE_LINE_TYPE ) )
			{
				final int num = this.getLineType(str2);
				if( num==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleEdgeLineTypeDirectly( id.intValue(), num ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_EDGE_LINE_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleEdgeLineColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_BACKGROUND_TRANSPARENT ) )
			{
				Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString( str2 ) );
				if( b==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleBackgroundTransparentDirectly( id.intValue(), b.booleanValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_RECTANGLE_BACKGROUND_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setRectangleBackgroundColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_START_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowStartXValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_START_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowStartYValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_END_X ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowEndXValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_END_Y ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowEndYValueDirectly( id.intValue(), value.floatValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_LINE_WIDTH ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowLineWidthDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_LINE_TYPE ) )
			{
				final int num = this.getLineType(str2);
				if( num==-1 )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowLineTypeDirectly( id.intValue(), num ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_HEAD_SIZE ) )
			{
				StringBuffer unit = new StringBuffer();
				Number value = SGUtilityText.getNumber( str2, unit );
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowHeadSizeDirectly( id.intValue(), value.floatValue(), unit.toString() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_COLOR ) )
			{
				Color cl = null;
				if( argn==2 )
				{
					cl = SGUtilityText.getColor(str2);
				}
				else if( argn==4 )
				{
					String str3 = (String)argsList.get(2);
					String str4 = (String)argsList.get(3);
					cl = SGUtilityText.getColor(str2,str3,str4);
				}
				else
				{
					return STATUS_FAILED;
				}

				if( cl==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowColorDirectly( id.intValue(), cl ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_START_TYPE ) )
			{
				final Number value = SGDrawingElementArrow.getSymbolTypeFromName(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowStartTypeDirectly( id.intValue(), value.intValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_END_TYPE ) )
			{
				final Number value = SGDrawingElementArrow.getSymbolTypeFromName(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				if( el.setArrowEndTypeDirectly( id.intValue(), value.intValue() ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_OPEN_ANGLE ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				final float angle = value.floatValue()*SGIConstants.RADIAN_DEGREE_RATIO;
				if( el.setArrowOpenAngleDirectly( id.intValue(), angle ) == false )
				{
					return STATUS_FAILED;
				}
			}
			else if( command.equals( COM_ARROW_CLOSE_ANGLE ) )
			{
				final Number value = SGUtilityText.getDouble(str2);
				if( value==null )
				{
					return STATUS_FAILED;
				}
				final float angle = value.floatValue()*SGIConstants.RADIAN_DEGREE_RATIO;
				if( el.setArrowCloseAngleDirectly( id.intValue(), angle ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return STATUS_SUCCEEDED;
		}

	}
	

	/**
	 * @author  okumura
	 */
	private class PropertyFileManager implements ActionListener {
		/**
		 * 
		 */
		private Document mPropertyFileDocument = null;

		/**
		 * Property file creator.
		 */
		private SGPropertyFileCreator mPropertyFileCreator = null;
		
		/**
		 * Property file chooser
		 */
		private SGPropertyFileChooserWizardDialog mPropertyFileChooserWizardDilaog = null;

		/**
		 * Multiple data file chooser
		 */
		private SGMultiDataFileChooserWizardDialog mMultiDataFileChooserWizardDialog = null;

		/**
		 * Multiple data file chooser for drag and drop
		 */
		private SGMultiDataFileChooserWizardDialog	mMultiDataFileChooserWizardDialogDD = null;


		private PropertyFileManager()
		{
		    this.mPropertyFileCreator = new SGPropertyFileCreator();
		    this.mPropertyFileChooserWizardDilaog = null;
			this.mMultiDataFileChooserWizardDialog = null;
		}

		/**
		 * save properties
		 * @param wnd - target window
		 * @param path - save file path
		 * @return status
		 */
		public int saveProperties( final SGDrawingWindow wnd, final String path )
		{
			// create a property file
			return this.mPropertyFileCreator.create( wnd, new File( path ) );
		}

		/**
		 * Show a dialog to save properties.
		 * @param wnd - target window
		 * @return status
		 */
		public int savePropertiesByDialog( final SGDrawingWindow wnd )
		{
			return this._savePropertiesByDialog(
					wnd, 
					getCurrentFileDirectory(), 
					SGMainFunctions.this.mCurrentPropertyFileName
					);
		}


		//
		private int _savePropertiesByDialog( final SGDrawingWindow wnd, final String dir, final String name )
		{
			// set the selected file name
			this.mPropertyFileCreator.setFile( dir, name );

			// create a property file
			final int ret = this.mPropertyFileCreator.create( wnd );
			if( ret!=OK_OPTION )
			{
				return ret;
			}

			File f = this.mPropertyFileCreator.getSelectedFile();
			if( f!=null )
			{
				updateCurrentFile( f, FILETYPE_PROPERTY );
			}

			return OK_OPTION;
		}

	    private boolean loadPropertyFromDialog( final SGDrawingWindow wnd )
	    {
	 
	        
	        // create wizard dialogs
	        this.createAllWizardDialogsToLoadPropertyFromToolBar( wnd );
	        
	        // set location
	        this.mPropertyFileChooserWizardDilaog.setCenter( wnd );
	        
	        // show the first wizard dialog
	        this.mPropertyFileChooserWizardDilaog.setVisible(true);
	        
	        // update the selected file name
	        File pf = this.mPropertyFileChooserWizardDilaog.getSelectedFile();
	        File df = this.mMultiDataFileChooserWizardDialog.getSelectedFile();
	        final long pUsed = this.mPropertyFileChooserWizardDilaog.lastUsed();
	        final long dUsed = this.mMultiDataFileChooserWizardDialog.lastUsed();
	        if( pf!=null & df!=null )
	        {
	            if( pUsed < dUsed )
	            {
	                updateCurrentFile( df, FILETYPE_DATA );
	            }
	            else
	            {
	                updateCurrentFile( pf, FILETYPE_PROPERTY );
	            }
	        }
	        else if( pf!=null )
	        {
	            updateCurrentFile( pf, FILETYPE_PROPERTY );
	        }
	        else if( df!=null )
	        {
	            updateCurrentFile( df, FILETYPE_DATA );
	        }
	        return true;
	    }
	    
		/**
		 * 
		 * @param e
		 * @return
		 */
		private boolean loadProperty( final ActionEvent e )
		{
			Object source = e.getSource();
			String command = e.getActionCommand();
			SGWizardDialog dg = (SGWizardDialog)source;

			if( source instanceof SGPropertyFileChooserWizardDialog )
			{
				SGPropertyFileChooserWizardDialog pdg
					= (SGPropertyFileChooserWizardDialog)dg;

				File f = pdg.getSelectedFile();
				if( f!=null )
				{
					updateCurrentFile( f, FILETYPE_PROPERTY );
					this.mMultiDataFileChooserWizardDialog.setFile(
						getCurrentFileDirectory(), SGMainFunctions.this.mCurrentDataFileName );
				}
				
				if( this.fromPropertyFileChooserDialog( e, dg ) == false )
				{
					return false;
				}
			}
			else if( source instanceof SGMultiDataFileChooserWizardDialog )
			{
				SGMultiDataFileChooserWizardDialog mdg
					= (SGMultiDataFileChooserWizardDialog)dg;

				File f = mdg.getSelectedFile();
				if( f!=null )
				{
					updateCurrentFile( f, FILETYPE_DATA );
					if( source.equals( this.mMultiDataFileChooserWizardDialog ) )
					{
						this.mPropertyFileChooserWizardDilaog.setFile(
							getCurrentFileDirectory(), SGMainFunctions.this.mCurrentPropertyFileName );
					}
				}

				if( this.fromMultiDataFileChooser(e) == false )
				{
					return false;
				}
			}

			// cancel or previous
			if( command.equals( SGDialog.CANCEL_BUTTON_TEXT ) )
			{
				dg.setVisible(false);
			}
			else if( command.equals( SGDialog.PREVIOUS_BUTTON_TEXT ) )
			{
				dg.showPrevious();
			}

			return true;
		}

		/**
		 * 
		 * @param idList
		 * @param cList
		 * @param mdg
		 * @return
		 */
		private boolean setMultiDataFileChooser(
			final ArrayList idList,
			final ArrayList nList,
			final ArrayList cList,
			final SGMultiDataFileChooserWizardDialog mdg )
		{
			final int dataNum = cList.size();

			// set the number of files
			mdg.setFileNumber( dataNum );

			final String NO = SGMultiDataFileChooserWizardDialog.NO_DATA;

			// figure id
			String[] idArray = new String[dataNum];
			for( int ii=0; ii<dataNum; ii++ )
			{
				idArray[ii] = ((Integer)idList.get(ii)).toString();
			}

			ArrayList noDataList = new ArrayList();
			
			// data name
			String[] nArray = new String[dataNum];
			for( int ii=0; ii<dataNum; ii++ )
			{
				Object obj = nList.get(ii);
				if( obj==null )
				{
					obj = NO;
					noDataList.add( new Integer(ii) );
				}
				nArray[ii] = (String)obj;
			}

			
			// data type
			String[] cArray = new String[dataNum];
			for( int ii=0; ii<dataNum; ii++ )
			{
				Object obj = cList.get(ii);
				if( noDataList.contains( new Integer(ii) ) )
				{
					obj = NO;
				}
				cArray[ii] = (String)obj;
			}


			// data name
			String[] fArray = new String[dataNum];
			for( int ii=0; ii<dataNum; ii++ )
			{
				if( noDataList.contains( new Integer(ii) ) )
				{
					fArray[ii] = NO;
				}
			}

			
			// set to the dialog
			mdg.setFigureIDArray( idArray );
			mdg.setDataNameArray( nArray );
			mdg.setDataTypeArray( cArray );
			mdg.setFileNameArray( fArray );
			
			return true;
		}

		
		/**
		 * 
		 * @return
		 */
		private boolean setPropertyFile(
			final SGDrawingWindow wnd,
			final Document doc,
			final Map dataMap )
		{
			// Element of the window
			final Element elWnd = getWindowElement( doc );
		
			// start progress
			wnd.startProgress();
			wnd.setProgressMessage( "Read Property" );
			
			// set the property to window from property file
			final boolean result = wnd.readProperty( elWnd, 0.0f, 1.0f );
	
			// end progress
			wnd.endProgress();
			
			int errcode;
			if( !result )
			{
				errcode = SGIConstants.PROPERTY_FILE_INCORRECT;
			}
			else
			{
				// create figure objects in a window
				errcode = createFiguresFromPropertyFile( elWnd, wnd, dataMap );
				
				if( errcode == SGIConstants.SUCCESSFUL_COMPLETION  )
				{
					// add history
					wnd.initPropertiesHistory();

					// initialize the history of save
					wnd.initSavedHistory();

					// set the saved flag
					wnd.setSaved(true);	
				}
			}
						
			// set the message
			String msg = null;
			switch( errcode )
			{
			case SGIConstants.SUCCESSFUL_COMPLETION:
				msg = MSG_SUCCESSFUL_COMPLETION;
				break;
			case SGIConstants.DATA_NUMBER_SHORTAGE:
				msg = MSG_DATA_NUMBER_SHORTAGE;
				break;
			case SGIConstants.DATA_NUMBER_EXCESS:
				msg = MSG_DATA_NUMBER_EXCESS;
				break;
			case  SGIConstants.FILE_OPEN_FAILURE:
				msg = MSG_FILE_OPEN_FAILURE;
				break;
			case SGIConstants.PROPERTY_FILE_INCORRECT:
				msg = MSG_PROPERTY_FILE_INVALID;
				break;
			case SGIConstants.DATA_FILE_INVALID:
				msg = MSG_DATA_FILE_INVALID;
				break;
			default:
				msg = MSG_UNKNOWN_ERROR_OCCURED;
			}
			
			// show the message dialog
			if( msg!=MSG_SUCCESSFUL_COMPLETION )
			{
				SGUtility.showMessageDialog( null, msg, "Property file", JOptionPane.ERROR_MESSAGE );
				return false;
			}
			return true;
		}
		
		/**
		 * 
		 * @return
		 */
		private boolean setPropertyFileFromEventDispatchThread(
			final SGDrawingWindow wnd,
			final Document doc,
			final Map dataMap )
		{
			// disable window
			wnd.setWaitCursor(true);

			Boolean result = Boolean.FALSE;
			
			try
			{
				result = (Boolean)Worker.post( new Task()
				{
					public Object run() throws Exception
					{
						if ( !setPropertyFile( wnd, doc, dataMap ) )
							return Boolean.FALSE;
						return Boolean.TRUE;
					}
				});
			
			} catch(Exception ex)
			{
				result = Boolean.FALSE;
			}

			// enable window again
			wnd.setWaitCursor(false);
			
			return result.booleanValue();
		}
		


		//
		private Element getWindowElement( Document doc )
		{
			Element root = doc.getDocumentElement();

			// get the node of window
			NodeList wList = root.getElementsByTagName( SGIRootObjectConstants.TAG_NAME_WINDOW );
			if( wList.getLength()==0 )
			{
				return null;
			}
			Element elWnd = (Element)wList.item(0);

			return elWnd;
		}

		

		
		/**
		 * 
		 * @param file
		 * @param com
		 * @param idList
		 * @param nList
		 * @param cList
		 * @return
		 */
		private boolean getInfoFromPropertyFile(
			final File file,
			final Component com,
			final ArrayList idList,
			final ArrayList nList,
			final ArrayList cList )
		{
			
			String path = SGUtility.getCanonicalPath( file.getPath() );
			if( path==null )
			{
				SGUtility.showMessageDialog( 
					com, MSG_FILE_OPEN_FAILURE, TITLE_FILE_OPEN_FAILURE, JOptionPane.ERROR_MESSAGE );
				return false;
			}

			// check validity of the file
			URL url = null;
			try
			{
				url = file.toURL();
			}
			catch( MalformedURLException ex )
			{
				return false;
			}

			
			// create a Document object
			Document doc = SGUtilityText.getDocument( url );
			if( doc==null )
			{
				SGUtility.showMessageDialog(
					com, MSG_PROPERTY_FILE_INVALID,
					TITLE_FILE_OPEN_FAILURE, JOptionPane.ERROR_MESSAGE );
				return false;
			}
			this.mPropertyFileDocument = doc;


			// figure ID
			NodeList figureNodeList
				= doc.getElementsByTagName( SGFigure.TAG_NAME_FIGURE );
			if( figureNodeList.getLength()==0 )
			{
				// if figure does not exist, return false
				SGUtility.showMessageDialog(
					com, MSG_PROPERTY_FILE_INVALID,
					TITLE_FILE_OPEN_FAILURE, JOptionPane.ERROR_MESSAGE );
				return false;
			}
			final int fnLength = figureNodeList.getLength();
			for( int ii=0; ii<fnLength; ii++ )
			{
				Node node = figureNodeList.item(ii);
				if( ( node instanceof Element ) == false )
				{
					continue;
				}

				Element figure = (Element)node;
				NodeList dataList = figure.getElementsByTagName(
					SGIFigureElementGraph.TAG_NAME_DATA );
				final int id = ii+1;
				final int dataNum = dataList.getLength();
				if( dataNum==0 )
				{
					idList.add( new Integer(id) );
				}
				else
				{
					for( int jj=0; jj<dataNum; jj++ )
					{
						idList.add( new Integer(id) );
					}
				}
			}

			
			// read the property file and get a list of candidates of data-type
			if( this.getDataInfoFromPropertyFile( doc, nList, cList ) == false )
			{
				SGUtility.showMessageDialog(
					com, MSG_PROPERTY_FILE_INVALID,
					TITLE_FILE_OPEN_FAILURE, JOptionPane.ERROR_MESSAGE );
				return false;
			}


			// if the candidate does not exist, return false
	/*		if( cList.size()==0 )
			{
				SGUtility.showMessageDialog(
					com, ERRMSG_PROPERTY_FILE_INCORRECT,
					TITLE_FILE_CHECK, JOptionPane.ERROR_MESSAGE );
				return false;
			}
	*/		if( cList.size() != idList.size() )
			{
				SGUtility.showMessageDialog(
					com, MSG_PROPERTY_FILE_INVALID,
					TITLE_FILE_OPEN_FAILURE, JOptionPane.ERROR_MESSAGE );
				return false;
			}
			
			
			
			return true;
		}
		
		
		/**
		 * determine the data-type from the property file
		 */
		private boolean getDataInfoFromPropertyFile(
			final Document doc, final ArrayList nList, final ArrayList cList )
		{

			// get root element - property
			Element root = doc.getDocumentElement();

			// get the node of window
			NodeList wList = root.getElementsByTagName( SGIRootObjectConstants.TAG_NAME_WINDOW );
			if( wList.getLength()==0 )
			{
				return false;
			}
			Element wnd = (Element)wList.item(0);

			// get the node list of figures
			NodeList fList = wnd.getElementsByTagName( SGFigure.TAG_NAME_FIGURE );
			for( int ii=0; ii<fList.getLength(); ii++ )
			{
				Node node = fList.item(ii);
				if( node instanceof Element )
				{
					Element figure = (Element)node;
					NodeList dList = figure.getElementsByTagName( SGIFigureElementGraph.TAG_NAME_DATA );
					final int dLength = dList.getLength();
					if( dLength==0 )
					{
						nList.add(null);
						cList.add(null);
					}
					else
					{
						for( int jj=0; jj<dLength; jj++ )
						{
							Node node_ = dList.item(jj);
							if( node_ instanceof Element )
							{
								Element data = (Element)node_;
								String name = data.getAttribute( SGIFigureElement.KEY_DATA_NAME );
								nList.add(name);
								String type = data.getAttribute( SGIFigureElement.KEY_DATA_TYPE );
								cList.add(type);
							}
						}
					}
				}
			}
			
			return true;
		}





		
		/**
		 * 
		 * @param e
		 * @param dg
		 * @return
		 */
		private boolean fromPropertyFileChooserDialog( ActionEvent e, SGWizardDialog dg )
		{
			String command = e.getActionCommand();
			Window owner = dg.getOwner();
			SGWizardDialog next = dg.getNext();
			if( ( dg instanceof SGPropertyFileChooserWizardDialog ) == false )
			{
				return false;
			}
			if( ( next instanceof SGMultiDataFileChooserWizardDialog ) == false )
			{
				return false;
			}
			SGPropertyFileChooserWizardDialog pdg = (SGPropertyFileChooserWizardDialog)dg;
			SGMultiDataFileChooserWizardDialog mdg = (SGMultiDataFileChooserWizardDialog)next;
			if( command.equals( SGDialog.NEXT_BUTTON_TEXT ) )
			{
				if( this.fromPropertyFileChooserDialogNext(pdg,mdg) == false )
				{
					return false;
				}
			}

			return true;
		}


		/**
		 * 
		 * @param e
		 * @return
		 */
		private boolean fromMultiDataFileChooser( final ActionEvent e )
		{
			Object source = e.getSource();
			SGWizardDialog dg = (SGWizardDialog)source;
			String command = e.getActionCommand();
			SGMultiDataFileChooserWizardDialog mdg = (SGMultiDataFileChooserWizardDialog)dg;
			Object owner = dg.getOwner();

			//
			if( command.equals( SGDialog.OK_BUTTON_TEXT ) )
			{
				SGDrawingWindow wnd = (SGDrawingWindow)owner;

				wnd.deleteImage();
				wnd.clearUndoBuffer();
				wnd.removeAllFigures();
				wnd.updateClientRect();

				// get the map of data file list
				TreeMap map = mdg.getSelectedFilePathListMap();

				// set properties
				if( !this.setPropertyFileFromEventDispatchThread( wnd, this.mPropertyFileDocument, map ) )
				{
					return false;
				}

				wnd.updateClientRect();
				
				// hide the wizard dialog
				dg.setVisible(false);

			}
			else if( command.equals( SGDialog.CANCEL_BUTTON_TEXT ) )
			{
				dg.setVisible(false);
			}

			return true;
		}

		/**
		 * 
		 * @param pdg
		 * @param mdg
		 * @return
		 */
		private boolean fromPropertyFileChooserDialogNext(
			final SGPropertyFileChooserWizardDialog pdg,
			final SGMultiDataFileChooserWizardDialog mdg )
		{

			// get file type from the selected file
			File file = pdg.getSelectedFileFromTextField();
			if( file.exists() == false )
			{
				SGUtility.showFileNotFoundMessageDialog( pdg );
				return false;
			}

			// set the selected file to the file chooser
			pdg.setSelectedFile( file );

			// get information from the property file
			ArrayList idList = new ArrayList();
			ArrayList nList = new ArrayList();
			ArrayList cList = new ArrayList();
			if( this.getInfoFromPropertyFile( file, pdg, idList, nList, cList ) == false )
			{
				return false;
			}

			// initialize the dialog
			mdg.init( idList.size() );		

			// set the id list and the class type list
			// to the multi data file chooser
			if( this.setMultiDataFileChooser( idList, nList, cList, mdg )==false )
			{
				return false;
			}
			
			// show the multi data chooser dialog
			pdg.showNext();

			return true;
		}
		

		/**
		 * 
		 */
		private boolean createAllWizardDialogsToLoadPropertyFromToolBar(
			final SGDrawingWindow owner )
		{
			
			//
			// create dialog objects
			//

			// dialog to load property file
			this.mPropertyFileChooserWizardDilaog
				= new SGPropertyFileChooserWizardDialog( owner, true );
			this.mPropertyFileChooserWizardDilaog.getPreviousButton().setVisible(false);
			this.mPropertyFileChooserWizardDilaog.getOKButton().setVisible(false);
			this.mPropertyFileChooserWizardDilaog.pack();


			// dialog to select data files
			this.mMultiDataFileChooserWizardDialog
				= new SGMultiDataFileChooserWizardDialog( owner, true );
			this.mMultiDataFileChooserWizardDialog.getNextButton().setVisible(false);
			this.mMultiDataFileChooserWizardDialog.pack();


			// create a file filter object
			SGExtensionFileFilter ff = new SGExtensionFileFilter();
			ff.setDescription( PROPERTY_FILE_DESCRIPTION );
			ff.addExtension( PROPERTY_FILE_EXTENSION );
			this.mPropertyFileChooserWizardDilaog.setFileFilter(ff);


			// get current path
			String path = getCurrentFileDirectory();

			// set the selected file name
			this.mPropertyFileChooserWizardDilaog.setFile( path, SGMainFunctions.this.mCurrentPropertyFileName );

			// set the selected file name
			this.mMultiDataFileChooserWizardDialog.setFile( path, SGMainFunctions.this.mCurrentDataFileName );

//	System.out.println(path);
//	System.out.println(this.mCurrentPropertyFileName);
//	System.out.println(this.mCurrentDataFileName);
//	System.out.println();


			//
			// next and previous dialog
			//

			this.mPropertyFileChooserWizardDilaog.setNext(
				this.mMultiDataFileChooserWizardDialog );
			this.mMultiDataFileChooserWizardDialog.setPrevious(
				this.mPropertyFileChooserWizardDilaog );				


			//
			// add action listener
			//

			this.mPropertyFileChooserWizardDilaog.addActionListener(this);
			this.mMultiDataFileChooserWizardDialog.addActionListener(this);


			return true;
		}

		//
		// Wizard
		//

		/**
		 * 
		 * @param propertyFile
		 * @param fileList
		 * @return
		 */
		private boolean showMultiDataFileChooserDialog(
			final File propertyFile,
			final ArrayList dataFileList,
			final Frame owner )
		{

			if( propertyFile.exists() == false )
			{
				SGUtility.showMessageDialog( 
					owner, MSG_FILE_OPEN_FAILURE, TITLE_FILE_OPEN_FAILURE, JOptionPane.ERROR_MESSAGE );
				return false;
			}

			
			// create data chooser dialog
			this.mMultiDataFileChooserWizardDialogDD
				= new SGMultiDataFileChooserWizardDialog(owner,true);

			SGMultiDataFileChooserWizardDialog dg
				= this.mMultiDataFileChooserWizardDialogDD;
			
			dg.getPreviousButton().setVisible(false);
			dg.getNextButton().setVisible(false);
			dg.pack();
			dg.addActionListener(this);


			// set directory
			this.mMultiDataFileChooserWizardDialogDD.setFile(
				getCurrentFileDirectory(), SGMainFunctions.this.mCurrentDataFileName );

			
			// get information from the property file
			ArrayList idList = new ArrayList();
			ArrayList nList = new ArrayList();
			ArrayList cList = new ArrayList();
			if( this.getInfoFromPropertyFile( propertyFile, owner, idList, nList, cList ) == false )
			{
				return false;
			}
			
			// initialize the dialog
			dg.init( idList.size() );		

			// set the id list and the class type list
			// to the multi data file chooser
			if( this.setMultiDataFileChooser( idList, nList, cList, dg ) == false )
			{
				return false;
			}
			
			// show dialog
			if( owner!=null )
			{
				dg.setCenter( owner );
			}
			else
			{
				dg.setLocationRelativeTo(null);
			}
			dg.setVisible(true);


			return true;
		}
		
		
		
		/**
		 * 
		 */
		public void actionPerformed( final ActionEvent e )
		{

			Object source = e.getSource();
			String command = e.getActionCommand();
			
			// Load Property
			if( source.equals( this.mPropertyFileChooserWizardDilaog )
				|| source.equals( this.mMultiDataFileChooserWizardDialog )
				|| source.equals( this.mMultiDataFileChooserWizardDialogDD ) )
			{
				this.loadProperty(e);
			}

		}

	}
	
	/**
	 * Execute a command.
	 * @param line - the command line
	 * @return
	 */
	public int exec( final String line )
	{
//System.out.println(line);

		int ret;
		SGDrawingWindow aWnd = this.mWindowManager.getActiveWindow();
		SGFigure aFig = this.mActiveFigure;

		ArrayList argsList = new ArrayList();
		final String command = this.parseCommand( line, argsList );
		if( command == null )
		{
			return STATUS_FAILED;
		}
		final int argn = argsList.size();


		// basic commands
		ret = this.execBasicCommand( command, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}


		// commands in the menu-bar
		ret = this.execMenuBarCommand( command, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}


		// set Window properties
		ret = this.mPropertyManager.setWindowProperties( command, aWnd, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}


		// set Figure properties
		ret = this.mPropertyManager.setFigureProperties( command, aFig, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}


		// set Figure Element properties
		ret = this.mPropertyManager.setFigureElementProperties( command, aFig, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}

//System.out.println(command);
//System.out.println(argsList);
//System.out.println();

		return STATUS_NOT_FOUND;
	}


	// execute the basic command
	private int execBasicCommand(
		final String command, final ArrayList argsList )
	{
		final int argn = argsList.size();
		final SGDrawingWindow aWnd = this.mWindowManager.getActiveWindow();

		int ret;

		if( command.equals( COM_CHWIN ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}

			// window id
			String str = (String)argsList.get(0);
			Number num = SGUtilityText.getInteger(str);
			if( num==null )
			{
				return STATUS_FAILED;
			}
			final int id = num.intValue();

			// set the active window
			SGDrawingWindow wnd = this.mWindowManager.getWindow(id);
			if( wnd==null )
			{
				return STATUS_FAILED;
			}
			this.mWindowManager.setActiveWindow( wnd );

			return STATUS_SUCCEEDED;
		}
		else if( command.equals( COM_CHFIG ) )
		{
			if( argn!=1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			
			// figure id
			String str = (String)argsList.get(0);
			Number num = SGUtilityText.getInteger(str);
			if( num==null )
			{
				return STATUS_FAILED;
			}
			final int id = num.intValue();

			SGFigure fig = aWnd.getFigure(id);
			if( fig==null )
			{
				return STATUS_FAILED;
			}
			this.mActiveFigure = fig;

			return STATUS_SUCCEEDED;
		}
		else if( command.equals( COM_SELECT_FIGURE ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}

			final int[] idArray = new int[argsList.size()];
			for( int ii=0; ii<argsList.size(); ii++ )
			{
				String str = (String)argsList.get(ii);
				Number num = SGUtilityText.getInteger(str);
				if( num==null )
				{
					return -1;
				}
				idArray[ii] = num.intValue();
			}

			aWnd.setSelectedFigure( idArray );

			return STATUS_SUCCEEDED;
		}
		// tree
		else if( command.equals( COM_TREE ) )
		{
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}

			StringBuffer sb = new StringBuffer();
			aWnd.createTree(sb);
			
			System.out.println(sb);
			
			return STATUS_SUCCEEDED;
		}
		// zoom
		else if( command.equals( COM_ZOOM ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}

			String str = (String)argsList.get(0);
			final Number num = SGUtilityText.getDouble(str);
			if( num==null )
			{
				return STATUS_FAILED;
			}
			
			if( aWnd.setZoomValue( num ) == false )
			{
				return STATUS_FAILED;
			}
			return STATUS_SUCCEEDED;
		}

		return STATUS_NOT_FOUND;
	}


	// commands in the menu-bar
	private int execMenuBarCommand(
		final String command, final ArrayList argsList )
	{
		int ret;

		// File
		ret = this.execMenuBarCommandFile( command, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}

		// Edit
		ret = this.execMenuBarCommandEdit( command, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}

		// Insert
		ret = this.execMenuBarCommandInsert( command, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}

		// Layout
		ret = this.execMenuBarCommandLayout( command, argsList );
		if( ret!= STATUS_NOT_FOUND )
		{
			return ret;
		}

		return STATUS_NOT_FOUND;
	}


	// commands in the menu-bar File
	private int execMenuBarCommandFile(
		final String command, final ArrayList argsList )
	{
		final int argn = argsList.size();
		final SGDrawingWindow aWnd = this.mWindowManager.getActiveWindow();

		// Create New Window
		if( command.equals( COM_WIN_OPEN ) )
		{
			// create a new window
			SGDrawingWindow wnd = this.mWindowManager.createNewWindow();
			if( wnd==null )
			{
				return STATUS_FAILED;
			}
			wnd.setVisible(true);

			return STATUS_SUCCEEDED;
		}
		// Close Window
		else if( command.equals( COM_WIN_CLOSE ) )
		{
			if( argn==0 )
			{
				if( aWnd!=null )
				{
					this.mWindowManager.closeWindow( aWnd );
				}
			}
			else if( argn==1 )
			{
				// window id
				String str = (String)argsList.get(0);
				Number num = SGUtilityText.getInteger(str);
				if( num==null )
				{
					return STATUS_FAILED;
				}
				final int id = num.intValue();

				// close
				this.mWindowManager.closeWindow(id);
			}
			else
			{
				return STATUS_FAILED;
			}

			return STATUS_SUCCEEDED;
		}
		// Draw Graph
		else if( command.equals( COM_DRAW_GRAPH ) )
		{
			if( argn<3 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			String str1 = (String)argsList.get(0);	// figure id
			String str2 = (String)argsList.get(1);	// file path
			String str3 = (String)argsList.get(2);	// data type

			// figure id
			Number num = SGUtilityText.getInteger(str1);
			if( num==null )
			{
				return STATUS_FAILED;
			}
			final int fid = num.intValue();

			// file path
			final String pathName = str2;
			File f = new File( pathName );
			if( f.exists() == false )
			{
				return STATUS_FAILED;
			}
			
			// data type
			final String dataType = str3;

			// create info list
			ArrayList infoList = new ArrayList();
			infoList.add( dataType );
			if( dataType.equals( SGDataTypeConstants.SXY_SAMPLING_DATA ) )
			{
				if( argsList.size() < 4 )
				{
					return STATUS_FAILED;
				}
				String str4 = (String)argsList.get(3);
				Number samplingRate = SGUtilityText.getDouble(str4);
				if( samplingRate==null )
				{
					return STATUS_FAILED;
				}
				infoList.add( samplingRate );
			}

			// draw the graph
			if( this.drawGraph( aWnd, fid, infoList, pathName, false ) == false )
			{
				return STATUS_FAILED;
			}

			return STATUS_SUCCEEDED;
		}
		// load property
		else if( command.equals( COM_LOAD_PROPERTY ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			String str1 = (String)argsList.get(0);	// file path

			File file = new File(str1);
			if( file.exists()==false )
			{
				return STATUS_FAILED;
			}
			String path = SGUtility.getCanonicalPath( file.getPath() );
			if( path==null )
			{
				return STATUS_FAILED;
			}

			// check validity of the file
			URL url = null;
			try
			{
				url = file.toURL();
			}
			catch( MalformedURLException ex )
			{
				return STATUS_FAILED;
			}

			// create a Document object
			Document doc = SGUtilityText.getDocument( url );
			if( doc==null )
			{
				return STATUS_FAILED;
			}

			//
			Element elWnd = this.mPropertyFileManager.getWindowElement( doc );
			NodeList nListFigure = elWnd.getElementsByTagName( SGFigure.TAG_NAME_FIGURE );
			final int figureNum = nListFigure.getLength();
			int cnt = 0;
			final int[] dataNumArray = new int[figureNum];
			for( int ii=0; ii<figureNum; ii++ )
			{
				Node node = nListFigure.item(ii);
				if( ( node instanceof Element ) == false )
				{
					return STATUS_FAILED;
				}
				Element elFigure = (Element)node;
				NodeList nListData = elFigure.getElementsByTagName( SGIFigureElementGraph.TAG_NAME_DATA );
				dataNumArray[ii] = nListData.getLength();
				cnt += dataNumArray[ii];
			}
			if( cnt!=argn-1 )
			{
				return STATUS_FAILED;
			}
			
			// list of path-names of data files
			ArrayList pathList = new ArrayList( argsList.subList( 1, argn ) );

			// create a data map
			Map dataMap = new TreeMap();
			cnt = 0;
			for( int ii=0; ii<figureNum; ii++ )
			{
				final int dataNum = dataNumArray[ii];
				ArrayList dataList = new ArrayList();
				for( int jj=0; jj<dataNum; jj++ )
				{
					dataList.add( pathList.get(cnt+jj) );
				}
				dataMap.put( new Integer(ii), dataList );
				cnt+=dataNum;
			}

			// set properties
			if( !this.mPropertyFileManager.setPropertyFileFromEventDispatchThread( aWnd, doc, dataMap ) )
			{
				return STATUS_FAILED;
			}

			return STATUS_SUCCEEDED;
		}
		// save property
		else if( command.equals( COM_SAVE_PROPERTY ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			if( aWnd.getVisibleFigureList().size()==0 )
			{
				return STATUS_FAILED;
			}
			String str1 = (String)argsList.get(0);	// path

			final int ret = this.mPropertyFileManager.saveProperties( aWnd, str1 );
			aWnd.setSaved( ret == OK_OPTION );

			return STATUS_SUCCEEDED;
		}
		// load data set
		else if( command.equals( COM_LOAD_DATA_SET ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			String str1 = (String)argsList.get(0);	// path
			File file = new File(str1);
			if( file.exists()==false )
			{
				return STATUS_FAILED;
			}

			if( !this.mDataSetManager.loadDataSetFromEventDispatchThread( aWnd, file ) )
			{
				return STATUS_FAILED;
			}

			return STATUS_SUCCEEDED;
		}
		// save data set
		else if( command.equals( COM_SAVE_DATA_SET ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			if( aWnd.getVisibleFigureList().size()==0 )
			{
				return STATUS_FAILED;
			}
			String str1 = (String)argsList.get(0);	// path
			final int ret = this.mDataSetManager.saveDataSet( aWnd, str1 );
			aWnd.setSaved( ret == OK_OPTION );

			return STATUS_SUCCEEDED;
		}
		// load background image
		else if( command.equals( COM_LOAD_IMAGE ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			String path = (String)argsList.get(0);	// path
			File fimg = new File( path );
			if( fimg.exists() == false )
			{
				return STATUS_FAILED;
			}
			if( aWnd.setImage( fimg ) == false )
			{
				return STATUS_FAILED;
			}

			return STATUS_SUCCEEDED;
		}
		// export as image
		else if( command.equals( COM_EXPORT_AS_IMAGE ) )
		{
			if( argn<2 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}

			String str0 = (String)argsList.get(0);	// type
			String str1 = (String)argsList.get(1);	// path

			Properties p = new Properties();
			for( int ii=2; ii<argn; ii++ )
			{
				String str = (String)argsList.get(ii);
				if( this.getProperties( str, p ) == false )
				{
					return STATUS_FAILED;
				}
			}

			return aWnd.exportAsImage(p, str0, str1);
			

		}
		// print
		else if( command.equals( COM_PRINT ) )
		{
			if( argn!=0 )
			{
				return STATUS_FAILED;
			}
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}

			// print in the silent mode
			if( aWnd.printImage( true ) == false )
			{
				return STATUS_FAILED;
			}
			
			return STATUS_SUCCEEDED;
		}
		// exit
		else if( command.equals( COM_EXIT ) )
		{
			if( argn!=0 )
			{
				return STATUS_FAILED;
			}
			this.exit();

			return STATUS_SUCCEEDED;
		}

		return STATUS_NOT_FOUND;
	}


	private boolean getProperties(
		final String str, final Properties p )
	{
		final int index = str.indexOf("=");
		if( index==-1 | index==0 | index==str.length()-1 )
		{
			return false;
		}

		final String back = str.substring( 0, index );
		final String front = str.substring( index+1, str.length() );
		p.setProperty( back, front );

		return true;
	}



	// commands in the menu-bar Edit
	private int execMenuBarCommandEdit(
		final String command, final ArrayList argsList )
	{
		final int argn = argsList.size();
		final SGDrawingWindow aWnd = this.mWindowManager.getActiveWindow();

		// Undo
		if( command.equals( COM_UNDO ) )
		{
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			aWnd.undo();
			aWnd.repaintContentPane();
			return STATUS_SUCCEEDED;
		}
		// Redo
		else if( command.equals( COM_REDO ) )
		{
			if( aWnd==null )
			{
				return STATUS_FAILED;
			}
			aWnd.redo();
			aWnd.repaintContentPane();
			return STATUS_SUCCEEDED;
		}
		// clear undo buffer
		else if( command.equals( COM_CLEAR_UNDO_BUFFER ) )
		{
			aWnd.clearUndoBuffer();
			return STATUS_SUCCEEDED;
		}
		// Cut
		else if( command.equals( COM_CUT ) )
		{
			aWnd.doCut();
			return STATUS_SUCCEEDED;
		}
		// Copy
		else if( command.equals( COM_COPY ) )
		{
			aWnd.doCopy();
			return STATUS_SUCCEEDED;
		}
		// Paste
		else if( command.equals( COM_PASTE ) )
		{
			aWnd.doPaste();
			return STATUS_SUCCEEDED;
		}
		// Delete
		else if( command.equals( COM_DELETE ) )
		{
			aWnd.doDelete();
			return STATUS_SUCCEEDED;
		}
		// Duplicate
		else if( command.equals( COM_DUPLICATE ) )
		{
			aWnd.doDuplicate();
			return STATUS_SUCCEEDED;
		}
		// Delete Background Image
		else if( command.equals( COM_DELETE_IMAGE) )
		{
			if( aWnd.deleteImage() == false )
			{
				return STATUS_FAILED;
			}
			return STATUS_SUCCEEDED;
		}

		return STATUS_NOT_FOUND;
	}


	/**
	 * 
	 * @param value
	 * @return
	 */
	private Number getNumber(
		final String value, final StringBuffer u )
	{
		return SGUtilityText.getNumber( value, u );
	}


	// commands in the menu-bar Insert
	private int execMenuBarCommandInsert(
		final String command, final ArrayList argsList )
	{
		final SGFigure aFig = this.mActiveFigure;
		if( aFig==null )
		{
			return STATUS_NOT_FOUND;
		}
		final int argn = argsList.size();

		if( command.equals( COM_INSERT_LABEL ) )
		{
			if( argn==1 )
			{
				String str1 = (String)argsList.get(0);
				if( str1.length()==0 )
				{
					return STATUS_FAILED;
				}
				aFig.addString(0,0,str1);
			}
			else if( argn==3 )
			{
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);
				String str3 = (String)argsList.get(2);

				StringBuffer unit1 = new StringBuffer();
				StringBuffer unit2 = new StringBuffer();

				Number nx = SGUtilityText.getNumber(str1,unit1);
				if( nx==null )
				{
					return STATUS_FAILED;
				}
				Number ny = SGUtilityText.getNumber(str2,unit2);
				if( ny==null )
				{
					return STATUS_FAILED;
				}

				final double nxPt = SGUtilityText.convertToPoint( nx.doubleValue(), unit1.toString() );
				final double nyPt = SGUtilityText.convertToPoint( ny.doubleValue(), unit2.toString() );

				final double x = aFig.getFigureX() + nxPt;
				final double y = aFig.getFigureY() + nyPt;
				aFig.addString( (int)x, (int)y, str3 );
			}
			else
			{
				return STATUS_FAILED;
			}

			return STATUS_SUCCEEDED;
		}
		else if(
			command.equals( COM_INSERT_AXIS_BREAK )
			| command.equals( COM_INSERT_SIGNIFICANT_DIFFERENCE )
			| command.equals( COM_INSERT_RECTANGLE )
			| command.equals( COM_INSERT_ELLIPSE )
			| command.equals( COM_INSERT_ARROW )
			| command.equals( COM_INSERT_LINE )
			| command.equals( COM_INSERT_TIMING_LINE ) )
		{
			if( argn==2 )
			{
				String str1 = (String)argsList.get(0);
				String str2 = (String)argsList.get(1);

				StringBuffer unit1 = new StringBuffer();
				StringBuffer unit2 = new StringBuffer();

				Number nx = SGUtilityText.getNumber(str1,unit1);
				if( nx==null )
				{
					return STATUS_FAILED;
				}
				Number ny = SGUtilityText.getNumber(str2,unit2);
				if( ny==null )
				{
					return STATUS_FAILED;
				}

				final double nxPt = SGUtilityText.convertToPoint( nx.doubleValue(), unit1.toString() );
				final double nyPt = SGUtilityText.convertToPoint( ny.doubleValue(), unit2.toString() );

				final float mag = aFig.getMagnification();
				final int x = (int)( aFig.getGraphRectX() + mag*nxPt );
				final int y = (int)( aFig.getGraphRectY() + mag*nyPt );

				if( command.equals( COM_INSERT_AXIS_BREAK ) )
				{
					aFig.addAxisBreakSymbol(x,y);
				}
				else if( command.equals( COM_INSERT_SIGNIFICANT_DIFFERENCE ) )
				{
					aFig.addSignificantDifferenceSymbol(x,y);
				}
				else if( command.equals( COM_INSERT_TIMING_LINE ) )
				{
					aFig.addTimingLine( x, y );
				}
				else if( command.equals( COM_INSERT_RECTANGLE ) )
				{
					aFig.addShape( SGIFigureElementShape.RECTANGLE, x, y );
				}
				else if( command.equals( COM_INSERT_ELLIPSE ) )
				{
					aFig.addShape( SGIFigureElementShape.ELLIPSE, x, y );
				}
				else if( command.equals( COM_INSERT_ARROW ) )
				{
					aFig.addShape( SGIFigureElementShape.ARROW, x, y );
				}
				else if( command.equals( COM_INSERT_LINE ) )
				{
					aFig.addShape( SGIFigureElementShape.LINE, x, y );
				}
				
				aFig.repaint();
			}
			else
			{
				return STATUS_FAILED;
			}

			return STATUS_SUCCEEDED;
		}

		return STATUS_NOT_FOUND;
	}


	// commands in the menu-bar Layout
	private int execMenuBarCommandLayout(
		final String command, final ArrayList argsList )
	{
		final int argn = argsList.size();
		final SGDrawingWindow aWnd = this.mWindowManager.getActiveWindow();
		if( aWnd==null )
		{
			return STATUS_FAILED;
		}

		if( command.equals( COM_PAPER_ALIGNMENT ) )
		{
			return STATUS_SUCCEEDED;
		}
		else if( command.equals( COM_PAPER_BOUNDING_BOX ) )
		{
			if( aWnd.setBoundingBox() == false )
			{
				return STATUS_FAILED;
			}
			return STATUS_SUCCEEDED;
		}
//		else if( command.equals( COM_TOOL_BAR_VISIBLE ) )
//		{
//			return SUCCEEDED;
//		}
//		else if( command.equals( COM_GRID_PLUS ) )
//		{
//			return SUCCEEDED;
//		}
//		else if( command.equals( COM_GRID_MINUS ) )
//		{
//			return SUCCEEDED;
//		}
//		else if( command.equals( COM_SNAP_TO_GRID ) )
//		{
//			return SUCCEEDED;
//		}
//		else if( command.equals( COM_ZOOM_IN ) )
//		{
//			return SUCCEEDED;
//		}
//		else if( command.equals( COM_ZOOM_OUT ) )
//		{
//			return SUCCEEDED;
//		}
//		else if( command.equals( COM_DEFAULT_ZOOM ) )
//		{
//			return SUCCEEDED;
//		}
//		else if( command.equals( COM_ZOOM_WAY_OUT ) )
//		{
//			return SUCCEEDED;
//		}
		else if( command.equals( COM_AUTO_ARRANGEMENT ) )
		{
			if( aWnd.alignFigures() == false )
			{
				return STATUS_FAILED;
			}
			return STATUS_SUCCEEDED;
		}
		else if( command.equals( COM_AUTO_ZOOM ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			String str = (String)argsList.get(0);
			Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str) );
			if( b==null )
			{
				return STATUS_FAILED;
			}
			aWnd.setAutoZoom( b.booleanValue() );
			return STATUS_SUCCEEDED;
		}
		else if( command.equals( COM_LOCK_FIGURE ) )
		{
			if( argn<1 )
			{
				return STATUS_FAILED;
			}
			String str = (String)argsList.get(0);
			Boolean b = SGUtilityText.getBoolean( SGUtilityText.getBooleanString(str) );
			if( b==null )
			{
				return STATUS_FAILED;
			}
			aWnd.setLocked( b.booleanValue() );
			return STATUS_SUCCEEDED;
		}

		return STATUS_NOT_FOUND;
	}




	// parse given string into the command and arguments
	private String parseCommand( final String line, final ArrayList argsList )
	{
		final int sIndex = line.indexOf('(');
		final int eIndex = line.indexOf(')');
		if( sIndex==-1 | eIndex==-1 )
		{
			return line;
		}
		if( sIndex > eIndex )
		{
			return line;
		}
		if( sIndex==0 )
		{
			return line;
		}

		String command = line.substring(0,sIndex);
		String args = line.substring(sIndex+1,eIndex);

		// args is not read from data file
		if( SGUtilityText.tokenize( args, argsList, false ) == false )
		{
			return line;
		}

		return command;
	}



	/**
	 * Get the simple path name from absolute path name.
	 * @param aPath - given absolute path name
	 * @return simple path name
	 */
	private String getSimpleFileName( final String aPath )
	{
		final String fs = FILE_SEPARATOR;
		StringTokenizer stk = new StringTokenizer( aPath, fs );
		String sub = null;
		while( stk.hasMoreTokens() )
		{
			sub = stk.nextToken();
		}
		if( sub==null )
		{
			throw new Error();
		}

		String name = null;
		StringTokenizer stk2 = new StringTokenizer( sub, Character.toString('.') );
		while( stk2.hasMoreTokens() )
		{
			name = stk2.nextToken();
			break;
		}
		if( name==null )
		{
			name = sub;
		}

		return name;
	}



	/**
	 * Returns a string for the name of new data.
	 * @param figure
	 * @param aPath
	 * @return
	 */
	private String getNewDataName(
		final SGFigure figure, final String aPath )
	{
		String name = this.getSimpleFileName( aPath );
		if( name==null )
		{
			return null;
		}

		List list = figure.getVisibleDataList();
		List dNameList = new ArrayList();
		for( int ii=0; ii<list.size(); ii++ )
		{
			SGData data = (SGData)list.get(ii);
			String dName = figure.getDataName(data);
			dNameList.add( dName );
		}
		
		name = SGUtilityText.getSerialName( dNameList, name );

		return name;
	}



	/**
	 * 
	 * @param wnd
	 * @return
	 */
	private boolean setWindowLocation( final SGDrawingWindow wnd )
	{
		// set the location
		final Dimension scSize = Toolkit.getDefaultToolkit().getScreenSize();
		final int x = (int)( scSize.getWidth() - wnd.getWidth() )/2;
		final int y = (int)( scSize.getHeight() - wnd.getHeight() )/2;
		wnd.setLocation(x,y);
	
		return true;
	}




	// create a data object from a file
	private int createDataFromFile(
		final String pathName, final ArrayList dataList, final SGDrawingWindow wnd )
			throws FileNotFoundException
	{

		// get information from the file
		ArrayList infoList = new ArrayList();
		int ret = this.readFile( pathName, infoList );
		if( ret!=0 )
		{
			return ret;
		}
		// create a data object
		SGData data = this.mDataCreator.create( pathName, infoList, wnd );
		if( data==null )
		{
			return -1;
		}
		dataList.add(data);

		return 0;
	}

	
	/**
	 * 
	 */
	private SGFigureIDSelectionWizardDialog mFigureIDSelectionWizardDialog = null;




	/**
	 * 
	 */
	private SGSingleDataFileChooserWizardDialog mSingleDataFileChooserWizardDilaog = null;



	/**
	 * 
	 */
	private SGXYDataTypeSelectionWizardDialog mXYDataTypeSelectionWizardDialog = null;



	/**
	 * 
	 */
	private SGXYDataTypeSelectionWizardDialog mXYDataTypeSelectionWizardDialogFromCLorDD = null;




	/**
	 * 
	 */
	private boolean createAllWizardDialogsForDataAdditionFromToolBar( final SGDrawingWindow owner )
	{

		//
		// create dialog objects
		//

		// dialog to select ID of window and figure
		this.mFigureIDSelectionWizardDialog
			= new SGFigureIDSelectionWizardDialog( owner, true );
		this.mFigureIDSelectionWizardDialog.getOKButton().setVisible(false);
		this.mFigureIDSelectionWizardDialog.getPreviousButton().setVisible(false);
		this.mFigureIDSelectionWizardDialog.pack();


		// dialog to select data files
		this.mSingleDataFileChooserWizardDilaog
			= new SGSingleDataFileChooserWizardDialog( owner, true );
		this.mSingleDataFileChooserWizardDilaog.getOKButton().setVisible(false);
		this.mSingleDataFileChooserWizardDilaog.pack();


		// set the selected file name
		String path = this.getCurrentFileDirectory();
		this.mSingleDataFileChooserWizardDilaog.setFile( path, this.mCurrentDataFileName );


		// dialog to select the data-type
		this.mXYDataTypeSelectionWizardDialog
			= new SGXYDataTypeSelectionWizardDialog( owner, true );
		this.mXYDataTypeSelectionWizardDialog.getNextButton().setVisible(false);
		this.mXYDataTypeSelectionWizardDialog.pack();


		//
		// set next and previous dialogs
		//
		
		this.mFigureIDSelectionWizardDialog.setNext( this.mSingleDataFileChooserWizardDilaog );
		this.mSingleDataFileChooserWizardDilaog.setPrevious( this.mFigureIDSelectionWizardDialog );
		this.mSingleDataFileChooserWizardDilaog.setNext( this.mXYDataTypeSelectionWizardDialog );
		this.mXYDataTypeSelectionWizardDialog.setPrevious( this.mSingleDataFileChooserWizardDilaog );


		//
		// add action listener
		//

		this.mFigureIDSelectionWizardDialog.addActionListener(this);
		this.mSingleDataFileChooserWizardDilaog.addActionListener(this);
		this.mXYDataTypeSelectionWizardDialog.addActionListener(this);

		return true;
	}



	/**
	 * Path name of the selected data file.
	 */
//	private String mCurrentFileDirectory = null;


	/**
	 * 
	 */
	private String mCurrentDataFileName = null;


	/**
	 * 
	 */
	private String mCurrentPropertyFileName = null;

	/**
	 * 
	 */
	private String mCurrentImageFileName = null;

	/**
	 * 
	 */
	private String mCurrentArchiveFileName = null;

	/**
	 * Returns the current file directory.
	 *
	 */
	private String getCurrentFileDirectory()
	{
		Preferences pref = Preferences.userNodeForPackage( this.getClass() );
		String currentDir = pref.get( PREF_KEY_CURRENT_DIRECTORY, null );

		boolean b = false;
		if( currentDir==null )
		{
			b = true;
		}
		else
		{
			File f = new File( currentDir );
			if( f.exists() == false )
			{
				b = true;
			}
		}

		if( b )
		{
			String path = USER_HOME;
			if( SGUtility.identifyOS( OS_NAME_WINDOWS ) )
			{
				path += FILE_SEPARATOR + MY_DOCUMENTS;
			}
			this.setCurrentFileDirectory(path);
			currentDir = path;
		}
		
		return currentDir;
	}



	/**
	 * 
	 * @param path
	 */
	private void setCurrentFileDirectory( final String path )
	{
		Preferences pref = Preferences.userNodeForPackage( this.getClass() );
		pref.put( PREF_KEY_CURRENT_DIRECTORY, path );
//		this.mCurrentFileDirectory = path;
//System.out.println(path);
	}





	/**
	 * 
	 */
	private int readFile(
		final String pathName,
		final ArrayList infoList )
	{

		final Window owner = this.mXYDataTypeSelectionWizardDialogFromCLorDD.getOwner();

		ArrayList cList = new ArrayList();
		try
		{
			if( this.mDataCreator.getFileTypeCandidateList( pathName, cList ) == false )
			{
				return -1;
			}
		}
		catch( FileNotFoundException ex )
		{
			return -1;
		}

		if( cList.size() == 0 )
		{
			return -1;
		}


		String fileType = null;

		// set candidate data-type to the dialog
		if( this.mXYDataTypeSelectionWizardDialogFromCLorDD.setEffectiveDataType(cList) == false )
		{
			return -1;
		}


		// set the location of wizard dialog
		this.mXYDataTypeSelectionWizardDialogFromCLorDD.setCenter( owner );


		// show a modal dialog to choose data-type from candidates
		this.mXYDataTypeSelectionWizardDialogFromCLorDD.setVisible(true);


		//
		// A modal dialog is shown.
		//

		// if canceled, return 1
		final int status = this.mXYDataTypeSelectionWizardDialogFromCLorDD.getCloseOption();
		if( status==SGIConstants.CANCEL_OPTION )
		{
			return 1;
		}

	
		// get selected file type
		fileType = this.mXYDataTypeSelectionWizardDialogFromCLorDD.getSelectedDataType();
		if( fileType == null )
		{
			return -1;
		}


		// add information to the list
		infoList.add(fileType);


		// create an array for data-retention
		if( fileType.equals(SGDataTypeConstants.SXY_SAMPLING_DATA) )
		{
			// add information of the sampling rate
			Double d = this.mXYDataTypeSelectionWizardDialogFromCLorDD.getSamplingRate();
			if( d==null )
			{
				return -1;
			}
			infoList.add(d);
		}
		else if( fileType.equals( SGDataTypeConstants.VXY_DATA ) )
		{
			boolean b = this.mXYDataTypeSelectionWizardDialogFromCLorDD.isPolarSelected();
			infoList.add( Boolean.valueOf(b) );
		}

		return 0;
	}


	/**
	 * Show the about dialog.
	 * @param wnd
	 */
	public boolean showAboutDialog( SGDrawingWindow wnd )
	{
	    if ( wnd == null )
	        wnd = this.mWindowManager.getActiveWindow();
	    if ( wnd == null )
	        return false;
	    
		SGAboutDialog dg = new SGAboutDialog(wnd,true, this.mAppProp.getVersionString());

		final int width = dg.getWidth();
		final int height = dg.getHeight();

		final int x = wnd.getX() + wnd.getWidth()/2 - width/2;
		final int y = wnd.getY() + wnd.getHeight()/2 - height/2;

		dg.setLocation(x,y);

		// show a modal dialog
		dg.setVisible(true);

		// dispose			
		dg.dispose();
		return true;
	}

	
	/**
	 * 
	 */
	public static final String TAG_NAME_FOCUSED_FIGURES = "FocusedFigures";

	
	/**
	 * 
	 * @return
	 */
	private boolean duplicateFocusedFigures( SGDrawingWindow wnd )
	{

		DOMImplementation domImpl = SGApplicationUtility.getDOMImplementation();
		if( domImpl==null )
		{
			return false;
		}
	
		// create a Document object
		Document document = domImpl.createDocument( "", TAG_NAME_FOCUSED_FIGURES, null );

		// create a DOM tree
		if( wnd.createDOMTree( document, SGIRootObjectConstants.FOCUSED_FIGURES_FOR_DUPLICATION ) == false )
		{
			return false;
		}

		// create a map of figure ID and data list
		TreeMap map = new TreeMap();

		ArrayList fList = wnd.getFocusedObjectsList();
		for( int ii=0; ii<fList.size(); ii++ )
		{
			SGFigure figure = (SGFigure)fList.get(ii);
			ArrayList dList = new ArrayList( figure.getVisibleDataList() );
			dList = (ArrayList)dList.clone();
			map.put( Integer.toString( figure.getID() ), dList );
		}


		// get root element - property
		Element root = document.getDocumentElement();

		// get the node of window
		NodeList wList = root.getElementsByTagName( SGIRootObjectConstants.TAG_NAME_WINDOW );
		if( wList.getLength()==0 )
		{
			return false;
		}
		Element elWnd = (Element)wList.item(0);

		int before = wnd.getFigureList().size();

		final int ret = this.createFiguresFromPropertyFile( elWnd, wnd, map );
		if( ret!=SGIConstants.SUCCESSFUL_COMPLETION )
		{
			return false;
		}

		int after = wnd.getFigureList().size();
		
		wnd.setChanged( before!=after );

		return true;
	}

	// call before discarded a window
	private int beforeDiscard( final SGDrawingWindow wnd )
	{
		final int ret = this.confirmBeforeDiscard( wnd );
		if( ret==JOptionPane.YES_OPTION )
		{

		}
		else if( ret==JOptionPane.NO_OPTION | ret==JOptionPane.CLOSED_OPTION )
		{
			// canceled and there is nothing to do
			return CANCEL_OPTION;
		}
		else if( ret==JOptionPane.CANCEL_OPTION )
		{
			// save the properties
			final int retSave = this.mPropertyFileManager.savePropertiesByDialog( wnd );
			return retSave;
		}

		return OK_OPTION;
	}


	// show the confirmation dialog for saving properties of the window
	private int confirmBeforeClosing( final SGDrawingWindow wnd )
	{
		final Object[] options = {
			MSG_CLOSE_WITHOUT_SAVING,
			SGDialog.CANCEL_BUTTON_TEXT,
			MSG_SAVE };
		return this._showConfirmationDialog( wnd, options, new SGCloseWindowConfirmPanel() );
	}


	// show the confirmation dialog for saving properties of the window
	private int confirmBeforeDiscard( final SGDrawingWindow wnd )
	{
		final Object[] options = {
//			MSG_DISCARD_WITHOUT_SAVING,
			MSG_CLOSE_WITHOUT_SAVING,
			SGDialog.CANCEL_BUTTON_TEXT,
			MSG_SAVE };
		return this._showConfirmationDialog( wnd, options, new SGCloseWindowConfirmPanel() );
	}


	//
	private int _showConfirmationDialog( Component component, Object[] options, Object message )
	{
		// beep
		Toolkit.getDefaultToolkit().beep();

		// show a dialog
		final int ret = JOptionPane.showOptionDialog(
			component,
			message,
			TITLE_CONFIRMATION,
			JOptionPane.YES_NO_CANCEL_OPTION,
			JOptionPane.WARNING_MESSAGE,
			null,
			options,
			options[2] );

		return ret;
	}

//	private class FigureCreator implements Runnable {
//	    private SGDrawingServer mServ;
//        private SGDrawingWindow mWnd;
//        private int mFigureID;
//        private SGData mData;
//        private Point mPos;
//        private String mName;
//        
//        // result
//        private SGFigure mFigure;
//        
//	    private boolean createNewFigure(
//	            final SGDrawingServer serv,
//	            final SGDrawingWindow wnd,
//	    		final int figureID,
//	    		final SGData data,
//	    		final Point pos,
//	    		final String name )
//	    {
//	        // initialize arguments
//	        mServ = serv;
//	        mWnd = wnd;
//	        mFigureID = figureID;
//	        mData = data;
//	        mPos = pos;
//	        mName = name;
//	        
//	        // initialize result
//	        mFigure = null;
//	        
//	        // start progress timer
//	        wnd.startProgress();
//
//	        // end progress timer
//	        wnd.endProgress();
//	        return true;
//	    }
//	    
//	    public void run()
//	    {
//			// create a figure instance
//			SGFigure figure = mServ.Main.createFigure( mFigureID, mWnd, mData );
//			if( mFigure==null )
//			{
//				return;
//			}
//
//			mWnd.setProgressValue(0.70f);
//
//			// add a new figure to the window
//			if( mWnd.addFigure( mFigure, pos ) == false )
//			{
//			    
//				return null;
//			}
//
//			wnd.setProgressValue(0.80f);
//
//			// add data object to the figure
//			if( !this.addData(data,name,figure) )
//			{
//			    wnd.endProgress();
//				return null;
//			}
//
//			wnd.setProgressValue(0.90f);
//			// init the history of the properties after the data is added
//			SGIFigureElement[] array = figure.getIFigureElementArray();
//			for( int ii=0; ii<array.length; ii++ )
//			{
//				if( array[ii].initPropertiesHistory() == false )
//				{
//				    wnd.endProgress();
//					return null;
//				}
//			}
//
//
//			wnd.setProgressValue(1.00f);
//			// draw back the figure into the paper
//			wnd.drawBackFigure( figure );
//
//
//			// initialize the history of the properties
//			// after drawn back
//			figure.initPropertiesHistory();
//
//
//			//
//			figure.setVisible(true);
//
//			wnd.setChanged(true);
//			wnd.notifyToRoot();
//
//
//			// show the window and figure
//			wnd.setVisible(true);
//
//
//			// update the items after the figure is set visible
//			wnd.updateItemsByFigureNumbers();
//
//			// terminate progress timer
//		    wnd.endProgress();
//
//			return figure;
//	    }
//	}
//
	
	private Document createChangeLogDocument()
	{
		Class inClass = getClass();
		URL url = inClass.getResource( RESOURCES_DIRNAME + "ChangeLog.xml" );
		if( url==null )
		{
			return null;
		}
		Document doc = SGUtilityText.getDocument(url.toString());
		return doc;
	}
	
	
	
	/**
	 * 
	 * @return
	 */
	private boolean showChangeLogDialog( Frame owner )
	{
		// create and set a dialog object
		SGChangeLogDialog dg = new SGChangeLogDialog(owner,true);
		dg.addActionListener( this );
		dg.setCenter(owner);

		// set a message
		String msg = "Change Log of Samurai Graph.";
		dg.setMessage(msg);

		// get the file of change log
		Class inClass = getClass();
		URL url = inClass.getResource( RESOURCES_DIRNAME + "ChangeLog.html" );
		if( url==null )
		{
			JOptionPane.showMessageDialog(
				owner,
				"Failed to get log information."
			);
			return false;
		}
		
		// set the html file
		dg.setPage( url );
		dg.pack();

		// show
		dg.setVisible(true);

		// dispose
		dg.dispose();

		return true;
	}



	/**
	 * 
	 */
	private boolean fromCLorDDWizardDialog( final ActionEvent e )
	{
		Object source = e.getSource();
		SGWizardDialog dg = (SGWizardDialog)source;
		String command = e.getActionCommand();
		return true;
	}



	/**
	 * 
	 * @param e
	 * @return
	 */
	private boolean addDataByToolBar( final ActionEvent e )
	{

		Object source = e.getSource();
		SGWizardDialog dg = (SGWizardDialog)source;
		SGWizardDialog prev = dg.getPrevious();
		SGWizardDialog next = dg.getNext();
		String command = e.getActionCommand();
		Window owner = dg.getOwner();

		
		// cancel or previous
		if( command.equals( SGDialog.CANCEL_BUTTON_TEXT ) )
		{
			dg.setVisible(false);
			return true;
		}
		else if( command.equals( SGDialog.PREVIOUS_BUTTON_TEXT ) )
		{
			dg.showPrevious();
			return true;
		}


		// figure ID
		if( source.equals( this.mFigureIDSelectionWizardDialog ) )
		{
			if( command.equals( SGDialog.NEXT_BUTTON_TEXT ) )
			{
				dg.showNext();
			}
		}
		// data file
		else if( source.equals( this.mSingleDataFileChooserWizardDilaog ) )
		{
			if( command.equals( SGDialog.NEXT_BUTTON_TEXT ) )
			{
				File file = this.mSingleDataFileChooserWizardDilaog.getSelectedFileFromTextField();
				if( file.exists() == false )
				{
					SGUtility.showFileNotFoundMessageDialog( this.mSingleDataFileChooserWizardDilaog );
					return false;
				}
				String pathName = file.getPath();

				ArrayList infoList = new ArrayList();
				ArrayList cList = new ArrayList();
				try
				{
					if( this.mDataCreator.getFileTypeCandidateList( pathName, cList ) == false )
					{
					    SGApplicationUtility.showDataFileInvalidMessageDialog(owner);
						return false;
					}
				}
				catch( FileNotFoundException ex )
				{
					SGUtility.showFileNotFoundMessageDialog(owner);
					return false;
				}
				
				if( cList.size() == 0 )
				{
				    SGApplicationUtility.showDataFileInvalidMessageDialog(owner);
					return false;
				}
				
				// set candidate data-type to the dialog
				if( this.mXYDataTypeSelectionWizardDialog.setEffectiveDataType(cList) == false )
				{
					SGApplicationUtility.showDataFileInvalidMessageDialog(owner);
					return false;
				}

				// set to the file chooser
				this.mSingleDataFileChooserWizardDilaog.setSelectedFile(file);

				dg.showNext();
			}
	
		}
		// data type
		else if( source.equals( this.mXYDataTypeSelectionWizardDialog ) )
		{

			if( command.equals( SGDialog.OK_BUTTON_TEXT ) )
			{
				// set invisible the dialog
				dg.setVisible(false);

				SGDrawingWindow wnd = dg.getOwnerWindow();
				
				// figure id
				final int figureID = this.mFigureIDSelectionWizardDialog.getFigureID();

				// path of the data file
				File f = this.mSingleDataFileChooserWizardDilaog.getSelectedFile();
				String pathName = f.getPath();

				// get selected file type
				String fileType = this.mXYDataTypeSelectionWizardDialog.getSelectedDataType();
				if( fileType == null )
				{
					SGUtility.showMessageDialog(
						owner,
						"Failed to get the data type.",
						TITLE_ERROR,
						JOptionPane.WARNING_MESSAGE
					);
					return false;
				}
				
				// add information to the list
				ArrayList infoList = new ArrayList();
				infoList.add(fileType);
				
				// create an array for data-retention
				if( fileType.equals( SGDataTypeConstants.SXY_SAMPLING_DATA ) )
				{
					// add the sampling rate to the infoList
					Double d = this.mXYDataTypeSelectionWizardDialog.getSamplingRate();
					if( d==null )
					{
						return false;
					}
					infoList.add(d);
				}
				else if( fileType.equals( SGDataTypeConstants.VXY_DATA ) )
				{
					boolean b = this.mXYDataTypeSelectionWizardDialog.isPolarSelected();
					infoList.add( Boolean.valueOf(b) );
				}

				// draw graph
				if( this.drawGraph( wnd, figureID, infoList, pathName, true ) == false )
				{
					return false;
				}

			}

		}


		return true;
	}

	
    private boolean drawGraphFromDialog( final SGDrawingWindow wnd )
    {
        // create wizard dialogs
        this.createAllWizardDialogsForDataAdditionFromToolBar( wnd );

        // set location
        this.mFigureIDSelectionWizardDialog.setCenter( wnd );

        // show the first wizard dialog
        this.mFigureIDSelectionWizardDialog.setVisible(true);

        // update the selected file name
        File f = this.mSingleDataFileChooserWizardDilaog.getSelectedFile();
        if( f ==null )
            return false;
        this.updateCurrentFile( f, FILETYPE_DATA );
        return true;
    }


	private boolean drawGraph(
		final SGDrawingWindow wnd,
		final int figureID,
		final ArrayList infoList,
		final String pathName,
		final boolean showDialog )
	{
		Boolean result = null;
		wnd.setWaitCursor(true);
		try
		{
			result = (Boolean)Worker.post( new Task() 
			{
				public Object run() throws Exception
				{
					if( wnd==null | infoList==null | pathName==null )
					{
						return Boolean.FALSE;
					}
					
					if( infoList.size()==0 )
					{
						return Boolean.FALSE;
					}
					
					final SGFigure fig = wnd.getFigure(figureID);
					
					// create a data object
					SGData data;
					try
					{
						data = SGMainFunctions.this.mDataCreator.create( pathName, infoList, wnd );
					}
					catch( FileNotFoundException ex )
					{
						if( showDialog )
						{
							SGUtility.showFileNotFoundMessageDialog( wnd );
						}
						return Boolean.FALSE;
					}
					if( data == null )
					{
						if( showDialog )
						{
							SGUtility.showMessageDialog(
									wnd,
									"Failed to create a data object.",
									TITLE_ERROR,
									JOptionPane.WARNING_MESSAGE
							);
						}
						return Boolean.FALSE;
					}
					
					// add data
					boolean createFigureFlag = (fig==null);
					if( fig!=null )
					{
						if( fig.isVisible() == false )
						{
							createFigureFlag = true;
						}
					}
					if( createFigureFlag )
					{
						String name = getSimpleFileName( pathName );
						if( name==null )
						{
							return Boolean.FALSE;
						}
						
						if( !SGMainFunctions.this.mFigureCreator.createNewFigure( wnd, figureID, data, new Point(), name ) )
						{
							return Boolean.FALSE;
						}
					}
					else
					{
						String name = getNewDataName( fig, pathName );
						if( name==null )
						{
							return Boolean.FALSE;
						}
						
						SGIProgressControl progress = (SGIProgressControl)wnd;
						progress.endProgress();
						progress.startProgress();
						progress.setProgressMessage( "Add Data" );
						if( !fig.addData(data,name, progress, 0.0f, 1.0f) )
						{
							progress.endProgress();
							if( showDialog )
							{
								SGApplicationUtility.showDataFileInvalidMessageDialog( wnd );
							}
						}
						progress.endProgress();
					}
					
					// repaint
					wnd.repaint();
					return Boolean.TRUE;
				}	
			});
		}catch(Exception ex)
		{
			result = Boolean.FALSE;
		}
		wnd.setWaitCursor(false);
		return result.booleanValue();
	}


	/**
	 * 
	 * @param f
	 * @param type
	 */
	private void updateCurrentFile( File f, int type )
	{
		String path = f.getPath();
		String parent = f.getParent();
		this.setCurrentFileDirectory(parent);
		String name = path.substring( parent.length() + 1 );
		if( type==FILETYPE_DATA )
		{
			this.mCurrentDataFileName = name;
		}
		else if( type==FILETYPE_PROPERTY )
		{
			this.mCurrentPropertyFileName = name;
		}
		else if( type==FILETYPE_IMAGE )
		{
			this.mCurrentImageFileName = name;
		}
		else if( type==FILETYPE_DATASET )
		{
			this.mCurrentArchiveFileName = name;
		}
	}


	
	/**
	 * 
	 */
	public void actionPerformed( final ActionEvent e )
	{

		Object source = e.getSource();
		String command = e.getActionCommand();

		//
		// from the command line or drag&drop
		//

		if( source.equals( this.mXYDataTypeSelectionWizardDialogFromCLorDD ) )
		{
			this.fromCLorDDWizardDialog(e);
		}
		

		//
		// from the tool bar
		//

		// New Graph
		if( source.equals( this.mFigureIDSelectionWizardDialog )
			|| source.equals( this.mSingleDataFileChooserWizardDilaog )
			|| source.equals( this.mXYDataTypeSelectionWizardDialog ) )
		{
			this.addDataByToolBar(e);
		}

	}


	/**
	 * Exit the application normally with confirmation.
	 */
	public void exit()
	{
		this.mWindowManager.closeAllWindow();
	}



	// Exit the application.
	private void exitApplication( final int status )
	{
		if( status!=0 )
		{
			String msg = "A fatal error has occured.\n"
				+ "The current application will be terminated.";
			String title = " Forced termination";
			SGUtility.showMessageDialog(
				null,
				msg,
				title,
				JOptionPane.ERROR_MESSAGE
			);
		}
		System.exit(status);
	}





	/**
	 * Open property file or dataset file.
	 * @param wnd
	 * @param fname
	 * @return
	 */
	public boolean openFile( SGDrawingWindow wnd, final String fname )
	{
	    if( wnd == null )
	        wnd = this.mWindowManager.getActiveWindow();
	    if( wnd == null )
	        return false;
	    
		// set unabled the window
		wnd.setWaitCursor(true);
		try
		{
			// analyze the file list
			ArrayList dataFileList = new ArrayList();
			File propertyFile = null;
			File archiveFile = null;
			File imageFile = null;
			File file = new File(fname);
			String path = file.getAbsolutePath();

			// property file?
			final boolean propertyFlag = path.endsWith( PROPERTY_FILE_EXTENSION );

			// archive file?
			final boolean archiveFlag = path.endsWith( ARCHIVE_FILE_EXTENSION );


			// as a property file
			if( propertyFlag )
			{
				propertyFile = file;
			}
			else if( archiveFlag )
			{
				archiveFile = file;
			}
			// as a data file
			else
			{
				return false;
			}
			// initialize progress value

			// use property file
			if( propertyFile!=null )
			{
				if( wnd.needsConfirmationBeforeDiscard() )
				{
					final int ret = this.beforeDiscard(wnd);
					if( ret==CANCEL_OPTION )
					{
						return true;
					}
				}

				if( this.mPropertyFileManager.showMultiDataFileChooserDialog(
					propertyFile, dataFileList, wnd ) == false )
				{
					return false;
				}
			}
			else if( archiveFile!=null )
			{
				if( wnd.needsConfirmationBeforeDiscard() )
				{
					final int ret = this.beforeDiscard(wnd);
					if( ret==CANCEL_OPTION )
					{
						return true;
					}
				}

				final boolean result = this.mDataSetManager.loadDataSetFromEventDispatchThread( wnd, archiveFile );
				wnd.setSaved( result );
			}
			else
			{
			    return false; // unreachable
			}
		}
		catch( Exception ex )
		{
			ex.printStackTrace();
		}
		finally
		{
			// set enabled the window
			wnd.setWaitCursor(false);
		}

		return true;
	}


	/**
	 * 
	 */
	private boolean fromDragAndDrop(
		final int x, final int y,
		final SGDrawingWindow wnd, final ArrayList fileList )
	{

		// set unabled the window
		wnd.setWaitCursor(true);

		try
		{
			// analyze the file list
			ArrayList dataFileList = new ArrayList();
			File propertyFile = null;
			File archiveFile = null;
			File imageFile = null;
			for( int ii=0; ii<fileList.size(); ii++ )
			{
				File file = (File)fileList.get(ii);
				String path = file.getAbsolutePath();

				// property file?
				final boolean propertyFlag = path.endsWith( PROPERTY_FILE_EXTENSION );

				// archive file?
				final boolean archiveFlag = path.endsWith( ARCHIVE_FILE_EXTENSION );

				// image file?
				boolean imageFlag = false;
				String[] imgArray = SGIImageConstants.DRAWABLE_IMAGE_EXTENSIONS;
				for( int jj=0; jj<imgArray.length; jj++ )
				{
					if( path.endsWith( imgArray[jj] ) | path.endsWith( imgArray[jj].toUpperCase() ) )
					{
						imageFlag = true;
						break;
					}
				}

				// as a property file
				if( propertyFlag )
				{
					propertyFile = file;
				}
				else if( archiveFlag )
				{
					archiveFile = file;
				}
				// as an image file
				else if( imageFlag )
				{
					imageFile = file;
				}
				// as a data file
				else
				{
					dataFileList.add(file);
				}
			}

			// use property file
			if( propertyFile!=null )
			{
				if( wnd.needsConfirmationBeforeDiscard() )
				{
					final int ret = this.beforeDiscard(wnd);
					if( ret==CANCEL_OPTION )
					{
						return true;
					}
				}

				if( this.mPropertyFileManager.showMultiDataFileChooserDialog(
					propertyFile, dataFileList, wnd ) == false )
				{
					return false;
				}
			}
			else if( archiveFile!=null )
			{
				if( wnd.needsConfirmationBeforeDiscard() )
				{
					final int ret = this.beforeDiscard(wnd);
					if( ret==CANCEL_OPTION )
					{
						return true;
					}
				}

				final boolean result = this.mDataSetManager.loadDataSet( wnd, archiveFile );
				wnd.setSaved( result );
			}
			else if( imageFile!=null )
			{
				if( wnd.setImage( imageFile ) == false )
				{
					return false;
				}
			}
			// add data
			else
			{
				if( this.addDataByDragAndDrop(
					x, y, wnd, dataFileList ) == false )
				{
					return false;
				}
			}
		}
		catch( Exception ex )
		{
			ex.printStackTrace();
		}
		finally
		{
			// set enabled the window
			wnd.setWaitCursor(false);
		}

		return true;
	}



	// Update the pattern of the tool bar in the preferences.
	private void updateToolBarPatternInPreferences( final String[] array )
	{
		Preferences pref = Preferences.userNodeForPackage( this.getClass() );

		// update the tool bar pattern
		String str = "";
		for( int ii=0; ii<array.length; ii++ )
		{
			str += array[ii];
			if( ii!=array.length-1 )
			{
				str += ",";
			}
		}
		pref.put( PREF_KEY_TOOL_BAR_PATTERN, str );
	}



	
	/**
	 * 
	 * @param elFigure
	 * @param figure
	 * @return
	 */
	private int createSingleFigureFromPropertyFile(
		final Element elFigure,
		final SGDrawingWindow wnd,
		final ArrayList dataList )
	{
		
		final int ic = SGIConstants.PROPERTY_FILE_INCORRECT;
		
		String str = null;
		Number num = null;
		Boolean b = null;
		Color cl = null;


		//
		// the name of instance
		//

		str = elFigure.getAttribute( SGFigure.KEY_FIGURE_TYPE );
		if( str.length()==0 )
		{
			return ic;
		}
//		String className;

		SGFigure figure = null;

		if( str.equals( SGIFigureTypeConstants.FIGURE_TYPE_SXY ) )
		{
			figure = new SGSXYFigure( wnd );
		}
		else if( str.equals( SGIFigureTypeConstants.FIGURE_TYPE_VXY ) )
		{
			figure = new SGVXYFigure( wnd );
		}
		else
		{
			return ic;
		}


		//
		// create a figure
		//

		final int figureID = wnd.getCurrentFigureId();
		figure.setID( figureID );


		//
		// create SGIFigureElement objects
		//

		if( figure.readProperty( elFigure ) == false )
		{
			return ic;
		}

		if( this.mFigureCreator.createFigureElementFromPropertyFile( figure, elFigure ) == ic )
		{
			return ic;
		}


		//
		// called after created SGIAxisElement object
		//

		// space axis line and numbers
		str = elFigure.getAttribute( SGFigure.KEY_SPACE_AXIS_LINE_AND_NUMBER );
		if( str.length()!=0 )
		{
			StringBuffer uSpaceLN = new StringBuffer();
			num = SGUtilityText.getNumber( str, uSpaceLN );
			if( num==null )
			{
				return ic;
			}
			final float spaceLN = num.floatValue();
			if( figure.setSpaceAxisLineAndNumber( spaceLN, uSpaceLN.toString() ) == false )
			{
				return ic;
			}
		}


		// space numbers and title
		str = elFigure.getAttribute( SGFigure.KEY_SPACE_NUMBER_AND_TITLE );
		if( str.length()!=0 )
		{
			StringBuffer uSpaceNT = new StringBuffer();
			num = SGUtilityText.getNumber( str, uSpaceNT );
			if( num==null )
			{
				return ic;
			}
			final float spaceNT = num.floatValue();
			if( figure.setSpaceNumberAndTitle( spaceNT, uSpaceNT.toString() ) == false )
			{
				return ic;
			}
		}


		// create data objects
		final int ret = this.createDataObjectsFromPropertyFile( elFigure, figure, dataList, wnd );
		if( ret!=SGIConstants.SUCCESSFUL_COMPLETION )
		{
			return ret;
		}


		// add the figure to the window
		wnd.addFigure( figure );


		//
		// initialize properties of the figure and figure elements
		//
		
		figure.initPropertiesHistory();


		return SGIConstants.SUCCESSFUL_COMPLETION;
	}

	// create the data object	
	private int createDataObjectsFromPropertyFile(
		final Element elFigure, final SGFigure figure, final ArrayList dataList, final SGIProgressControl progress  )
	{

		final int ic = SGIConstants.PROPERTY_FILE_INCORRECT;
		final int di = SGIConstants.DATA_FILE_INVALID;

		NodeList nList = elFigure.getElementsByTagName( SGIFigureElementGraph.TAG_NAME_DATA );
		if( nList.getLength()!=dataList.size() )
		{
			return ic;
		}

		final int len = nList.getLength();
		for( int ii=0; ii<len; ii++ )
		{

			//
			// get information from the Element object
			//

			Node node = nList.item(ii);
			if( ( node instanceof Element ) == false )
			{
				continue;
			}

			Element elData = (Element)node;

			ArrayList infoList = new ArrayList();

			// class
			String dataType = elData.getAttribute( SGIFigureElement.KEY_DATA_TYPE );
			if( dataType==null )
			{
				return ic;
			}
			infoList.add(dataType);

			
			String className = null;
			if( dataType.equals( SGDataTypeConstants.SXY_DATA ) )
			{
				className = SGSXYData.class.getName();
			}
			else if( dataType.equals( SGDataTypeConstants.VXY_DATA ) )
			{
				className = SGVXYData.class.getName();

				String str = elData.getAttribute( SGIFigureElementGraph.KEY_POLAR );
				if( str.length()==0 )
				{
					return ic;
				}
				Boolean b = SGUtilityText.getBoolean(str);
				if( b==null )
				{
					return ic;
				}
				infoList.add(b);
			}
			else if( dataType.equals( SGDataTypeConstants.SXY_MULTIPLE_DATA ) )
			{
				className = SGSXYMultipleData.class.getName();
			}
			else if( dataType.equals( SGDataTypeConstants.SXY_SAMPLING_DATA ) )
			{
				className = SGSXYSamplingData.class.getName();

				String str = elData.getAttribute( SGIFigureElementGraph.KEY_SAMPLING_RATE );
				if( str.length()==0 )
				{
					return ic;
				}
				Double d = SGUtilityText.getDouble(str);
				if( d==null )
				{
					return ic;
				}
				infoList.add(d);
			}
			else if( dataType.equals( SGDataTypeConstants.SXY_DATE_DATA ) )
			{
				className = SGSXYDateData.class.getName();
			}
			else
			{
				return ic;
			}


			// get the Class object
			Class cl = null;
			try
			{
				cl = Class.forName(className);
			}
			catch( ClassNotFoundException ex )
			{
				return ic;
			}
			if( cl==null )
			{
				return ic;
			}


			// create a new SGData object
			SGData data = null;
			try
			{
				data = (SGData)cl.newInstance();
			}
			catch( Exception ex )
			{
				return ic;
			}
			if( data==null )
			{
				return ic;
			}


			// get the data file
			if( dataList.size() == 0 )
			{
				return SGIConstants.DATA_NUMBER_SHORTAGE;
			}


			// get an object from dataList
			Object obj = dataList.get(ii);
			if( obj instanceof String )
			{
				String fileName = (String)obj;
			
				// check the type of data
				ArrayList cList = new ArrayList();
				Frame owner = figure.getWindow();

				try
				{
					if( this.mDataCreator.getFileTypeCandidateList( fileName, cList ) == false )
					{
						return di;
					}
					if( cList.size()==0 || cList.contains(dataType)==false )
					{
						return di;
					}
			
					// create data object
					SGData data_ = this.mDataCreator.create( fileName, infoList, progress );
					if( data_ == null )
					{
						return di;
					}
					if( data.setData(data_) == false )
					{
						return di;
					}
				}
				catch( FileNotFoundException ex )
				{
					return SGIConstants.FILE_OPEN_FAILURE;
				}

			}
			else if( obj instanceof SGData )
			{
				SGData data_ = (SGData)obj;
				if( data.setData( data_ ) == false )
				{
					return ic;
				}
			}
			else
			{
				return ic;
			}


			//
			if( figure.createDataObjectFromPropertyFile( elData, data ) == false )
			{
				return ic;
			}
			
		}
		
		
		SGIFigureElement[] array = figure.getIFigureElementArray();
		for( int ii=0; ii<array.length; ii++ )
		{
			array[ii].initPropertiesHistory();
		}

		return SGIConstants.SUCCESSFUL_COMPLETION;

	}


	/**
	 * 
	 * @param elWnd
	 * @param wnd
	 * @return
	 */
	private int createFiguresFromPropertyFile(
		final Element elWnd,
		final SGDrawingWindow wnd,
		final Map dataMap)
	{
		final SGIProgressControl progress = (SGIProgressControl)wnd;
		final int ic = SGIConstants.PROPERTY_FILE_INCORRECT;

		Set keySet = dataMap.keySet();
		ArrayList dataListList = new ArrayList( dataMap.values() );

		NodeList nList = elWnd.getElementsByTagName( SGFigure.TAG_NAME_FIGURE );
		final int len = nList.getLength();
		
		for( int ii=0; ii<len; ii++ )
		{
			Node node = nList.item(ii);
			if( node instanceof Element )
			{
				Element el = (Element)node;
				ArrayList dataList = (ArrayList)dataListList.get(ii);
				if( dataList.contains( SGMultiDataFileChooserWizardDialog.NO_DATA ) )
				{
					dataList = new ArrayList();
				}
				final int ret = this.createSingleFigureFromPropertyFile(
					el, wnd, dataList );
				if( ret!=SGIConstants.SUCCESSFUL_COMPLETION )
				{
					return ret;
				}
			}
		}
		
		return SGIConstants.SUCCESSFUL_COMPLETION;
	}

	
	

	/**
	 * 
	 */
	private boolean addDataByDragAndDrop(
		final int x, final int y,
		final SGDrawingWindow wnd, final ArrayList fileList )
	{

		// dialog to select data-type
		this.mXYDataTypeSelectionWizardDialogFromCLorDD
			= new SGXYDataTypeSelectionWizardDialog( wnd, true );
		this.mXYDataTypeSelectionWizardDialogFromCLorDD.addActionListener(this);
		this.mXYDataTypeSelectionWizardDialogFromCLorDD.mPreviousButton.setVisible(false);
		this.mXYDataTypeSelectionWizardDialogFromCLorDD.mNextButton.setVisible(false);
		this.mXYDataTypeSelectionWizardDialogFromCLorDD.pack();


		// create data objects
		final int ii = 0;
//		for( int ii=0; ii<fileList.size(); ii++ )
		{
			File file = (File)fileList.get(ii);

			ArrayList list = new ArrayList();
			int ret;
			try
			{
				ret = this.createDataFromFile( file.getPath(), list, wnd );
			}
			catch( FileNotFoundException ex )
			{
				SGUtility.showFileNotFoundMessageDialog( wnd );
				return false;
			}
			if( ret==-1 )
			{
				SGApplicationUtility.showDataFileInvalidMessageDialog(wnd);
				return false;
			}

			if( ret==1 )
			{
				return true;
			}

			if( list.size() == 0 )
			{
				return false;
			}
			SGData data = (SGData)list.get(0);

			String path = file.getAbsolutePath();

			Object com = wnd.getComponent(x,y);
			if( com instanceof SGDrawingWindow )
			{
				// get current figure ID
				int figureID = wnd.getCurrentFigureId();

				SGFigure figure = wnd.getFigure( figureID );
				if( figure==null )
				{
					String name = this.getSimpleFileName( path );
					if( name==null )
					{
						return false;
					}

					Point pos = new Point(x, y);
					if( !this.mFigureCreator.createNewFigure( wnd, figureID, data, pos, name ) )
					{
						return false;
					}
				}
				else
				{
					String name = this.getNewDataName( figure, path );
					if( name==null )
					{
						return false;
					}
					
					SGIProgressControl progress = (SGIProgressControl)wnd;
					progress.startProgress();
					progress.setProgressMessage( "Add Data" );
					// add data to the figure
					if( figure.addData( data, name, progress, 0.0f, 1.0f ) == false )
					{
						progress.endProgress();
					    SGApplicationUtility.showDataTypeInvalidMessageDialog( wnd );
						return false;
					}
					progress.endProgress();
				}

			}
			else if( com instanceof SGFigure )
			{
				SGFigure figure = (SGFigure)com;
				String name = this.getNewDataName( figure, path );
				if( name==null )
				{
					return false;
				}

				SGIProgressControl progress = (SGIProgressControl)wnd;
				progress.startProgress();
				progress.setProgressMessage( "Add Data" );
				// add data to the figure
				if( figure.addData( data, name, progress, 0.0f, 1.0f ) == false )
				{
					progress.endProgress();
				    SGApplicationUtility.showDataTypeInvalidMessageDialog( wnd );
					return false;
				}
				progress.endProgress();
			}
			else
			{
				return false;
			}

		}

		return true;
	}

    /**
     * A class for drag and drop.
     */
    private class DropEventHandler extends Thread
    {
    	/**
    	 * 
    	 */
    	private DropTargetDropEvent mDropTargetDropEvent = null;
    
    
    	/**
    	 * 
    	 */
    	private ArrayList mDroppedFileList = null;
    
    
    	/**
    	 * 
    	 * @param m
    	 */
    	private DropEventHandler( DropTargetDropEvent dtde, ArrayList list )
    	{
    		super();
    		this.mDropTargetDropEvent = dtde;
    		this.mDroppedFileList = list;
    		
    		this.start();
    	}
    
    
    	/**
    	 * 
    	 */
    	public void run()
    	{
    		DropTargetDropEvent dtde = this.mDropTargetDropEvent;
    		DropTarget tg = (DropTarget)dtde.getSource();
    		Component com = tg.getComponent();
    		SGDrawingWindow wnd = (SGDrawingWindow)com;
    		Point p = dtde.getLocation();
    
    		fromDragAndDrop( p.x, p.y, wnd, this.mDroppedFileList );
    
    		wnd.getContentPane().repaint();
    
    		this.mDropTargetDropEvent = null;
    		this.mDroppedFileList = null;
    	}
    
    }
    
 
}