#include "config.h"
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/msg.h>
#include <signal.h>
#include <limits.h>

#include "saphire.h"
#include "saphire_inner.h"

//////////////////////////////////////////////////////////
// 宣言
//////////////////////////////////////////////////////////
char gTmpDir[PATH_MAX];
    // 一時ファイルの置き場

string_obj* gErrMsg;
    // エラーメッセージ
vector_obj* gSaphireProgCompletions;
    // 補完で使うプログラムの名前 vector of string
vector_obj* gSaphireExternalProgCompletions;
    // 補完で使うプログラムの名前 vector of string
BOOL gKitutukiExit;
    // exitコマンドが実行されたらTRUE

enum eLineField gLineField = kLF;
    // saphireが文字列処理の内部コマンドで使う改行コード
enum eKanjiCode gKanjiCode = kUtf8;
    // saphireが文字列処理の内部コマンドや配列の添え字
    // で使う漢字コード


char* gStatmentKindStrs[kCommand+1] = {
    "object", "ref", "var", "global", "export", 
    "ary", "ary_add", "ary_erase", 
    "hash", "hash_add", "hash_erase",
    "def", "inherit", "self", "stackframe", "class",
    "sweep", "show",
    "subshell", "msleep", "true",  "false", "[",  
    "index", 
    "rindex", 
    "length",
    "lc",  "uc", "chomp", "eval", "fg", 
    "bg", "jobs", 
    "rehash", 
    "kanjicode", "linefield", "calc", 
     "print",  "puts",
    "load",
    "while", "break", "exit", "if", "split", 
    "x", "join", "select", "lines", "rows",
    "match", "scan", "sub",  "gsub",
    "read", "close",
    "cd", 
    "add", "del",  "selector", "max", "min", "extname", 
    "parentname", "noextname", 
    "compile", "raise",
    "popd", "pushd",
    "++",
    "--",
    "+",
    "-",
    "*",
    "/",
    "mod",
    "pow",
    "ssort",
    "range",
    "printf",
    "pomch",
    "ptee",
    "pcat",
    "return",
    "minitscr",
    "mendwin",
    "mclear",
    "mclear_immediately",
    "mrefresh",
    "mmove",
    "mmove_immediately",
    "mprintw",
    "yeild",
    "each",
    "foreach",
    "abs",
    "fabs",
    "p",
    "try",
    "envspace",
    "quote",
    "time",
//    "parse",
    "fselector",
    "cmdhistory",
    "unset",
    "umask",
    "errmsg",
    "runinfo",
    "inner_command",
    "command"
};

hash_obj* gFuncs;
    // ユーザー関数 sFunction*が入っている
    // キーは関数名

hash_obj* gClasses;
    // クラス sClass*が入っている
    // キーはクラス名

hash_obj* gGlobals;
    // グローバル変数 キーは変数名 内容はstring_obj*
hash_obj* gArrays;
    // グローバル変数の配列 キーは変数名
    // 内容は配列(vector_obj*) 配列の中にはstring_obj*
    // が入っている
hash_obj* gHashs;
    // グローバル変数の配列 キーは変数名
    // 内容はハッシュ(hash_obj*) ハッシュの中にはstring_obj*
    // が入っている

vector_obj* gStackFrame;
vector_obj* gStackFrameArrays;
vector_obj* gStackFrameHashs;
vector_obj* gStackFrameObjects;
vector_obj* gStackFrameRefs;
    // ローカル変数のスタック
    // 内容はhash_obj*
    // これのキーはローカル変数名
    // 内容はstring_obj*
enum eAppType gAppType;
    // アプリケーションの種類
BOOL gJobControl;
    // ジョブのコントロールをするかどうか
vector_obj* gJobs;
    // ジョブ sJob*の配列

char* gMagicNumber = "SAPHIRE_OBJ";  
    // オブジェクトファイルのマジックナンバー

hash_obj* gReadBuffers;
    // キー ファイル名
    // 内容 sRFd*
    // read用のファイルのリードバッファー

string_obj* gGlobalPipe;
    // グローバルパイプの読み込みんだデータ

string_obj* gGlobalPipeNum[10];
    // ナンバーグローバルパイプの読み込みんだデータ

hash_obj* gRegexs;
    // 正規表現のキャッシュ
    // キーは正規表現の文字列
    // 内容はregex_t
hash_obj* gRegexsI;
    // 正規表現のキャッシュ(ignore case)
    // キーは正規表現の文字列
    // 内容はregex_t
hash_obj* gRegexsM;
    // 正規表現のキャッシュ(multi line)
    // キーは正規表現の文字列
    // 内容はregex_t
hash_obj* gRegexsIM;
    // 正規表現のキャッシュ(ignore case|multi line)
    // キーは正規表現の文字列
    // 内容はregex_t

hash_obj* gInnerCommands;
    // 外部登録の内部コマンド
    // saphireを組み込んだアプリケーションで
    // 定義された内部コマンドのデータ
    // キーはコマンド名
    // 内容はsInnerCommand

vector_obj* gDirStack;
    // popd, pushdで使われるディレクトリスタック
    // 内容は string_obj*

vector_obj* gTmpFiles;
    // プロセス置換、ヘアドキュメントで使われる一時ファイルを行末で
    // 消すために保存しておく配列

vector_obj* gStackTraceFName;
vector_obj* gStackTraceLineNum;
    // スタックの記録

BOOL gSaphireBackground;
    // Saphireがバックグラウンドで動いているかどうか

hash_obj* gObjects;
    // オブジェクト キーはオブジェクト名 内容はsObject*

hash_obj* gRefs;
    // リファレンス

hash_obj* gGlobalPool;
hash_obj* gHashPool;
hash_obj* gAryPool;
hash_obj* gClassPool;
hash_obj* gObjectPool;
hash_obj* gFunPool;

//////////////////////////////////////////////////////////////////////////
// メモリチェッカー
/////////////////////////////////////////////////////////////////////////
static hash_obj* gMemChecker;

void memchecker_init()
{
    gMemChecker = HASH_NEW(8096);
}

void memchecker_final()
{
    hash_delete(gMemChecker);
}

// オブジェクトの種類を返す 0:無し 1:
int memchecker_is_enable_mem(void* obj)
{
    char buf[512];
    snprintf(buf, 512, "%d", (unsigned int)obj);
    return (int)hash_item(gMemChecker, buf);
}

void* memchecker_malloc(int size, enum eMemCheckerKind obj_kind)
{
    void* obj = MALLOC(size);

    char buf[512];
    snprintf(buf, 512, "%d", (unsigned int)obj);
    hash_put(gMemChecker, buf, (void*)obj_kind);

    return obj;
}

void memchecker_free(void* obj)
{
    char buf[512];
    snprintf(buf, 512, "%d", (unsigned int)obj);
    hash_erase(gMemChecker, buf);
}

sObject* sObject_new(char* name, char* klass_name, sObject* parent, int ref_count) 
{
    sObject* self = memchecker_malloc(sizeof(sObject), kMCObject);

    self->mName = STRING_NEW(name);
    self->mClassName = STRING_NEW(klass_name);
    self->mGlobals = HASH_NEW(15);
    self->mHashs = HASH_NEW(15);
    self->mArrays = HASH_NEW(15);
    self->mObjects = HASH_NEW(15);
    self->mRefs = HASH_NEW(15);
    self->mClasses = HASH_NEW(15);

    self->mParent = parent;

    self->mMethods = HASH_NEW(15);

    self->mRefCount = ref_count;
    if(ref_count == 0) {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self);
        hash_put(gObjectPool, buf, self);
    }

    return self;
}

void sObject_delete(sObject* self)
{
    string_delete(self->mName);
    string_delete(self->mClassName);

    hash_it* it = hash_loop_begin(self->mGlobals);
    while(it) {
        sGlobal_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(self->mGlobals);
    it = hash_loop_begin(self->mHashs);
    while(it) {
        hash_obj* h = hash_loop_item(it);

        hash_it* it2 = hash_loop_begin(h);
        while(it2) {
            string_delete(hash_loop_item(it2));
            it2 = hash_loop_next(it2);
        }

        sHash_delete(h);

        it = hash_loop_next(it);
    }
    hash_delete(self->mHashs);

    it = hash_loop_begin(self->mArrays);
    while(it) {
        vector_obj* v = hash_loop_item(it);
        int i;
        for(i=0; i<vector_size(v); i++) {
            string_delete(vector_item(v, i));
        }

        sAry_delete(v);
        it = hash_loop_next(it);
    }
    hash_delete(self->mArrays);
    it = hash_loop_begin(self->mObjects);
    while(it) {
        sObject_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(self->mObjects);
    it = hash_loop_begin(self->mRefs);
    while(it) {
        sRef_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(self->mRefs);
    it = hash_loop_begin(self->mClasses);
    while(it) {
        sClass_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(self->mClasses);
    it = hash_loop_begin(self->mMethods);
    while(it) {
        sFunction_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(self->mMethods);

    memchecker_free(self);
    FREE(self);
}

//////////////////////////////////////////////////////////
// 内部関数
//////////////////////////////////////////////////////////

// 引数がラインフィールドをポイントしているか2
BOOL is_line_field2(enum eLineField lf, char* p)
{
    if(lf == kLF && *p == '\n') {
        return TRUE;
    }
    else if(lf == kCR && *p == '\r') {
        return TRUE;
    }
    else if(lf == kCRLF && *p == '\r' && *(p+1) == '\n') {
        return TRUE;
    }
    else if(lf == kBel && *p == '\a') {
        return TRUE;
    }

    return FALSE;
}

// 改行コードを取り除く
BOOL string_chomp(string_obj* str)
{
    char* s = string_c_str(str);
    const int len = strlen(s);

    if(len >= 2 && s[len-2] == '\r' && s[len-1] == '\n')
    {
        string_trunc(str, len-2);

        return TRUE;
    }
    else if(len >= 1 && (s[len-1] == '\r' || s[len-1] == '\n' || s[len-1] == '\a')) 
    {
        string_trunc(str, len-1);
        return TRUE;
    }

    return FALSE;
}

// 改行コードを取り除く
BOOL string_chomp2(string_obj* str, enum eLineField lf)
{
    char* s = string_c_str(str);
    const int len = strlen(s);

    if(lf == kCRLF) {
        if(len >= 2 && s[len-2] == '\r' && s[len-1] == '\n')
        {
            string_trunc(str, len-2);

            return TRUE;
        }
    }
    else if(lf == kCR) {
        if(len >= 1) {
            if(s[len-1] == '\r') {
                string_trunc(str, len-1);
                return TRUE;
            }
        }
    }
    else if(lf == kLF) {
        if(len >= 1) {
            if(s[len-1] == '\n') {
                string_trunc(str, len-1);
                return TRUE;
            }
        }
    }
    else {
        if(len >= 1) {
            if(s[len-1] == '\a') {
                string_trunc(str, len-1);
                return TRUE;
            }
        }
    }

    return FALSE;
}

// 改行コードを付ける(改行コードそのまま)
void string_pomch(string_obj* str, enum eLineField lf)
{
    char* s = string_c_str(str);
    const int len = strlen(s);

    if(len >= 2 && s[len-2] == '\r' && s[len-1] == '\n') {
    }
    else if(len >= 1 && (s[len-1] == '\r' || s[len-1] == '\n' || s[len-1] == '\a')) {
    }
    else {
        if(lf == kCR) {
            string_push_back2(str, '\r');
        }
        else if(lf == kCRLF) {
            string_push_back(str, "\r\n");
        }
        else if(lf == kLF) {
            string_push_back2(str, '\n');
        }
        else {
            string_push_back2(str, '\a');
        }
    }
}

// エラーメッセージを設定
void err_msg(char* msg, char* sname, int line)
{
    char* tmp = MALLOC(strlen(sname) + 128 + strlen(msg));
    snprintf(tmp, strlen(sname) + 128 + strlen(msg), "%s %d: %s\n", sname,  line, msg);

    string_put(gErrMsg, tmp);
    FREE(tmp);
}

// コマンド名からコマンドIDを探す
int get_command_kind(char* arg0)
{
    int i;
    for(i=0; i<kInnerCommand; i++) {
        if(strcmp(arg0, gStatmentKindStrs[i]) == 0) {
            return i;
        }
    }

    return kCommand;
}

// gotoのラベルを通すための関数
void null_fun()
{
}

// 文字列をクォートして返す
void saphire_get_quoted_fname(char* fname, string_obj* quoted_fname)
{
    char* p = fname;
    while(*p) {
        if(*p==' ' || *p=='*' || *p=='>' || *p=='&' || *p=='~'
                || *p=='#' || *p =='$' || *p=='(' || *p==')'
                || *p=='\\' || *p=='|' || *p=='[' || *p==']' || *p=='{' 
                || *p=='}' || *p==';'
                || *p=='\'' || *p=='"' || *p=='<' || *p=='>' || *p=='?' 
                || *p=='%' || *p=='@' || *p=='\n' || *p=='\t' || *p=='\r' || *p=='\a')
        {
            string_push_back2(quoted_fname, '\\');
            string_push_back2(quoted_fname, *p++);
        }
        else {
            string_push_back2(quoted_fname, *p++);
        }
    }
}

// 文字列をクォートして返す
void bash_get_quoted_fname(char* fname, string_obj* quoted_fname)
{
    char* p = fname;
    while(*p) {
        if(*p==';' || *p=='&' || *p=='(' || *p==')' || *p=='|'
                || *p=='^' || *p =='<' || *p=='>' || *p=='?'
                || *p=='*' || *p=='[' || *p==']' || *p=='$' || *p=='`' 
                || *p=='"' || *p=='\''
                || *p=='{' || *p=='}' || *p=='\t' || *p==' ' || *p == '~'
                || *p=='#' || *p=='\\')
        {
            string_push_back2(quoted_fname, '\\');
            string_push_back2(quoted_fname, *p++);
        }
        else {
            string_push_back2(quoted_fname, *p++);
        }
    }
}
/////////////////////////////////////////////////////////
// ローカル変数、グローバル変数に参照、代入
/////////////////////////////////////////////////////////
void saphire_set_local_var(char* name, char* value)
{
    hash_obj* top_stack 
        = vector_item(gStackFrame, vector_size(gStackFrame)-1);

    string_obj* var = hash_item(top_stack, name);

    if(var) {
        string_put(var, value);
    }
    else {
        hash_put(top_stack , name, sVar_new(value));
    }
}

void saphire_init_stack_frame()
{
    int i;
    for(i=0; i<vector_size(gStackFrame); i++) {
        hash_obj* h = vector_item(gStackFrame, i);
        hash_it* it = hash_loop_begin(h);
        while(it != NULL) {
            sVar_delete(hash_loop_item(it));
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_clear(gStackFrame);
    vector_add(gStackFrame, HASH_NEW(30));
    for(i=0; i<vector_size(gStackFrameArrays); i++) {
        hash_obj* h = vector_item(gStackFrameArrays, i);
        hash_it* it = hash_loop_begin(h);
        while(it != NULL) {
            vector_obj* v = hash_loop_item(it);
            int i;
            for(i=0; i<vector_size(v); i++) {
                string_delete(vector_item(v, i));
            }
            sAry_delete(v);
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_clear(gStackFrameArrays);
    vector_add(gStackFrameArrays, HASH_NEW(30));
    for(i=0; i<vector_size(gStackFrameHashs); i++) {
        hash_obj* h = vector_item(gStackFrameHashs, i);
        hash_it* it = hash_loop_begin(h);
        while(it != NULL) {
            hash_obj* h2 = hash_loop_item(it);
            hash_it* it2 = hash_loop_begin(h2);
            while(it2) {
                string_delete(hash_loop_item(it2));
                it2 = hash_loop_next(it2);
            }
            sHash_delete(h2);
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_clear(gStackFrameHashs);
    vector_add(gStackFrameHashs, HASH_NEW(30));
    for(i=0; i<vector_size(gStackFrameObjects); i++) {
        hash_obj* h = vector_item(gStackFrameObjects, i);
        hash_it* it = hash_loop_begin(h);
        while(it != NULL) {
            sObject_delete(hash_loop_item(it));
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_clear(gStackFrameObjects);
    vector_add(gStackFrameObjects, HASH_NEW(30));
    for(i=0; i<vector_size(gStackFrameRefs); i++) {
        hash_obj* h = vector_item(gStackFrameRefs, i);
        hash_it* it = hash_loop_begin(h);
        while(it != NULL) {
            sRef_delete(hash_loop_item(it));
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_clear(gStackFrameRefs);
    vector_add(gStackFrameRefs, HASH_NEW(30));
}

void saphire_set_global_var(char* name, char* value)
{
    string_obj* global = hash_item(gGlobals, name);

    if(global) {
        string_put(global, value);
    }
    else {
        hash_put(gGlobals, name, sGlobal_new(value, 1));
    }
}

void saphire_set_global_var2(char* name, char* value, sObject* object)
{
    string_obj* global = hash_item(object->mGlobals, name);

    if(global) {
        string_put(global, value);
    }
    else {
        hash_put(object->mGlobals, name, sGlobal_new(value, 1));
    }
}


char* saphire_get_global_var(char* name)
{
    string_obj* global = hash_item(gGlobals, name);

    if(global) {
        return string_c_str(global);
    }
    else {
        return NULL;
    }
}

hash_obj* saphire_get_hash(char* name)
{
    return hash_item(gHashs, name);
}

vector_obj* saphire_get_array(char* name)
{
    return hash_item(gArrays, name);
}

void saphire_set_array(char* name, vector_obj* array)
{
    vector_obj* v = hash_item(gArrays, name);

    if(v) {
        hash_erase(gArrays, name);
    }

    hash_put(gArrays, name, array);
}

char* saphire_get_local_var(char* name)
{
    hash_obj* top_stack 
        = vector_item(gStackFrame, vector_size(gStackFrame)-1);

    string_obj* var = hash_item(top_stack, name);

    if(var) {
        return string_c_str(var);
    }
    else {
        return NULL;
    }
}

//////////////////////////////////////////////////////////
// シグナル処理
//////////////////////////////////////////////////////////
volatile BOOL gKitutukiSigInt = FALSE;
volatile BOOL gKitutukiSigTstp = FALSE;
volatile BOOL gKitutukiSigCont = FALSE;
volatile BOOL gSigUser = FALSE;

void sig_int()
{
    gKitutukiSigInt = TRUE; 

#ifdef MDEBUG
fprintf(stderr, "SIGINT!!\n");
#endif
}

void sig_int_optc()
{
    sigchld_block(1);
    gKitutukiSigInt = TRUE; 
fprintf(stderr, "SIGINT!!\n");
    sigchld_block(0);
}

void sig_tstp()
{
    gKitutukiSigInt = TRUE; 
#ifdef MDEBUG
fprintf(stderr, "SIGTSTP!!\n");
#endif
}

void sig_tstp_optc()
{
    sigchld_block(1);
    gKitutukiSigTstp = TRUE;
    kill(getpid(), SIGSTOP);
    sigchld_block(0);
}

void sig_cont_optc()
{
    sigchld_block(1);
    gKitutukiSigCont = TRUE;
//    killpg(getpgid(), SIGCONT);
    sigchld_block(0);
}

void sig_user()
{
    gSigUser = TRUE;
}

void sigchld_action(int signum, siginfo_t* info, void* ctx)
{
}

void sigchld_block(int block)
{
    sigset_t sigset;

    sigemptyset(&sigset);
    sigaddset(&sigset, SIGCHLD);

    if(sigprocmask(block?SIG_BLOCK:SIG_UNBLOCK, &sigset, NULL) != 0)
    {
        fprintf(stderr, "error\n");
        exit(1);
    }
}

void sigttou_block(int block)
{
    sigset_t sigset;

    sigemptyset(&sigset);
    sigaddset(&sigset, SIGTTOU);

    if(sigprocmask(block?SIG_BLOCK:SIG_UNBLOCK, &sigset, NULL) != 0)
    {
        fprintf(stderr, "error\n");
        exit(1);
    }
}

void saphire_restore_signal_default()
{
    struct sigaction sa;

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_DFL;
    if(sigaction(SIGPIPE, &sa, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }
    if(sigaction(SIGCHLD, &sa, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }
    if(sigaction(SIGINT, &sa, NULL) < 0) {
        perror("sigaction2");
        exit(1);
    }
    if(sigaction(SIGCONT, &sa, NULL) < 0) {
        perror("sigaction3");
        exit(1);
    }
    if(sigaction(SIGWINCH, &sa, NULL) < 0) {
        perror("sigaction4");
        exit(1);
    }

    if(sigaction(SIGPROF, &sa, NULL) < 0) {
        perror("sigaction5");
        exit(1);
    }

    if(sigaction(SIGTTIN, &sa, NULL) < 0) {
        perror("sigaction6");
        exit(1);
    }
    if(sigaction(SIGTTOU, &sa, NULL) < 0) {
        perror("sigaction7");
        exit(1);
    }
    if(sigaction(SIGTSTP, &sa, NULL) < 0) {
        perror("sigaction8");
        exit(1);
    }
    if(sigaction(SIGQUIT, &sa, NULL) < 0) {
        perror("sigaction9");
        exit(1);
    }
    if(sigaction(SIGUSR1, &sa, NULL) < 0) {
        perror("sigaction10");
        exit(1);
    }
}

void (*saphire_set_signal_other)() = NULL;

void saphire_set_signal()
{
    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    sa.sa_sigaction = sigchld_action;
    sa.sa_flags = SA_RESTART|SA_SIGINFO;
    if(sigaction(SIGCHLD, &sa, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_flags = SA_SIGINFO;
    sa.sa_handler = sig_int;
    if(sigaction(SIGINT, &sa, NULL) < 0) {
        perror("sigaction2");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    if(sigaction(SIGCONT, &sa, NULL) < 0) {
        perror("sigaction3");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    if(sigaction(SIGWINCH, &sa, NULL) < 0) {
        perror("sigaction4");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGTTOU, &sa, NULL) < 0) {
        perror("sigaction5");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGTTIN, &sa, NULL) < 0) {
        perror("sigaction6");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = sig_tstp;
    sa.sa_flags = 0;
    if(sigaction(SIGTSTP, &sa, NULL) < 0) {
        perror("sigaction7");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGQUIT, &sa, NULL) < 0) {
        perror("sigaction8");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = sig_user;
    sa.sa_flags = 0;
    if(sigaction(SIGUSR1, &sa, NULL) < 0) {
        perror("sigaction9");
        exit(1);
    }
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGPIPE, &sa, NULL) < 0) {
        perror("sigaction10");
        exit(1);
    }

/*
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_DFL;
    sa.sa_flags = SA_RESTART;
    if(sigaction(SIGSTOP, &sa, NULL) < 0) {
        perror("sigaction11");
        exit(1);
    }
*/

    if(saphire_set_signal_other) saphire_set_signal_other();
}

void saphire_set_signal_optc()
{
    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    sa.sa_sigaction = sigchld_action;
    sa.sa_flags = SA_SIGINFO|SA_RESTART;
    if(sigaction(SIGCHLD, &sa, NULL) < 0) {
        perror("sigaction1");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_flags = SA_SIGINFO;
    sa.sa_handler = sig_int_optc;
    //sa.sa_handler = SIG_IGN; //sig_int_optc;
    if(sigaction(SIGINT, &sa, NULL) < 0) {
        perror("sigaction2");
        exit(1);
    }
    sa.sa_flags = SA_SIGINFO;

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = sig_cont_optc;
    if(sigaction(SIGCONT, &sa, NULL) < 0) {
        perror("sigaction3");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    if(sigaction(SIGWINCH, &sa, NULL) < 0) {
        perror("sigaction4");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGTTOU, &sa, NULL) < 0) {
        perror("sigaction5");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGTTIN, &sa, NULL) < 0) {
        perror("sigaction6");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = sig_tstp_optc;
    sa.sa_flags = 0;
    if(sigaction(SIGTSTP, &sa, NULL) < 0) {
        perror("sigaction7");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGQUIT, &sa, NULL) < 0) {
        perror("sigaction8");
        exit(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = sig_user;
    sa.sa_flags = 0;
    if(sigaction(SIGUSR1, &sa, NULL) < 0) {
        perror("sigaction9");
        exit(1);
    }
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    if(sigaction(SIGPIPE, &sa, NULL) < 0) {
        perror("sigaction10");
        exit(1);
    }

/*
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_DFL;
    sa.sa_flags = SA_RESTART;
    if(sigaction(SIGSTOP, &sa, NULL) < 0) {
        perror("sigaction11");
        exit(1);
    }
*/

    if(saphire_set_signal_other) saphire_set_signal_other();
}

//////////////////////////////////////////////////////////
// ジョブ
//////////////////////////////////////////////////////////
sJob* sJob_new() 
{
    sJob* self = (sJob*)MALLOC(sizeof(sJob));
    
    self->mName = STRING_NEW("");
    self->mPGroup = 0;
    self->mPIDs = VECTOR_NEW(30);
    self->mTty = NULL;

    return self;
}

void sJob_delete(sJob* self)
{
    string_delete(self->mName);
    vector_delete(self->mPIDs);
    if(self->mTty) FREE(self->mTty);
    FREE(self);
}

/////////////////////////////////////////////////////////
// リダイレクト
/////////////////////////////////////////////////////////
sRedirect* sRedirect_new(enum eRedirect type, int fd)
{
    sRedirect* self = (sRedirect*)MALLOC(sizeof(sRedirect));
    self->mType = type;
    self->mFd = fd;
    self->mFName = STRING_NEW("");
    return self;
}

sRedirect* sRedirect_new2(sRedirect* redirect)
{
    sRedirect* self = (sRedirect*)MALLOC(sizeof(sRedirect));
    self->mType = redirect->mType;
    self->mFd = redirect->mFd;
    self->mFName = STRING_NEW(string_c_str(redirect->mFName));
    return self;
}

void sRedirect_delete(sRedirect* self)
{
    string_delete(self->mFName);

    FREE(self);
}

void sRedirect_save(sRedirect* self, int fd)
{
    unsigned char uc;
    int n;
    
    uc = self->mType;
    if(write(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("write1");
        exit(1);
    }
    
    n = self->mFd;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write2");
        exit(1);
    }
}

void sRedirect_view(sRedirect* self)
{
    printf("fd %d\n", self->mFd);
}

sRedirect* sRedirect_load(int fd)
{
    unsigned char uc;
    int n;
    
    sRedirect* self = (sRedirect*)MALLOC(sizeof(sRedirect));
    
    if(read(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("read");
        exit(1);
    }
    self->mType = uc;
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mFd = n;

    self->mFName = STRING_NEW("");
    
    return self;
}

/////////////////////////////////////////////////////////
// 引数
/////////////////////////////////////////////////////////
#ifdef MDEBUG

sArg* sArg_new_debug(int kind, MANAGED void* body, const char* fname, int line, const char* func)
{
    sArg* self = CheckMemLeak_Malloc(sizeof(sArg), fname, line, func);

    self->mKind = kind;
    self->mBody = body;

    return self;
}


sArg* sArg_new2_debug(sArg* arg, const char* fname, int line, const char* func)
{
    sArg* self = CheckMemLeak_Malloc(sizeof(sArg), fname, line, func);

    self->mKind = arg->mKind;
    switch(self->mKind) {
    case 0:
        self->mBody = STRING_NEW(string_c_str(arg->mBody));
        break;

    case 2:
        self->mBody = STATMENTS_NEW2(arg->mBody);
        break;

    case 4:
        self->mBody = sAtCommand_new2(arg->mBody);
        break;

    case 5:
        self->mBody = sBlock_new2(arg->mBody);
        break;

    case 6:
        self->mBody = STATMENTS_NEW2(arg->mBody);
        break;

    case 7:
        self->mBody = STRING_NEW(string_c_str(arg->mBody));
        break;

    case 8:
        self->mBody = STATMENTS_NEW2(arg->mBody);
        break;

    case 9:
        self->mBody = sRedirect_new2(arg->mBody);
    }

    return self;
}

#else

sArg* sArg_new(int kind, MANAGED void* body)
{
    sArg* self = MALLOC(sizeof(sArg));

    self->mKind = kind;
    self->mBody = body;

    return self;
}


sArg* sArg_new2(sArg* arg)
{
    sArg* self = MALLOC(sizeof(sArg));

    self->mKind = arg->mKind;
    switch(self->mKind) {
    case 0:
        self->mBody = STRING_NEW(string_c_str(arg->mBody));
        break;

    case 2:
        self->mBody = STATMENTS_NEW2(arg->mBody);
        break;

    case 4:
        self->mBody = sAtCommand_new2(arg->mBody);
        break;

    case 5:
        self->mBody = sBlock_new2(arg->mBody);
        break;

    case 6:
        self->mBody = STATMENTS_NEW2(arg->mBody);
        break;

    case 7:
        self->mBody = STRING_NEW(string_c_str(arg->mBody));
        break;

    case 8:
        self->mBody = STATMENTS_NEW2(arg->mBody);
        break;

    case 9:
        self->mBody = sRedirect_new2(arg->mBody);
    }

    return self;
}

#endif

void sArg_delete(sArg* self)
{
    switch(self->mKind) {
    case 0:
        string_delete(self->mBody);
        break;

    case 2:
        sStatments_delete(self->mBody);
        break;

    case 4:
        sAtCommand_delete(self->mBody);
        break;

    case 5:
        sBlock_delete(self->mBody);
        break;

    case 6:
        sStatments_delete(self->mBody);
        break;

    case 7:
        string_delete(self->mBody);
        break;

    case 8:
        sStatments_delete(self->mBody);
        break;

    case 9:
        sRedirect_delete(self->mBody);
    }

    FREE(self);
}

void sArg_save(sArg* self, int fd)
{
    int n = self->mKind;
    if(write(fd, &n,sizeof(int)) < 0) {
        perror("write3");
        exit(1);
    }

    switch(self->mKind) {
        case 0: {
            string_obj* str = self->mBody;
            int n = string_size(str);

            if(write(fd, &n, sizeof(int)) < 0) {
                perror("write4");
                exit(1);
            }

            if(write(fd, string_c_str(str), n) < 0) {
                perror("write5");
                exit(1);
            }
            }
            break;

        case 2:
            sStatments_save(self->mBody, fd);
            break;

        case 4:
            sAtCommand_save(self->mBody, fd);
            break;

        case 5:
            sBlock_save(self->mBody, fd);
            break;

        case 6:
            sStatments_save(self->mBody, fd);
            break;

        case 7: {
            string_obj* str = self->mBody;
            int n = string_size(str);

            if(write(fd, &n, sizeof(int)) < 0) {
                perror("write6");
                exit(1);
            }

            if(write(fd, string_c_str(str), n) < 0) {
                perror("write7");
                exit(1);
            }
            }
            break;

        case 8:
            sStatments_save(self->mBody, fd);
            break;

        case 9:
            sRedirect_save(self->mBody, fd);
            break;
    }
}

sArg* sArg_load(int fd)
{
    sArg* self = MALLOC(sizeof(sArg));

    int n;
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mKind = n;

    switch(self->mKind) {
    case 0: {
        int m;
        if(read(fd, &m, sizeof(int)) < 0) {
            perror("read");
            exit(1);
        }
        char* buf = MALLOC(sizeof(char)*(m+1));
        if(read(fd, buf, m) < 0) {
            perror("read");
            exit(1);
        }
        buf[m] = 0;

        self->mBody = STRING_NEW(buf);
        FREE(buf);
        }
        break;

    case 2:
        self->mBody = sStatments_load(fd);
        break;

    case 4:
        self->mBody = sAtCommand_load(fd);
        break;

    case 5:
        self->mBody = sBlock_load(fd);
        break;

    case 6:
        self->mBody = sStatments_load(fd);
        break;

    case 7: {
        int m;
        if(read(fd, &m, sizeof(int)) < 0) {
            perror("read");
            exit(1);
        }
        char* buf = MALLOC(sizeof(char)*(m+1));
        if(read(fd, buf, m) < 0) {
            perror("read");
            exit(1);
        }
        buf[m] = 0;

        self->mBody = STRING_NEW(buf);
        FREE(buf);
        }
        break;

    case 8:
        self->mBody = sStatments_load(fd);
        break;

    case 9:
        self->mBody = sRedirect_load(fd);
    }

    return self;
}

//////////////////////////////////////////////////////////
// プロセス置換
//////////////////////////////////////////////////////////
sPSub* sPSub_new(char* fname, sStatments* statments)
{
    sPSub* self = MALLOC(sizeof(sPSub));

    self->mFName = STRING_NEW(fname);
    self->mStatments = STATMENTS_NEW2(statments);

    return self;
}

void sPSub_delete(sPSub* self)
{
    string_delete(self->mFName);
    sStatments_delete(self->mStatments);
    FREE(self);
}

//////////////////////////////////////////////////////////
// sSubshell
//////////////////////////////////////////////////////////
sSubshell* sSubshell_new(MANAGED sStatments* contents)
{
    sSubshell* self = MALLOC(sizeof(sSubshell));

    self->contents = contents;

    return self;
}

void sSubshell_delete(sSubshell* self)
{
    sStatments_delete(self->contents);
    FREE(self);
}

sSubshell* sSubshell_new2(sSubshell* subshell)
{
    sSubshell* self = MALLOC(sizeof(sSubshell));

    self->contents = STATMENTS_NEW2(subshell->contents);

    return self;
}

void sSubshell_save(sSubshell* self, int fd)
{
    sStatments_save(self->contents, fd);
}

sSubshell* sSubshell_load(int fd)
{
    sSubshell* self = MALLOC(sizeof(sSubshell));
    
    self->contents = sStatments_load(fd);
    
    return self;
}

void sSubshell_view(sSubshell* self)
{
    sStatments_view(self->contents);
}

//////////////////////////////////////////////////////////
// sWhile
//////////////////////////////////////////////////////////
sWhile* sWhile_new(MANAGED sStatments* joukensiki, MANAGED sStatments* contents)
{
    sWhile* self = MALLOC(sizeof(sWhile));

    self->joukensiki = joukensiki;
    self->contents = contents;

    return self;
}

sWhile* sWhile_new2(sWhile* while_)
{
    sWhile* self = MALLOC(sizeof(sWhile));

    self->joukensiki = STATMENTS_NEW2(while_->joukensiki);
    self->contents = STATMENTS_NEW2(while_->contents);

    return self;
}

void sWhile_delete(sWhile* self)
{
    sStatments_delete(self->joukensiki);
    sStatments_delete(self->contents);
    FREE(self);
}

void sWhile_view(sWhile* self)
{
    sStatments_view(self->joukensiki);
    sStatments_view(self->contents);
}

void sWhile_save(sWhile* self, int fd)
{
    sStatments_save(self->joukensiki, fd);
    sStatments_save(self->contents, fd);
}

sWhile* sWhile_load(int fd)
{
    sWhile* self = MALLOC(sizeof(sWhile));
    
    self->joukensiki = sStatments_load(fd);
    self->contents = sStatments_load(fd);
    
    return self;
}

//////////////////////////////////////////////////////////
// sIf
//////////////////////////////////////////////////////////
sIf* sIf_new(MANAGED vector_obj* joukensiki_list, MANAGED vector_obj* contents_list)
{
    sIf* self = MALLOC(sizeof(sIf));
    
    self->joukensiki_list = joukensiki_list;
    self->contents_list = contents_list;

    return self;
}

void sIf_delete(sIf* self)
{
    int i;
    for(i=0; i<vector_size(self->joukensiki_list); i++) {
        sStatments_delete(vector_item(self->joukensiki_list, i));
    }
    vector_delete(self->joukensiki_list);
    for(i=0; i<vector_size(self->contents_list); i++) {
        sStatments_delete(vector_item(self->contents_list, i));
    }
    vector_delete(self->contents_list);

    FREE(self);
}

sIf* sIf_new2(sIf* if_)
{
    sIf* self = MALLOC(sizeof(sIf));
    
    self->joukensiki_list = VECTOR_NEW(10);
    self->contents_list = VECTOR_NEW(10);

    int i;
    for(i=0; i<vector_size(if_->joukensiki_list); i++) {
        vector_add(self->joukensiki_list, STATMENTS_NEW2(vector_item(if_->joukensiki_list, i)));
    }
    for(i=0; i<vector_size(if_->contents_list); i++) {
        vector_add(self->contents_list, STATMENTS_NEW2(vector_item(if_->contents_list, i)));
    }

    return self;
}

void sIf_view(sIf* self)
{
    int i;
    for(i=0; i<vector_size(self->joukensiki_list); i++) {
        sStatments_view(vector_item(self->joukensiki_list, i));
    }

    for(i=0; i<vector_size(self->contents_list); i++) {
        sStatments_view(vector_item(self->contents_list, i));
    }
}

void sIf_save(sIf* self, int fd)
{
    int n;
    unsigned char c;
    int i;
    
    n = vector_size(self->joukensiki_list);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write8");
        exit(1);
    }
    for(i=0; i<n; i++) {
        sStatments_save(vector_item(self->joukensiki_list, i), fd);
    }
    
    n = vector_size(self->contents_list);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write9");
        exit(1);
    }
    for(i=0; i<n; i++) {
        sStatments_save(vector_item(self->contents_list, i), fd);
    }
}

sIf* sIf_load(int fd)
{
    int n;
    int i;

    sIf* self = MALLOC(sizeof(sIf));
    
    self->joukensiki_list = VECTOR_NEW(50);
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        vector_add(self->joukensiki_list, sStatments_load(fd));
    }

    self->contents_list = VECTOR_NEW(50);
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        vector_add(self->contents_list, sStatments_load(fd));
    }
    
    return self;
}

/////////////////////////////////////////////////////////
// コマンド ls -al
/////////////////////////////////////////////////////////
#ifdef MDEBUG
sCommand* sCommand_new_debug(const char* fname, int line, const char* func)
{
    sCommand* self = CheckMemLeak_Malloc(sizeof(sCommand), fname, line, func);

    self->mKind = kCommand;
    self->mArgs = VECTOR_NEW(50);

    self->mRedirects = VECTOR_NEW(10);

    self->mExtra = NULL;

    return self;
}
#else
sCommand* sCommand_new()
{
    sCommand* self = MALLOC(sizeof(sCommand));

    self->mKind = kCommand;
    self->mArgs = VECTOR_NEW(50);

    self->mRedirects = VECTOR_NEW(10);

    self->mExtra = NULL;

    return self;
}
#endif

void sCommand_delete(sCommand* self)
{
    int i;
    for(i=0; i<vector_size(self->mArgs); i++) {
        sArg_delete(vector_item(self->mArgs, i));
    }
    vector_delete(self->mArgs);
    for(i=0; i<vector_size(self->mRedirects); i++) {
        sRedirect_delete(vector_item(self->mRedirects, i));
    }
    vector_delete(self->mRedirects);

    switch(self->mKind) {
    case kIf:
        sIf_delete(self->mExtra);
        break;
        
    case kWhile:
        sWhile_delete(self->mExtra);
        break;
        
    case kSubshell:
        sSubshell_delete(self->mExtra);
        break;
    }

    FREE(self);
}

sCommand* sCommand_new2(sCommand* command)
{
    sCommand* self = MALLOC(sizeof(sCommand));

    self->mKind = command->mKind;

    self->mArgs = VECTOR_NEW(50);
    int i;
    for(i=0; i<vector_size(command->mArgs); i++) {
        vector_add(self->mArgs, ARG_NEW2(vector_item(command->mArgs, i)));
    }
    self->mRedirects = VECTOR_NEW(10);
    for(i=0; i<vector_size(command->mRedirects); i++) {
        vector_add(self->mRedirects, sRedirect_new2(vector_item(command->mRedirects, i)));
    }
    /// mExtra ///
    switch(command->mKind) {
    case kIf:
        self->mExtra = sIf_new2(command->mExtra);
        break;
        
    case kWhile:
        self->mExtra = sWhile_new2(command->mExtra);
        break;
        
    case kSubshell:
        self->mExtra = sSubshell_new2(command->mExtra);
        break;

    default:
        self->mExtra = NULL;
        break;
    }

    return self;
}


void sCommand_view(sCommand* self)
{
    printf("self->mKind %s\n", gStatmentKindStrs[self->mKind]);

    int i;
    for(i=0; i<vector_size(self->mArgs); i++) {
        sArg* arg = vector_item(self->mArgs, i);
        if(arg->mKind == 0)
            printf("self->mArgs[%d] %s\n", i, string_c_str(arg->mBody));
    }
    for(i=0; i<vector_size(self->mRedirects); i++) {
        printf("self->mRedirects[%d]\n", i);
        sRedirect_view(vector_item(self->mRedirects, i));
    }

    /// mExtra ///
    switch(self->mKind) {
    case kIf:
        sIf_view(self->mExtra);
        break;
        
    case kWhile:
        sWhile_view(self->mExtra);
        break;
        
    case kSubshell:
        sSubshell_view(self->mExtra);
        break;
    }
}

void sCommand_save(sCommand* self, int fd)
{
    int i;
    unsigned char uc;
    int n;
    
    /// mKind ///
    uc = self->mKind;
    if(write(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("write10");
        exit(1);
    }
    
    /// mArgs ///
    n = vector_size(self->mArgs);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write11");
        exit(1);
    }
    
    for(i=0; i<vector_size(self->mArgs); i++) {
        sArg* arg = vector_item(self->mArgs, i);
        sArg_save(arg, fd);
    }

    /// mRedirects ///
    n = vector_size(self->mRedirects);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write12");
        exit(1);
    }

    for(i=0; i<n; i++) {
        sRedirect_save(vector_item(self->mRedirects, i), fd);
    }

    /// mExtra ///
    switch(self->mKind) {
    case kIf:
        sIf_save(self->mExtra, fd);
        break;
        
    case kWhile:
        sWhile_save(self->mExtra, fd);
        break;

    case kSubshell:
        sSubshell_save(self->mExtra, fd);
        break;
    }
}

sCommand* sCommand_load(int fd)
{
    int n, m;
    unsigned char uc;
    int i;
    
    sCommand* self = MALLOC(sizeof(sCommand));
    
    /// mKind ///
    if(read(fd, &uc, sizeof(unsigned char)) < 0) {
        perror("read");
        exit(1);
    }
    self->mKind = uc;
    
    /// mArgs ///
    self->mArgs = VECTOR_NEW(30);
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        vector_add(self->mArgs, sArg_load(fd));
    }

    /// mRedirects ///
    self->mRedirects = VECTOR_NEW(10);
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    for(i=0; i<n; i++) {
        vector_add(self->mRedirects, sRedirect_load(fd));
    }

    
    /// mExtra ///
    switch(self->mKind) {
    case kIf:
        self->mExtra = sIf_load(fd);
        break;
        
    case kWhile:
        self->mExtra = sWhile_load(fd);
        break;

    case kSubshell:
        self->mExtra = sSubshell_load(fd);
        break;
        
    default:
        self->mExtra = NULL;
    }
    
    return self;
}

/////////////////////////////////////////////////////////
// 文 ls -al | grep aaa
/////////////////////////////////////////////////////////
#if !defined(MDEBUG)
sStatment* sStatment_new()
{
    sStatment* self = MALLOC(sizeof(sStatment));

    self->mCommands = VECTOR_NEW(30);
    self->mTerminated = kTNormal;
    self->mFName = STRING_NEW("");
    self->mLine = 0;
    self->mTitle = STRING_NEW("");
    self->mNotEvaled = NULL;
    self->mBackground = FALSE;
    self->mRCodeReverse = FALSE;

    self->mGlobalPipeIn = FALSE;
    self->mGlobalPipeOut = FALSE;

    self->mGlobalPipeInNum = 0;
    self->mGlobalPipeOutNum = 0;

    self->mSTDINPipe = FALSE;

    return self;
}
#else
sStatment* sStatment_new_debug(const char* fname, int line, const char* func_name)
{
    sStatment* self = CheckMemLeak_Malloc(sizeof(sStatment), fname, line, func_name);

    self->mCommands = VECTOR_NEW(30);
    self->mTerminated = kTNormal;
    self->mFName = STRING_NEW("");
    self->mLine = 0;
    self->mTitle = STRING_NEW("");
    self->mNotEvaled = NULL;
    self->mBackground = FALSE;
    self->mRCodeReverse = FALSE;

    self->mGlobalPipeIn = FALSE;
    self->mGlobalPipeOut = FALSE;

    self->mGlobalPipeInNum = 0;
    self->mGlobalPipeOutNum = 0;

    self->mSTDINPipe = FALSE;

    return self;
}
#endif


void sStatment_delete(sStatment* self)
{
    int i;
    for(i=0; i<vector_size(self->mCommands); i++) {
        sCommand_delete(vector_item(self->mCommands, i));
    }
    vector_delete(self->mCommands);

    if(self->mNotEvaled) string_delete(self->mNotEvaled);

    string_delete(self->mFName);
    string_delete(self->mTitle);
    FREE(self);
}

#if !defined(MDEBUG)
sStatment* sStatment_new2(sStatment* statment)
{
    sStatment* self = MALLOC(sizeof(sStatment));

    self->mCommands = VECTOR_NEW(30);
    int i;
    for(i=0; i<vector_size(statment->mCommands); i++) {
        vector_add(self->mCommands, sCommand_new2(vector_item(statment->mCommands, i)));
    }
    self->mTerminated = statment->mTerminated;
    self->mFName = STRING_NEW(string_c_str(statment->mFName));
    self->mLine = statment->mLine;
    self->mTitle = STRING_NEW(string_c_str(statment->mTitle));
    if(statment->mNotEvaled) {
        self->mNotEvaled = STRING_NEW(string_c_str(statment->mNotEvaled));
    }
    else {
        self->mNotEvaled = NULL;
    }
    self->mBackground = statment->mBackground;
    self->mRCodeReverse = statment->mRCodeReverse;

    self->mGlobalPipeIn = statment->mGlobalPipeIn;
    self->mGlobalPipeOut = statment->mGlobalPipeOut;

    self->mGlobalPipeInNum = statment->mGlobalPipeInNum;
    self->mGlobalPipeOutNum = statment->mGlobalPipeOutNum;

    self->mSTDINPipe = statment->mSTDINPipe;

    return self;
}
#else
sStatment* sStatment_new2_debug(sStatment* statment, const char* fname, int line, const char* func_name)
{
    sStatment* self = CheckMemLeak_Malloc(sizeof(sStatment), fname, line, func_name);

    self->mCommands = VECTOR_NEW(30);
    int i;
    for(i=0; i<vector_size(statment->mCommands); i++) {
        vector_add(self->mCommands, sCommand_new2(vector_item(statment->mCommands, i)));
    }
    self->mTerminated = statment->mTerminated;
    self->mFName = STRING_NEW(string_c_str(statment->mFName));
    self->mLine = statment->mLine;
    self->mTitle = STRING_NEW(string_c_str(statment->mTitle));
    if(statment->mNotEvaled) {
        self->mNotEvaled = STRING_NEW(string_c_str(statment->mNotEvaled));
    }
    else {
        self->mNotEvaled = NULL;
    }
    self->mBackground = statment->mBackground;
    self->mRCodeReverse = statment->mRCodeReverse;

    self->mGlobalPipeIn = statment->mGlobalPipeIn;
    self->mGlobalPipeOut = statment->mGlobalPipeOut;

    self->mGlobalPipeInNum = statment->mGlobalPipeInNum;
    self->mGlobalPipeOutNum = statment->mGlobalPipeOutNum;

    self->mSTDINPipe = statment->mSTDINPipe;

    return self;
}
#endif


void sStatment_view(sStatment* self)
{
    printf("command number %d\n", vector_size(self->mCommands));
    int i;
    for(i=0; i<vector_size(self->mCommands); i++) {
        printf("command %d ---\n", i);
        sCommand_view(vector_item(self->mCommands, i));
    }

    if(self->mTerminated == kTNormal)
        printf("self->mTerminated kTNormal\n");
    else if(self->mTerminated == kTOrOr)
        printf("self->mTerminated kTOrOr\n");
    else if(self->mTerminated == kTAndAnd)
        printf("self->mTerminated kTAndAnd\n");

    printf("self->mFName %s\n", string_c_str(self->mFName));
    printf("self->mLine %d\n", self->mLine);
    printf("self->mTitle %s\n", string_c_str(self->mTitle));

    if(self->mNotEvaled) 
        printf("self->mNotEvaled %s\n", string_c_str(self->mNotEvaled));
    else
        printf("self->mNotEvaled NULL\n");

    if(self->mBackground) 
        printf("self->mBackground true\n");
    else
        printf("self->mBackground false\n");

    printf("self->mGlobalPipeIn %d\n", self->mGlobalPipeIn);
    printf("self->mGlobalPipeOut %d\n", self->mGlobalPipeOut);

    printf("self->mSTDINPipe %d\n", self->mSTDINPipe);
}

void sStatment_save(sStatment* self, int fd)
{
    int n;
    unsigned char c;
    int i;
    
    n = vector_size(self->mCommands);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write13");
        exit(1);
    }
    
    for(i=0; i<n; i++) {
        sCommand* item = vector_item(self->mCommands, i);
        sCommand_save(item, fd);
    }
    
    c = self->mTerminated;
    if(write(fd, &c, sizeof(unsigned char)) < 0) {
        perror("write14");
        exit(1);
    }
    
    n = string_size(self->mFName);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write15");
        exit(1);
    }
    int wret = write(fd, string_c_str(self->mFName), n);
    
    n = self->mLine;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write16");
        exit(1);
    }
    
    n = string_size(self->mTitle);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write17");
        exit(1);
    }
    if(write(fd, string_c_str(self->mTitle), n) < 0) {
        perror("write18");
        exit(1);
    }
    
    if(self->mNotEvaled) {
        n = 1;
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write19");
            exit(1);
        }
        
        n = string_size(self->mNotEvaled);
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write20");
            exit(1);
        }
        if(write(fd, string_c_str(self->mNotEvaled), n) < 0) {
            perror("write21");
            exit(1);
        }
    }
    else {
        n = 0;
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write22");
            exit(1);
        }
    }
    
    n = self->mBackground;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write23");
        exit(1);
    }

    // mGlobalPipeIn //
    n = self->mGlobalPipeIn;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write24");
        exit(1);
    }

    // mGlobalPipeOut //
    n = self->mGlobalPipeOut;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write25");
        exit(1);
    }

    // mGlobalPipeInNum //
    n = self->mGlobalPipeInNum;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write26");
        exit(1);
    }

    // mGlobalPipeOutNum //
    n = self->mGlobalPipeOutNum;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write27");
        exit(1);
    }

    // mSTDINPipe //
    n = self->mSTDINPipe;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write28");
        exit(1);
    }
}

sStatment* sStatment_load(int fd)
{
    int n;
    unsigned char c;
    int i;
    
    sStatment* self = MALLOC(sizeof(sStatment));
    
    self->mCommands = VECTOR_NEW(30);
    self->mRCodeReverse = FALSE;
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    for(i=0; i<n; i++) {
        vector_add(self->mCommands, sCommand_load(fd));
    }
    
    if(read(fd, &c, sizeof(unsigned char)) < 0) {
        perror("read");
        exit(1);
    }
    self->mTerminated = c;
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    char* buf = MALLOC(n+1);
    if(read(fd, buf, n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->mFName = STRING_NEW(buf);
    FREE(buf);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mLine = n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    buf = MALLOC(n+1);
    if(read(fd, buf, n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->mTitle = STRING_NEW(buf);
    FREE(buf);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    if(n == 0) {
        self->mNotEvaled = NULL;
    }
    else {
        if(read(fd, &n, sizeof(int)) < 0) {
            perror("read");
            exit(1);
        }
        buf = MALLOC(n+1);
        if(read(fd, buf, n) < 0) {
            perror("read");
            exit(1);
        }
        buf[n] = 0;
        self->mNotEvaled = STRING_NEW(buf);
        FREE(buf);
    }
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mBackground = n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mGlobalPipeIn = n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mGlobalPipeOut = n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mGlobalPipeInNum = n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mGlobalPipeOutNum = n;

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mSTDINPipe = n;

    return self;
}

/////////////////////////////////////////////////////////
// 複文 ls -al | grep aaa; pwd; cd aaa
/////////////////////////////////////////////////////////
#if defined(MDEBUG)
sStatments* sStatments_new_debug(const char* fname, int line, const char* func_name)
{
    sStatments* self = CheckMemLeak_Malloc(sizeof(sStatments), fname, line, func_name);

    self->mStatments = VECTOR_NEW(30);
    self->mSource = STRING_NEW("");

    return self;
}
#else
sStatments* sStatments_new()
{
    sStatments* self = MALLOC(sizeof(sStatments));

    self->mStatments = VECTOR_NEW(30);
    self->mSource = STRING_NEW("");

    return self;
}
#endif

void sStatments_delete(sStatments* self)
{
    int i;
    for(i=0; i<vector_size(self->mStatments); i++) {
        sStatment_delete(vector_item(self->mStatments, i));
    }
    vector_delete(self->mStatments);
    string_delete(self->mSource);
    FREE(self);
}

#ifdef MDEBUG

sStatments* sStatments_new2_debug(sStatments* statments, const char* fname, int line, const char* func_name)
{
    sStatments* self = CheckMemLeak_Malloc(sizeof(sStatments), fname, line, func_name);

    self->mStatments = VECTOR_NEW(30);
    int i;
    for(i=0; i < vector_size(statments->mStatments); i++) {
        vector_add(self->mStatments, STATMENT_NEW2(vector_item(statments->mStatments, i)));
    }

    self->mSource = STRING_NEW(string_c_str(statments->mSource));

    return self;
}

#else

sStatments* sStatments_new2(sStatments* statments)
{
    sStatments* self = MALLOC(sizeof(sStatments));

    self->mStatments = VECTOR_NEW(30);
    int i;
    for(i=0; i < vector_size(statments->mStatments); i++) {
        vector_add(self->mStatments, STATMENT_NEW2(vector_item(statments->mStatments, i)));
    }

    self->mSource = STRING_NEW(string_c_str(statments->mSource));

    return self;
}

#endif

void sStatments_view(sStatments* self)
{

    int i;
    for(i=0; i<vector_size(self->mStatments); i++)
        sStatment_view(vector_item(self->mStatments, i));
}

void sStatments_save(sStatments* self, int fd)
{
    int i;
    int n;
    
    n = vector_size(self->mStatments);
    
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write29");
        exit(1);
    }
    
    for(i=0; i<vector_size(self->mStatments); i++) {
        sStatment_save(vector_item(self->mStatments, i), fd);
    }

    n = string_length(self->mSource);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write30");
        exit(1);
    }
    if(write(fd, string_c_str(self->mSource), sizeof(char)*n) < 0) {
        perror("write31");
        exit(1);
    }
}

sStatments* sStatments_load(int fd)
{
    int n;
    int i;
    sStatments* self = MALLOC(sizeof(sStatments));
    
    self->mStatments = VECTOR_NEW(30);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    
    for(i=0; i<n; i++) {
        vector_add(self->mStatments, sStatment_load(fd));
    }
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    char* buf = MALLOC(sizeof(char)*(n+1));
    if(read(fd, buf, sizeof(char)*n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;

    self->mSource = STRING_NEW(buf);
    FREE(buf);
    
    return self;
}

/////////////////////////////////////////////////////////
// クラス
/////////////////////////////////////////////////////////
sClass* sClass_new(char* name, sStatments* statments, sClass* parent, int ref_count)
{
    sClass* self = memchecker_malloc(sizeof(sClass), kMCClass);
    self->name = STRING_NEW(name);
    if(statments) {
        self->statments = STATMENTS_NEW2(statments);
    }
    else {
        self->statments = STATMENTS_NEW();
    }
    self->parent = parent;
    self->input = FALSE;

    self->mRefCount = ref_count;
    if(ref_count == 0) {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self);
        hash_put(gClassPool, buf, self);
    }

    return self;
}

void sClass_delete(sClass* self)
{
    string_delete(self->name);
    sStatments_delete(self->statments);

    if(self->parent) {
        sClass* parent = self->parent;
        while(parent) {
            sClass* parent2 = parent;
            parent = parent->parent;
            parent2->parent = NULL;
            sClass_delete(parent2);
        }
    }
    memchecker_free(self);
    FREE(self);
}

void sClass_save(sClass* self, int fd)
{
    int n = string_length(self->name);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write32");
        exit(1);
    }
    if(write(fd, string_c_str(self->name), sizeof(char)*n) < 0) {
        perror("write33");
        exit(1);
    }

    sStatments_save(self->statments, fd);

    if(self->parent) {
        char c = 1;
        if(write(fd, &c, sizeof(char)) < 0) {
            perror("write34");
            exit(1);
        }
        
        sClass_save(self->parent, fd);
    }
    else {
        char c = 0;
        if(write(fd, &c, sizeof(char)) < 0) {
            perror("write35");
            exit(1);
        }
    }
}

sClass* sClass_load(int fd)
{
    sClass* self = memchecker_malloc(sizeof(sClass), kMCClass);
    
    int n;
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    char* buf = MALLOC(sizeof(char)*(n+1));
    if(read(fd, buf, sizeof(char)*n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->name = STRING_NEW(buf);
    FREE(buf);

    self->statments = sStatments_load(fd);

    char c;
    if(read(fd, &c, sizeof(char)) < 0) {
        perror("read");
        exit(1);
    }
    if(c) { 
        self->parent = sClass_load(fd);
    }
    self->input = FALSE;

    return self;
}

void sClass_view(sClass* self)
{
    printf("name %s\n", string_c_str(self->name));
}

/////////////////////////////////////////////////////////
// 関数
/////////////////////////////////////////////////////////
sFunction* sFunction_new(char* name, sStatments* statments, char* arg_name, sFunction* parent, int ref_count)
{
    sFunction* self = memchecker_malloc(sizeof(sFunction), kMCFunction);

    self->name = STRING_NEW(name);
    if(arg_name) {
        self->arg_name = STRING_NEW(arg_name);
    }
    else {
        self->arg_name = STRING_NEW("");
    }
    self->parent = parent;
    
    self->statments = STATMENTS_NEW2(statments);
    self->input = FALSE;

    self->mRefCount = ref_count;
    if(ref_count == 0) {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self);
        hash_put(gFunPool, buf, self);
    }

    return self;
}

void sFunction_delete(sFunction* self)
{
    string_delete(self->name);
    string_delete(self->arg_name);
    sStatments_delete(self->statments);

    if(self->parent) {
        sFunction* parent = self->parent;
        while(parent) {
            sFunction* parent2 = parent;
            parent = parent->parent;
            parent2->parent = NULL;
            sFunction_delete(parent2);
        }
    }

    memchecker_free(self);
    FREE(self);
}

void sFunction_save(sFunction* self, int fd)
{
    int n = string_length(self->name);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write36");
        exit(1);
    }
    if(write(fd, string_c_str(self->name), sizeof(char)*n) < 0) {
        perror("write37");
        exit(1);
    }
    n = string_length(self->arg_name);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write38");
        exit(1);
    }
    if(write(fd, string_c_str(self->arg_name), sizeof(char)*n) < 0) {
        perror("write39");
        exit(1);
    }

    sStatments_save(self->statments, fd);

    if(self->parent) {
        char c = 1;
        if(write(fd, &c, sizeof(char)) < 0) {
            perror("write39-1");
            exit(1);
        }
        sFunction_save(self->parent, fd);
    }
    else {
        char c = 0;
        if(write(fd, &c, sizeof(char)) < 0) {
            perror("write39-2");
            exit(1);
        }
    }
/*
    n = vector_size(self->args);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }

    int i;
    for(i=0; i<vector_size(self->args); i++) {
        string_obj* str = vector_item(self->args, i);
        n = string_length(str);
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write");
            exit(1);
        }
        if(write(fd, string_c_str(str), sizeof(char)*n) < 0) {
            perror("write");
            exit(1);
        }
    }
*/
}

sFunction* sFunction_load(int fd)
{
    sFunction* self = memchecker_malloc(sizeof(sFunction), kMCFunction);
    
    int n;
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    char* buf = MALLOC(sizeof(char)*(n+1));
    if(read(fd, buf, sizeof(char)*n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->name = STRING_NEW(buf);
    FREE(buf);

    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    buf = MALLOC(sizeof(char)*(n+1));
    if(read(fd, buf, sizeof(char)*n) < 0) {
        perror("read");
        exit(1);
    }
    buf[n] = 0;
    self->arg_name = STRING_NEW(buf);
    FREE(buf);

    self->statments = sStatments_load(fd);

    char exists;
    if(read(fd, &exists, sizeof(char)) < 0) {
        perror("read");
        exit(1);
    }

    if(exists) {
        self->parent = sFunction_load(fd);
    }
/*
    self->args = VECTOR_NEW(10);
    
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }

    int i;
    for(i=0; i<n; i++) {
        int m;
        if(read(fd, &m, sizeof(int)) < 0) {
            perror("read");
            exit(1);
        }

        char* buf = MALLOC(sizeof(char)*(m+1));
        if(read(fd, buf, sizeof(char)*m) < 0) {
            perror("read");
            exit(1);
        }
        buf[m] = 0;

        vector_add(self->args, STRING_NEW(buf));
    }
*/
    self->input = FALSE;

    return self;
}

void sFunction_view(sFunction* self)
{
    printf("name %s\n", string_c_str(self->name));
    printf("arg_name %s\n", string_c_str(self->arg_name));
}

//////////////////////////////////////////////////////
// saphireが組み込まれたアプリケーションから
// 登録された内部コマンド
/////////////////////////////////////////////////////
sInnerCommand* sInnerCommand_new(char* name, fInnerCommand fun)
{
    sInnerCommand* self = MALLOC(sizeof(sInnerCommand));

    self->mName = STRING_NEW(name);
    self->mFun = fun;

    return self;
}

void sInnerCommand_delete(sInnerCommand* self)
{
    string_delete(self->mName);
    FREE(self);
}

/////////////////////////////////////////////////////////
// プログラム名補完候補を再読み込み
/////////////////////////////////////////////////////////
static BOOL name_sort(void* left, void* right)
{
    string_obj* l = left;
    string_obj* r = right;

    return strcmp(string_c_str(l), string_c_str(r)) < 0;
}

static void saphire_rehash_external_prog()
{
    hash_obj* check = HASH_NEW(3000);

    int i;
    for(i=0; i<vector_size(gSaphireExternalProgCompletions); i++) {
        string_delete(vector_item(gSaphireExternalProgCompletions, i));
    }
    vector_clear(gSaphireExternalProgCompletions);

    /// 内部コマンド ///
    for(i=0; i<kCommand; i++) {
        vector_add(gSaphireExternalProgCompletions, STRING_NEW(gStatmentKindStrs[i]));
    }

    /// 外部登録の内部コマンド ///
    hash_it* it = hash_loop_begin(gInnerCommands);
    while(it != NULL) {
        sInnerCommand* command = hash_loop_item(it);
        vector_add(gSaphireExternalProgCompletions, STRING_NEW(string_c_str(command->mName)));
        it = hash_loop_next(it);
    }

    /// $PATHにあるファイルを全てgSaphireExternalProgCompletionsに入れていく ///
    char* path = getenv("PATH");
    if(path == NULL) {
        fprintf(stderr, "$PATH is NULL\n");
        exit(1);
    }

    char* p = path;
    char buf[4096];
    char* p2 = buf;

    while(*p) {
        if(*p != ':') {
            *p2++ = *p;
        }
        else {
            *p2 = 0;

            DIR* dir = opendir(buf);
            if(dir) {
                struct dirent* entry;
                while(entry = readdir(dir)) {
                    char path2[PATH_MAX];
                    snprintf(path2, PATH_MAX, "%s/%s", buf, entry->d_name);
                    
                    struct stat stat_;
                    memset(&stat_, 0, sizeof(struct stat));
                    stat(path2, &stat_);

                    if(strcmp(entry->d_name, ".") != 0
                            && strcmp(entry->d_name, "..") != 0
                            &&
                            (stat_.st_mode & S_IXUSR
                             ||stat_.st_mode & S_IXGRP
                             ||stat_.st_mode & S_IXOTH))
                    {
                        if(hash_item(check, entry->d_name) == NULL) {
                            hash_put(check, entry->d_name, (void*)1);
#if defined(__CYGWIN__)
                            if(strstr(entry->d_name, ".exe") 
                                == entry->d_name + strlen(entry->d_name) 
                                    -4) 
                            {
                                char buf[128];
                                memcpy(buf, entry->d_name
                                        , strlen(entry->d_name) -4);
                                buf[strlen(entry->d_name) -4] = 0;
                                vector_add(gSaphireExternalProgCompletions, STRING_NEW(buf));
                            }
                            else {
                                vector_add(gSaphireExternalProgCompletions
                                        , STRING_NEW(entry->d_name));
                            }
                            
#else
                            vector_add(gSaphireExternalProgCompletions
                                        , STRING_NEW(entry->d_name));
#endif
                        }
                    }
                }

                closedir(dir);
            }

            p2 = buf;
        }

        p++;
    }
    *p2 = 0;

    DIR* dir = opendir(buf);
    if(dir) {
        struct dirent* entry;
        while(entry = readdir(dir)) {
            if(strcmp(entry->d_name, ".") != 0
                    && strcmp(entry->d_name, "..") != 0)
            {
                vector_add(gSaphireExternalProgCompletions, STRING_NEW(entry->d_name));
            }
        }

        closedir(dir);
    }

    (void)vector_sort(gSaphireExternalProgCompletions, name_sort);

    hash_delete(check);
}

static void saphire_rehash_user_fun()
{
    /// ユーザーオリジナル関数 ///
    hash_it* it = hash_loop_begin(gFuncs);
    while(it != NULL) {
        sFunction* fun = hash_loop_item(it);
        vector_add(gSaphireProgCompletions, STRING_NEW(string_c_str(fun->name)));
        it = hash_loop_next(it);
    }
}

void saphire_rehash_object()
{
    /// ユーザーオリジナル関数 ///
    hash_it* it = hash_loop_begin(gObjects);
    while(it != NULL) {
        sObject* object = hash_loop_item(it);
        string_obj* str = STRING_NEW(string_c_str(object->mName));
        string_push_back(str, "->");
        vector_add(gSaphireProgCompletions, str);
        it = hash_loop_next(it);
    }
}

void saphire_rehash_class()
{
    /// ユーザーオリジナル関数 ///
    hash_it* it = hash_loop_begin(gClasses);
    while(it != NULL) {
        sClass* klass = hash_loop_item(it);
        string_obj* str = STRING_NEW(string_c_str(klass->name));
        vector_add(gSaphireProgCompletions, str);
        it = hash_loop_next(it);
    }
}

void saphire_rehash()
{
    saphire_rehash_external_prog();

    int i;
    for(i=0; i<vector_size(gSaphireProgCompletions); i++) {
        string_delete(vector_item(gSaphireProgCompletions, i));
    }
    vector_clear(gSaphireProgCompletions);

    for(i=0; i< vector_size(gSaphireExternalProgCompletions); i++) {
        vector_add(gSaphireProgCompletions, STRING_NEW(string_c_str(vector_item(gSaphireExternalProgCompletions, i))));
    }

    saphire_rehash_user_fun();
    saphire_rehash_object();
    saphire_rehash_class();
}

void saphire_rehash2()
{
    int i;
    for(i=0; i<vector_size(gSaphireProgCompletions); i++) {
        string_delete(vector_item(gSaphireProgCompletions, i));
    }
    vector_clear(gSaphireProgCompletions);

    for(i=0; i< vector_size(gSaphireExternalProgCompletions); i++) {
        vector_add(gSaphireProgCompletions, STRING_NEW(string_c_str(vector_item(gSaphireExternalProgCompletions, i))));
    }

    saphire_rehash_user_fun();
    saphire_rehash_object();
    saphire_rehash_class();
}

///////////////////////////////////////////////////////
// ランタイムスクリプト実行
///////////////////////////////////////////////////////
static void read_rc_file(BOOL run_user_runtimescript)
{
    char rc_fname[PATH_MAX];
    snprintf(rc_fname, PATH_MAX,"%s/saphire.sa", SYSCONFDIR);

    if(access(rc_fname, R_OK) == 0) {
        sWFd* pipeout = WFD_NEW(STDOUT_FILENO);
        sRFd* pipein = RFD_NEW(STDIN_FILENO);
        saphire_init_stack_frame();
        if(saphire_load(rc_fname
           , pipeout, pipein, STDERR_FILENO) < 0)
        {
            fprintf(stderr, "%s", string_c_str(gErrMsg));
            exit(1);
        }
        if(!sWFd_flash(pipeout)) {
            fprintf(stderr, "can't flash rc");
            exit(1);
        }
        sWFd_delete(pipeout);
        sRFd_delete(pipein);
    }
    else {
        fprintf(stderr, "can't find %s file\n", rc_fname);
        exit(1);
    }

    /// ユーザーのランタイムスクリプト ////
    if(run_user_runtimescript) {
        char* user_rc_fname = getenv("SAPHIRE_USERRC");
        if(user_rc_fname) {
            if(access(user_rc_fname, R_OK) == 0) {
                sWFd* pipeout = WFD_NEW(STDOUT_FILENO);
                sRFd* pipein = RFD_NEW(STDIN_FILENO);
                saphire_init_stack_frame();
                if(saphire_load(user_rc_fname
                    , pipeout, pipein, STDERR_FILENO) < 0)
                {
                    fprintf(stderr, "%s", string_c_str(gErrMsg));
                    exit(1);
                }
                if(!sWFd_flash(pipeout)) {
                    fprintf(stderr, "can't flash rc");
                    exit(1);
                }
                sWFd_delete(pipeout);
                sRFd_delete(pipein);
            }
        }
    }
}

static void read_rc_obj_file(BOOL run_user_runtimescript)
{
    char rc_fname[PATH_MAX];
    snprintf(rc_fname, PATH_MAX, "%s/saphire.sao", SYSCONFDIR);

    if(access(rc_fname, R_OK) == 0) {
        sWFd* pipeout = WFD_NEW(STDOUT_FILENO);
        sRFd* pipein = RFD_NEW(STDIN_FILENO);
        saphire_init_stack_frame();
        if(saphire_load_obj(rc_fname
           , pipeout, pipein, STDERR_FILENO, NULL) < 0)
        {
            fprintf(stderr, "%s", string_c_str(gErrMsg));
            exit(1);
        }
        if(!sWFd_flash(pipeout)) {
            fprintf(stderr, "can't flash rc");
            exit(1);
        }
        sWFd_delete(pipeout);
        sRFd_delete(pipein);
    }
    else {
        fprintf(stderr, "can't find %s file\n", rc_fname);
        exit(1);
    }

    /// ユーザーのランタイムスクリプト ////
    if(run_user_runtimescript) {
        char* user_rc_fname = getenv("SAPHIRE_USERRC");
        if(user_rc_fname) {
            if(access(user_rc_fname, R_OK) == 0) {
                sWFd* pipeout = WFD_NEW(STDOUT_FILENO);
                sRFd* pipein = RFD_NEW(STDIN_FILENO);
                saphire_init_stack_frame();
                if(saphire_load_obj(user_rc_fname
                    , pipeout, pipein, STDERR_FILENO, NULL) < 0)
                {
                    fprintf(stderr, "%s", string_c_str(gErrMsg));
                    exit(1);
                }
                sWFd_flash(pipeout);

                sWFd_delete(pipeout);
                sRFd_delete(pipein);
            }
        }
    }
}

////////////////////////////////////////////////////////
// 初期化
////////////////////////////////////////////////////////

BOOL gRunningOnTerminal;

void saphire_init(enum eAppType app_type, BOOL job_control, enum eRuntimeScript runtime_script, BOOL run_user_runtimescript)
{
    memchecker_init();

    gGlobalPool = HASH_NEW(50);
    gHashPool = HASH_NEW(50);
    gAryPool = HASH_NEW(50);
    gClassPool = HASH_NEW(50);
    gObjectPool = HASH_NEW(50);
    gFunPool = HASH_NEW(50);

    gGlobalPipe = STRING_NEW("");
    int i;
    for(i=0; i<10; i++) {
        gGlobalPipeNum[i] = STRING_NEW("");
    }

    // ナンバーグローバルパイプの読み込みんだデータ

    gAppType = app_type;
    gJobControl = job_control;

    gArrays = HASH_NEW(100);
    gHashs = HASH_NEW(100);
    gGlobals = HASH_NEW(100);
    gFuncs = HASH_NEW(50);
    gClasses = HASH_NEW(50);
    gObjects = HASH_NEW(50);
    gRefs = HASH_NEW(50);

    gJobs = VECTOR_NEW(50);

    gReadBuffers = HASH_NEW(100);

    setenv("SAPHIRE_VERSION", "3.6.2", 1);
    setenv("SAPHIRE_PROMPT", "> ", 1);
    setenv("SAPHIRE_DOCDIR", DOCDIR, 1);
    char* home = getenv("HOME");
    if(home) {
        char path[PATH_MAX];
        snprintf(path, PATH_MAX, "%s/.saphire_history", home);
        setenv("SAPHIRE_HISTFILE", path, 1);
    }
    setenv("SAPHIRE_HISTSIZE", "1000", 1);
    if(home) {
        char path[PATH_MAX];
        snprintf(path, PATH_MAX, "%s/.saphire.sao", home);
        setenv("SAPHIRE_USERRC", path, 1);
    }
    
    gSaphireProgCompletions = VECTOR_NEW(100);
    gSaphireExternalProgCompletions = VECTOR_NEW(100);

    gRegexs = HASH_NEW(100);
    gRegexsI = HASH_NEW(100);
    gRegexsM = HASH_NEW(100);
    gRegexsIM = HASH_NEW(100);
    
    gStackFrame = VECTOR_NEW(5);
    vector_add(gStackFrame, HASH_NEW(30));
    gStackFrameArrays = VECTOR_NEW(5);
    vector_add(gStackFrameArrays, HASH_NEW(30));
    gStackFrameHashs = VECTOR_NEW(5);
    vector_add(gStackFrameHashs, HASH_NEW(30));
    gStackFrameObjects = VECTOR_NEW(5);
    vector_add(gStackFrameObjects, HASH_NEW(30));
    gStackFrameRefs = VECTOR_NEW(5);
    vector_add(gStackFrameRefs, HASH_NEW(30));

    gInnerCommands = HASH_NEW(50);
    gKitutukiExit = -1;

    gTmpFiles = VECTOR_NEW(10);
    gDirStack = VECTOR_NEW(10);
    gErrMsg = STRING_NEW("");

    gStackTraceFName = VECTOR_NEW(30);
    gStackTraceLineNum = VECTOR_NEW(30);

//    saphire_rehash();

    /// シグナルの設定 ///
    if(app_type == kATOptC) {
        saphire_set_signal_optc();
    }
    else {
        saphire_set_signal();
    }

    /// ランタイムスクリプト読み込み ///
    switch(runtime_script) {
        case kRSNoRead:
            break;

        case kRSSource:
            read_rc_file(run_user_runtimescript);
            break;

        case kRSObject:
            read_rc_obj_file(run_user_runtimescript);
            break;
    }
    if(tcgetpgrp(0) == getpgid(0)) {
        gSaphireBackground = FALSE;
        //mreset_tty();
    }
    else {
        gSaphireBackground = TRUE;
    }

    char* env = getenv("SAPHIRE_TMPDIR");
    if(env) 
        xstrncpy(gTmpDir, env, PATH_MAX);
    else {
        snprintf(gTmpDir, PATH_MAX, "/tmp");
    }

    //tmpfile_all_sweep();
    gSaphireVarCompletions = VECTOR_NEW(100);

    char* term_env = getenv("TERM");
    if(term_env != NULL && strcmp(term_env, "") != 0) {
        gRunningOnTerminal = TRUE;
        mcurses_init(kTKUtf8);
    }
    else {
        gRunningOnTerminal = FALSE;
    }
    kanji_init();
}

void saphire_final()
{
    hash_it* it = hash_loop_begin(gGlobalPool);
    while(it != NULL) {
        sGlobal_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gGlobalPool);

    it = hash_loop_begin(gHashPool);
    while(it != NULL) {
        hash_obj* h = hash_loop_item(it);
        
        hash_it* it2 = hash_loop_begin(h);
        while(it2) {
            string_delete(hash_loop_item(it2));
            it2 = hash_loop_next(it2);
        }

        sHash_delete(h);
        it = hash_loop_next(it);
    }
    hash_delete(gHashPool);

    it = hash_loop_begin(gAryPool);
    while(it != NULL) {
        vector_obj* ary = hash_loop_item(it);
        int i;
        for(i=0; i<vector_size(ary); i++) {
            string_delete(vector_item(ary, i));
        }
        sAry_delete(ary);
        it = hash_loop_next(it);
    }
    hash_delete(gAryPool);

    it = hash_loop_begin(gClassPool);
    while(it != NULL) {
        sClass_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gClassPool);

    it = hash_loop_begin(gObjectPool);
    while(it != NULL) {
        sObject_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gObjectPool);

    it = hash_loop_begin(gFunPool);
    while(it != NULL) {
        sFunction_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gFunPool);

    string_delete(gErrMsg);
    int i;
    for(i=0; i<vector_size(gStackTraceFName); i++) {
        string_delete(vector_item(gStackTraceFName, i));
    }
    vector_delete(gStackTraceFName);
    vector_delete(gStackTraceLineNum);
    for(i=0; i<vector_size(gDirStack); i++) {
        string_delete(vector_item(gDirStack, i));
    }
    vector_delete(gDirStack);

    it = hash_loop_begin(gInnerCommands);
    while(it != NULL) {
        sInnerCommand* command = hash_loop_item(it);
        sInnerCommand_delete(command);
        it = hash_loop_next(it);
    }
    hash_delete(gInnerCommands);
    it = hash_loop_begin(gArrays);
    while(it) {
        vector_obj* v = hash_loop_item(it);
        int i;
        for(i=0; i<vector_size(v); i++) {
            string_delete(vector_item(v, i));
        }
        sAry_delete(v);
        it = hash_loop_next(it);
    }
    hash_delete(gArrays);
    it = hash_loop_begin(gHashs);
    while(it) {
        hash_obj* h = hash_loop_item(it);

        hash_it* it2 = hash_loop_begin(h);
        while(it2) {
            string_delete(hash_loop_item(it2));
            it2 = hash_loop_next(it2);
        }

        sHash_delete(h);

        it = hash_loop_next(it);
    }
    hash_delete(gHashs);
    it = hash_loop_begin(gGlobals);
    while(it) {
        sGlobal_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gGlobals);
    it = hash_loop_begin(gFuncs);
    while(it) {
        sFunction_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gFuncs);
    it = hash_loop_begin(gClasses);
    while(it) {
        sClass_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gClasses);
    it = hash_loop_begin(gObjects);
    while(it) {
        sObject_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gObjects);
    it = hash_loop_begin(gRefs);
    while(it) {
        sRef_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gRefs);

    string_delete(gGlobalPipe);
    for(i=0; i<10; i++) {
        string_delete(gGlobalPipeNum[i]);
    }
    it = hash_loop_begin(gReadBuffers);
    while(it) {
        sRFd* fd = (sRFd*)hash_loop_item(it);
        sRFd_close(fd);
        sRFd_delete(fd);
        it = hash_loop_next(it);
    }
    hash_delete(gReadBuffers);
    
    it = hash_loop_begin(gRegexs);
    while(it != NULL) {
        onig_free(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gRegexs);

    it = hash_loop_begin(gRegexsI);
    while(it != NULL) {
        onig_free(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gRegexsI);

    it = hash_loop_begin(gRegexsM);
    while(it != NULL) {
        onig_free(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gRegexsM);

    it = hash_loop_begin(gRegexsIM);
    while(it != NULL) {
        onig_free(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(gRegexsIM);

    for(i=0; i<vector_size(gJobs); i++) {
        sJob_delete(vector_item(gJobs, i));
    }
    vector_delete(gJobs);
    

    /// プロセス置換で使った一時ファイルを掃除 ///
    int k;
    for(k=0; k<vector_size(gTmpFiles); k++) {
        string_obj* fname = vector_item(gTmpFiles, k);

        if(unlink(string_c_str(fname)) < 0) {
            perror("unlink");
            exit(1);
        }
        string_delete(vector_item(gTmpFiles, i));
    }
    vector_delete(gTmpFiles);

    for(i=0; i<vector_size(gSaphireExternalProgCompletions); i++) {
        string_delete(vector_item(gSaphireExternalProgCompletions, i));
    }
    vector_delete(gSaphireExternalProgCompletions);
    for(i=0; i<vector_size(gSaphireProgCompletions); i++) {
        string_delete(vector_item(gSaphireProgCompletions, i));
    }
    vector_delete(gSaphireProgCompletions);
    for(i=0; i<vector_size(gSaphireVarCompletions); i++) {
        string_delete(vector_item(gSaphireVarCompletions, i));
    }
    vector_delete(gSaphireVarCompletions);

    for(i=0; i<vector_size(gStackFrame); i++) {
        hash_obj* h = vector_item(gStackFrame, i);
        it = hash_loop_begin(h);
        while(it != NULL) {
            sVar_delete(hash_loop_item(it));
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_delete(gStackFrame);

    for(i=0; i<vector_size(gStackFrameArrays); i++) {
        hash_obj* h = vector_item(gStackFrameArrays, i);
        it = hash_loop_begin(h);
        while(it != NULL) {
            vector_obj* v = hash_loop_item(it);
            int i;
            for(i=0; i<vector_size(v); i++) {
                string_delete(vector_item(v, i));
            }
            sAry_delete(v);
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_delete(gStackFrameArrays);

    for(i=0; i<vector_size(gStackFrameHashs); i++) {
        hash_obj* h = vector_item(gStackFrameHashs, i);
        it = hash_loop_begin(h);
        while(it != NULL) {
            hash_obj* h2 = hash_loop_item(it);
            hash_it* it2 = hash_loop_begin(h2);
            while(it2) {
                string_delete(hash_loop_item(it2));
                it2 = hash_loop_next(it2);
            }
            sHash_delete(h2);
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_delete(gStackFrameHashs);

    for(i=0; i<vector_size(gStackFrameObjects); i++) {
        hash_obj* h = vector_item(gStackFrameObjects, i);
        it = hash_loop_begin(h);
        while(it != NULL) {
            sObject_delete(hash_loop_item(it));
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_delete(gStackFrameObjects);

    for(i=0; i<vector_size(gStackFrameRefs); i++) {
        hash_obj* h = vector_item(gStackFrameRefs, i);
        it = hash_loop_begin(h);
        while(it != NULL) {
            sRef_delete(hash_loop_item(it));
            it = hash_loop_next(it);
        }
        hash_delete(h);
    }
    vector_delete(gStackFrameRefs);

    memchecker_final();

    if(gRunningOnTerminal) {
        mcurses_final();
    }
    kanji_final();
}

int saphire_gc_sweep(sWFd* nextout)
{
    int count = 0;

    hash_it* it = hash_loop_begin(gGlobalPool);
    while(it != NULL) {
        sGlobal* global = hash_loop_item(it);
        char* key = hash_loop_key(it);
        if(global->mRefCount == 0) {
            sGlobal_delete((string_obj*)global);
            char buf[BUFSIZ];
            snprintf(buf, BUFSIZ, "%p deleted\n", global);
            if(!sWFd_push_back(nextout, buf)) {
                return -1;
            }
            it = hash_loop_next(it);
            hash_erase(gGlobalPool, key);
            count++;
        }
        else {
            it = hash_loop_next(it);
        }
    }

    it = hash_loop_begin(gHashPool);
    while(it != NULL) {
        sHash* hash = hash_loop_item(it);
        char* key = hash_loop_key(it);
        if(hash->mRefCount == 0) {
            hash_it* it2 = hash_loop_begin((hash_obj*)hash);
            while(it2) {
                string_delete(hash_loop_item(it2));
                it2 = hash_loop_next(it2);
            }
            sHash_delete((hash_obj*)hash);

            char buf[BUFSIZ];
            snprintf(buf, BUFSIZ, "%p deleted\n", hash);
            if(!sWFd_push_back(nextout, buf)) {
                return -1;
            }

            it = hash_loop_next(it);
            hash_erase(gHashPool, key);
            count++;
        }
        else {
            it = hash_loop_next(it);
        }
    }

    it = hash_loop_begin(gAryPool);
    while(it != NULL) {
        sAry* ary = hash_loop_item(it);
        char* key = hash_loop_key(it);
        if(ary->mRefCount == 0) {
            vector_obj* v = (vector_obj*)ary;
            int i;
            for(i=0; i<vector_size(v); i++) {
                string_delete(vector_item(v, i));
            }
            sAry_delete(v);

            char buf[BUFSIZ];
            snprintf(buf, BUFSIZ, "%p deleted\n", v);
            if(!sWFd_push_back(nextout, buf)) {
                return -1;
            }

            it = hash_loop_next(it);
            hash_erase(gAryPool, key);
            count++;
        }
        else {
            it = hash_loop_next(it);
        }
    }

    it = hash_loop_begin(gClassPool);
    while(it != NULL) {
        sClass* klass = hash_loop_item(it);
        char* key = hash_loop_key(it);
        if(klass->mRefCount == 0) {
            sClass_delete(klass);

            char buf[BUFSIZ];
            snprintf(buf, BUFSIZ, "%p deleted\n", klass);
            if(!sWFd_push_back(nextout, buf)) {
                return -1;
            }

            it = hash_loop_next(it);
            hash_erase(gClassPool, key);

            count++;
        }
        else {
            it = hash_loop_next(it);
        }
    }

    it = hash_loop_begin(gObjectPool);
    while(it != NULL) {
        sObject* object = hash_loop_item(it);
        char* key = hash_loop_key(it);
        if(object->mRefCount == 0) {
            sObject_delete(object);

            char buf[BUFSIZ];
            snprintf(buf, BUFSIZ, "%p deleted\n", object);
            if(!sWFd_push_back(nextout, buf)) {
                return -1;
            }

            it = hash_loop_next(it);
            hash_erase(gObjectPool, key);
            count++;
        }
        else {
            it = hash_loop_next(it);
        }
    }

    it = hash_loop_begin(gFunPool);
    while(it != NULL) {
        sFunction* fun = hash_loop_item(it);
        char* key = hash_loop_key(it);
        if(fun->mRefCount == 0) {
            sFunction_delete(fun);

            char buf[BUFSIZ];
            snprintf(buf, BUFSIZ, "%p deleted\n", fun);
            if(!sWFd_push_back(nextout, buf)) {
                return -1;
            }

            it = hash_loop_next(it);
            hash_erase(gFunPool, key);
            count++;
        }
        else {
            it = hash_loop_next(it);
        }
    }

    return count;
}

/////////////////////////////////////////////////////////////////////////
// ファイルディスクリプタ
/////////////////////////////////////////////////////////////////////////

#ifdef MDEBUG
sRFd* sRFd_new_debug(int fd, const char* fname, int line, const char* func_name)
{
    sRFd* self = (sRFd*)CheckMemLeak_Malloc(sizeof(sRFd), fname, line, func_name);

    self->mMallocSize = 1024;
    self->mBuffer = MALLOC(self->mMallocSize);
    xstrncpy(self->mBuffer, "", 1024);
    self->mPtr = self->mBuffer;
    self->mFd = fd;

    return self;
}

sRFd* sRFd_new2_debug(int fd, char* str, const char* fname, int line, const char* func_name)
{
    sRFd* self = (sRFd*)CheckMemLeak_Malloc(sizeof(sRFd), fname, line, func_name);

    self->mBuffer = STRDUP(str);
    self->mMallocSize = strlen(str) + 1;
    self->mPtr = self->mBuffer;
    self->mFd = fd;

    return self;
}

#else

sRFd* sRFd_new(int fd)
{
    sRFd* self = (sRFd*)MALLOC(sizeof(sRFd));

    self->mMallocSize = 1024;
    self->mBuffer = MALLOC(self->mMallocSize);
    xstrncpy(self->mBuffer, "", 1024);
    self->mPtr = self->mBuffer;
    self->mFd = fd;

    return self;
}

sRFd* sRFd_new2(int fd, char* str)
{
    sRFd* self = (sRFd*)MALLOC(sizeof(sRFd));

    self->mBuffer = STRDUP(str);
    self->mMallocSize = strlen(str) + 1;
    self->mPtr = self->mBuffer;
    self->mFd = fd;

    return self;
}

#endif

sRFd* sRFd_new3(int fd, MANAGED char* buffer, int malloc_size)
{
    sRFd* self = (sRFd*)MALLOC(sizeof(sRFd));

    self->mMallocSize = malloc_size;
    self->mBuffer = buffer;
    self->mPtr = self->mBuffer;
    self->mFd = fd;

    return self;
}

sRFd* sRFd_new_from_wfd(int fd, sWFd* nextout)
{
    sRFd* self = (sRFd*)MALLOC(sizeof(sRFd));

    self->mMallocSize = nextout->mMallocSize;
    self->mBuffer = nextout->mBuffer;
    self->mPtr = self->mBuffer;
    self->mFd = fd;

    nextout->mBuffer = NULL;

    return self;
}

void sRFd_delete(sRFd* self)
{
    FREE(self->mBuffer);
    FREE(self);
}

BOOL sRFd_close(sRFd* self)
{
    if(self->mFd == -1) {
    }
    else {
        if(self->mFd == 0) {
            return TRUE;
        }
        else {
            return close(self->mFd) == 0;
        }
    }
}

// 0:正常終了 1:EOF -1:エラー
int sRFd_read_all(sRFd* self, string_obj* str)
{
    if(self->mFd == -1) {
        if(!string_put_cancelable(str, self->mPtr)) {
            return -1;
        }
        self->mPtr = self->mBuffer + strlen(self->mBuffer);
    }
    else {
        /// 読み込む
        while(1) {
            char buf[BUFSIZ+1];
            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(r == 0) {
                if(strlen(self->mBuffer) == 0) {
                    return 1;
                }
                break;
            }
            buf[r] = 0;

            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }

            int len = strlen(self->mBuffer) + strlen(buf) + 1;
            if(len > self->mMallocSize) {
                self->mMallocSize = len * 2;

                int n = self->mPtr - self->mBuffer;
                self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                self->mPtr = self->mBuffer + n;

                xstrncat(self->mBuffer, buf, self->mMallocSize);
            }
            else {
                xstrncat(self->mBuffer, buf, self->mMallocSize);
            }
        }

        if(!string_put_cancelable(str, self->mPtr)) {
            self->mPtr = self->mBuffer + strlen(self->mBuffer);
            return -1;
        }

        self->mPtr = self->mBuffer + strlen(self->mBuffer);
    }

    return 0;
}

// 0:正常終了 1:EOF -1:エラー
int sRFd_read_all_preserve(sRFd* self, string_obj* str)
{
    if(self->mFd == -1) {
        if(!string_put_cancelable(str, self->mPtr)) {
            return -1;
        }
    }
    else {
        /// 読み込む
        while(1) {
            char buf[BUFSIZ+1];
            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(r == 0) {
                if(strlen(self->mBuffer) == 0) {
                    return 1;
                }
                break;
            }
            buf[r] = 0;

            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }

            int len = strlen(self->mBuffer) + strlen(buf) + 1;
            if(len > self->mMallocSize) {
                self->mMallocSize = len * 2;

                int n = self->mPtr - self->mBuffer;
                self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                self->mPtr = self->mBuffer + n;

                xstrncat(self->mBuffer, buf, self->mMallocSize);
            }
            else {
                xstrncat(self->mBuffer, buf, self->mMallocSize);
            }
        }

        /// 読み込んだバッファを書き込む
        if(!string_put_cancelable(str, self->mBuffer)) {
            return -1;
        }
    }

    return 0;
}

// 0:正常終了 1:EOF -1:エラー
int sRFd_read_all_to_buffer(sRFd* self)
{
    if(self->mFd == -1) {
    }
    else {
        /// 読み込む
        while(1) {
            char buf[BUFSIZ+1];
            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(r == 0) {
                if(self->mPtr == self->mBuffer + strlen(self->mBuffer)) {
                    return 1;
                }
                break;
            }
            buf[r] = 0;

            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }

            int len = strlen(self->mBuffer) + strlen(buf) + 1;
            if(len < self->mMallocSize) {
                xstrncat(self->mBuffer, buf, self->mMallocSize);
            }
            else {
                self->mMallocSize = len * 2;

                int n = self->mPtr - self->mBuffer;
                self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                self->mPtr = self->mBuffer + n;

                xstrncat(self->mBuffer, buf, self->mMallocSize);
            }
        }
    }

    return 0;
}

// 0: 正常終了 1: EOF -1:エラー
int sRFd_read_oneline(sRFd* self, string_obj* str, enum eLineField lf)
{
    if(self->mFd == -1) {
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return -1;
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファから一行を得て返す ///
        if(lf == kLF) {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\r' && *(self->mPtr + 1) == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\r') {
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\a') {
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
    }
    else {
        while(1) {
            /// 改行が含まれていればbreakする ///
            if(lf == kCR) {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kLF) {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\n') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kCRLF) {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r' && *(p+1) == '\n') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\a') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }

            char buf[BUFSIZ+1];

            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            buf[r] = 0;

            /// EOF ///
            if(r == 0) {
                break;
            }
            else {
                int len = strlen(self->mBuffer) + strlen(buf) + 1;
                if(len < self->mMallocSize) {
                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
                else {
                    self->mMallocSize = len * 2;

                    int n = self->mPtr - self->mBuffer;
                    self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                    self->mPtr = self->mBuffer + n;

                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
            }
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファから一行を得て返す ///
        if(lf == kLF) {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\r' && *(self->mPtr + 1) == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\r') {
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else {
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\a') {
                    string_push_back2(str, *self->mPtr++);
                    break;
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
    }
}

// 0: 正常終了 1: EOF -1:エラー
int sRFd_read_oneline_preserve(sRFd* self, string_obj* str, enum eLineField lf)
{
    if(self->mFd == -1) {
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return -1;
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファから一行を得て返す ///
        if(lf == kLF) {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\n') {
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r' && *(p + 1) == '\n') {
                    string_push_back2(str, *p++);
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r') {
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\a') {
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
    }
    else {
        while(1) {
            /// 改行が含まれていればbreakする ///
            if(lf == kCR) {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kLF) {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\n') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kCRLF) {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r' && *(p+1) == '\n') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else {
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\a') {
                        flg = 1;
                        break;
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }

            char buf[BUFSIZ+1];

            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            buf[r] = 0;

            /// EOF ///
            if(r == 0) {
                break;
            }
            else {
                int len = strlen(self->mBuffer) + strlen(buf) + 1;
                if(len < self->mMallocSize) {
                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
                else {
                    self->mMallocSize = len * 2;

                    int n = self->mPtr - self->mBuffer;
                    self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                    self->mPtr = self->mBuffer + n;

                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
            }
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファから一行を得て返す ///
        if(lf == kLF) {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\n') {
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r' && *(p + 1) == '\n') {
                    string_push_back2(str, *p++);
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r') {
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else {
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\a') {
                    string_push_back2(str, *p++);
                    break;
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
    }
}

// 0: 正常終了 1: EOF -1:エラー
int sRFd_read_oneline_num(sRFd* self, string_obj* str, int line_num, enum eLineField lf)
{
    if(self->mFd == -1) {
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return -1;
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn行を得て返す ///
        if(lf == kLF) {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\r' && *(self->mPtr + 1) == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr ==  '\r') {
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\a') {
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
    }
    else {
        while(1) {
            /// 改行がn個含まれていればbreakする ///
            if(lf == kCR) {
                int flg = 0;
                int c = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kLF) {
                int c;
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\n') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kCRLF) {
                int c;
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r' && *(p+1) == '\n') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else {
                int flg = 0;
                char* p = self->mPtr;
                int c = 0;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\a') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }

            char buf[BUFSIZ+1];

            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            buf[r] = 0;

            /// EOF ///
            if(r == 0) {
                break;
            }
            else {
                int len = strlen(self->mBuffer) + strlen(buf) + 1;
                if(len < self->mMallocSize) {
                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
                else {
                    self->mMallocSize = len * 2;

                    int n = self->mPtr - self->mBuffer;
                    self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                    self->mPtr = self->mBuffer + n;

                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
            }
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn行を得て返す ///
        if(lf == kLF) {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\r' && *(self->mPtr + 1) == '\n') {
                    string_push_back2(str, *self->mPtr++);
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\r') {
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
        else {
            int c = 0;
            while(*self->mPtr) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*self->mPtr == '\a') {
                    string_push_back2(str, *self->mPtr++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *self->mPtr++);
                }
            }

            return 0;
        }
    }
}

// 0: 正常終了 1: EOF -1:エラー
int sRFd_read_oneline_preserve_num(sRFd* self, string_obj* str, int line_num, enum eLineField lf)
{
    if(self->mFd == -1) {
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return -1;
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn行を得て返す ///
        if(lf == kLF) {
            int c = 0;
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\n') {
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            char* p = self->mPtr;
            int c = 0;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r' && *(p + 1) == '\n') {
                    string_push_back2(str, *p++);
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            int c = 0;
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r') {
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else {
            char* p = self->mPtr;
            int c = 0;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\a') {
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
    }
    else {
        while(1) {
            /// 改行がn個含まれていればbreakする ///
            if(lf == kCR) {
                int flg = 0;
                int c = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kLF) {
                int c;
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\n') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else if(lf == kCRLF) {
                int c;
                int flg = 0;
                char* p = self->mPtr;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\r' && *(p+1) == '\n') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }
            else {
                int flg = 0;
                char* p = self->mPtr;
                int c = 0;
                while(*p) {
                    if(gKitutukiSigInt) {
                        gKitutukiSigInt = FALSE;
                        return -1;
                    }
                    if(*p == '\a') {
                        c++;
                        if(c == line_num) {
                            flg = 1;
                            break;
                        }
                    }
                    else {
                        p++;
                    }
                }
                if(flg) break;
            }

            char buf[BUFSIZ+1];

            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            buf[r] = 0;

            /// EOF ///
            if(r == 0) {
                break;
            }
            else {
                int len = strlen(self->mBuffer) + strlen(buf) + 1;
                if(len < self->mMallocSize) {
                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
                else {
                    self->mMallocSize = len * 2;

                    int n = self->mPtr - self->mBuffer;
                    self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                    self->mPtr = self->mBuffer + n;

                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
            }
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn行を得て返す ///
        if(lf == kLF) {
            int c = 0;
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\n') {
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else if(lf == kCRLF) {
            char* p = self->mPtr;
            int c = 0;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r' && *(p + 1) == '\n') {
                    string_push_back2(str, *p++);
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        
        else if(lf == kCR) {
            int c = 0;
            char* p = self->mPtr;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\r') {
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
        else {
            char* p = self->mPtr;
            int c = 0;
            while(*p) {
                if(gKitutukiSigInt) {
                    gKitutukiSigInt = FALSE;
                    return -1;
                }
                if(*p == '\a') {
                    string_push_back2(str, *p++);
                    c++;
                    if(c == line_num) {
                        break;
                    }
                }
                else {
                    string_push_back2(str, *p++);
                }
            }

            return 0;
        }
    }
}


// 0: 正常終了 1: EOF -1:エラー
int sRFd_read_onechar_preserve_num(sRFd* self, string_obj* str, int char_num)
{
    if(self->mFd == -1) {
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return -1;
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn文字を得て返す ///
        char* p = self->mPtr;
        int c = 0;
        while(*p) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            string_push_back2(str, *p++);
            c++;
            if(c >= char_num) {
                break;
            }
        }

        return 0;
    }
    else {
        while(1) {
            /// 文字がchar_num個含まれていればbreakする ///
            if(strlen(self->mPtr) >= char_num) {
                break;
            }

            char buf[BUFSIZ+1];

            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            buf[r] = 0;

            /// EOF ///
            if(r == 0) {
                break;
            }
            else {
                int len = strlen(self->mBuffer) + strlen(buf) + 1;
                if(len < self->mMallocSize) {
                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
                else {
                    self->mMallocSize = len * 2;

                    int n = self->mPtr - self->mBuffer;
                    self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                    self->mPtr = self->mBuffer + n;

                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
            }
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn個を得て返す ///
        int c = 0;
        char* p = self->mPtr;
        while(*p) {
            string_push_back2(str, *p++);
            c++;
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            if(c >= char_num) {
                break;
            }
        }

        return 0;
    }
}

// 0: 正常終了 1: EOF -1:エラー
int sRFd_read_onechar_num(sRFd* self, string_obj* str, int char_num)
{
    if(self->mFd == -1) {
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return -1;
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn文字を得て返す ///
        int c = 0;
        while(*self->mPtr) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            string_push_back2(str, *self->mPtr++);
            c++;
            if(c >= char_num) {
                break;
            }
        }




        return 0;
    }
    else {
        while(1) {
            /// 文字がchar_num個含まれていればbreakする ///
            if(strlen(self->mPtr) >= char_num) {
                break;
            }

            char buf[BUFSIZ+1];

            int r = read(self->mFd, buf, BUFSIZ);

            if(r < 0) {
                return -1;
            }
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            buf[r] = 0;

            /// EOF ///
            if(r == 0) {
                break;
            }
            else {
                int len = strlen(self->mBuffer) + strlen(buf) + 1;
                if(len < self->mMallocSize) {
                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
                else {
                    self->mMallocSize = len * 2;

                    int n = self->mPtr - self->mBuffer;
                    self->mBuffer = REALLOC(self->mBuffer, self->mMallocSize);
                    self->mPtr = self->mBuffer + n;

                    xstrncat(self->mBuffer, buf, self->mMallocSize);
                }
            }
        }

        /// バッファーが無い
        if(*self->mPtr == 0) {
            return 1;
        }

        /// 取ったバッファからn個を得て返す ///
        int c = 0;
        while(*self->mPtr) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return -1;
            }
            string_push_back2(str, *self->mPtr++);
            c++;
            if(c >= char_num) {
                break;
            }
        }

        return 0;
    }
}

void sRFd_set_buffer(sRFd* self, char* buffer)
{
    FREE(self->mBuffer);
    self->mBuffer = STRDUP(buffer);
    self->mMallocSize = strlen(buffer) + 1;
    self->mPtr = self->mBuffer;
}

#ifndef MDEBUG

sWFd* sWFd_new(int fd)
{
    sWFd* self = MALLOC(sizeof(sWFd));

    self->mFd = fd;
    self->mBuffer = MALLOC(1024);
    *self->mBuffer = 0;
    self->mP = self->mBuffer;
    self->mMallocSize = 1024;

    return self;
}

#else

sWFd* sWFd_new_debug(int fd, const char* fname, int line, const char* func)
{
    sWFd* self = CheckMemLeak_Malloc(sizeof(sWFd), fname, line, func);

    self->mFd = fd;
    self->mBuffer = MALLOC(1024);
    *self->mBuffer = 0;
    self->mP = self->mBuffer;
    self->mMallocSize = 1024;

    return self;
}

#endif

void sWFd_delete(sWFd* self)
{
    if(self->mBuffer) FREE(self->mBuffer);   // self->mBufferが他に代入されて、saphire_vm.cでsRFd_newに渡している。NULLが入る可能性があり
    FREE(self);
}

BOOL sWFd_close(sWFd* self)
{
    if(self->mFd != -1) {
        if(!sWFd_flash(self)) {
            return FALSE;
        }
        if(self->mFd == 1) {
            return TRUE;
        }
        else {
            if(close(self->mFd) < 0) {
                return FALSE;
            }
        }
    }

    return TRUE;
}

BOOL sWFd_push_back(sWFd* self, char* str)
{
    const int len = strlen(str);
    const int new_size = (self->mP - self->mBuffer) + len + 1;
    if(new_size < self->mMallocSize) {
        while(*str) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                return FALSE;
            }
            *(self->mP)++ = *str++;
        }
        *(self->mP) = 0;
    }
    else {
        while(new_size > self->mMallocSize) {
            self->mMallocSize *= 2;
       }

        char* old_buffer = self->mBuffer;
        self->mBuffer = (char*)MALLOC(self->mMallocSize);

        if(!memcpy_cancelable(self->mBuffer, old_buffer, self->mP - old_buffer))
        {
            FREE(old_buffer);
            return FALSE;
        }
        self->mP = self->mBuffer + (self->mP - old_buffer);

        while(*str) {
            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                FREE(old_buffer);
                return FALSE;
            }
            *self->mP++ = *str++;
        }
        *self->mP = 0;

        FREE(old_buffer);
    }

    if(self->mFd == 1) {
        return sWFd_flash(self);
    }
    else {
        if(self->mP - self->mBuffer > BUFSIZ) {
            return sWFd_flash(self);
        }
    }

    return TRUE;
}

BOOL sWFd_flash(sWFd* self)
{
    if(self->mFd != -1) {
        char* p = self->mBuffer;
        char* str_size = p + strlen(p);

        while(*p) {
            int size;
            if(str_size - p < BUFSIZ) {
                size = strlen(p);
            }
            else {
                size = BUFSIZ;
            }

            if(write(self->mFd, p, size) < 0) {
                if(errno == EINTR) {
                    self->mP = self->mBuffer;
                    *self->mBuffer = 0;
                    return FALSE;
                }
                else if(errno = EPIPE) {
                    break;
                }
                else {
//printf("self->mFd %d\n", self->mFd);
                    perror("write(sWFd_flash)2");
                    return FALSE;
                }
            }

            if(gKitutukiSigInt) {
                gKitutukiSigInt = FALSE;
                self->mP = self->mBuffer;
                *self->mBuffer = 0;
                return FALSE;
            }

            p+=size;
        }

        self->mP = self->mBuffer;
        *self->mBuffer = 0;
    }

    return TRUE;
}

vector_obj* gSaphireVarCompletions;               // 環境変数補完候補

void saphire_make_var_completion_list()
{
    int i;
    for(i=0; i<vector_size(gSaphireVarCompletions); i++) {
        string_delete(vector_item(gSaphireVarCompletions, i));
    }
    vector_clear(gSaphireVarCompletions);

    /// 環境変数 ///
#if HAVE_DECL_ENVIRON
    extern char **environ;

    char** p;
    for(p = environ; *p; p++) {
        char env_name[1024];

        char* p2 = env_name;
        char* p3 = *p;

        while(*p3 != '=') {
            p3++;
        }

        if(p3 - *p < 1024) {
            memcpy(env_name, *p, p3 - *p);
            env_name[p3 - *p] = 0;
        }

        vector_add(gSaphireVarCompletions, STRING_NEW(env_name));
    }
#endif

    hash_it* it = hash_loop_begin(gGlobals);
    while(it) {
        char* key = hash_loop_key(it);

        char env_name[1024];
        xstrncpy(env_name, key, 1024);

        vector_add(gSaphireVarCompletions, STRING_NEW(env_name));

        it = hash_loop_next(it);
    }

    it = hash_loop_begin(gArrays);
    while(it) {
        char* key = hash_loop_key(it);

        char env_name[1024];
        xstrncpy(env_name, key, 1024);

        vector_add(gSaphireVarCompletions, STRING_NEW(env_name));

        it = hash_loop_next(it);
    }

    it = hash_loop_begin(gHashs);
    while(it) {
        char* key = hash_loop_key(it);

        char env_name[1024];
        xstrncpy(env_name, key, 1024);

        vector_add(gSaphireVarCompletions, STRING_NEW(env_name));

        it = hash_loop_next(it);
    }

    (void)vector_sort(gSaphireVarCompletions, name_sort);
}

/////////////////////////////////////////////////////////////////////////
// ブロック
/////////////////////////////////////////////////////////////////////////
sBlock* sBlock_new(MANAGED sStatments* statments, MANAGED vector_obj* args, int count)
{
    sBlock* self = MALLOC(sizeof(sBlock));

    self->mStatments = statments;
    self->mArgs = args;
    self->mCount = count;

    return self;
}

void sBlock_delete(sBlock* self)
{
    sStatments_delete(self->mStatments);
    int i;
    for(i=0; i<vector_size(self->mArgs); i++) {
        string_delete(vector_item(self->mArgs, i));
    }
    vector_delete(self->mArgs);
    FREE(self);
}
    

sBlock* sBlock_new2(sBlock* block)
{
    sBlock* self = MALLOC(sizeof(sBlock));

    self->mStatments = STATMENTS_NEW2(block->mStatments);
    self->mArgs = VECTOR_NEW(10);
    int i;
    for(i=0; i<vector_size(block->mArgs); i++) {
        vector_add(self->mArgs, STRING_NEW(string_c_str(vector_item(block->mArgs, i))));
    }
    self->mCount = block->mCount;

    return self;
}

sBlock* sBlock_load(int fd)
{
    sBlock* self = MALLOC(sizeof(sBlock));
    
    self->mStatments = sStatments_load(fd);

    self->mArgs = VECTOR_NEW(50);
    int n;
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    int i;
    for(i=0; i<n; i++) {
        int n;
        if(read(fd, &n, sizeof(int)) < 0) {
            perror("read");
            exit(1);
        }
        
        char* buf = MALLOC(n+1);
        if(read(fd, buf, n) < 0) {
            perror("read");
            exit(1);
        }
        buf[n] = 0;
        string_obj* buf2 = STRING_NEW(buf);
        
        vector_add(self->mArgs, buf2);
    }

    /// mCount ///
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mCount = n;

    return self;
}

void sBlock_save(sBlock* self, int fd)
{
    sStatments_save(self->mStatments, fd);

    /// mArgs ///
    int n = vector_size(self->mArgs);
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }

    int i;
    for(i=0; i<n; i++) {
        char* str = string_c_str(vector_item(self->mArgs, i));

        int n = strlen(str);
        if(write(fd, &n, sizeof(int)) < 0) {
            perror("write");
            exit(1);
        }
        if(write(fd, str, n) < 0) {
            perror("write");
            exit(1);
        }
    }

    /// mCount ///
    n = self->mCount;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
}

vector_obj* saphire_methods_names(sObject* object)
{
    vector_obj* v = VECTOR_NEW(50);
    hash_it* it = hash_loop_begin(object->mMethods);
    while(it) {
        sFunction* fun = hash_loop_item(it);

        vector_add(v, STRING_NEW(string_c_str(fun->name)));

        it = hash_loop_next(it);
    }

    vector_add(v, STRING_NEW("class"));

    return v;
}

sObject* saphire_get_object(char* name)
{
    return hash_item(gObjects, name);
}

sRef* saphire_get_ref(char* name)
{
    return hash_item(gRefs, name);
}

string_obj* sVar_new(char* str)
{
    const int len = strlen(str);
    
    string_obj* self = memchecker_malloc(sizeof(string_obj), kMCVar);
    
    self->mMallocLen = (len < 128) ? 128 : (len + 1) * 2;
    self->mStr = (char*)MALLOC(sizeof(char)*self->mMallocLen);
    xstrncpy(self->mStr, str, self->mMallocLen);
    
    self->mLen = len;

    return self;
}

void sVar_delete(string_obj* self)
{
    FREE(self->mStr);
    memchecker_free(self);
    FREE(self);
}

string_obj* sGlobal_new(char* str, int ref_count)
{
    const int len = strlen(str);
    
    string_obj* self = memchecker_malloc(sizeof(sGlobal), kMCGlobal);
    
    self->mMallocLen = (len < 128) ? 128 : (len + 1) * 2;
    self->mStr = (char*)MALLOC(sizeof(char)*self->mMallocLen);
    xstrncpy(self->mStr, str, self->mMallocLen);
    
    self->mLen = len;

    ((sGlobal*)self)->mRefCount = ref_count;
    if(ref_count == 0) {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self);
        hash_put(gGlobalPool, buf, self);
    }

    return self;
}

void sGlobal_delete(string_obj* self)
{
    FREE(self->mStr);
    memchecker_free(self);
    FREE(self);
}

vector_obj* sAry_new(int first_size, int ref_count)
{
   vector_obj* self = (vector_obj*)memchecker_malloc(sizeof(sAry), kMCArray);

   self->mTable = (void**)MALLOC(sizeof(void*) * first_size);
   self->mTableSize = first_size;
   self->mCount = 0;

   ((sAry*)self)->mRefCount = ref_count;
   if(ref_count == 0) {
       char buf[512];
       snprintf(buf, 512, "%d", (unsigned int)self);
       hash_put(gAryPool, buf, self);
   }

   return self;
}

void sAry_delete(vector_obj* self)
{
    FREE(self->mTable);
    memchecker_free(self);
    FREE(self);
}

hash_obj* sHash_new(int size, int ref_count)
{
   hash_obj* self = (hash_obj*)memchecker_malloc(sizeof(sHash), kMCHash);
   
   self->mTableSize = size;
   self->mTable = (hash_it**)MALLOC(sizeof(hash_it*) * size);
   memset(self->mTable, 0, sizeof(hash_it*)*size);

   self->mEntryIt = NULL;

   self->mCounter = 0;

   ((sHash*)self)->mRefCount = ref_count;
    if(ref_count == 0) {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self);
        hash_put(gHashPool, buf, self);
    }

   return self;
}

void sHash_delete(hash_obj* self)
{
    memchecker_free(self);
    hash_delete(self);
}

sRef* sRef_new(void* mem, enum eMemCheckerKind kind)
{
    sRef* self = memchecker_malloc(sizeof(sRef), kMCRef);

    self->mMem = mem;
    self->mKind = kind;

    int kind2 = memchecker_is_enable_mem(self->mMem);

    switch(kind2) {
    case kMCArray: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sAry* ary = hash_item(gAryPool, buf);
        if(ary) {
            ary->mRefCount++;
        }
        }
        break;
        
    case kMCObject: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sObject* object = hash_item(gObjectPool, buf);
        if(object) {
            object->mRefCount++;
        }
        }
        break;
        
    case kMCHash: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sHash* hash = hash_item(gHashPool, buf);
        if(hash) {
            hash->mRefCount++;
        }
        }
        break;
        
    case kMCGlobal: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sGlobal* global = hash_item(gGlobalPool, buf);
        if(global) {
            global->mRefCount++;
        }
        }
        break;
        
    case kMCFunction: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sFunction* fun = hash_item(gFunPool, buf);
        if(fun) {
            fun->mRefCount++;
        }
        }
        break;
        
    case kMCClass: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sClass* klass = hash_item(gClassPool, buf);
        if(klass) {
            klass->mRefCount++;
        }
        }
        break;
    }

    return self;
}

void sRef_delete(sRef* self)
{
    int kind2 = memchecker_is_enable_mem(self->mMem);

    switch(kind2) {
    case kMCArray: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sAry* ary = hash_item(gAryPool, buf);
        if(ary) {
            ary->mRefCount--;
        }
        }
        break;
        
    case kMCObject: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sObject* object = hash_item(gObjectPool, buf);
        if(object) {
            object->mRefCount--;
        }
        }
        break;
        
    case kMCHash: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sHash* hash = hash_item(gHashPool, buf);
        if(hash) {
            hash->mRefCount--;
        }
        }
        break;
        
    case kMCGlobal: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sGlobal* global = hash_item(gGlobalPool, buf);
        if(global) {
            global->mRefCount--;
        }
        }
        break;
        
    case kMCFunction: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sFunction* fun = hash_item(gFunPool, buf);
        if(fun) {
            fun->mRefCount--;
        }
        }
        break;
        
    case kMCClass: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sClass* klass = hash_item(gClassPool, buf);
        if(klass) {
            klass->mRefCount--;
        }
        }
        break;
    }
    memchecker_free(self);
    FREE(self);
}

void sRef_inc_refcount(sRef* self) 
{
    int kind2 = memchecker_is_enable_mem(self->mMem);

    switch(kind2) {
    case kMCArray: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sAry* ary = hash_item(gAryPool, buf);
        if(ary) {
            ary->mRefCount++;
        }
        }
        break;
        
    case kMCObject: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sObject* object = hash_item(gObjectPool, buf);
        if(object) {
            object->mRefCount++;
        }
        }
        break;
        
    case kMCHash: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sHash* hash = hash_item(gHashPool, buf);
        if(hash) {
            hash->mRefCount++;
        }
        }
        break;
        
    case kMCGlobal: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sGlobal* global = hash_item(gGlobalPool, buf);
        if(global) {
            global->mRefCount++;
        }
        }
        break;
        
    case kMCFunction: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sFunction* fun = hash_item(gFunPool, buf);
        if(fun) {
            fun->mRefCount++;
        }
        }
        break;
        
    case kMCClass: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sClass* klass = hash_item(gClassPool, buf);
        if(klass) {
            klass->mRefCount++;
        }
        }
        break;
    }
}

void sRef_dec_refcount(sRef* self) 
{
    int kind2 = memchecker_is_enable_mem(self->mMem);

    switch(kind2) {
    case kMCArray: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sAry* ary = hash_item(gAryPool, buf);
        if(ary) {
            ary->mRefCount--;
        }
        }
        break;
        
    case kMCObject: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sObject* object = hash_item(gObjectPool, buf);
        if(object) {
            object->mRefCount--;
        }
        }
        break;
        
    case kMCHash: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sHash* hash = hash_item(gHashPool, buf);
        if(hash) {
            hash->mRefCount--;
        }
        }
        break;
        
    case kMCGlobal: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sGlobal* global = hash_item(gGlobalPool, buf);
        if(global) {
            global->mRefCount--;
        }
        }
        break;
        
    case kMCFunction: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sFunction* fun = hash_item(gFunPool, buf);
        if(fun) {
            fun->mRefCount--;
        }
        }
        break;
        
    case kMCClass: {
        char buf[512];
        snprintf(buf, 512, "%d", (unsigned int)self->mMem);
        sClass* klass = hash_item(gClassPool, buf);
        if(klass) {
            klass->mRefCount--;
        }
        }
        break;
    }
}

sAtCommand* sAtCommand_new(MANAGED sStatments* statments, enum eLineField lf)
{
    sAtCommand* self = MALLOC(sizeof(sAtCommand));

    self->mStatments = statments;
    self->mLineField = lf;
    
    return self;
}

void sAtCommand_delete(sAtCommand* self)
{
    sStatments_delete(self->mStatments);
    FREE(self);
}

sAtCommand* sAtCommand_new2(sAtCommand* at_command)
{
    sAtCommand* self = MALLOC(sizeof(sAtCommand));
    
    self->mStatments = STATMENTS_NEW2(at_command->mStatments);
    self->mLineField = at_command->mLineField;

    return self;
}

sAtCommand* sAtCommand_save(sAtCommand* self, int fd)
{
    sStatments_save(self->mStatments, fd);

    int n = self->mLineField;
    if(write(fd, &n, sizeof(int)) < 0) {
        perror("write");
        exit(1);
    }
}

sAtCommand* sAtCommand_load(int fd)
{
    sAtCommand* self = MALLOC(sizeof(sAtCommand));

    self->mStatments = sStatments_load(fd);

    int n;
    if(read(fd, &n, sizeof(int)) < 0) {
        perror("read");
        exit(1);
    }
    self->mLineField = n;

    return self;
}

void dig_stack_frame(char* sname, int sline, vector_obj** fun_argv_before, vector_obj* argv2)
{
    // スタックを掘る
    vector_add(gStackFrame, HASH_NEW(30));
    vector_add(gStackFrameHashs, HASH_NEW(30));
    vector_add(gStackFrameArrays, HASH_NEW(30));
    vector_add(gStackFrameObjects, HASH_NEW(30));
    vector_add(gStackFrameRefs, HASH_NEW(30));

    /// スタックフレームの記録を付ける
    vector_add(gStackTraceFName, STRING_NEW(sname));
    vector_add(gStackTraceLineNum, (void*)sline);

    /// 引数を渡す ///
    *fun_argv_before = hash_item(gArrays, "ARGV");
    hash_put(gArrays, "ARGV", argv2);
}

void burry_stack_frame(vector_obj* fun_argv_before, string_obj** sname_strace, int* line_strace)
{
    // スタックを埋める
    hash_obj* h = vector_pop_back(gStackFrame);
    hash_it* it = hash_loop_begin(h);
    while(it != NULL) {
        sVar_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(h);
    h =vector_pop_back(gStackFrameArrays);
    it = hash_loop_begin(h);
    while(it != NULL) {
        vector_obj* v = hash_loop_item(it);
        int i;
        for(i=0; i<vector_size(v); i++) {
            string_delete(vector_item(v, i));
        }
        sAry_delete(v);
        it = hash_loop_next(it);
    }
    hash_delete(h);
    h =vector_pop_back(gStackFrameObjects);
    it = hash_loop_begin(h);
    while(it != NULL) {
        sObject_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(h);
    h =vector_pop_back(gStackFrameHashs);
    it = hash_loop_begin(h);
    while(it != NULL) {
        hash_obj* h2 = hash_loop_item(it);
        hash_it* it2 = hash_loop_begin(h2);
        while(it2) {
            string_delete(hash_loop_item(it2));
            it2 = hash_loop_next(it2);
        }
        sHash_delete(h2);
        it = hash_loop_next(it);
    }
    hash_delete(h);
    h =vector_pop_back(gStackFrameRefs);
    it = hash_loop_begin(h);
    while(it != NULL) {
        sRef_delete(hash_loop_item(it));
        it = hash_loop_next(it);
    }
    hash_delete(h);

    /// 引数を戻す
    vector_obj* argv2 = hash_item(gArrays, "ARGV");
    int k;
    for(k=0; k<vector_size(argv2); k++) {
        string_delete(vector_item(argv2, k));
    }
    sAry_delete(argv2);
    
    if(fun_argv_before) {
        hash_put(gArrays, "ARGV", fun_argv_before);
    }
    else {
        hash_erase(gArrays, "ARGV");
    }

    /// スタックフレームの記録を消す
    *sname_strace = vector_pop_back(gStackTraceFName);
    *line_strace = (int)vector_pop_back(gStackTraceLineNum);
}

BOOL memcpy_cancelable(char* mem, char* mem2, size_t size)
{
    char* p = mem;
    char* p2 = mem2;
    while(1) {
        if(gKitutukiSigInt) {
            gKitutukiSigInt = FALSE;
            return FALSE;
        }

        if(mem + size - p < BUFSIZ) {
            memcpy(p, p2, mem + size - p);
            break;
        }
        else {
            memcpy(p, p2, BUFSIZ);
            p += BUFSIZ;
            p2 += BUFSIZ;
        }
    }

    return TRUE;
}

BOOL string_put_cancelable(string_obj* self, char* str)
{
    const int len = strlen(str);

    if(len+1 < self->mMallocLen) {
        if(!memcpy_cancelable(self->mStr, str, len + 1)) {
            *self->mStr = 0;
            self->mLen = 0;
            return FALSE;
        }
        self->mLen = len;
    }
    else {
        FREE(self->mStr);
        
        self->mMallocLen = (len < 16) ? 16 : (len + 1) * 2;
        self->mStr = (char*)MALLOC(sizeof(char)*self->mMallocLen);
       
        if(!memcpy_cancelable(self->mStr, str, len + 1)) {
            *self->mStr = 0;
            self->mLen = 0;
            return FALSE;
        }
        self->mLen = len;
    }

    return TRUE;
}

BOOL string_push_back_cancelable(string_obj* self, char* str)
{
    char* new_str;
    const int str_len = strlen(str);

    if(self->mLen + str_len + 1 < self->mMallocLen) {
        if(!memcpy_cancelable(self->mStr + strlen(self->mStr), str, str_len + 1))
        {
            *self->mStr = 0;
            self->mLen = 0;
            return FALSE;
        }
        self->mLen += str_len;
    }
    else {
        int malloc_len = (self->mLen + str_len + 1) * 2;
        new_str = (char*)MALLOC(malloc_len);

        if(!memcpy_cancelable(new_str, self->mStr, strlen(self->mStr) + 1) ){
            FREE(new_str);
            return FALSE;
        }
        if(!memcpy_cancelable(new_str + strlen(new_str), str, strlen(str) + 1))
        {
            FREE(new_str);
            return FALSE;
        }

        FREE(self->mStr);

        self->mMallocLen = malloc_len;
        self->mStr = new_str;
        self->mLen = self->mLen + str_len;
    }

    return TRUE;
}

static BOOL quick_sort_cancelable(vector_obj* self, int left, int right, sort_if fun)
{
    int i;
    int j;
    void* center_item;

    if(left < right) {
        center_item = self->mTable[(left+right) / 2];

        i = left;
        j = right;

        do { 
            while(1) {
                int ret = fun(self->mTable[i], center_item);
                if(ret < 0) return FALSE;
                if(self->mTable[i]==center_item || !ret)
                {
                    break;
                }
                i++;
                if(gKitutukiSigInt) {
                    return FALSE;
                }
            }
                     
            while(1) {
                int ret = fun(center_item, self->mTable[j]);
                if(ret < 0) return FALSE;
                if(center_item==self->mTable[j] || !ret)
                {
                    break;
                }
                j--;
                if(gKitutukiSigInt) {
                    return FALSE;
                }
            }

            if(i <= j) {
                void* tmp = self->mTable[i]; // swap
                self->mTable[i] = self->mTable[j];
                self->mTable[j] = tmp;

                i++;
                j--;
            }
            if(gKitutukiSigInt) {
                return FALSE;
            }
        } while(i <= j);

        if(!quick_sort_cancelable(self, left, j, fun)) {
            return FALSE;
        }
        if(!quick_sort_cancelable(self, i, right, fun)) {
            return FALSE;
        }
    }

    return TRUE;
}

BOOL vector_sort_cancelable(vector_obj* self, sort_if fun)
{
    if(!quick_sort_cancelable(self, 0, self->mCount-1, fun)) {
        gKitutukiSigInt = FALSE;
        return FALSE;
    }

    return TRUE;
}

string_obj* string_new_cancelable(char* str)
{
    const int len = strlen(str);
    
    string_obj* self = (string_obj*)MALLOC(sizeof(string_obj));
    
    self->mMallocLen = (len < 127) ? 128 : (len + 1) * 2;
    self->mStr = (char*)MALLOC(sizeof(char)*self->mMallocLen);
    if(!memcpy_cancelable(self->mStr, str, strlen(str) + 1)) {
        *self->mStr = 0;
        self->mLen = 0;
        return NULL;
    }
    self->mLen = len;

    return self;
}

int str_kanjilen_cancelable(enum eKanjiCode code, char* mbs)
{
    if(code == kUtf8) {
        int n = 0;
        char* p = mbs;

        char* max = strlen(mbs) + mbs;
        while(p < max) {
            if(((unsigned char)*p) > 127) {
                int size = ((*p & 0x80) >> 7) + ((*p & 0x40) >> 6) + ((*p & 0x20) >> 5) + ((*p & 0x10) >> 4);
                p+= size;
                n++;
            }
            else {
                p++;
                n++;
            }
            if(gKitutukiSigInt) {
                return -1;
            }
        }

        return n;
    }
    else {
        int result = 0;
        char* p = mbs;
        while(*p) {
            if(is_kanji(code, *p) && *(p+1) != 0) {
                p+=2;
                result++;
            }
            else {
                p++;
                result++;
            }
            if(gKitutukiSigInt) {
                return -1;
            }
        }
        return result;
    }
}
