/**
 * $Id: sc_hash.c,v 1.7 2004/08/05 13:54:30 jklein Exp $
 * $Author: jklein $
 * $Log: sc_hash.c,v $
 * Revision 1.7  2004/08/05 13:54:30  jklein
 * Fixed so that Scale can be compiled on Mac OSX(10.3 or higher).
 *
 * Revision 1.6  2004/08/04 08:55:28  jklein
 * fixed off-by-one bug in sc_buffer.c
 *
 * Revision 1.5  2004/08/03 05:49:50  jklein
 * Add sxtable_t structure.
 *
 * Revision 1.4  2004/07/10 17:04:48  jklein
 * Added few function to sxarray_t.
 *
 * Revision 1.3  2004/06/21 16:44:17  jklein
 * Added iterator.
 *
 */


#include "scale/hash.h"
#include "scale/alloc.h"
#include "scale/xtable.h"


typedef struct _sc_hash_elt shash_elt_t;

struct _sc_hash_elt {
	spointer key;
	spointer obj;

	shash_elt_t *next;	/* this is for hash_elt which has same hash-key */
};

struct _sc_hash_table {
	shash_elt_t **table;
	size_t count;
	size_t hashsize;
	hashfunc_t hash;
	sc_compf_t cmp;
};


static void
_s_hash_remove_by_key(shash_t *htable,
					  const spointer key,
					  unsigned int hashkey)
{
	shash_elt_t *elt = NULL, *prev = NULL;

	for(elt = htable->table[hashkey]; elt; elt=elt->next) {
		if(htable->cmp(elt->key, key) == 0) {
			if(prev)
				prev->next = elt->next;
			else
				htable->table[hashkey] = elt->next;

			s_freen(elt, sizeof(shash_elt_t));

			htable->count--;
			break;
		}
		prev = elt;
	}
}


shash_t *
s_hash_alloc(size_t hsize, hashfunc_t hfunc, sc_compf_t cmpf)
{
	shash_t *htable;

	htable = typed_malloc(shash_t);
	htable->hashsize = (hsize != 0)?hsize:DEFAULT_HASH_SIZE;
	htable->table = s_calloc(htable->hashsize, sizeof(shash_elt_t *));
	htable->hash = hfunc;
	htable->cmp = cmpf;
	htable->count = 0;

	return htable;
}


spointer
s_hash_get(shash_t *htable, const spointer key)
{
	unsigned int hashkey;
	shash_elt_t *elt;

	hashkey = htable->hash(key, htable->hashsize);

	for(elt = htable->table[hashkey]; elt; elt=elt->next) {
		if(htable->cmp(elt->key, key) == 0)
			return elt->obj;
	}
	
	return NULL;
}


void
s_hash_put(shash_t *htable, spointer key, spointer obj)
{
	shash_elt_t *next, *elt;
	unsigned int hashkey;

	elt = typed_malloc(shash_elt_t);
	elt->key = key;
	elt->obj = obj;

	hashkey = htable->hash(key, htable->hashsize);

	/* remove same key before adding new one. */
	_s_hash_remove_by_key(htable, key, hashkey);

	next = htable->table[hashkey];
	htable->table[hashkey] = elt;
	htable->table[hashkey]->next = next;
	htable->count++;
}



/* This function doesn't remove object.
 * It only takes out from a hash-table.
 * So, you have to free it in you own reponsibility. */
void
s_hash_remove(shash_t *htable, const spointer key)
{
	_s_hash_remove_by_key(htable, key, htable->hash(key, htable->hashsize));
}


size_t
s_hash_size(shash_t *htable)
{
	return htable->count;
}


void 
s_hash_free(shash_t *htable)
{
	s_hash_clear(htable);
	s_freen(htable->table, sizeof(shash_elt_t *) * htable->hashsize);
	s_freen(htable, sizeof(shash_t));
}


void
s_hash_clear(shash_t *htable)
{
	shash_elt_t *elt;
	int i;

	for (i=0; i<htable->hashsize; i++) {
		
		if ((elt = htable->table[i]) != NULL) {
			s_freen(elt, sizeof(shash_elt_t));
			htable->table[i] = NULL;
		}
	}

	htable->count = 0;
}


/* 
 * Alfred V. Aho, Ravi Sethi, Jeffrey D. Ullman 
 * ĸإѥ飲 ˡġ
 * Ҽڡ
 * ºݤ˼¸ơХ󥹤ɤϥåͤ뤳Ȥǧ
 */
unsigned int
s_string_hashfunc(const spointer key, size_t hashsize)
{
	unsigned int h, g;
	const unsigned char *p;

	h = g = 0;
	for (p=key; *p; ++p) {
		h = (h << 4) + (*p);
		if ((g = h & 0xf0000000)) {
			h = h ^ (g >> 24);
			h = h ^ g;
		}
	}
	return h % hashsize;
}


/* for iterator */

typedef struct {
	shash_elt_t *next;
	size_t cindex;
	size_t done;
} hash_itr_manager_t;


static spointer
_s_hash_iterator_next(siterator_t *itr)
{
	hash_itr_manager_t *mng = (hash_itr_manager_t *)itr->data;
	shash_t *htable = (shash_t *)itr->listdata;
	spointer key;

	if (s_hash_size(htable) == mng->done)
		return NULL;

	key = mng->next->key;
	mng->next = mng->next->next;

	/* find next element */
	while (mng->next == NULL && mng->cindex < htable->hashsize) {
		mng->next = htable->table[++mng->cindex];
	}

	mng->done++;

	return key;
}


static void
_s_hash_iterator_clear(siterator_t *itr)
{
	hash_itr_manager_t *mng = (hash_itr_manager_t *)itr->data;
	shash_t *htable = (shash_t *)itr->listdata;

	mng->done = 0;
	mng->cindex = 0;

	mng->next = htable->table[mng->cindex];

	/* find first element */
	while (mng->next == NULL && mng->cindex < htable->hashsize) {
		mng->next = htable->table[++mng->cindex];
	}
}


static void
_s_hash_iterator_free(siterator_t *itr)
{
	s_freen(itr->data, sizeof(hash_itr_manager_t));
	s_freen(itr, sizeof(siterator_t));
}


siterator_t *
s_hash_keys(shash_t *htable)
{
	siterator_t *itr;

	itr = typed_malloc(siterator_t);
	itr->listdata = htable;
	itr->next  = _s_hash_iterator_next;
	itr->clear = _s_hash_iterator_clear;
	itr->free  = _s_hash_iterator_free;

	itr->data = typed_malloc(hash_itr_manager_t);

	itr->clear(itr);

	return itr;
}


/* for xtable_t */


#define GET_HASH(xt) ((shash_t *)xt->tabledata)


static void
_s_hash_xtable_free(sxtable_t *xt)
{
	shash_t *hash = GET_HASH(xt);

	if (xt->is_origin)
		s_hash_free(hash);
	s_freen(xt, sizeof(sxtable_t));
}


static size_t
_s_hash_xtable_size(const sxtable_t *xt)
{
	return s_hash_size(GET_HASH(xt));
}


static void
_s_hash_xtable_put(sxtable_t *xt, spointer key, spointer contents)
{
	s_hash_put(GET_HASH(xt), key, contents);
}


static spointer
_s_hash_xtable_get(const sxtable_t *xt, const spointer key)
{
	return s_hash_get(GET_HASH(xt), key);
}


static void
_s_hash_xtable_remove(sxtable_t *xt, const spointer key)
{
	s_hash_remove(GET_HASH(xt), key);
}


static siterator_t *
_s_hash_xtable_keys(const sxtable_t *xt)
{
	return s_hash_keys(GET_HASH(xt));
}


sxtable_t *
s_hash_attach_xtable(shash_t *hash)
{
	sxtable_t *xt;

	xt = typed_malloc(sxtable_t);
	xt->tabledata = hash;
	xt->is_origin = false;
	xt->free = _s_hash_xtable_free;
	xt->size = _s_hash_xtable_size;
	xt->put  = _s_hash_xtable_put;
	xt->get  = _s_hash_xtable_get;
	xt->remove = _s_hash_xtable_remove;
	xt->keys = _s_hash_xtable_keys;

	return xt;
}


sxtable_t *
s_hash_xtable(size_t hsize, hashfunc_t hfunc, sc_compf_t cmpf)
{
	sxtable_t *xt;
	shash_t *hash;

	hash = s_hash_alloc(hsize, hfunc, cmpf);
	xt = s_hash_attach_xtable(hash);
	xt->is_origin = true;

	return xt;
}

