/**
 * $Id: bench.c,v 1.3 2004/08/05 13:54:30 jklein Exp $
 * $Author: jklein $
 * $Log: bench.c,v $
 * Revision 1.3  2004/08/05 13:54:30  jklein
 * Fixed so that Scale can be compiled on Mac OSX(10.3 or higher).
 *
 * Revision 1.2  2004/08/03 05:49:50  jklein
 * Add sxtable_t structure.
 *
 * Revision 1.1  2004/08/02 16:39:13  jklein
 * Fixed bug in the btree.c
 *
*/


/*
  ٥ޡ٤Υƥȥ
  ٥ޡƱˡñ٥ƥȤͤƤ롣
*/


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include "scale/commons.h"
#include "scale/alloc.h"
#include "scale/hash.h"
#include "scale/list.h"
#include "scale/heap.h"
#include "scale/vector.h"
#include "scale/btree.h"
#include "scale/stack.h"
#include "scale/queue.h"
#include "scale/buffer.h"


/* test-data list */
size_t tdsize = -1;
char **tdlist = NULL;


double 
bench_xarray(sxarray_t *xarr)
{
	clock_t stime, etime;
	int i;

	stime = clock();

	for (i=0; i<tdsize; ++i)
		s_xarray_add(xarr, tdlist[i]);
	for (i=tdsize-1; i>=0; --i)
		s_xarray_remove(xarr, i);

	etime = clock();

	return (etime - stime) / (double)CLOCKS_PER_SEC;
}


double 
bench_insert_at_xarray(sxarray_t *xarr)
{
	clock_t stime, etime;
	int i;

	stime = clock();

	for (i=0; i<tdsize; ++i)
		s_xarray_insert_at(xarr, 0, tdlist[i]);
	for (i=tdsize-1; i>=0; --i)
		s_xarray_remove(xarr, i);

	etime = clock();

	return (etime - stime) / (double)CLOCKS_PER_SEC;
}


int
bench_init(size_t tdc)
{
	char buf[32];
	int i;

	tdsize = tdc;
	memset(buf, 0x00, 32);
	buf[0] = 'a' - 1; /* init to before the 'a'. */

	tdlist = (char **)s_malloc(sizeof(char *) * tdsize);

	for (i=0; i<tdsize; i++) {
		snprintf(buf, 32, "%d", i);
		tdlist[i] = strdup(buf);
		
	}
	return 1;
}


void
finalize()
{
	int i;

	for (i=0; i<tdsize; ++i)
		free(tdlist[i]);
	s_freen(tdlist, sizeof(char *) * tdsize);
}


double 
bench_vector(size_t n)
{
	sxarray_t *xarr;
	double time;

	xarr = s_vector_xarrayn(n);
	time = bench_xarray(xarr);
	s_xarray_free(xarr);
	return time;
}


double
bench_list()
{
	sxarray_t *xarr;
	double time;

	xarr = s_list_xarray();
	time = bench_xarray(xarr);
	s_xarray_free(xarr);

	return time;
}


double
bench_heap()
{
	sheap_t *heap;
	double etime, stime;
	int i;

	heap = s_heap_alloc(s_string_compare);

	stime = clock();

	for (i=0; i<tdsize; ++i) 
		s_heap_push(heap, tdlist[i]);

	for (i=0; i<tdsize; ++i) {
		s_heap_pop(heap);
	}

	etime = clock();

	s_heap_free(heap);
	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_stack()
{
	sstack_t *stack;
	double stime, etime;
	int i;

	stack = s_stack_alloc();

	stime = clock();
	for (i=0; i<tdsize; ++i)
		s_stack_push(stack, tdlist[i]);
	for (i=0; i<tdsize; ++i)
		s_stack_pop(stack);
	etime = clock();

	s_stack_free(stack);
	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_hash(int hashsize)
{
	shash_t *hash;
	double stime, etime;
	int i;

	hash = s_hash_alloc(hashsize, s_string_hashfunc,
						s_string_compare);

	stime = clock();
	for (i=0; i<tdsize; ++i)
		s_hash_put(hash, tdlist[i], tdlist[i]);

	for (i=0; i<tdsize; ++i)
		s_hash_remove(hash, tdlist[i]);
	etime = clock();

	s_hash_free(hash);
	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_btree()
{
	sbtree_t *bt;
	double stime, etime;
	int i;

	bt = s_btree_alloc(s_string_compare);

	stime = clock();
	for (i=0; i<tdsize; ++i)
		s_btree_add(bt, tdlist[i], tdlist[i]);

	for (i=0; i<tdsize; ++i) 
		s_btree_remove(bt, tdlist[i]);
	etime = clock();

	s_btree_free(bt);
	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_insert_at_vector()
{
	sxarray_t *xarr;
	double time;

	xarr = s_vector_xarray();
	time = bench_insert_at_xarray(xarr);
	s_xarray_free(xarr);
	return time;
}


double
bench_sbuffer_strcat()
{
	sbuffer_t *sbuf = s_buffer_allocn(0x10);
	double stime, etime;
	int i;

	stime = clock();
	for (i=0; i<tdsize; i++)
		s_buffer_strcat(sbuf, tdlist[i]);
	etime = clock();

	s_buffer_free(sbuf);
	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_sbuffer_chrrep()
{
	sbuffer_t *sbuf = s_buffer_allocn(0x10);
	double stime, etime;
	int i;

	stime = clock();
	for (i=0; i<tdsize; i++)
		s_buffer_strcat(sbuf, tdlist[i]);
	for (i=0; i<tdsize; i++)
		s_buffer_chrrep(sbuf, (i%10)+'0', (i%10)+'1');
	etime = clock();
	
	s_buffer_free(sbuf);
	return (etime - stime) / CLOCKS_PER_SEC;
}



double
bench_sbuffer_strrep()
{
	sbuffer_t *sbuf = s_buffer_allocn(0x10);
	double stime, etime;
	int i;

	stime = clock();
	for (i=0; i<tdsize; i++)
		s_buffer_strcat(sbuf, tdlist[i]);
	for (i=0; i<tdsize; i++)
		s_buffer_strrep(sbuf, tdlist[i], "A");
	etime = clock();	

	s_buffer_free(sbuf);
	return (etime - stime) / CLOCKS_PER_SEC;
}


void
bench_sbuffer()
{
	int benchlist[] = {
		5000, 10000, -1
	};
	int i = 0, benchsize;

	while ( (benchsize = benchlist[i++]) != -1) {
		bench_init(benchsize);
		printf("[ count = %d ]\n", benchsize);
		printf("  strcat : %.3f\n", bench_sbuffer_strcat());
		printf("  chrrep : %.3f\n", bench_sbuffer_chrrep());
		printf("  strrep : %.3f\n", bench_sbuffer_strrep());
		printf("\n");
		finalize();
	}
}


void
bench_arrays(void)
{
	int benchlist[] = {
		100000, 500000, 1000000, -1
	};
	int i = 0, benchsize;

	printf("Array Benchmark\n");
	
	while ( (benchsize = benchlist[i++]) != -1) {
		bench_init(benchsize);
		printf("[ count = %d ]\n", benchsize);
		printf("  Vector        : %.4f\n", bench_vector(64));
//		printf("  Vector(ins at): %.4f\n", bench_insert_at_vector());
		printf("  Vector(large) : %.4f\n", bench_vector(tdsize));
		printf("  List          : %.4f\n", bench_list());
		printf("  Heap          : %.4f\n", bench_heap());
		printf("  Stack         : %.4f\n", bench_stack());
		finalize();
	}
}


void
bench_tables(void)
{
	int benchlist[] = {
		100000, 500000, -1
	};
	int i = 0, benchsize;

	printf("Table Benchmark\n");
	while ( (benchsize = benchlist[i++]) != -1) {
		bench_init(benchsize);
		printf("[ count = %d ]\n", benchsize);
		printf("  Hash          : %.4f\n", bench_hash(tdsize/2));
		printf("  Hash(large)   : %.4f\n", bench_hash(tdsize*2));
		printf("  BTree         : %.4f\n", bench_btree());
		finalize();
	}
}


int
main(int argc, char *argv[])
{
	bench_arrays();
	bench_tables();
	bench_sbuffer();

	return EXIT_SUCCESS;
}
