
#include "scale/hash.h"
#include "scale/alloc.h"


typedef struct _sc_hash_elt shash_elt_t;

struct _sc_hash_elt {
	spointer key;
	spointer obj;

	shash_elt_t *next;	/* this is for hash_elt which has same hash-key */
};

struct _sc_hash_table {
	shash_elt_t **table;
	size_t count;
	size_t hashsize;
	hashfunc_t hash;
	sc_compf_t cmp;
};

struct _sc_hash_iterate {
	const shash_t *htable;
	shash_elt_t *next;
	size_t current_index;
	size_t done;
};


shash_iterate_t *
s_hash_keys(const shash_t *htable)
{
	shash_iterate_t *itr;
	
	itr = s_malloc(sizeof(shash_iterate_t));

	itr->htable = htable;
	s_hash_iterate_clear(itr);

	return itr;
}


spointer
s_hash_iterate_next(shash_iterate_t *itr)
{
	spointer key;
	
	if (itr->next == NULL)
		return NULL;

	if (itr->htable->count <= itr->done)
		return NULL;

	key = itr->next->key;
	itr->next = itr->next->next;

	while (itr->next == NULL &&
		   ++itr->current_index < itr->htable->hashsize) {
		itr->next = itr->htable->table[itr->current_index];
	}

	itr->done++;

	return key;
}



void
s_hash_iterate_clear(shash_iterate_t *itr)
{
	itr->current_index = 0;	
	itr->next = itr->htable->table[itr->current_index];
	itr->done = 0;

	while (itr->next == NULL &&
		   ++itr->current_index < itr->htable->hashsize) {
		itr->next = itr->htable->table[itr->current_index];
	}
}

void
s_hash_iterate_free(shash_iterate_t *itr)
{
	s_freen(itr, sizeof(shash_iterate_t));
}


shash_t *
s_hash_alloc(size_t hsize, hashfunc_t hfunc, sc_compf_t cmpf)
{
	shash_t *htable;

	htable = s_malloc(sizeof(shash_t));
	htable->hashsize = (hsize != 0)?hsize:DEFAULT_HASH_SIZE;
	htable->table = s_calloc(htable->hashsize, sizeof(shash_elt_t *));
	htable->hash = hfunc;
	htable->cmp = cmpf;
	htable->count = 0;

	return htable;
}


spointer
s_hash_get(shash_t *htable, const spointer key)
{
	unsigned int hashkey;
	shash_elt_t *elt;

	hashkey = htable->hash(key, htable->hashsize);

	for(elt = htable->table[hashkey]; elt; elt=elt->next) {
		if(htable->cmp(elt->key, key) == 0)
			return elt->obj;
	}
	
	return NULL;
}


void
s_hash_put(shash_t *htable, spointer key, spointer obj)
{
	shash_elt_t *next, *elt;
	unsigned int hashkey;

	elt = s_malloc(sizeof(shash_elt_t));
	elt->key = key;
	elt->obj = obj;

	/* remove same key before adding new one. */
	s_hash_remove(htable, key);

	hashkey = htable->hash(key, htable->hashsize);

	next = htable->table[hashkey];
	htable->table[hashkey] = elt;
	htable->table[hashkey]->next = next;
	htable->count++;
}


/* This function doesn't remove object.
 * It only takes out from a hash-table.
 * So, you have to free it in you own reponsibility.
 */
void
s_hash_remove(shash_t *htable, const spointer key)
{
	shash_elt_t *elt, *prev;
	unsigned int hashkey;

	elt = prev = NULL;

	hashkey = htable->hash(key, htable->hashsize);

	for(elt = htable->table[hashkey]; elt; elt=elt->next) {
		if(htable->cmp(elt->key, key) == 0) {
			if(prev)
				prev->next = elt->next;
			else
				htable->table[hashkey] = elt->next;

			s_freen(elt, sizeof(shash_elt_t));

			htable->count--;
			break;
		}
		prev = elt;
	}
}


size_t
s_hash_size(shash_t *htable)
{
	return htable->count;
}


void 
s_hash_free(shash_t *htable)
{
	s_hash_clear(htable);
	s_freen(htable->table, sizeof(shash_elt_t *) * htable->hashsize);
	s_freen(htable, sizeof(shash_t));
}


void
s_hash_clear(shash_t *htable)
{
	shash_elt_t *elt;
	int i;

	for (i=0; i<htable->hashsize; i++) {
		
		if ((elt = htable->table[i]) != NULL) {
			s_freen(elt, sizeof(shash_elt_t));
			htable->table[i] = NULL;
		}
	}

	htable->count = 0;
}


/* 
 * Alfred V. Aho, Ravi Sethi, Jeffrey D. Ullman 
 * ĸإѥ飲 ˡġ
 * Ҽڡ
 * ºݤ˼¸ơХ󥹤ɤϥåͤ뤳Ȥǧ
 * 
 * char * to hashkey.
 */
unsigned int
s_string_hashfunc(const spointer key, size_t hashsize)
{
	unsigned int h, g;
	const char *p;

	h = g = 0;
	for (p=key; *p; p++) {
		h = (h << 4) + (*p);
		if ((g = h & 0xf0000000)) {
			h = h ^ (g >> 24);
			h = h ^ g;
		}
	}
	return h % hashsize;
}


