
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <getopt.h>
#include "scale/commons.h"
#include "scale/alloc.h"
#include "scale/list.h"
#include "scale/vector.h"
#include "scale/buffer.h"
#include "scale/hash.h"
#include "scale/stack.h"
#include "scale/commons.h"
#include "scale/queue.h"


#define FUNC_START() printf("*** %s ***\n", __func__)

typedef int (*testfunc)(void);


int
test_list(void)
{
	slist_t *list;
	slist_t *sublist;

	FUNC_START();

	list = s_list_alloc();

	s_list_add(list, "A");
	s_list_add(list, "B");
	s_list_add(list, "C");

	assert(list->count == 3);
	assert(strcmp(s_list_get(list, 0), "A") == 0);
	assert(strcmp(s_list_get(list, 1), "B") == 0);	
	assert(strcmp(s_list_get(list, 2), "C") == 0);

	assert(strcmp(s_list_remove(list, 1), "B") == 0);
	assert(list->count == 2);

	assert(strcmp(s_list_get(list, 0), "A") == 0);
	assert(strcmp(s_list_get(list, 1), "C") == 0);

	assert(s_list_find(list, s_string_compare, "A") == 0);
	assert(s_list_find(list, s_string_compare, "C") == 1);

	sublist = s_list_alloc();
	s_list_add(sublist, "D");
	s_list_add(sublist, "E");
	assert(sublist->count == 2);
	assert(s_list_find(sublist, s_string_compare, "D") == 0);
	assert(s_list_find(sublist, s_string_compare, "E") == 1);

	assert(s_list_join(list, sublist));

	assert(list->count == 4);

	assert(strcmp(s_list_get(list, 0), "A") == 0);
	assert(strcmp(s_list_get(list, 1), "C") == 0);
	assert(strcmp(s_list_get(list, 2), "D") == 0);
	assert(strcmp(s_list_get(list, 3), "E") == 0);
	
	assert(s_list_find(list, s_string_compare, "A") == 0);
	assert(s_list_find(list, s_string_compare, "C") == 1);
	assert(s_list_find(list, s_string_compare, "D") == 2);
	assert(s_list_find(list, s_string_compare, "E") == 3);

	s_list_free(list);
	s_list_free(sublist);
	return 1;
}


int
test_vector(void)
{
	svector_t *vect;
	svector_t *subvect;

	FUNC_START();
	
	vect = s_vector_alloc(s_string_compare);

	s_vector_add(vect, "A");
	s_vector_add(vect, "B");
	s_vector_add(vect, "C");

	assert(s_vector_size(vect) == 3);
	assert(strcmp(s_vector_get(vect, 0), "A") == 0);
	assert(strcmp(s_vector_get(vect, 1), "B") == 0);
	assert(strcmp(s_vector_get(vect, 2), "C") == 0);

	assert(s_vector_find(vect, s_string_compare, "A") == 0);
	assert(s_vector_find(vect, s_string_compare, "B") == 1);
	assert(s_vector_find(vect, s_string_compare, "C") == 2);

	assert(strcmp(s_vector_remove(vect, 1), "B") == 0);
	assert(s_vector_size(vect) == 2);

	assert(strcmp(s_vector_get(vect, 0), "A") == 0);
	assert(strcmp(s_vector_get(vect, 1), "C") == 0);


	subvect = s_vector_alloc(s_string_compare);
	s_vector_add(subvect, "a");
	s_vector_add(subvect, "b");
	s_vector_add(subvect, "c");
	assert(s_vector_size(subvect) == 3);

	assert(s_vector_join(vect, subvect));

	assert(s_vector_size(vect) == 5);
	assert(strcmp(s_vector_get(vect, 0), "A") == 0);
	assert(strcmp(s_vector_get(vect, 1), "C") == 0);
	assert(strcmp(s_vector_get(vect, 2), "a") == 0);
	assert(strcmp(s_vector_get(vect, 3), "b") == 0);
	assert(strcmp(s_vector_get(vect, 4), "c") == 0);

	s_vector_free(vect);
	s_vector_free(subvect);

	return 1;
}


int
test_buffer(void)
{
	sbuffer_t *sbuf;

	FUNC_START();

	sbuf = s_buffer_alloc(1024);

	assert(s_buffer_strlen(sbuf) == 0);

	s_buffer_strcat(sbuf, "ABC");
	s_buffer_strcat(sbuf, "DEF");

	assert(strcmp(sbuf->string, "ABCDEF") == 0);

	s_buffer_strcpy(sbuf, "abc");

	assert(strcmp(sbuf->string, "abc") == 0);

	assert(s_buffer_sprintf(sbuf, "NO.%d", 1) == 4);
	assert(strcmp(sbuf->string, "NO.1") == 0);

	s_buffer_free(sbuf);
	return 1;
}


int
test_hash(void)
{
	shash_t *hash;
	
	FUNC_START();

	hash = s_hash_alloc(128, s_string_hashfunc, s_string_compare);

	assert(s_hash_size(hash) == 0);

	s_hash_put(hash, "a", "A");
	s_hash_put(hash, "b", "B");
	s_hash_put(hash, "c", "C");

	assert(s_hash_size(hash) == 3);
	assert(strcmp(s_hash_get(hash, "a"), "A") == 0);
	assert(strcmp(s_hash_get(hash, "b"), "B") == 0);
	assert(strcmp(s_hash_get(hash, "c"), "C") == 0);

	s_hash_remove(hash, "b");

	assert(s_hash_size(hash) == 2);
	assert(strcmp(s_hash_get(hash, "a"), "A") == 0);
	assert(s_hash_get(hash, "b") == NULL);
	assert(strcmp(s_hash_get(hash, "c"), "C") == 0);

	{ /* test iterater */
		shash_iterate_t *itr = s_hash_keys(hash);
		void *key, *value;
		size_t count = 0;
		while ((key = s_hash_iterate_next(itr)) != NULL) {
			value = s_hash_get(hash, key);
			++count;

			if ((strcmp(key, "a") && strcmp(value, "A")) ||
				(strcmp(key, "c") && strcmp(value, "C"))) {
				/* OK */
			}
			else {
				/* NG */
				assert(false);
			}
		}
		assert(count == 2);

		s_hash_iterate_free(itr);
	}

	s_hash_free(hash);

	return 1;
}


int
test_stack(void)
{
	sstack_t *stack;

	FUNC_START();

	stack = s_stack_allocn(10000);

	s_stack_push(stack, "A");
	s_stack_push(stack, "B");
	s_stack_push(stack, "C");
	
	assert(s_stack_size(stack) == 3);
	assert(strcmp(s_stack_pop(stack), "C") == 0);
	assert(strcmp(s_stack_pop(stack), "B") == 0);
	assert(strcmp(s_stack_pop(stack), "A") == 0);
	assert(s_stack_size(stack) == 0);

	s_stack_push(stack, "a");
	s_stack_push(stack, "b");
	
	assert(strcmp(s_stack_peek(stack), "b") == 0);
	assert(s_stack_size(stack) == 2);
	assert(strcmp(s_stack_pop(stack), "b") == 0);
	assert(strcmp(s_stack_pop(stack), "a") == 0);
	assert(s_stack_size(stack) == 0);

	s_stack_free(stack);

	return 1;
}


int
test_queue(void)
{
	squeue_t *queue;

	FUNC_START();

	queue = s_queue_alloc();

	s_queue_enqueue(queue, "A");
	s_queue_enqueue(queue, "B");
	s_queue_enqueue(queue, "C");

	assert(s_queue_size(queue) == 3);
	assert(strcmp(s_queue_dequeue(queue), "A") == 0);
	assert(strcmp(s_queue_dequeue(queue), "B") == 0);
	assert(strcmp(s_queue_dequeue(queue), "C") == 0);
	assert(s_queue_size(queue) == 0);

	
	s_queue_enqueue(queue, "a");
	s_queue_enqueue(queue, "b");
	assert(s_queue_size(queue) == 2);
	assert(strcmp(s_queue_front(queue), "a") == 0);
	assert(s_queue_size(queue) == 2);
	assert(strcmp(s_queue_dequeue(queue), "a") == 0);
	assert(strcmp(s_queue_dequeue(queue), "b") == 0);
	assert(s_queue_size(queue) == 0);

	s_queue_free(queue);

	return 1;
}


int
main(int argc, char *argv[])
{
	int kflg = 0;
	char c;
	int has_error = 0;
	testfunc funcs[] = {
		test_queue,
		test_stack,
		test_hash,
		test_buffer,
		test_vector,
		test_list,
		NULL
	};
	testfunc *f;

	if ((c = getopt(argc, argv, "k"))) {
		switch (c) {
		case 'k':
			kflg++;
			break;
		}
	}

	for (f=funcs; *f; f++) {
		if (!(*f)()) {
			if (! kflg) {
				printf("Test failed\n");
				return EXIT_FAILURE;
			}
			else {
				has_error = 1;
			}
		}
	}

	if (has_error) {
		fprintf(stderr, "\n");
		fprintf(stderr, "Test program contains ERROR.\n");
		fprintf(stderr, "May be you want to take off '-k' option.\n");
	}

	return EXIT_SUCCESS;
}
