/***************************************************************************
 *   Copyright (C) 2005 by TAM (Teppei Tamra)   *
 *   tam-t@par.odn.ne.jp   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "skim_scim_wnn_items.h"
#include <qlabel.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qurl.h>
#include <qtooltip.h>
#include <kfiledialog.h>
#include <skim/skimkeygrabber.h>

using namespace SkimScimWnnItems;

// Item

SkimScimWnnItem::SkimScimWnnItem(scim::String _label,scim::String _name,scim::String _tip) : QObject(0)
{
    label = QString::fromLocal8Bit(_label.c_str());
    name = _name;
    tip = QString::fromLocal8Bit(_tip.c_str());

    valueWidget = 0;
    labelWidget = 0;

}

SkimScimWnnItem::~SkimScimWnnItem()
{
}

QWidget * SkimScimWnnItem::getValueWidget()
{
    if (!valueWidget) {
        createValueWidget();
        QToolTip::add(valueWidget,tip);
    }
    static_cast<QLabel *>(getLabelWidget())->setBuddy(valueWidget);
    return valueWidget;
}

QWidget * SkimScimWnnItem::getLabelWidget()
{
    if (!labelWidget) createLabelWidget();
    return labelWidget;
}


void SkimScimWnnItem::readConfig(scim::ConfigPointer cfg)
{
}

void SkimScimWnnItem::saveConfig(scim::ConfigPointer cfg)
{
}

QString SkimScimWnnItem::getLabel()
{
    return label;
}

void SkimScimWnnItem::createLabelWidget()
{
    QString l = label;
    l.replace(QChar('_'),QChar('&'));
    labelWidget = new QLabel(0,"label");
    labelWidget->setText(l);
    labelWidget->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
    labelWidget->hide();
}

void SkimScimWnnItem::createValueWidget()
{
    valueWidget = new QLabel(0,"value");
    static_cast<QLabel *>(valueWidget)->setText(label);
    valueWidget->hide();
}

void SkimScimWnnItem::createTipWidget()
{
}

// Container

SkimScimWnnContainer::SkimScimWnnContainer()
{
}


SkimScimWnnContainer::~SkimScimWnnContainer()
{
}

void SkimScimWnnContainer::append(SkimScimWnnItem *item)
{
}

// SkimScimWnn

SkimScimWnn::SkimScimWnn(scim::String _label,scim::String _name,scim::String _tip)
: SkimScimWnnItem(_label,_name,_tip),SkimScimWnnContainer()
{
}

SkimScimWnn::~SkimScimWnn()
{
}

void SkimScimWnn::readConfig(scim::ConfigPointer cfg)
{
    for(unsigned int i = 0;i < items.size();i ++) {
        items[i]->readConfig(cfg);
    }
    return;
}

void SkimScimWnn::saveConfig(scim::ConfigPointer cfg)
{
    for(unsigned int i = 0;i < items.size();i ++) {
        items[i]->saveConfig(cfg);
    }
    return;
}

void SkimScimWnn::append(SkimScimWnnItem *item)
{
    items.append(item);
    connect(item,SIGNAL(changed()),this,SIGNAL(changed()));
    item->getValueWidget()->reparent(getValueWidget(),QPoint());
    item->getValueWidget()->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
    static_cast<QTabWidget *>(getValueWidget())->addTab(item->getValueWidget(),item->getLabel());
    item->getValueWidget()->show();
}

void SkimScimWnn::createValueWidget()
{
    valueWidget = new QTabWidget(0,"Tab");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
    valueWidget->hide();
}


// Page

SkimScimWnnPage::SkimScimWnnPage(scim::String _label,scim::String _name,scim::String _tip)
: SkimScimWnnItem(_label,_name,_tip),SkimScimWnnContainer()
{
}

SkimScimWnnPage::~SkimScimWnnPage()
{
}

void SkimScimWnnPage::append(SkimScimWnnItem *item)
{
    items.append(item);
    connect(item,SIGNAL(changed()),this,SIGNAL(changed()));
    item->getLabelWidget()->reparent(getValueWidget(),QPoint());
    item->getLabelWidget()->setSizePolicy(QSizePolicy::Minimum,QSizePolicy::Minimum);
    item->getValueWidget()->reparent(getValueWidget(),QPoint());
    item->getValueWidget()->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    table->addWidget(item->getLabelWidget(),items.count() - 1,0);
    table->addWidget(item->getValueWidget(),items.count() - 1,1);
    item->getValueWidget()->show();
    item->getLabelWidget()->show();
}

void SkimScimWnnPage::readConfig(scim::ConfigPointer cfg)
{
    for(unsigned int i = 0;i < items.size();i ++) {
        items[i]->readConfig(cfg);
    }
    return;
}

void SkimScimWnnPage::saveConfig(scim::ConfigPointer cfg)
{
    for(unsigned int i = 0;i < items.size();i ++) {
        items[i]->saveConfig(cfg);
    }
    return;
}

void SkimScimWnnPage::createValueWidget()
{
    valueWidget = new QWidget(0,"Page");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
    valueWidget->hide();
    table = new QGridLayout(valueWidget,0,0,4);
    spacer = new QSpacerItem(1,1,QSizePolicy::Minimum,QSizePolicy::Expanding);
    //table->addItem(spacer,99,0);
    table->addMultiCell(spacer,99,100,0,1);
}



// String


SkimScimWnnEntryItem::SkimScimWnnEntryItem(scim::String _label,scim::String _name,scim::String _tip,scim::String _default)
: SkimScimWnnItem(_label,_name,_tip)
{
    stringData = _default;
}

SkimScimWnnEntryItem::~SkimScimWnnEntryItem()
{
}

void SkimScimWnnEntryItem::readConfig(scim::ConfigPointer cfg)
{
    stringData = cfg->read(scim::String(name.latin1()),scim::String(stringData.local8Bit().data()));
    static_cast<QLineEdit *>(getValueWidget())->setText(stringData);
    return;
}

void SkimScimWnnEntryItem::saveConfig(scim::ConfigPointer cfg)
{
    cfg->write(scim::String(name.latin1()),scim::String(stringData.local8Bit().data()));
    return;
}

void SkimScimWnnEntryItem::createValueWidget()
{
    valueWidget = new QLineEdit(0,"String");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    static_cast<QLineEdit *>(valueWidget)->setText(stringData);
    valueWidget->hide();
    connect(valueWidget,SIGNAL(textChanged(const QString &)),this,SLOT(textChanged(const QString &)));
}

void SkimScimWnnEntryItem::textChanged(const QString &text)
{
    stringData = text;
    emit changed();
}

// Bool


SkimScimWnnBoolItem::SkimScimWnnBoolItem(scim::String _label,scim::String _name,scim::String _tip,bool _default)
: SkimScimWnnItem(_label,_name,_tip)
{
    boolData = _default;
}

SkimScimWnnBoolItem::~SkimScimWnnBoolItem()
{
}

void SkimScimWnnBoolItem::readConfig(scim::ConfigPointer cfg)
{
    boolData = cfg->read(scim::String(name.latin1()),boolData);
    static_cast<QCheckBox *>(getValueWidget())->setChecked(boolData);
    return;
}

void SkimScimWnnBoolItem::saveConfig(scim::ConfigPointer cfg)
{
    cfg->write(scim::String(name.latin1()),boolData);
    return;
}

void SkimScimWnnBoolItem::createValueWidget()
{
    valueWidget = new QCheckBox(0,"Bool");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    static_cast<QCheckBox *>(valueWidget)->setChecked(boolData);
    valueWidget->hide();
    connect(valueWidget,SIGNAL(clicked()),this,SLOT(clicked()));
}

void SkimScimWnnBoolItem::clicked()
{
    boolData = static_cast<QCheckBox *>(valueWidget)->isOn();
    emit changed();
}


// File


SkimScimWnnFileItem::SkimScimWnnFileItem(scim::String _label,scim::String _name,scim::String _tip,scim::String _default)
: SkimScimWnnItem(_label,_name,_tip)
{
    stringData = _default;
}

SkimScimWnnFileItem::~SkimScimWnnFileItem()
{
}

void SkimScimWnnFileItem::readConfig(scim::ConfigPointer cfg)
{
    stringData = cfg->read(scim::String(name.latin1()),scim::String(stringData.local8Bit().data()));
    if (!getValueWidget()) return;
    le->setText(stringData);
    return;
}

void SkimScimWnnFileItem::saveConfig(scim::ConfigPointer cfg)
{
    cfg->write(scim::String(name.latin1()),scim::String(stringData.local8Bit().data()));
    return;
}

void SkimScimWnnFileItem::createValueWidget()
{
    valueWidget = new QWidget(0,"File");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    le = new QLineEdit(valueWidget,"File-LineEdit");
    le->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    QPushButton *b = new QPushButton(valueWidget,"File-Button");
    b->setSizePolicy(QSizePolicy::Minimum,QSizePolicy::Minimum);
    b->setText("...");
    QHBoxLayout *l = new QHBoxLayout(valueWidget);
    l->addWidget(le);
    l->addWidget(b);
    le->setText(stringData);
    valueWidget->hide();
    connect(le,SIGNAL(textChanged(const QString &)),this,SLOT(textChanged(const QString &)));
    connect(b,SIGNAL(clicked()),this,SLOT(buttonClicked()));
}

void SkimScimWnnFileItem::textChanged(const QString &text)
{
    stringData = text;
    emit changed();
}

void SkimScimWnnFileItem::buttonClicked()
{
    QString d = KFileDialog::getOpenFileName(QUrl(stringData).dirPath());
    if (d != stringData) {
        le->setText(d);
        stringData = d; // ס
    }
}

// Key


SkimScimWnnKeyItem::SkimScimWnnKeyItem(scim::String _label,scim::String _name,scim::String _tip,scim::String _default)
: SkimScimWnnItem(_label,_name,_tip)
{
    stringData = _default;
}

SkimScimWnnKeyItem::~SkimScimWnnKeyItem()
{
}

void SkimScimWnnKeyItem::readConfig(scim::ConfigPointer cfg)
{
    stringData = cfg->read(scim::String(name.latin1()),scim::String(stringData.latin1()));
    if (!getValueWidget()) return;
    le->setText(stringData);
    return;
}

void SkimScimWnnKeyItem::saveConfig(scim::ConfigPointer cfg)
{
    cfg->write(scim::String(name.latin1()),scim::String(stringData.latin1()));
    return;
}

void SkimScimWnnKeyItem::createValueWidget()
{
    valueWidget = new QWidget(0,"Key");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    le = new QLineEdit(valueWidget,"Key-LineEdit");
    le->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    QPushButton *b = new QPushButton(valueWidget,"Key-Button");
    b->setSizePolicy(QSizePolicy::Minimum,QSizePolicy::Minimum);
    b->setText("...");
    QHBoxLayout *l = new QHBoxLayout(valueWidget);
    l->addWidget(le);
    l->addWidget(b);
    le->setText(stringData);
    valueWidget->hide();
    connect(le,SIGNAL(textChanged(const QString &)),this,SLOT(textChanged(const QString &)));
    connect(b,SIGNAL(clicked()),this,SLOT(buttonClicked()));
}

void SkimScimWnnKeyItem::textChanged(const QString &text)
{
    stringData = text;
    emit changed();
}

void SkimScimWnnKeyItem::buttonClicked()
{
    SkimShortcutListEditor *d = new SkimShortcutListEditor();
    d->setModal(true);
    QStringList s = QStringList::split(",",stringData);
    d->setStringList(s);
    if (d->exec() == QDialog::Accepted) {
        le->setText(d->getCombinedString());
        stringData = d->getCombinedString();
    }
    delete d;
}

// Int


SkimScimWnnIntItem::SkimScimWnnIntItem(scim::String _label,scim::String _name,scim::String _tip,int _default,int lower,int upper)
: SkimScimWnnItem(_label,_name,_tip)
{
    intData = _default;
    upperValue = upper;
    lowerValue = lower;
}

SkimScimWnnIntItem::~SkimScimWnnIntItem()
{
}

void SkimScimWnnIntItem::readConfig(scim::ConfigPointer cfg)
{
    intData = cfg->read(scim::String(name.latin1()),intData);
    static_cast<QSpinBox *>(getValueWidget())->setValue(intData);
    return;
}

void SkimScimWnnIntItem::saveConfig(scim::ConfigPointer cfg)
{
    cfg->write(scim::String(name.latin1()),intData);
    return;
}

void SkimScimWnnIntItem::createValueWidget()
{
    valueWidget = new QSpinBox(lowerValue,upperValue,1,0,"Int");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    static_cast<QSpinBox *>(valueWidget)->setValue(intData);
    valueWidget->hide();
    connect(valueWidget,SIGNAL(valueChanged(int)),this,SLOT(valueChanged(int)));
}

void SkimScimWnnIntItem::valueChanged(int value)
{
    intData = value;
    emit changed();
}


// Select


SkimScimWnnSelectItem::SkimScimWnnSelectItem(scim::String _label,scim::String _name,scim::String _tip,scim::String _default,std::vector<scim::String> _list)
: SkimScimWnnItem(_label,_name,_tip)
{
    stringData = _default;
    for(unsigned int i = 0;i < _list.size();i ++) list.append(_list[i]);
}

SkimScimWnnSelectItem::~SkimScimWnnSelectItem()
{
}

void SkimScimWnnSelectItem::readConfig(scim::ConfigPointer cfg)
{
    stringData = cfg->read(scim::String(name.latin1()),scim::String(stringData.local8Bit().data()));
    static_cast<QComboBox *>(getValueWidget())->setCurrentText(stringData);
    return;
}

void SkimScimWnnSelectItem::saveConfig(scim::ConfigPointer cfg)
{
    cfg->write(scim::String(name.latin1()),scim::String(stringData.local8Bit().data()));
    return;
}

void SkimScimWnnSelectItem::createValueWidget()
{
    valueWidget = new QComboBox(0,"Select");
    valueWidget->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Minimum);
    static_cast<QComboBox *>(valueWidget)->insertStringList(list);
    static_cast<QComboBox *>(valueWidget)->setCurrentText(stringData);
    valueWidget->hide();
    connect(valueWidget,SIGNAL(activated(int)),this,SLOT(activated(int)));
}

void SkimScimWnnSelectItem::activated(int pos)
{
    stringData = list[pos];
    emit changed();
}



