package jp.sf.amateras.mirage;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import jp.sf.amateras.mirage.annotation.PrimaryKey;
import jp.sf.amateras.mirage.bean.BeanDesc;
import jp.sf.amateras.mirage.bean.BeanDescFactory;
import jp.sf.amateras.mirage.bean.PropertyDesc;
import jp.sf.amateras.mirage.naming.DefaultNameConverter;
import jp.sf.amateras.mirage.naming.NameConverter;
import jp.sf.amateras.mirage.parser.Node;
import jp.sf.amateras.mirage.parser.SqlContext;
import jp.sf.amateras.mirage.parser.SqlContextImpl;
import jp.sf.amateras.mirage.parser.SqlParserImpl;
import jp.sf.amateras.mirage.provider.ConnectionProvider;
import jp.sf.amateras.mirage.type.DefaultValueType;
import jp.sf.amateras.mirage.type.ValueType;

public class SqlManagerImpl implements SqlManager {

//	private static final Logger logger = Logger.getLogger(SqlManagerImpl.class.getName());

	private NameConverter nameConverter = new DefaultNameConverter();

	private SqlExecuter executer = new SqlExecuter();

	public SqlManagerImpl(){
		addValueType(new DefaultValueType());
		setNameConverter(new DefaultNameConverter());
	}

	@Override
	public void setNameConverter(NameConverter nameConverter) {
		this.nameConverter = nameConverter;
		this.executer.setNameConverter(nameConverter);
	}

	@Override
	public void setConnectionProvider(ConnectionProvider connectionProvider) {
		this.executer.setConnectionProvider(connectionProvider);
	}

	protected Node prepareNode(String sqlPath) {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		InputStream in = cl.getResourceAsStream(sqlPath);
		if (in == null) {
			throw new RuntimeException(String.format(
					"resource: %s is not found.", sqlPath));
		}

		String sql = null;
		try {
			byte[] buf = new byte[1024 * 8];
			int length = 0;
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			while((length = in.read(buf)) != -1){
				out.write(buf, 0, length);
			}

			sql = new String(out.toByteArray(), "UTF-8");

		} catch(Exception ex){
			throw new RuntimeException(String.format("Failed to load SQL from: %s", sqlPath));
		}

		Node node = new SqlParserImpl(sql).parse();
		return node;
	}

	protected SqlContext prepareSqlContext(Object param){
		SqlContext context = new SqlContextImpl();

		if (param != null) {
			BeanDesc beanDesc = BeanDescFactory.getBeanDesc(param);
			for (int i = 0; i < beanDesc.getPropertyDescSize(); i++) {
				PropertyDesc pd = beanDesc.getPropertyDesc(i);
				context.addArg(pd.getPropertyName(), pd.getValue(param), pd
						.getPropertyType());
			}
		}

		return context;
	}

	@Override
	public int executeUpdate(String sqlPath) {
		return executeUpdate(sqlPath, null);
	}

	@Override
	public int executeUpdate(String sqlPath, Object param) {
		Node node = prepareNode(sqlPath);
		SqlContext context = prepareSqlContext(param);
		node.accept(context);

		return executer.executeUpdateSql(context.getSql(), context.getBindVariables());

	}

	@Override
	public <T> List<T> getResultList(Class<T> clazz, String sqlPath) {
		return getResultList(clazz, sqlPath, null);
	}

	@Override
	public <T> List<T> getResultList(Class<T> clazz, String sqlPath, Object param) {
		Node node = prepareNode(sqlPath);
		SqlContext context = prepareSqlContext(param);
		node.accept(context);

		return executer.getResultList(clazz, context.getSql(), context.getBindVariables());
	}

	@Override
	public <T> T getSingleResult(Class<T> clazz, String sqlPath) {
		return getSingleResult(clazz, sqlPath, null);
	}

	@Override
	public <T> T getSingleResult(Class<T> clazz, String sqlPath, Object param) {
		Node node = prepareNode(sqlPath);
		SqlContext context = prepareSqlContext(param);
		node.accept(context);

		return executer.getSingleResult(clazz, context.getSql(), context.getBindVariables());
	}

	@Override
	public int deleteEntity(Object entity) {
		StringBuilder sb = new StringBuilder();
		sb.append("DELETE FROM ").append(nameConverter.entityToTable(entity.getClass().getName()));
		sb.append(" WHERE ");

		List<Object> params = new ArrayList<Object>();
		boolean hasPrimaryKey = false;

		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(entity.getClass());

		for(int i=0;i<beanDesc.getPropertyDescSize();i++){
			PropertyDesc pd = beanDesc.getPropertyDesc(i);
			if(pd.getAnnotation(PrimaryKey.class) != null && pd.isReadable()){
				if(!params.isEmpty()){
					sb.append(" AND ");
				}
				sb.append(nameConverter.propertyToColumn(pd.getPropertyName())).append("=?");
				try {
					params.add(pd.getValue(entity));
				} catch(Exception ex){
					throw new RuntimeException(ex);
				}
				hasPrimaryKey = true;
			}
		}

		if(hasPrimaryKey == false){
			throw new RuntimeException(
					"Primary key is not found: " + entity.getClass().getName());
		}

		return executer.executeUpdateSql(sb.toString(), params.toArray());
	}

	@Override
	public int insertEntity(Object entity) {
		List<Object> params = new ArrayList<Object>();
		StringBuilder sb = new StringBuilder();
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(entity.getClass());

		sb.append("INSERT INTO ").append(nameConverter.entityToTable(entity.getClass().getName())).append(" (");
		{
			int count = 0;
			for(int i=0;i<beanDesc.getPropertyDescSize();i++){
				PropertyDesc pd = beanDesc.getPropertyDesc(i);
				if(pd.getAnnotation(PrimaryKey.class) == null && !pd.isTransient() && pd.isReadable() ){
					if(count != 0){
						sb.append(", ");
					}
					sb.append(nameConverter.propertyToColumn(pd.getPropertyName()));
					count++;
				}
			}
		}
		sb.append(") VALUES (");
		{
			int count = 0;
			for(int i=0;i<beanDesc.getPropertyDescSize();i++){
				PropertyDesc pd = beanDesc.getPropertyDesc(i);
				PrimaryKey primaryKey = pd.getAnnotation(PrimaryKey.class);
				if((primaryKey == null || primaryKey.persistent()) && !pd.isTransient() && pd.isReadable() ){
					if(count != 0){
						sb.append(", ");
					}
					sb.append("?");

					try {
						params.add(pd.getValue(entity));
					} catch(Exception ex){
						throw new RuntimeException(ex);
					}

					count++;
				}
			}
		}
		sb.append(")");

		return executer.executeUpdateSql(sb.toString(), params.toArray());
	}

	@Override
	public int updateEntity(Object entity) {
		List<Object> params = new ArrayList<Object>();
		StringBuilder sb = new StringBuilder();

		sb.append("UPDATE ").append(nameConverter.entityToTable(entity.getClass().getName())).append(" SET ");

		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(entity.getClass());
		{
			int count = 0;
			for (int i = 0; i < beanDesc.getPropertyDescSize(); i++) {
				PropertyDesc pd = beanDesc.getPropertyDesc(i);
				if(pd.getAnnotation(PrimaryKey.class) == null && !pd.isTransient() && pd.isReadable() ){
					if (count != 0) {
						sb.append(", ");
					}
					sb.append(nameConverter.propertyToColumn(pd.getPropertyName())).append(" = ?");
					try {
						params.add(pd.getValue(entity));
					} catch (Exception ex) {
						throw new RuntimeException(ex);
					}
					count++;
				}
			}
		}
		sb.append(" WHERE ");
		{
			int count = 0;
			for (int i = 0; i < beanDesc.getPropertyDescSize(); i++) {
				PropertyDesc pd = beanDesc.getPropertyDesc(i);
				if(pd.getAnnotation(PrimaryKey.class) != null && pd.isReadable() ){
					if(count != 0){
						sb.append(" AND ");
					}
					sb.append(nameConverter.propertyToColumn(pd.getPropertyName())).append(" = ? ");
					try {
						params.add(pd.getValue(entity));
					} catch(Exception ex){
						throw new RuntimeException(ex);
					}
					count++;
				}
			}
			if(count == 0){
				throw new RuntimeException(
						"Primary key is not found: " + entity.getClass().getName());
			}
		}

		return executer.executeUpdateSql(sb.toString(), params.toArray());
	}

	@Override
	public <T> T findEntity(Class<T> clazz, Object... id) {
		// assemble SQL
		StringBuilder sb = new StringBuilder();
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(clazz);

		sb.append("SELECT * FROM ");
		sb.append(nameConverter.entityToTable(clazz.getName()));
		sb.append(" WHERE ");

		int count = 0;

		for(int i=0; i<beanDesc.getPropertyDescSize(); i++){
			PropertyDesc pd = beanDesc.getPropertyDesc(i);
			if(pd.getAnnotation(PrimaryKey.class) != null){
				if(count != 0){
					sb.append(" AND ");
				}
				sb.append(nameConverter.propertyToColumn(pd.getPropertyName()));
				sb.append(" = ?");
				count++;
			}
		}
		if(count == 0){
			throw new RuntimeException(
					"Primary key is not found: " + clazz.getName());
		}

		// execute SQL
		return executer.getSingleResult(clazz, sb.toString(), id);
	}

	@Override
	public void addValueType(ValueType valueType) {
		this.executer.addValueType(valueType);
	}

	@Override
	public <T> int getCount(String sqlPath) {
		return getCount(sqlPath, null);
	}

	@Override
	public <T> int getCount(String sqlPath, Object param) {
		Node node = prepareNode(sqlPath);
		SqlContext context = prepareSqlContext(param);
		node.accept(context);
		String sql = "SELECT COUNT(*) FROM (" + context.getSql() + ")";

		return executer.getSingleResult(Integer.class, sql, context.getBindVariables());
	}
}
