/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.telephony;

/**
 * Contains disconnect call causes generated by the framework and the RIL.
 * @hide
 */
public class DisconnectCause {

    /** The disconnect cause is not valid (Not received a disconnect cause) */
    public static final int NOT_VALID                      = -1;
    /** Has not yet disconnected */
    public static final int NOT_DISCONNECTED               = 0;
    /** An incoming call that was missed and never answered */
    public static final int INCOMING_MISSED                = 1;
    /** Normal; Remote hangup*/
    public static final int NORMAL                         = 2;
    /** Normal; Local hangup */
    public static final int LOCAL                          = 3;
    /** Outgoing call to busy line */
    public static final int BUSY                           = 4;
    /** Outgoing call to congested network */
    public static final int CONGESTION                     = 5;
    /** Not presently used */
    public static final int MMI                            = 6;
    /** Invalid dial string */
    public static final int INVALID_NUMBER                 = 7;
    /** Cannot reach the peer */
    public static final int NUMBER_UNREACHABLE             = 8;
    /** Cannot reach the server */
    public static final int SERVER_UNREACHABLE             = 9;
    /** Invalid credentials */
    public static final int INVALID_CREDENTIALS            = 10;
    /** Calling from out of network is not allowed */
    public static final int OUT_OF_NETWORK                 = 11;
    /** Server error */
    public static final int SERVER_ERROR                   = 12;
    /** Client timed out */
    public static final int TIMED_OUT                      = 13;
    /** Client went out of network range */
    public static final int LOST_SIGNAL                    = 14;
    /** GSM or CDMA ACM limit exceeded */
    public static final int LIMIT_EXCEEDED                 = 15;
    /** An incoming call that was rejected */
    public static final int INCOMING_REJECTED              = 16;
    /** Radio is turned off explicitly */
    public static final int POWER_OFF                      = 17;
    /** Out of service */
    public static final int OUT_OF_SERVICE                 = 18;
    /** No ICC, ICC locked, or other ICC error */
    public static final int ICC_ERROR                      = 19;
    /** Call was blocked by call barring */
    public static final int CALL_BARRED                    = 20;
    /** Call was blocked by fixed dial number */
    public static final int FDN_BLOCKED                    = 21;
    /** Call was blocked by restricted all voice access */
    public static final int CS_RESTRICTED                  = 22;
    /** Call was blocked by restricted normal voice access */
    public static final int CS_RESTRICTED_NORMAL           = 23;
    /** Call was blocked by restricted emergency voice access */
    public static final int CS_RESTRICTED_EMERGENCY        = 24;
    /** Unassigned number */
    public static final int UNOBTAINABLE_NUMBER            = 25;
    /** MS is locked until next power cycle */
    public static final int CDMA_LOCKED_UNTIL_POWER_CYCLE  = 26;
    /** Drop call*/
    public static final int CDMA_DROP                      = 27;
    /** INTERCEPT order received, MS state idle entered */
    public static final int CDMA_INTERCEPT                 = 28;
    /** MS has been redirected, call is cancelled */
    public static final int CDMA_REORDER                   = 29;
    /** Service option rejection */
    public static final int CDMA_SO_REJECT                 = 30;
    /** Requested service is rejected, retry delay is set */
    public static final int CDMA_RETRY_ORDER               = 31;
    /** Unable to obtain access to the CDMA system */
    public static final int CDMA_ACCESS_FAILURE            = 32;
    /** Not a preempted call */
    public static final int CDMA_PREEMPTED                 = 33;
    /** Not an emergency call */
    public static final int CDMA_NOT_EMERGENCY             = 34;
    /** Access Blocked by CDMA network */
    public static final int CDMA_ACCESS_BLOCKED            = 35;
    /** Unknown error or not specified */
    public static final int ERROR_UNSPECIFIED              = 36;
    /**
     * Only emergency numbers are allowed, but we tried to dial
     * a non-emergency number.
     */
    // TODO: This should be the same as NOT_EMERGENCY
    public static final int EMERGENCY_ONLY                 = 37;
    /**
     * The supplied CALL Intent didn't contain a valid phone number.
     */
    public static final int NO_PHONE_NUMBER_SUPPLIED       = 38;
    /**
     * Our initial phone number was actually an MMI sequence.
     */
    public static final int DIALED_MMI                     = 39;
    /**
     * We tried to call a voicemail: URI but the device has no
     * voicemail number configured.
     */
    public static final int VOICEMAIL_NUMBER_MISSING       = 40;
    /**
     * This status indicates that InCallScreen should display the
     * CDMA-specific "call lost" dialog.  (If an outgoing call fails,
     * and the CDMA "auto-retry" feature is enabled, *and* the retried
     * call fails too, we display this specific dialog.)
     *
     * TODO: this is currently unused, since the "call lost" dialog
     * needs to be triggered by a *disconnect* event, rather than when
     * the InCallScreen first comes to the foreground.  For now we use
     * the needToShowCallLostDialog field for this (see below.)
     */
    public static final int CDMA_CALL_LOST                 = 41;
    /**
     * This status indicates that the call was placed successfully,
     * but additionally, the InCallScreen needs to display the
     * "Exiting ECM" dialog.
     *
     * (Details: "Emergency callback mode" is a CDMA-specific concept
     * where the phone disallows data connections over the cell
     * network for some period of time after you make an emergency
     * call.  If the phone is in ECM and you dial a non-emergency
     * number, that automatically *cancels* ECM, but we additionally
     * need to warn the user that ECM has been canceled (see bug
     * 4207607.))
     *
     * TODO: Rethink where the best place to put this is. It is not a notification
     * of a failure of the connection -- it is an additional message that accompanies
     * a successful connection giving the user important information about what happened.
     *
     * {@hide}
     */
    public static final int EXITED_ECM                     = 42;

    /**
     * The outgoing call failed with an unknown cause.
     */
    public static final int OUTGOING_FAILURE               = 43;

    /**
     * The outgoing call was canceled by the {@link android.telecom.ConnectionService}.
     */
    public static final int OUTGOING_CANCELED              = 44;

    /**
     * The call, which was an IMS call, disconnected because it merged with another call.
     */
    public static final int IMS_MERGED_SUCCESSFULLY        = 45;

    /**
     * Stk Call Control modified DIAL request to USSD request.
     * {@hide}
     */
    public static final int DIAL_MODIFIED_TO_USSD          = 46;
    /**
     * Stk Call Control modified DIAL request to SS request.
     * {@hide}
     */
    public static final int DIAL_MODIFIED_TO_SS            = 47;
    /**
     * Stk Call Control modified DIAL request to DIAL with modified data.
     * {@hide}
     */
    public static final int DIAL_MODIFIED_TO_DIAL          = 48;

    /**
     * The call was terminated because CDMA phone service and roaming have already been activated.
     * {@hide}
     */
    public static final int CDMA_ALREADY_ACTIVATED         = 49;

    /** call failed due to LTE to 3G/2G handover not feasible */
    public static final int HO_NOT_FEASIBLE = 50;

    public static final int NO_CIRCUIT_AVAIL = 51;
    public static final int NO_ROUTE_TO_DESTINAON = 52;
    public static final int OPERATOR_DETERMINED_BARRING = 53;
    public static final int CALL_FAIL_NO_USER_RESPONDING = 54;
    public static final int CALL_FAIL_NO_ANSWER_FROM_USER = 55;
    public static final int CALL_FAIL_DESTINATION_OUT_OF_ORDER = 56;
    public static final int BEARER_CAPABILITY_NOT_AUTHORIZED = 57;
    public static final int CHANNEL_UNACCEPTABLE = 58;
    public static final int CALL_REJECTED = 59;
    public static final int NUMBER_CHANGED = 60;
    public static final int PREEMPTION = 61;
    public static final int FACILITY_REJECTED = 62;
    public static final int RESP_TO_STATUS_ENQUIRY = 63;
    public static final int NORMAL_UNSPECIFIED = 64;
    public static final int NETWORK_OUT_OF_ORDER = 65;
    public static final int TEMPORARY_FAILURE = 66;
    public static final int SWITCHING_EQUIPMENT_CONGESTION = 67;
    public static final int ACCESS_INFORMATION_DISCARDED = 68;
    public static final int REQUESTED_CIRCUIT_OR_CHANNEL_NOT_AVAILABLE = 69;
    public static final int RESOURCES_UNAVAILABLE_OR_UNSPECIFIED = 70;
    public static final int QOS_UNAVAILABLE = 71;
    public static final int REQUESTED_FACILITY_NOT_SUBSCRIBED = 72;
    public static final int INCOMING_CALLS_BARRED_WITHIN_CUG = 73;
    public static final int BEARER_CAPABILITY_UNAVAILABLE = 74;
    public static final int SERVICE_OPTION_NOT_AVAILABLE = 75;
    public static final int BEARER_SERVICE_NOT_IMPLEMENTED = 76;
    public static final int REQUESTED_FACILITY_NOT_IMPLEMENTED = 77;
    public static final int ONLY_DIGITAL_INFORMATION_BEARER_AVAILABLE = 78;
    public static final int SERVICE_OR_OPTION_NOT_IMPLEMENTED = 79;
    public static final int INVALID_TRANSACTION_IDENTIFIER = 80;
    public static final int USER_NOT_MEMBER_OF_CUG = 81;
    public static final int INCOMPATIBLE_DESTINATION = 82;
    public static final int INVALID_TRANSIT_NW_SELECTION = 83;
    public static final int SEMANTICALLY_INCORRECT_MESSAGE = 84;
    public static final int INVALID_MANDATORY_INFORMATION = 85;
    public static final int MESSAGE_TYPE_NON_IMPLEMENTED = 86;
    public static final int MESSAGE_TYPE_NOT_COMPATIBLE_WITH_PROTOCOL_STATE = 87;
    public static final int INFORMATION_ELEMENT_NON_EXISTENT = 88;
    public static final int CONDITIONAL_IE_ERROR = 89;
    public static final int MESSAGE_NOT_COMPATIBLE_WITH_PROTOCOL_STATE = 90;
    public static final int RECOVERY_ON_TIMER_EXPIRED = 91;
    public static final int PROTOCOL_ERROR_UNSPECIFIED = 92;
    public static final int INTERWORKING_UNSPECIFIED = 93;
    public static final int LOCAL_LOW_BATTERY = 94;
    public static final int LOW_BATTERY = 95;

    /** EMERGENCY call failed with temporary fail cause */
    public static final int EMERGENCY_TEMP_FAILURE         = 96;
    /** EMERGENCY call failed with permanent fail cause */
    public static final int EMERGENCY_PERM_FAILURE         = 97;

    public static final int NON_SELECTED_USER_CLEARING = 98;

    /**
     * Call was rejected due to number being blacklisted by user.
     * {@@hide}
     */
    public static final int CALL_BLACKLISTED = 400;

    //*********************************************************************************************
    // When adding a disconnect type:
    // 1) Please assign the new type the next id value below.
    // 2) Increment the next id value below to a new value.
    // 3) Update MAXIMUM_VALID_VALUE to the new disconnect type.
    // 4) Update toString() with the newly added disconnect type.
    // 5) Update android.telecom.DisconnectCauseUtil with any mappings to a telecom.DisconnectCause.
    //
    // NextId: 99
    //*********************************************************************************************

    /** Smallest valid value for call disconnect codes. */
    public static final int MINIMUM_VALID_VALUE = NOT_DISCONNECTED;

    /** Largest valid value for call disconnect codes. */
    public static final int MAXIMUM_VALID_VALUE = CALL_BLACKLISTED;

    /** Private constructor to avoid class instantiation. */
    private DisconnectCause() {
        // Do nothing.
    }

    /** Returns descriptive string for the specified disconnect cause. */
    public static String toString(int cause) {
        switch (cause) {
        case NOT_DISCONNECTED:
            return "NOT_DISCONNECTED";
        case INCOMING_MISSED:
            return "INCOMING_MISSED";
        case NORMAL:
            return "NORMAL";
        case LOCAL:
            return "LOCAL";
        case BUSY:
            return "BUSY";
        case CONGESTION:
            return "CONGESTION";
        case INVALID_NUMBER:
            return "INVALID_NUMBER";
        case NUMBER_UNREACHABLE:
            return "NUMBER_UNREACHABLE";
        case SERVER_UNREACHABLE:
            return "SERVER_UNREACHABLE";
        case INVALID_CREDENTIALS:
            return "INVALID_CREDENTIALS";
        case OUT_OF_NETWORK:
            return "OUT_OF_NETWORK";
        case SERVER_ERROR:
            return "SERVER_ERROR";
        case TIMED_OUT:
            return "TIMED_OUT";
        case LOST_SIGNAL:
            return "LOST_SIGNAL";
        case LIMIT_EXCEEDED:
            return "LIMIT_EXCEEDED";
        case INCOMING_REJECTED:
            return "INCOMING_REJECTED";
        case POWER_OFF:
            return "POWER_OFF";
        case OUT_OF_SERVICE:
            return "OUT_OF_SERVICE";
        case ICC_ERROR:
            return "ICC_ERROR";
        case CALL_BARRED:
            return "CALL_BARRED";
        case FDN_BLOCKED:
            return "FDN_BLOCKED";
        case CS_RESTRICTED:
            return "CS_RESTRICTED";
        case CS_RESTRICTED_NORMAL:
            return "CS_RESTRICTED_NORMAL";
        case CS_RESTRICTED_EMERGENCY:
            return "CS_RESTRICTED_EMERGENCY";
        case UNOBTAINABLE_NUMBER:
            return "UNOBTAINABLE_NUMBER";
        case CDMA_LOCKED_UNTIL_POWER_CYCLE:
            return "CDMA_LOCKED_UNTIL_POWER_CYCLE";
        case CDMA_DROP:
            return "CDMA_DROP";
        case CDMA_INTERCEPT:
            return "CDMA_INTERCEPT";
        case CDMA_REORDER:
            return "CDMA_REORDER";
        case CDMA_SO_REJECT:
            return "CDMA_SO_REJECT";
        case CDMA_RETRY_ORDER:
            return "CDMA_RETRY_ORDER";
        case CDMA_ACCESS_FAILURE:
            return "CDMA_ACCESS_FAILURE";
        case CDMA_PREEMPTED:
            return "CDMA_PREEMPTED";
        case CDMA_NOT_EMERGENCY:
            return "CDMA_NOT_EMERGENCY";
        case CDMA_ACCESS_BLOCKED:
            return "CDMA_ACCESS_BLOCKED";
        case EMERGENCY_ONLY:
            return "EMERGENCY_ONLY";
        case NO_PHONE_NUMBER_SUPPLIED:
            return "NO_PHONE_NUMBER_SUPPLIED";
        case DIALED_MMI:
            return "DIALED_MMI";
        case VOICEMAIL_NUMBER_MISSING:
            return "VOICEMAIL_NUMBER_MISSING";
        case CDMA_CALL_LOST:
            return "CDMA_CALL_LOST";
        case EXITED_ECM:
            return "EXITED_ECM";
        case DIAL_MODIFIED_TO_USSD:
            return "DIAL_MODIFIED_TO_USSD";
        case DIAL_MODIFIED_TO_SS:
            return "DIAL_MODIFIED_TO_SS";
        case DIAL_MODIFIED_TO_DIAL:
            return "DIAL_MODIFIED_TO_DIAL";
        case ERROR_UNSPECIFIED:
            return "ERROR_UNSPECIFIED";
        case OUTGOING_FAILURE:
            return "OUTGOING_FAILURE";
        case OUTGOING_CANCELED:
            return "OUTGOING_CANCELED";
        case IMS_MERGED_SUCCESSFULLY:
            return "IMS_MERGED_SUCCESSFULLY";
        case CDMA_ALREADY_ACTIVATED:
            return "CDMA_ALREADY_ACTIVATED";
        case NON_SELECTED_USER_CLEARING:
            return "NON_SELECTED_USER_CLEARING";
        case HO_NOT_FEASIBLE:
            return "HO_NOT_FEASIBLE";
        case NO_CIRCUIT_AVAIL:
            return "NO_CIRCUIT_AVAIL";
        case NO_ROUTE_TO_DESTINAON:
            return "NO_ROUTE_TO_DESTINAON";
        case OPERATOR_DETERMINED_BARRING:
            return "OPERATOR_DETERMINED_BARRING";
        case CALL_FAIL_NO_USER_RESPONDING:
            return "CALL_FAIL_NO_USER_RESPONDING";
        case CALL_FAIL_NO_ANSWER_FROM_USER:
            return "CALL_FAIL_NO_ANSWER_FROM_USER";
        case CALL_FAIL_DESTINATION_OUT_OF_ORDER:
            return "CALL_FAIL_DESTINATION_OUT_OF_ORDER";
        case BEARER_CAPABILITY_NOT_AUTHORIZED:
            return "BEARER_CAPABILITY_NOT_AUTHORIZED";
        case CHANNEL_UNACCEPTABLE:
            return "CHANNEL_UNACCEPTABLE";
        case CALL_REJECTED:
            return "CALL_REJECTED";
        case NUMBER_CHANGED:
            return "NUMBER_CHANGED";
        case PREEMPTION:
            return "PREEMPTION";
        case FACILITY_REJECTED:
            return "FACILITY_REJECTED";
        case RESP_TO_STATUS_ENQUIRY:
            return "RESP_TO_STATUS_ENQUIRY";
        case NORMAL_UNSPECIFIED:
            return "NORMAL_UNSPECIFIED";
        case NETWORK_OUT_OF_ORDER:
            return "NETWORK_OUT_OF_ORDER";
        case TEMPORARY_FAILURE:
            return "TEMPORARY_FAILURE";
        case SWITCHING_EQUIPMENT_CONGESTION:
            return "SWITCHING_EQUIPMENT_CONGESTION";
        case ACCESS_INFORMATION_DISCARDED:
            return "ACCESS_INFORMATION_DISCARDED";
        case REQUESTED_CIRCUIT_OR_CHANNEL_NOT_AVAILABLE:
            return "REQUESTED_CIRCUIT_OR_CHANNEL_NOT_AVAILABLE";
        case RESOURCES_UNAVAILABLE_OR_UNSPECIFIED:
            return "RESOURCES_UNAVAILABLE_OR_UNSPECIFIED";
        case QOS_UNAVAILABLE:
            return "QOS_UNAVAILABLE";
        case REQUESTED_FACILITY_NOT_SUBSCRIBED:
            return "REQUESTED_FACILITY_NOT_SUBSCRIBED";
        case INCOMING_CALLS_BARRED_WITHIN_CUG:
            return "INCOMING_CALLS_BARRED_WITHIN_CUG";
        case BEARER_CAPABILITY_UNAVAILABLE:
            return "BEARER_CAPABILITY_UNAVAILABLE";
        case SERVICE_OPTION_NOT_AVAILABLE:
            return "SERVICE_OPTION_NOT_AVAILABLE";
        case BEARER_SERVICE_NOT_IMPLEMENTED:
            return "BEARER_SERVICE_NOT_IMPLEMENTED";
        case REQUESTED_FACILITY_NOT_IMPLEMENTED:
            return "REQUESTED_FACILITY_NOT_IMPLEMENTED";
        case ONLY_DIGITAL_INFORMATION_BEARER_AVAILABLE:
            return "ONLY_DIGITAL_INFORMATION_BEARER_AVAILABLE";
        case SERVICE_OR_OPTION_NOT_IMPLEMENTED:
            return "SERVICE_OR_OPTION_NOT_IMPLEMENTED";
        case INVALID_TRANSACTION_IDENTIFIER:
            return "INVALID_TRANSACTION_IDENTIFIER";
        case USER_NOT_MEMBER_OF_CUG:
            return "USER_NOT_MEMBER_OF_CUG";
        case INCOMPATIBLE_DESTINATION:
            return "INCOMPATIBLE_DESTINATION";
        case INVALID_TRANSIT_NW_SELECTION:
            return "INVALID_TRANSIT_NW_SELECTION";
        case SEMANTICALLY_INCORRECT_MESSAGE:
            return "SEMANTICALLY_INCORRECT_MESSAGE";
        case INVALID_MANDATORY_INFORMATION:
            return "INVALID_MANDATORY_INFORMATION";
        case MESSAGE_TYPE_NON_IMPLEMENTED:
            return "MESSAGE_TYPE_NON_IMPLEMENTED";
        case MESSAGE_TYPE_NOT_COMPATIBLE_WITH_PROTOCOL_STATE:
            return "MESSAGE_TYPE_NOT_COMPATIBLE_WITH_PROTOCOL_STATE";
        case INFORMATION_ELEMENT_NON_EXISTENT:
            return "INFORMATION_ELEMENT_NON_EXISTENT";
        case CONDITIONAL_IE_ERROR:
            return "CONDITIONAL_IE_ERROR";
        case MESSAGE_NOT_COMPATIBLE_WITH_PROTOCOL_STATE:
            return "MESSAGE_NOT_COMPATIBLE_WITH_PROTOCOL_STATE";
        case RECOVERY_ON_TIMER_EXPIRED:
            return "RECOVERY_ON_TIMER_EXPIRED";
        case PROTOCOL_ERROR_UNSPECIFIED:
            return "PROTOCOL_ERROR_UNSPECIFIED";
        case INTERWORKING_UNSPECIFIED:
            return "INTERWORKING_UNSPECIFIED";
        case LOCAL_LOW_BATTERY:
            return "LOCAL_LOW_BATTERY";
        case LOW_BATTERY:
            return "LOW_BATTERY";
        case EMERGENCY_TEMP_FAILURE:
            return "EMERGENCY_TEMP_FAILURE";
        case EMERGENCY_PERM_FAILURE:
            return "EMERGENCY_PERM_FAILURE";
        case CALL_BLACKLISTED:
            return "CALL_BLACKLISTED";
        default:
            return "INVALID: " + cause;
        }
    }
}
