/* Copyright 2016 Braden Farmer
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.farmerbb.taskbar.activity;

import android.annotation.SuppressLint;
import android.app.ActivityOptions;
import android.content.ComponentName;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.graphics.Rect;
import android.hardware.display.DisplayManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.preference.Preference;
import android.preference.PreferenceActivity;
import android.provider.Settings;
import android.support.v4.content.LocalBroadcastManager;
import android.view.Display;
import android.view.Gravity;
import android.view.View;
import android.view.WindowManager;

import com.farmerbb.taskbar.BuildConfig;
import com.farmerbb.taskbar.MainActivity;
import com.farmerbb.taskbar.R;
import com.farmerbb.taskbar.util.AppEntry;
import com.farmerbb.taskbar.util.FreeformHackHelper;
import com.farmerbb.taskbar.util.PinnedBlockedApps;
import com.farmerbb.taskbar.util.U;

public class ContextMenuActivity extends PreferenceActivity implements Preference.OnPreferenceClickListener {

    String packageName;
    String componentName;
    String appName;

    boolean showStartMenu = false;
    boolean shouldHideTaskbar = false;
    boolean isStartButton = false;

    @SuppressLint("RtlHardcoded")
    @SuppressWarnings("deprecation")
    @Override
    protected void onPostCreate(Bundle savedInstanceState) {
        super.onPostCreate(savedInstanceState);

        LocalBroadcastManager.getInstance(this).sendBroadcast(new Intent("com.farmerbb.taskbar.CONTEXT_MENU_APPEARING"));

        showStartMenu = getIntent().getBooleanExtra("launched_from_start_menu", false);
        isStartButton = !getIntent().hasExtra("package_name") && !getIntent().hasExtra("app_name");

        // Determine where to position the dialog on screen
        WindowManager.LayoutParams params = getWindow().getAttributes();
        SharedPreferences pref = U.getSharedPreferences(this);

        DisplayManager dm = (DisplayManager) getSystemService(DISPLAY_SERVICE);
        Display display = dm.getDisplay(Display.DEFAULT_DISPLAY);

        if(showStartMenu) {
            int x = getIntent().getIntExtra("x", 0);
            int y = getIntent().getIntExtra("y", 0);
            int offset = getResources().getDimensionPixelSize(R.dimen.context_menu_offset);

            switch(pref.getString("position", "bottom_left")) {
                case "bottom_left":
                case "bottom_vertical_left":
                    params.gravity = Gravity.BOTTOM | Gravity.LEFT;
                    params.x = x;
                    params.y = display.getHeight() - y - offset;
                    break;
                case "bottom_right":
                case "bottom_vertical_right":
                    params.gravity = Gravity.BOTTOM | Gravity.LEFT;
                    params.x = x - getResources().getDimensionPixelSize(R.dimen.context_menu_width) + offset + offset;
                    params.y = display.getHeight() - y - offset;
                    break;
            }
        } else {
            int x = getIntent().getIntExtra("x", display.getWidth());
            int y = getIntent().getIntExtra("y", display.getHeight());
            int offset = getResources().getDimensionPixelSize(R.dimen.icon_size);

            switch(pref.getString("position", "bottom_left")) {
                case "bottom_left":
                    params.gravity = Gravity.BOTTOM | Gravity.LEFT;
                    params.x = isStartButton ? 0 : x;
                    params.y = offset;
                    break;
                case "bottom_vertical_left":
                    params.gravity = Gravity.BOTTOM | Gravity.LEFT;
                    params.x = offset;
                    params.y = display.getHeight() - y - (isStartButton ? 0 : offset);
                    break;
                case "bottom_right":
                    params.gravity = Gravity.BOTTOM | Gravity.RIGHT;
                    params.x = display.getWidth() - x;
                    params.y = offset;
                    break;
                case "bottom_vertical_right":
                    params.gravity = Gravity.BOTTOM | Gravity.RIGHT;
                    params.x = offset;
                    params.y = display.getHeight() - y - (isStartButton ? 0 : offset);
                    break;
            }
        }

        params.width = getResources().getDimensionPixelSize(R.dimen.context_menu_width);
        params.dimAmount = 0;

        getWindow().setAttributes(params);

        View view = findViewById(android.R.id.list);
        if(view != null) view.setPadding(0, 0, 0, 0);

        // Generate options to show on the menu, depending on which icon was clicked
        if(isStartButton) {
            addPreferencesFromResource(R.xml.pref_context_menu_open_settings);
            findPreference("open_taskbar_settings").setOnPreferenceClickListener(this);

            if(!getIntent().getBooleanExtra("dont_show_quit", false)) {
                addPreferencesFromResource(R.xml.pref_context_menu_quit);
                findPreference("quit_taskbar").setOnPreferenceClickListener(this);
            }
        } else {
            appName = getIntent().getStringExtra("app_name");
            packageName = getIntent().getStringExtra("package_name");
            componentName = getIntent().getStringExtra("component_name");

            if(getResources().getConfiguration().screenWidthDp >= 600
                    && Build.VERSION.SDK_INT <= Build.VERSION_CODES.M)
                setTitle(appName);
            else {
                addPreferencesFromResource(R.xml.pref_context_menu_header);
                findPreference("header").setTitle(appName);
            }

            if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.N
                    && pref.getBoolean("freeform_hack", false)
                    && isInMultiWindowMode()) {
                String windowSizePref = pref.getString("window_size", "standard");

                if(!windowSizePref.equals("standard")) {
                    addPreferencesFromResource(R.xml.pref_context_menu_window_size_standard);
                    findPreference("window_size_standard").setOnPreferenceClickListener(this);
                }

                if(!windowSizePref.equals("fullscreen")) {
                    addPreferencesFromResource(R.xml.pref_context_menu_window_size_fullscreen);
                    findPreference("window_size_fullscreen").setOnPreferenceClickListener(this);
                }

                if(!windowSizePref.equals("phone_size")) {
                    addPreferencesFromResource(R.xml.pref_context_menu_window_size_phone_size);
                    findPreference("window_size_phone_size").setOnPreferenceClickListener(this);
                }
            }

            final PackageManager pm = getPackageManager();
            Intent homeIntent = new Intent(Intent.ACTION_MAIN);
            homeIntent.addCategory(Intent.CATEGORY_HOME);
            ResolveInfo defaultLauncher = pm.resolveActivity(homeIntent, PackageManager.MATCH_DEFAULT_ONLY);

            if(!packageName.equals(BuildConfig.APPLICATION_ID)
                    && !packageName.equals(defaultLauncher.activityInfo.packageName)) {
                PinnedBlockedApps pba = PinnedBlockedApps.getInstance(this);

                if(pba.isPinned(componentName)) {
                    addPreferencesFromResource(R.xml.pref_context_menu_pin);
                    findPreference("pin_app").setOnPreferenceClickListener(this);
                    findPreference("pin_app").setTitle(R.string.unpin_app);
                } else if(pba.isBlocked(componentName)) {
                    addPreferencesFromResource(R.xml.pref_context_menu_block);
                    findPreference("block_app").setOnPreferenceClickListener(this);
                    findPreference("block_app").setTitle(R.string.unblock_app);
                } else {
                    final int MAX_NUM_OF_COLUMNS =
                            pref.getString("position", "bottom_left").contains("vertical")
                                    ? getResources().getInteger(R.integer.num_of_columns_vertical)
                                    : getResources().getInteger(R.integer.num_of_columns);

                    if(pba.getPinnedApps().size() < MAX_NUM_OF_COLUMNS) {
                        addPreferencesFromResource(R.xml.pref_context_menu_pin);
                        findPreference("pin_app").setOnPreferenceClickListener(this);
                        findPreference("pin_app").setTitle(R.string.pin_app);
                    }

                    addPreferencesFromResource(R.xml.pref_context_menu_block);
                    findPreference("block_app").setOnPreferenceClickListener(this);
                    findPreference("block_app").setTitle(R.string.block_app);
                }
            }

            addPreferencesFromResource(R.xml.pref_context_menu);

            findPreference("app_info").setOnPreferenceClickListener(this);
            findPreference("uninstall").setOnPreferenceClickListener(this);
        }
    }

    @Override
    public boolean onPreferenceClick(Preference p) {
        boolean appIsValid = true;

        if(!isStartButton) {
            try {
                getPackageManager().getPackageInfo(getIntent().getStringExtra("package_name"), 0);
            } catch(PackageManager.NameNotFoundException e) {
                appIsValid = false;
            }
        }

        if(appIsValid) switch(p.getKey()) {
            case "app_info":
                Intent intent = new Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS, Uri.parse("package:" + getIntent().getStringExtra("package_name")));
                if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && isInMultiWindowMode())
                    intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

                startFreeformActivity();
                startActivity(intent);

                showStartMenu = false;
                shouldHideTaskbar = true;
                break;
            case "uninstall":
                if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && isInMultiWindowMode()) {
                    Intent intent2 = new Intent(ContextMenuActivity.this, DummyActivity.class);
                    intent2.putExtra("uninstall", getIntent().getStringExtra("package_name"));

                    startFreeformActivity();
                    startActivity(intent2);
                } else {
                    startFreeformActivity();
                    startActivity(new Intent(Intent.ACTION_DELETE, Uri.parse("package:" + getIntent().getStringExtra("package_name"))));
                }

                showStartMenu = false;
                shouldHideTaskbar = true;
                break;
            case "open_taskbar_settings":
                startFreeformActivity();
                startActivity(new Intent(this, MainActivity.class));

                showStartMenu = false;
                shouldHideTaskbar = true;
                break;
            case "quit_taskbar":
                sendBroadcast(new Intent("com.farmerbb.taskbar.QUIT"));

                showStartMenu = false;
                shouldHideTaskbar = true;
                break;
            case "pin_app":
                PinnedBlockedApps pba = PinnedBlockedApps.getInstance(this);
                if(pba.isPinned(componentName))
                    pba.removePinnedApp(this, componentName);
                else {
                    Intent throwaway = new Intent();
                    throwaway.setComponent(ComponentName.unflattenFromString(componentName));

                    pba.addPinnedApp(this, new AppEntry(
                            packageName,
                            componentName,
                            appName,
                            throwaway.resolveActivityInfo(getPackageManager(), 0).loadIcon(getPackageManager()),
                            true));
                }
                break;
            case "block_app":
                PinnedBlockedApps pba2 = PinnedBlockedApps.getInstance(this);
                if(pba2.isBlocked(componentName))
                    pba2.removeBlockedApp(this, componentName);
                else {
                    Intent throwaway = new Intent();
                    throwaway.setComponent(ComponentName.unflattenFromString(componentName));

                    pba2.addBlockedApp(this, new AppEntry(
                            packageName,
                            componentName,
                            appName,
                            null,
                            false));
                }
                break;
            case "window_size_standard":
                U.launchStandard(this, generateIntent());

                showStartMenu = false;
                shouldHideTaskbar = true;
                break;
            case "window_size_fullscreen":
                U.launchFullscreen(this, generateIntent());

                showStartMenu = false;
                shouldHideTaskbar = true;
                break;
            case "window_size_phone_size":
                U.launchPhoneSize(this, generateIntent());

                showStartMenu = false;
                shouldHideTaskbar = true;
                break;
        }

        finish();
        return true;
    }

    @Override
    protected void onPause() {
        super.onPause();
        if(!isFinishing()) finish();
    }

    @Override
    public void finish() {
        LocalBroadcastManager.getInstance(this).sendBroadcast(new Intent("com.farmerbb.taskbar.CONTEXT_MENU_DISAPPEARING"));

        if(showStartMenu)
            LocalBroadcastManager.getInstance(this).sendBroadcast(new Intent("com.farmerbb.taskbar.TOGGLE_START_MENU_ALT"));
        else if(shouldHideTaskbar) {
            SharedPreferences pref = U.getSharedPreferences(this);
            if(pref.getBoolean("hide_taskbar", true) && !pref.getBoolean("in_freeform_workspace", false))
                LocalBroadcastManager.getInstance(this).sendBroadcast(new Intent("com.farmerbb.taskbar.HIDE_TASKBAR"));
        }

        super.finish();
    }

    @SuppressWarnings("deprecation")
    private void startFreeformActivity() {
        SharedPreferences pref = U.getSharedPreferences(this);
        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.N
                && pref.getBoolean("freeform_hack", false)
                && isInMultiWindowMode()
                && !FreeformHackHelper.getInstance().isFreeformHackActive()) {
            DisplayManager dm = (DisplayManager) getSystemService(DISPLAY_SERVICE);
            Display display = dm.getDisplay(Display.DEFAULT_DISPLAY);

            Intent intent = new Intent(this, InvisibleActivityFreeform.class);
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_LAUNCH_ADJACENT);
            startActivity(intent, ActivityOptions.makeBasic().setLaunchBounds(new Rect(display.getWidth(), display.getHeight(), display.getWidth() + 1, display.getHeight() + 1)).toBundle());
        }
    }

    private Intent generateIntent() {
        Intent intent = new Intent();
        intent.setComponent(ComponentName.unflattenFromString(componentName));
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP);

        SharedPreferences pref = U.getSharedPreferences(this);
        if(pref.getBoolean("disable_animations", false) && !showStartMenu)
            intent.addFlags(Intent.FLAG_ACTIVITY_NO_ANIMATION);

        return intent;
    }
}