/*
 * QB Forth
 * Quantum and Basic Forth system for Arduino
 *
 * Author titoi2
 *
 * 2011/03/18	0.0.0 Start
 * 2011/04/05	0.1.1 Create complete interpreter
 * 2011/04/10	0.1.2 Create : compiler
 * 2011/04/15	0.1.3 motor controls
 *
 */

#include <avr/pgmspace.h> 
#include <Servo.h>

/********************************
 *	DEFINITIONS		*
 ********************************/

// Version
#define	VERSION_MAJOR	0
#define	VERSION_MINOR	1
#define	VERSION_REV	4

// compile switch
//#define CHECK_STACK_UNDERFLOW
//#define CHECK_STACK_OVERFLOW
//#define __DEBUG_CODE_ENABLE__

// constants
#define	IMMEDIATE_MASK	0x40
#define	LINE_BREAK_CODE	0x0D	// LINE BREAK CODE is CR


// Arduino control definitions

#define PIN_MOTOR_R_VREF      5
#define PIN_MOTOR_L_VREF      6
#define PIN_MOTOR_R_CONTROL1  8
#define PIN_MOTOR_R_CONTROL2  9
#define PIN_MOTOR_L_CONTROL1  10
#define PIN_MOTOR_L_CONTROL2  11

#define MOTORTYPE_MOTOR_R    0
#define MOTORTYPE_MOTOR_L    1

#define MOTORACTION_MOTOR_STOP		0
#define MOTORACTION_MOTOR_FORWARD	1
#define MOTORACTION_MOTOR_REVERSE	2
#define MOTORACTION_MOTOR_BRAKE		3

#define MESSAGE_ACK		0x30
#define MESSAGE_ERROR		0x31
#define MESSAGE_NACK		0x32


#define SERIAL_COMMUNICATION_BAUD_RATE		115200


/********************************
 *      external C functions    *
 ********************************/
// interface 
extern int Serial_available()				__asm__("__Serial_available");
extern int Serial_read()				__asm__("__Serial_read");
extern void Serial_print(char *buf)			__asm__("__Serial_print");
extern void Serial_println(char *buf)			__asm__("__Serial_printlnStr");
extern void Serial_println()				__asm__("__Serial_println");
extern void Serial_printfmt(int data, int fmt)		__asm__("__Serial_printfmt");
extern void Serial_printlnfmt(int data, int fmt)	__asm__("__Serial_printlnfmt");


extern void wiring_digitalWrite(int, int)		__asm__("__wiring_digitalWrite");
extern void wiring_analogWrite(int, int)		__asm__("__wiring_analogWrite");

// implementation
int Serial_available()
{
  return Serial.available();
}

int Serial_read()
{
  return  Serial.read();
}

void Serial_print(char *buf)
{
  Serial.print(buf);  
}
void Serial_println(char *buf)
{
  Serial.println(buf);	
}
void Serial_println()
{
  Serial.println();	
}
void Serial_printfmt(int data, int fmt)
{
  Serial.print(data, fmt);	
}
void Serial_printlnfmt(int data, int fmt)
{
  Serial.println(data, fmt);
}

void wiring_digitalWrite(int pin, int value)
{
    digitalWrite((uint8_t)pin, (uint8_t)value);
}
void wiring_analogWrite(int pin, int value)
{
    analogWrite((uint8_t)pin, value);
}



/********************************
 *	CONSTANTS		*
 ********************************/

// MESSAGE max length 31 bytes

PROGMEM static const prog_uchar  opening_messages1[] = 
	"QBForth ";

PROGMEM static const prog_uchar  opening_messages2[] = 
	" Copyright (C) 2011 titoi2";

PROGMEM static const prog_uchar  opening_messages3[] = 
	"QBForth is Quantum & Basic ";

PROGMEM static const prog_uchar  opening_messages4[] = 
	"Forth for ARDUINO UNO";

// ERROR MESSAGES
typedef PROGMEM const prog_uchar	*tcpuc;
PROGMEM static const prog_uchar  *error_messages_table[] = {
	(tcpuc)"Unknown Error",				//  0
	(tcpuc)"Empty Stack",				//  1
	(tcpuc)"Dictionary full",			//  2
	(tcpuc)"Error #3",				//  3
	(tcpuc)"Is'nt unique",				//  4
	(tcpuc)"Error #5",				//  5
	(tcpuc)"Error #6",				//  6
	(tcpuc)"Full Stack",				//  7
	(tcpuc)"Error #8",				//  8
	(tcpuc)"Error #9",				//  9
	(tcpuc)"Error #10",				//  10
	(tcpuc)"Error #11",				//  11
	(tcpuc)"Error #12",				//  12
	(tcpuc)"Error #13",				//  13
	(tcpuc)"Vector unassigned",			//  14
	(tcpuc)"Compilation only, use in definition",	//  15
	(tcpuc)"Execution only",			//  16
	(tcpuc)"Conditionals not paired",		//  17
	(tcpuc)"Definition not finished",		//  18
	(tcpuc)"In protected dictionary",		//  19
	(tcpuc)"Error #20",				//  20
	(tcpuc)"Error #21",				//  21
	(tcpuc)"Declare vocabulary",			//  22
	(tcpuc)"Name expected",				//  23
	(tcpuc)"Error #24",				//  24
	(tcpuc)"Field error"				//  25
};


/********************************
 *      WORK AREA               *
 ********************************/

// data area is data stack and variables
// data stack from high address to low
// variable use low address to high

#define	DATA_AREA_SIZE	256
unsigned char qbfDataArea[DATA_AREA_SIZE]		__asm__("__qbfDataArea");


// Pascal String
// token buffer tokenbuffer[0] is length

#define	TOKEN_BUFFER_SIZE	34
static unsigned char tokenbuffer[TOKEN_BUFFER_SIZE]	__asm__("__tokenbuffer");


// init return stack pointer save
static unsigned int	save_rsp			__asm__("__save_rsp");

// Word Pad is definition of a new word buffer

#define	WORD_PAD_SIZE	256
unsigned char qbfWordPad[WORD_PAD_SIZE]			__asm__("__qbfWordPad");


// USER VARIABLE 
static unsigned int	state	__asm__("__state");	// STATE is true when in compilation mode
static unsigned int	latest	__asm__("__latest");	// last word nfa
static unsigned int	base	__asm__("__base");	// current number-conversion radix {{2...36}}
static unsigned int	dp	__asm__("__dp");	// dictionary pointer
static unsigned int	csp	__asm__("__csp");	// current stack pointer

static unsigned char	tib[82]	__asm__("__tib");	// text in buffer
static unsigned int	tiblen	__asm__("__tiblen");	// text in buffer size = 80
static unsigned int	to_in	__asm__("__to_in");	// text in buffer cursor
static unsigned int	span	__asm__("__span");	// text in buffer data count

/********************************
 *      inner C functions       *
 ********************************/
// interface 
extern void ErrorUndefinedWord(char *buf)		__asm__("__ErrorUndefinedWord");
extern void OpeningMessagePrint()			__asm__("__OpeningMessagePrint");
extern void ErrorMessagePrint(int n)			__asm__("__ErrorMessagePrint");
extern void PrintOk()					__asm__("__PrintOk");
extern void *GetDataStackLast()				__asm__("__GetDataStackLast");
extern unsigned long ToNumSingle(unsigned char *target)	__asm__("__ToNumSingle");
extern unsigned long WordFind(unsigned char *target)	__asm__("__WordFind");
extern unsigned int WordGet(unsigned int deli)		__asm__("__WordGet");
extern void	WordNamePrint( char *nm)  		__asm__("__WordNamePrint");
extern void	Query()			  		__asm__("__Query");

// for Arduino control
void setMotorSpeed(int power_l,int power_r)		__asm__("__setMotorSpeed");
void controlMotor(int motortype,int motoraction)	__asm__("__controlMotor");
void led(int f)						__asm__("__led");



// for DEBUG
extern void DebugPrintNum(int n)			__asm__("__DebugPrintNum");
extern void DebugRamDump(unsigned char *adrs, int n)	__asm__("__DebugRamDump");
extern void DebugStackDump(int sp)			__asm__("__DebugStackDump");


// implementation 

#define	OUTPUT_STR_BUFFER_SIZE	32

static char strbuf[OUTPUT_STR_BUFFER_SIZE];	// output str buffer

// *** PROGMEN str to Serial
static void pmStrOutput(char *mes)
{
	strncpy_P(strbuf, mes, sizeof(strbuf));
	strbuf[sizeof(strbuf)-1] = '\0';
	Serial.print(strbuf);
}

void	WordNamePrint( char *nm)
{
	int len = sizeof(strbuf)-1;
	int nmlen = pgm_read_byte(nm) & 0x1F;
	if (len > nmlen ) len = nmlen;
	strncpy_P(strbuf, nm+1, len);
	strbuf[len] = '\0';
	Serial.println(strbuf);
}

void OpeningMessagePrint()
{
	pmStrOutput((char*)opening_messages1);
	Serial_printfmt(VERSION_MAJOR, DEC);
	Serial_print(",");
	Serial_printfmt(VERSION_MINOR, DEC);
	Serial_print(",");
	Serial_printfmt(VERSION_REV, DEC);
        pmStrOutput((char*)opening_messages2);
	Serial.println();
        pmStrOutput((char*)opening_messages3);
        pmStrOutput((char*)opening_messages4);
	Serial.println();
}

void ErrorMessagePrint(int n)
{
	PROGMEM static const prog_uchar  str[] = "ErrorMessagePrint:";
	pmStrOutput((char*)str);
	Serial_printfmt(n, DEC);
	Serial.println();


//	pmStrOutput((char*)error_messages_table[  n ]);
//	Serial.println();
}

void PrintOk()
{
	PROGMEM static const prog_uchar  ok[] = " OK";
	pmStrOutput((char*)ok);
//	Serial.println();
}

void ErrorUndefinedWord(char *buf)
{
	PROGMEM static const prog_uchar  undefineword[] = "Undefined Word";
	PROGMEM static const prog_uchar  leftgyu[] = ">>>>";
	PROGMEM static const prog_uchar  rightgyu[] = "<<<<";
	pmStrOutput((char*)undefineword);
	pmStrOutput((char*)leftgyu);
	Serial_print(buf+1);  // skip length
        pmStrOutput((char*)rightgyu);
	Serial.println();
}




void *GetDataStackLast()
{
  return &qbfDataArea[DATA_AREA_SIZE];
}


// ToNumSingle:
//	convert string tu num
// result : find then r25:r24=num,r22=1
//               else r25:r24=any,r22=0
unsigned long ToNumSingle(unsigned char *target)
{
	int targetlen = target[0];

#if 0
	Serial_printfmt(targetlen, DEC);
	PROGMEM static const prog_uchar  pbgw[] = "TO NUM STR:[";
	pmStrOutput((char*)pbgw);
	Serial_print((char*)target+1);
	Serial.println("]");
#endif

	if (base > 36) base = 10;
	int i=1;
	char sgn = 0;
	if ( target[1] == '-' ) {
		sgn = 1;
		i++;
	}
	unsigned int num = 0;
	char dplf = -1;
	for ( ; i<=targetlen; i++) {
		unsigned char tc = target[i];
		unsigned char nm;
		if ( (tc >= '0' ) && (tc <= '9' ) ) {
			nm = tc - '0';
		}
		else if ( (tc >= 'a' ) && (tc <= 'z' ) ) {
			nm = tc - 'a' + 10;
		}
		else if ( (tc >= 'A' ) && (tc <= 'Z' ) ) {
			nm = tc - 'A' + 10;
		}
		else {
			if ((tc == '.') && ( dplf == -1)) {
				dplf++;
				continue;
			}
			else {
				// error
#if 0
	PROGMEM static const prog_uchar  pbgwe[] = "ToNum ERROR:[";
	pmStrOutput((char*)pbgwe);
	Serial.print((char*)target+1);
	Serial.println("]");
#endif
				return 0;
			}
		}
		num *= 10;
		num += nm;
	}
	if (sgn) {
		num = -num;
	}
#if 0
	PROGMEM static const prog_uchar  pbgws[] = "ToNum:[";
	pmStrOutput((char*)pbgws);
	Serial_printfmt(num, DEC);
	Serial.println("]");
#endif
	return ((unsigned long)num << 16) + 1;
}



// WordFind:
//	find target from dictionary
// result : find then r25:r24=cfa,r22=flag
//               else r25:r24=target,r22=0
unsigned long WordFind(unsigned char *target)
{
	unsigned int	nfa = latest;
	prog_uchar	*pmp = (prog_uchar  *)(nfa);
	unsigned long	res = (long)target << 16;
	int targetlen = target[0];

	while (nfa) {
		prog_uchar pgwk = pgm_read_byte(pmp++);
		int pglen = pgwk & 0x1F;
		prog_uchar *lfa = pmp + pglen;
		if ((int)lfa & 1) lfa++;	// align word

		unsigned int imm = (pgwk & IMMEDIATE_MASK) ? 1 : 0xFFFF;
		if ( pglen == targetlen ) {
			int matchflag = 1;
			for (int i=1; i<= targetlen; i++) {
				unsigned char tc = target[i];
				if ( (tc >= 'a' ) && (tc <= 'z' ) ) {
					tc -= 0x20;	// to Capital
				}
				unsigned char pgc = pgm_read_byte(pmp++);;
				if ( (pgc >= 'a' ) && (pgc <= 'z' ) ) {
					pgc -= 0x20;	// to Capital
				}

				if ( tc != pgc ) {
					matchflag = 0;	// unmatched
					break;
				}
			}
			if (matchflag) {
				res = imm;
				if ((int)pmp & 1) pmp++;	// align word
				pmp += 2;			// skip LFA
				res += (long)pmp << 16;		// add cfa
				return res;
			}
		}
		// next
		nfa = pgm_read_word(lfa);	// next nfa
		pmp = (prog_uchar  *)nfa;
	}
	return res;
}

// WordGet:
//	
// result : r25:r24=token buffer address
//	token is pascal string and zero term
unsigned int WordGet(unsigned int deli)
{
	tokenbuffer[0] = 0;
	int	ix = 1;
	int c;

	do {
		if (to_in >= span) {
			return (int)tokenbuffer;
		}
		c = tib[to_in++];
	} while ( c <= deli );
	tokenbuffer[ix++] = c;
	while (1) {
		if (to_in >= span) {
			break;
		}
		c = tib[to_in++];
		if ( c  <= deli ) break;
		if (ix < (sizeof(tokenbuffer)-1) ) {
			tokenbuffer[ix++] = c;
		}
	}
	tokenbuffer[0] = ix - 1;
	tokenbuffer[ix] = 0;
#if 0
	Serial_printfmt(ix, HEX);
	PROGMEM static const prog_uchar  pbgw[] = "GET WORD:[";
	pmStrOutput((char*)pbgw);
	Serial_print((char*)tokenbuffer+1);
	Serial.println("]");
#endif
	return (int)tokenbuffer;
}


void	Query()
{
	span = 0;
	to_in = 0;
	while (1) {
		if (Serial.available() <= 0) continue;

		int c = Serial.read();
		if ( c == LINE_BREAK_CODE ) {
			tib[span] = '\0';
			return;
		}
		else if ( c == 9 ) {
			c = ' ';	// tab to space
		}
		else if (c < ' ' ) {
			continue;
		}
		tib[span++] = (unsigned char)c;
		if (span >= tiblen) {
			tib[span] = '\0';
			return;
		}
	}
}



// for Arduino control functions


// for Debug
void led(int f)
{
	if (f) {
		digitalWrite(13, HIGH);
	}
	else {
		digitalWrite(13, LOW);
	}
	
}

void setMotorSpeed(int power_l,int power_r)
{
    analogWrite(PIN_MOTOR_L_VREF,power_l);
    analogWrite(PIN_MOTOR_R_VREF,power_r);
    delay(10);
}

void controlMotor(int motortype,int motoraction)
{
    int pinno_1;
    int pinno_2;
    
    const unsigned char cont[4][2] = {
	{ LOW	,	LOW	},	// STOP
	{ HIGH	,	LOW	},	// FORWARD
	{ LOW	,	HIGH	},	// REVERSE
	{ HIGH	,	HIGH	}	// BRAKE
    };
    
    if(motortype == MOTORTYPE_MOTOR_R){
	pinno_1 = PIN_MOTOR_R_CONTROL1;
	pinno_2 = PIN_MOTOR_R_CONTROL2;
    } else {
	pinno_1 = PIN_MOTOR_L_CONTROL1;
	pinno_2 = PIN_MOTOR_L_CONTROL2;
    }
    
    digitalWrite(pinno_1,cont[motoraction][0]);
    digitalWrite(pinno_2,cont[motoraction][1]);  
}



void DebugPrintNum(int n)
{
#ifdef __DEBUG_CODE_ENABLE__
	PROGMEM static const prog_uchar  debugnum[] = "DEBUG NUM HEX:";
	PROGMEM static const prog_uchar  debugok[] = " OK";
	pmStrOutput((char*)debugnum);
	Serial_printfmt(n, HEX);
        pmStrOutput((char*)debugok);
	Serial.println();
#endif
}

void DebugRamDump(unsigned char *adrs, int n)
{
#ifdef __DEBUG_CODE_ENABLE__
	PROGMEM static const prog_uchar  debugdump[] = "DEBUG RAM DUMP:";
	pmStrOutput((char*)debugdump);
	Serial_printfmt((int)adrs, HEX);
	for (int i = 0; i<n; i++) {
		if ( (i  % 16) == 0 ) Serial.println();
		int x = *adrs++;
		Serial_printfmt((x >> 4) & 0xF, HEX);
		Serial_printfmt(x & 0xF, HEX);
		Serial_print(" ");
	}
	Serial.println();
#endif
}

void DebugStackDump(int sp)
{
#ifdef __DEBUG_CODE_ENABLE__
	PROGMEM static const prog_uchar  debugdump[] = "DEBUG STACK DUMP DEPTH:";
	pmStrOutput((char*)debugdump);
	int last = (int)GetDataStackLast();
	int len = (last - sp) / 2;
	int *adrs = (int*)sp;
	Serial_printfmt(len, DEC);
	Serial.println();
	for (int i = 0; i<len; i++) {
		int x = *adrs++;
		Serial_printfmt((x >> 12) & 0xF, HEX);
		Serial_printfmt((x >> 8) & 0xF, HEX);
		Serial_printfmt((x >> 4) & 0xF, HEX);
		Serial_printfmt(x & 0xF, HEX);
		Serial.println();
	}
	Serial.println();
#endif
}

/****************************************
 *		Reference		*
 *	AVR micom register MAP		*
 ****************************************/
//	R0	fixed (temporary register)
//	R1	fixed (zero register)
//	R3:R2	saved
//	R5:R4	saved
//	R7:R6	saved
//	R9:R8	saved		9th parameter
//	R11:R10	saved		8th parameter
//	R13:R12	saved		7th parameter
//	R15:R14	saved		6th parameter
//	R17:R16	saved		5th parameter
//	R19:R18	clobbered	4th parameter	64-bit
//	R21:R20	clobbered	3rd parameter	64-bit
//	R23:R22	clobbered	2nd parameter	32-bit, 64-bit
//	R25:R24	clobbered	1st parameter	8-bit, 16-bit, 32-bit, 64-bit
//	R27:R26 (X register)	clobbered
//	R29:R28 (Y register)	saved, stack frame pointer
//	R31:R30 (Z register)	clobbered


/********************************
 *      setup                   *
 ********************************/
void setup() {
  tiblen = 80;
  span = 0;
  to_in = 0;
  tib[0] = 0;
  tokenbuffer[0] = 0;
  latest = 0;
  state = 0;	// interpretin state
  save_rsp = -1;
  csp = -1;
  base = 10;
  dp = 0;



    /////////////////////////////////////////////////
    // GPIO SETUP
    /////////////////////////////////////////////////

    // pin direction setup
    pinMode(2,OUTPUT);
    pinMode(3,OUTPUT);
    pinMode(4,OUTPUT);
    pinMode(5,OUTPUT);    // Vref (R ch)
    pinMode(6,OUTPUT);    // Vref (L ch)
    pinMode(7,OUTPUT);
  
    pinMode(8,OUTPUT);    // IN1 (Rch)
    pinMode(9,OUTPUT);    // IN2 (Rch)
    pinMode(10,OUTPUT);    // IN1 (Rch)
    pinMode(11,OUTPUT);    // IN2 (Rch)

    pinMode(12,OUTPUT);
    pinMode(13,OUTPUT);

    // pin output value setup
    digitalWrite(8,LOW);
    digitalWrite(9,LOW);
    digitalWrite(10,LOW);
    digitalWrite(11,LOW);


    /////////////////////////////////////////////////
    // MISC SETUP
    /////////////////////////////////////////////////

    // initialize peripherals
    analogWrite(PIN_MOTOR_R_VREF,0);
    analogWrite(PIN_MOTOR_L_VREF,0);







//  Serial.begin(9600);
  Serial.begin(SERIAL_COMMUNICATION_BAUD_RATE);
  Serial.i
  Serial.flush();

}


/********************************
 *      loop                    *
 ********************************/
void loop()
{
// register assignment
// R3:R2		IP Instruction Pointer
// R5:R4		W Work
// R29:R28 (Y register)	SP data stack pointer

__asm__ __volatile__ (

/********************************
 *      MACROS                  *
 ********************************/

// DS_PUSH hi,lo
// push to data stack
//	SP -= 2
"\
.macro DS_PUSH hi,lo							\n\
	st	-y,\\hi		; hi					\n\
	st	-y,\\lo		; lo					\n\
.endm									\n"

// DS_POP hi,lo
// pop from data stack
//	SP += 2
"\
.macro DS_POP hi,lo							\n\
	ld	\\lo,y+		; lo					\n\
	ld	\\hi,y+		; hi					\n\
.endm									\n"

// DS_PEEK hi,lo
// peek data stack
"\
.macro DS_PEEK hi,lo							\n\
	ld	\\lo,y		; lo					\n\
	ldd	\\hi,y+1	; hi					\n\
.endm									\n"


// LD_PM_B rx, <- (z+)
// load from progmem byte
//	Z REG += 1
"\
.macro LD_PM_B rx							\n\
	lpm	\\rx,z+		; rx <- (z+)				\n\
.endm									\n"


// LD_PM_W hi,lo <- (z+)
// load from progmem word
//	Z REG += 2
"\
.macro LD_PM_W hi,lo							\n\
	lpm	\\lo,z+		; lo <- (z+)				\n\
	lpm	\\hi,z+		; hi <- (z+)				\n\
.endm									\n"


/********************************
 *      MAIN ROUTINE            *
 ********************************/
// INIT START
"\
INIT:									\n\
	call	__OpeningMessagePrint					\n\
	in	r24,__SP_L__		; save init return stack pointer\n\
	sts	__save_rsp,r24						\n\
	in	r25,__SP_H__						\n\
	sts	__save_rsp+1,r25					\n\
	call	__GetDataStackLast					\n\
	movw	r28,r24			; set data stack pointer	\n\
	ldi	r30,lo8(__BUILTIN_LAST_WARD)	; last ward address set	\n\
	ldi	r31,hi8(__BUILTIN_LAST_WARD)				\n\
	LD_PM_W	r25,r24							\n\
	ldi	r26,lo8(__latest)					\n\
	ldi	r27,hi8(__latest)					\n\
	st	x+,r24							\n\
	st	x,r25							\n\
;	ldi	r24,lo8(__dic)		; dictionary pointer set	\n\
;	ldi	r25,hi8(__dic)						\n\
	ldi	r24,lo8(__qbfWordPad)	; dictionary on RAM pointer set	\n\
	ldi	r25,hi8(__qbfWordPad)					\n\
	ldi	r26,lo8(__dp)						\n\
	ldi	r27,hi8(__dp)						\n\
	st	x+,r24							\n\
	st	x,r25							\n\
									\n\
	ldi	r24,lo8(CLD_01)		; set IP			\n\
	mov	r2,r24							\n\
	ldi	r24,hi8(CLD_01)						\n\
	mov	r3,r24							\n\
	jmp	NEXT							\n\
CLD_01:									\n\
	.word	CF_COLD							\n\
"



/********************************
 *      DEBUG ROUTINE           *
 ********************************/
// void PutDebugNum(r25:r24)
"\
PutDebugNum:								\n\
	push	r22							\n\
	push	r23							\n\
	push	r24							\n\
	push	r25							\n\
	push	r26							\n\
	push	r27							\n\
	push	r30							\n\
	push	r31							\n\
	call	__DebugPrintNum						\n\
	pop	r31							\n\
	pop	r30							\n\
	pop	r27							\n\
	pop	r26							\n\
	pop	r25							\n\
	pop	r24							\n\
	pop	r23							\n\
	pop	r22							\n\
	ret								\n\
"



/************************************************************************
 *      WORD DICTIONARY							*
 ************************************************************************/
"\
; CONSTANTS DEFINITIONS							\n\
.equ	WORD_MSB	,0x80						\n\
.equ	WORD_IMMIDIATE	,0x40						\n\
.equ	WORD_SMUDGE	,0x20						\n\
.equ	WORD_LENGTH_MASK,0x1F						\n\
									\n\
; TRACE FLAG for DEBUG							\n\
.equ	WORD_TRACE,0							\n\
"

// DICTIONARY MACROS
#define TO_LABEL(x)  #x
#define	NF_(x)	TO_LABEL(NF_##x)		//	Name Field 
#define	NF(x)	NF_(x)
//#define	LF_(x)	TO_LABEL(LF_##x)	//	Link Field 
//#define	LF(x)	LF_(x)
#define	CF_(x)	TO_LABEL(CF_##x)		//	Code Field 
#define	CF(x)	CF_(x)
#define	PF_(x)	TO_LABEL(PF_##x)		//	Parameter Field 
#define	PF(x)	PF_(x)
#define	SET_LAST_WORD	".equ\tDIC_LAST_WORD," NF(WORD_NAME) "\t\n"
#define WORDNAME_(x)	TO_LABEL(#x)
#define WORDNAME__(x)	WORDNAME_(x)
#define WORDNAME	WORDNAME__(WORD_NAME)

/************************************************************************
 *      BUILTIN DICTIONARY ENTRY START					*
 ************************************************************************/

".equ\tDIC_LAST_WORD	,0\n"	// LAST WORD NFA

/************************************************
 * WORD COLD					*
 *	Initial process TODO			*
 ************************************************/
#define	WORD_NAME	COLD

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_ABORT						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD


/************************************************
 * WORD DEBUG_RAM_DUMP ( adrs n --- )		*
 ************************************************/
#define	WORD_NAME	DEBUG_RAM_DUMP

NF(WORD_NAME) ":							\n\
	.byte	14 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	call	__DebugRamDump						\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD DEBUG_STACK_DUMP ( --- )		*
 ************************************************/
#define	WORD_NAME	DEBUG_STACK_DUMP

NF(WORD_NAME) ":							\n\
	.byte	16 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	movw	r24,r28							\n\
	call	__DebugStackDump					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD DEBUG_DIC_DUMP ( --- )			*
 ************************************************/
#define	WORD_NAME	DEBUG_DIC_DUMP

NF(WORD_NAME) ":							\n\
	.byte	14 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	ldi	r24,lo8(__qbfWordPad)	; dictionary on RAM pointer set	\n\
	ldi	r25,hi8(__qbfWordPad)					\n\
	ldi	r22,64							\n\
	ldi	r23,0							\n\
	call	__DebugRamDump						\n\
	jmp	NEXT							\n"
SET_LAST_WORD



/************************************************
 * WORD STATE ( -- a-addr )			*
 ************************************************/
#define	WORD_NAME	STATE

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r24,lo8(__state)					\n\
	ldi	r25,hi8(__state)					\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD LATEST ( -- a-addr )			*
 ************************************************/
#define	WORD_NAME	LATEST

NF(WORD_NAME) ":							\n\
	.byte	6 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r24,lo8(__latest)					\n\
	ldi	r25,hi8(__latest)					\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD BASE ( -- a-addr )			*
 ************************************************/
#define	WORD_NAME	BASE

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r24,lo8(__base)						\n\
	ldi	r25,hi8(__base)						\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD DP ( -- a-addr )			*
 ************************************************/
#define	WORD_NAME	DP

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r24,lo8(__dp)						\n\
	ldi	r25,hi8(__dp)						\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD CSP ( -- a-addr )			*
 ************************************************/
#define	WORD_NAME	CSP

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r24,lo8(__csp)						\n\
	ldi	r25,hi8(__csp)						\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD




/************************************************
 * WORD EXIT         				*
 ************************************************/
#define	WORD_NAME	EXIT

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	pop	r2		; pop IP				\n\
	pop	r3							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD : ( -- )				*
 ************************************************/
#define	WORD_NAME	COLON

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\":\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	;;								\n\
	;; TODO Compiler						\n\
	;; 								\n\
	.word	CF_Q_EXEC		; ?EXEC				\n\
	.word	CF_DPSTORE_		; DP! init			\n\
	.word	CF_S_CSP		; !CSP				\n\
	.word	CF_P_CREATE		; (CREATE)			\n\
	.word	CF_Q_CSP		; ?CSP				\n\
	.word	CF_RBRACKET_		; ]				\n\
	.word	CF_P_S_CODE		; (;CODE)			\n\
DOCOL:									\n\
	push	r3			; push IP			\n\
	push	r2							\n\
	ldi	r24,2							\n\
	add	r4,r24			; W += 2			\n\
	adc	r5,__zero_reg__						\n\
	movw	r2,r4			; IP = W			\n\
NEXT:									\n\
	movw	r30,r2			; W = (IP+)			\n\
	LD_PM_W	r5,r4							\n\
	movw	r2,r30							\n\
	movw	r30,r4			; JMP (W)			\n\
	LD_PM_W	r25,r24							\n\
	movw	r30,r24							\n\
	lsr	r31							\n\
	ror	r30							\n\
	ijmp								\n"
SET_LAST_WORD

/************************************************
 * WORD (;CODE) ( -- )				*
 ************************************************/
#define	WORD_NAME	P_S_CODE

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB						\n\
	.ascii	\"(;CODE)\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_FROM_R	; R>					\n\
;	.word	CF_LATEST						\n\
;	.word	CF_FETCH_						\n\
	.word	CF_WORDPAD_CFA						\n\
 .word	CF_DEBUG_STACK_DUMP	; debug				\n\
	.word	CF_STORE_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD


/************************************************
 * WORD ; ( -- ) IMMIDIATE			*
 ************************************************/
#define	WORD_NAME	SEMICOLON

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	\";\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
 .word	CF_DEBUG_DIC_DUMP	; debug				\n\
	.word	CF_Q_CSP						\n\
	.word	CF_COMPILE						\n\
	.word	CF_EXIT			; EXIT = ;S			\n\
;	.word	CF_SMUDGE						\n\
 .word	CF_DEBUG_DIC_DUMP	; debug				\n\
	.word	CF_LBRACKET_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD EXECUTE ( a -- )			*
 ************************************************/
#define	WORD_NAME	EXECUTE

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r5,r4			; pop W	= cfa			\n\
	movw	r30,r4			; jmp (W)			\n\
	LD_PM_W	r25,r24							\n\
	movw	r30,r24							\n\
	lsr	r31							\n\
	ror	r30							\n\
	ijmp								\n"
SET_LAST_WORD

/************************************************
 * WORD BRANCH	( --- )				*
 ************************************************/
#define	WORD_NAME	BRANCH

NF(WORD_NAME) ":							\n\
	.byte	6 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	movw	r30,r2			; IP += (IP)			\n\
	LD_PM_W	r25,r24							\n\
	add	r2,r24							\n\
	adc	r3,r25							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ?BRANCH	( --- )				*
 ************************************************/
#define	WORD_NAME	Q_BRANCH

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB						\n\
	.ascii	\"?BRANCH\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	or	r24,r25							\n\
	breq	Q_BRANCH_01						\n\
	ldi	r24,2							\n\
	add	r2,r24			; IP += 2			\n\
	adc	r3,__zero_reg__						\n\
	rjmp	Q_BRANCH_02						\n\
Q_BRANCH_01:								\n\
	movw	r30,r2			; IP += (IP)			\n\
	LD_PM_W	r25,r24							\n\
	add	r2,r24							\n\
	adc	r3,r25							\n\
Q_BRANCH_02:								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD LIT	( --- n )			*
 ************************************************/
#define	WORD_NAME	LIT

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1				 			\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi	r24,lo8(" NF(WORD_NAME) ")				\n\
	ldi	r25,hi8(" NF(WORD_NAME) ")				\n\
;	call	__WordNamePrint						\n\
.endif									\n\
	movw	r30,r2			; PUSH	(IP+)			\n\
	LD_PM_W	r25,r24							\n\
	movw	r2,r30							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD PM@	( pm-adrs --- n )		*
 ************************************************/
#define	WORD_NAME	PMFETCH_

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	\"PM@\"							\n\
	.p2align 1				 			\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi	r24,lo8(" NF(WORD_NAME) ")				\n\
	ldi	r25,hi8(" NF(WORD_NAME) ")				\n\
;	call	__WordNamePrint						\n\
.endif									\n\
	DS_POP	r31,r30			; DS_POP Z			\n\
	LD_PM_W	r25,r24							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD NFA ( pfa -- nfa )			*
 ************************************************/
#define	WORD_NAME	NFA

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	sbiw	r24,5							\n\
	movw	r30,r24							\n\
NFA_00:									\n\
	lpm	r24,z							\n\
	and	r24,r24							\n\
	brlt	NFA_01							\n\
	sbiw	r30,1							\n\
	rjmp	NFA_00							\n\
NFA_01:									\n\
	DS_PUSH	r31,r30							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD LFA ( pfa -- lfa )			*
 ************************************************/
#define	WORD_NAME	LFA

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	sbiw	r24,4							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD CFA ( pfa -- cfa )			*
 ************************************************/
#define	WORD_NAME	CFA

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	sbiw	r24,2							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD PFA ( nfa -- pfa )			*
 ************************************************/
#define	WORD_NAME	PFA

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	movw	r30,r24							\n\
	lpm	r24,z							\n\
	and	r24,WORD_LENGTH_MASK					\n\
	add	r30,r24							\n\
	adc	r31,__zero_reg__					\n\
	mov	r24,r30							\n\
	andi	r24,1							\n\
	breq	PFA_00							\n\
	adiw	r30,1			; it's LFA			\n\
PFA_00:									\n\
	adiw	r30,4			; LFA to PFA			\n\
	DS_PUSH	r31,r30							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD NFA2CFA ( nfa -- cfa )			*
 ************************************************/
#define	WORD_NAME	NFA2CFA

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	movw	r30,r24							\n\
	lpm	r24,z							\n\
	and	r24,WORD_LENGTH_MASK					\n\
	add	r30,r24							\n\
	adc	r31,__zero_reg__					\n\
	mov	r24,r30							\n\
	andi	r24,1							\n\
	breq	NFA2CFA_00						\n\
	adiw	r30,1			; it's LFA			\n\
NFA2CFA_00:								\n\
	adiw	r30,2			; LFA to CFA			\n\
	DS_PUSH	r31,r30							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD WORDPAD_CFA ( --- cfa )			*
 ************************************************/
#define	WORD_NAME	WORDPAD_CFA

NF(WORD_NAME) ":							\n\
	.byte	11 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r24,lo8(__qbfWordPad)	; dictionary on RAM pointer set	\n\
	ldi	r25,hi8(__qbfWordPad)					\n\
	movw	r26,r24							\n\
	ld	r24,x							\n\
	andi	r24,WORD_LENGTH_MASK					\n\
	inc	r24							\n\
	add	r26,r24							\n\
	adc	r27,__zero_reg__					\n\
	mov	r24,r26							\n\
	andi	r24,1							\n\
	breq	WORDPAD_CFA_00						\n\
	adiw	r26,1			; it's LFA			\n\
WORDPAD_CFA_00:								\n\
	adiw	r26,2			; LFA to CFA			\n\
	DS_PUSH	r27,r26							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD 1+ ( n --- n+1 )			*
 ************************************************/
#define	WORD_NAME	ONE_PLUS_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"1+\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	adiw	r24,1							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD 2+ ( n --- n+2 )			*
 ************************************************/
#define	WORD_NAME	TWO_PLUS_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"2+\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	adiw	r24,2							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD 1- ( n --- n-1 )			*
 ************************************************/
#define	WORD_NAME	ONE_MINUS_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"1-\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	sub	r24,1							\n\
	sbc	r25,__zero_reg__					\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD 2- ( n --- n-2 )			*
 ************************************************/
#define	WORD_NAME	TWO_MINUS_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"2-\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	sub	r24,2							\n\
	sbc	r25,__zero_reg__					\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD


/************************************************
 * WORD ?COMP         				*
 ************************************************/
#define	WORD_NAME	Q_COMP

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	\"?COMP\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r26,lo8(__state)					\n\
	ldi	r27,hi8(__state)					\n\
	ld	r24,x+							\n\
	ld	r25,x							\n\
	or	r24,r25							\n\
	brne	Q_COMP_01						\n\
	ldi	r24,15		; Compilation only, use in definition	\n\
	clr	r25							\n\
	call	__ErrorMessagePrint					\n\
Q_COMP_01:								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ?EXEC					*
 ************************************************/
#define	WORD_NAME	Q_EXEC

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	\"?EXEC\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi	r24,lo8(" NF(WORD_NAME) ")				\n\
	ldi	r25,hi8(" NF(WORD_NAME) ")				\n\
	call	__WordNamePrint						\n\
.endif									\n\
	ldi	r26,lo8(__state)					\n\
	ldi	r27,hi8(__state)					\n\
	ld	r24,x+							\n\
	ld	r25,x							\n\
	or	r24,r25							\n\
	breq	Q_EXEC_01						\n\
	ldi	r24,16		; Execution only			\n\
	clr	r25							\n\
	call	__ErrorMessagePrint					\n\
Q_EXEC_01:								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ?PAIRS					*
 ************************************************/
#define	WORD_NAME	Q_PAIRS

NF(WORD_NAME) ":							\n\
	.byte	6 + WORD_MSB						\n\
	.ascii	\"?PAIRS\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	DS_POP	r23,r22							\n\
	cp	r22,r24							\n\
	brne	Q_PAIRS_01						\n\
	cp	r23,r25							\n\
	breq	Q_PAIRS_02						\n\
Q_PAIRS_01:								\n\
	ldi	r24,13		; Error #13				\n\
	clr	r25							\n\
	call	__ErrorMessagePrint					\n\
Q_PAIRS_02:								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ?STACK	TODO				*
 ************************************************/
#define	WORD_NAME	Q_STACK

NF(WORD_NAME) ":							\n\
	.byte	6 + WORD_MSB						\n\
	.ascii	\"?STACK\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ?CSP					*
 ************************************************/
#define	WORD_NAME	Q_CSP

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	\"?CSP\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_SPFETCH_						\n\
 .word	CF_DUP						\n\
 .word	CF_DOT_						\n\
	.word	CF_CSP							\n\
	.word	CF_FETCH_						\n\
 .word	CF_DUP						\n\
 .word	CF_DOT_						\n\
	.word	CF_MINUS_						\n\
	.word	CF_LIT							\n\
	.word	18			;Definition not finished	\n\
	.word	CF_Q_ERROR						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD !CSP					*
 ************************************************/
#define	WORD_NAME	S_CSP

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	\"!CSP\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_SPFETCH_						\n\
	.word	CF_CSP							\n\
	.word	CF_STORE_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD ERROR ( n --- )			*
 ************************************************/
#define	WORD_NAME	ERROR

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	call	__ErrorMessagePrint					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ?ERROR ( f n --- )			*
 ************************************************/
#define	WORD_NAME	Q_ERROR

NF(WORD_NAME) ":							\n\
	.byte	6 + WORD_MSB						\n\
	.ascii	\"?ERROR\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_SWAP							\n\
	.word	CF_Q_BRANCH						\n\
	.word	Q_ERROR_00 - .						\n\
	.word	CF_ERROR						\n\
	.word	CF_BRANCH						\n\
	.word	Q_ERROR_01 - .						\n\
Q_ERROR_00:								\n\
	.word	CF_DROP							\n\
Q_ERROR_01:								\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD


/************************************************
 * WORD COMPILE	( --- )				*
 ************************************************/
#define	WORD_NAME	COMPILE

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_Q_COMP						\n\
	.word	CF_FROM_R	; R>					\n\
	.word	CF_DUP							\n\
	.word	CF_TWO_PLUS_						\n\
	.word	CF_TO_R		; >R					\n\
	.word	CF_PMFETCH_	; FETCH FROM PROGMEM			\n\
	.word	CF_COMMA_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD BEGIN	( --- a 3 )	IMMIDIATE	*
 ************************************************/
#define	WORD_NAME	BEGIN

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_Q_COMP						\n\
	.word	CF_LT_MARK						\n\
	.word	CF_LIT							\n\
	.word	3							\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD AGAIN	( a 3 --- )	IMMIDIATE	*
 ************************************************/
#define	WORD_NAME	AGAIN

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_LIT							\n\
	.word	3							\n\
	.word	CF_Q_PAIRS						\n\
	.word	CF_BRANCH		;;; TODO			\n\
	.word	CF_LT_MARK						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD FIND ( a1 -- a2 f ) like ANS Forth	*
 *	a1: search string address		*
 *	a2: found compilation address		*
 *	f:  1 immidiate				*
 *	   -1 normal				*
 *	    0 not found (a1=a2)			*
 ************************************************/
#define	WORD_NAME	FIND

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	call	__WordFind						\n\
	; resut 							\n\
	; r25,r24 found THEN cfa					\n\
	;	        ELSE terget					\n\
	; r22	 0 : not found						\n\
	;	-1 : normal						\n\
	;	 1 : immediate						\n\
	DS_PUSH	r25,r24							\n\
	DS_PUSH	r23,r22							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD IF	( --- a 1 )	IMMIDIATE	*
 ************************************************/
#define	WORD_NAME	IF

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_Q_COMP	; ?COMP					\n\
	.word	CF_COMPILE						\n\
	.word	CF_Q_BRANCH						\n\
	.word	CF_GT_MARK						\n\
	.word	CF_LIT							\n\
	.word	1							\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD ELSE	( a1 1 --- a2 1 ) IMMIDIATE	*
 ************************************************/
#define	WORD_NAME	ELSE

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_LIT							\n\
	.word	1							\n\
	.word	CF_Q_PAIRS	; ?PAIRS				\n\
	.word	CF_COMPILE						\n\
	.word	CF_BRANCH						\n\
	.word	CF_GT_MARK						\n\
	.word	CF_SWAP							\n\
	.word	CF_GT_RESOLVE						\n\
	.word	CF_LIT							\n\
	.word	1							\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD THEN	( a 1 ---  )	IMMIDIATE	*
 ************************************************/
#define	WORD_NAME	THEN

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_LIT							\n\
	.word	1							\n\
	.word	CF_Q_PAIRS	; ?PAIRS				\n\
	.word	CF_GT_RESOLVE						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD


/************************************************
 * WORD (CREATE) ( -- )				*
 ************************************************/
#define	WORD_NAME	P_CREATE

NF(WORD_NAME) ":							\n\
	.byte	8 + WORD_MSB						\n\
	.ascii	\"(CREATE)\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
; redefinition check TODO						\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_HERE							\n\
	.word	CF_LIT				; BL			\n\
	.word	' '				;			\n\
	.word	CF_WORD				; WORD ( c-addr1 )	\n\
	.word	CF_Q_DUP						\n\
	.word	CF_Q_BRANCH						\n\
	.word	P_CREATE_02 - .						\n\
	.word	CF_DUP							\n\
	.word	CF_CFETCH_			; Length		\n\
	.word	CF_DUP							\n\
	.word	CF_ZERO_GT_			; 0> IF			\n\
	.word	CF_Q_BRANCH						\n\
	.word	P_CREATE_00 - .						\n\
	.word	CF_ONE_PLUS_			; len++			\n\
	.word	CF_DUP				;			\n\
	.word	CF_LIT							\n\
	.word	1							\n\
	.word	CF_AND							\n\
	.word	CF_Q_BRANCH			; IF			\n\
	.word	P_CREATE_01 - .						\n\
	.word	CF_ONE_PLUS_			; len odd then ++	\n\
P_CREATE_01:								\n\
	.word	CF_DUP							\n\
	.word	CF_ALLOT						\n\
	.word	CF_SWAP							\n\
	.word	CF_DUP							\n\
	.word	CF_DUP							\n\
	.word	CF_CFETCH_						\n\
	.word	CF_LIT							\n\
	.word	0x80				; MSB SET		\n\
	.word	CF_OR							\n\
	.word	CF_SWAP							\n\
	.word	CF_CSTORE_						\n\
	.word	CF_SWAP							\n\
	.word	CF_ROT							\n\
	.word	CF_SWAP							\n\
	.word	CF_CMOVE		; copy word to dic buffer	\n\
	.word	CF_LATEST						\n\
	.word	CF_FETCH_						\n\
	.word	CF_COMMA_						\n\
	.word	CF_HERE							\n\
	.word	CF_TWO_PLUS_						\n\
	.word	CF_COMMA_						\n\
 .word	CF_DEBUG_DIC_DUMP	; debug				\n\
	.word	CF_EXIT							\n\
P_CREATE_00:					; name undefined	\n\
	.word	CF_DROP							\n\
	.word	CF_DROP							\n\
	.word	CF_DROP							\n\
P_CREATE_02:					; name undefined	\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD


/************************************************
 * WORD LITERAL	( a --- n ) IMMIDIATE	TODO	*
 ************************************************/
#define	WORD_NAME	LITERAL

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1				  			\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_STATE						\n\
	.word	CF_FETCH_						\n\
;	.word	CF_Q_BRANCH						\n\
;	.word							\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD HERE ( --- dic-address )		*
 ************************************************/
#define	WORD_NAME	HERE

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	  						\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_DP							\n\
	.word	CF_FETCH_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD ALLOT ( n --- )				*
 ************************************************/
#define	WORD_NAME	ALLOT

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	  						\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_DP							\n\
 .word CF_DEBUG_STACK_DUMP	\n\
	.word	CF_P_STORE_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD <MARK ( --- a )				*
 ************************************************/
#define	WORD_NAME	LT_MARK

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	\"<MARK\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_HERE							\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD >MARK ( --- a )				*
 ************************************************/
#define	WORD_NAME	GT_MARK

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	\">MARK\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_HERE							\n\
	.word	CF_LIT							\n\
	.word	0							\n\
	.word	CF_COMMA_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD <RESOLVE ( a --- )			*
 ************************************************/
#define	WORD_NAME	LT_RESOLVE

NF(WORD_NAME) ":							\n\
	.byte	8 + WORD_MSB						\n\
	.ascii	\"<RESOLVE\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_HERE							\n\
	.word	CF_MINUS_						\n\
	.word	CF_COMMA_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD >RESOLVE ( a --- )			*
 ************************************************/
#define	WORD_NAME	GT_RESOLVE

NF(WORD_NAME) ":							\n\
	.byte	8 + WORD_MSB						\n\
	.ascii	\">RESOLVE\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_HERE							\n\
	.word	CF_OVER							\n\
	.word	CF_MINUS_						\n\
	.word	CF_SWAP							\n\
	.word	CF_STORE_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD DUP         				*
 ************************************************/
#define	WORD_NAME	DUP

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	  						\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_PEEK	r25,r24							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD SWAP         				*
 ************************************************/
#define	WORD_NAME	SWAP

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	  						\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	DS_POP	r23,r22							\n\
	DS_PUSH	r25,r24							\n\
	DS_PUSH	r23,r22							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD OVER         				*
 ************************************************/
#define	WORD_NAME	OVER

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	  						\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	DS_PEEK	r23,r22							\n\
	DS_PUSH	r25,r24							\n\
	DS_PUSH	r23,r22							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ROT					*
 ************************************************/
#define	WORD_NAME	ROT

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	  						\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	DS_POP	r23,r22							\n\
	DS_POP	r21,r20							\n\
	DS_PUSH	r23,r22							\n\
	DS_PUSH	r25,r24							\n\
	DS_PUSH	r21,r20							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD DROP         				*
 ************************************************/
#define	WORD_NAME	DROP

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	 			 			\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ?DUP         				*
 ************************************************/
#define	WORD_NAME	Q_DUP

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	\"?DUP\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_PEEK	r25,r24							\n\
	tst	r24							\n\
	brne	QDUP_00							\n\
	tst	r25							\n\
	breq	QDUP_01							\n\
QDUP_00:								\n\
	DS_PUSH	r25,r24							\n\
QDUP_01:								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD DEPTH         				*
 ************************************************/
#define	WORD_NAME	DEPTH

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1	 			 			\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	call	__GetDataStackLast					\n\
	movw	r22,r28							\n\
	sub	r24,r22							\n\
	sbc	r25,r23							\n\
	lsr	r25			; word count			\n\
	ror	r24							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD . ( n --- ) TODO			*
 ************************************************/
#define	WORD_NAME	DOT_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\".\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	ldi	r26,lo8(__base)						\n\
	ldi	r27,hi8(__base)						\n\
	ld	r22,x							\n\
	ldi	r22,10		; debug					\n\
	clr	r23							\n\
	call	__Serial_printfmt					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD +					*
 ************************************************/
#define	WORD_NAME	PLUS_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\"+\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	add	r24,r22							\n\
	adc	r25,r23							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD -					*
 ************************************************/
#define	WORD_NAME	MINUS_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\"-\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	sub	r24,r22							\n\
	sbc	r25,r23							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ABS ( n --- n )				*
 ************************************************/
#define	WORD_NAME	ABS

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	tst	r25							\n\
	brge	ABS_00							\n\
	clr	r22							\n\
	clr	r23							\n\
	sub	r22,r24							\n\
	sbc	r23,r25							\n\
	DS_PUSH	r23,r22							\n\
	jmp	NEXT							\n\
ABS_00:									\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD <					*
 ************************************************/
#define	WORD_NAME	LT_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\"<\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	sub	r24,r22							\n\
	sbc	r25,r23							\n\
	brlt	LT_01							\n\
	clr	r24							\n\
	rjmp	LT_02							\n\
LT_01:									\n\
	ldi	r24,1							\n\
LT_02:									\n\
	clr	r25							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD >					*
 ************************************************/
#define	WORD_NAME	GT_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\">\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	sub	r22,r24							\n\
	sbc	r23,r25							\n\
	brlt	GT_01							\n\
	clr	r24							\n\
	rjmp	GT_02							\n\
GT_01:									\n\
	ldi	r24,1							\n\
GT_02:									\n\
	clr	r25							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD


/************************************************
 * WORD =					*
 ************************************************/
#define	WORD_NAME	EQ_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\"=\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	sub	r24,r22							\n\
	brne	EQ_01							\n\
	sub	r25,r23							\n\
	brne	EQ_01							\n\
	clr	r24							\n\
	rjmp	EQ_02							\n\
EQ_01:									\n\
	ldi	r24,1							\n\
EQ_02:									\n\
	clr	r25							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD 0<					*
 ************************************************/
#define	WORD_NAME	ZERO_LT_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"0<\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	tst	r25							\n\
	brlt	ZERO_LT_01						\n\
	clr	r24							\n\
	rjmp	ZERO_LT_02						\n\
ZERO_LT_01:								\n\
	ldi	r24,1							\n\
ZERO_LT_02:								\n\
	clr	r25							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD 0>					*
 ************************************************/
#define	WORD_NAME	ZERO_GT_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"0>\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	tst	r25							\n\
	brlt	ZERO_GT_02		; < 0				\n\
	or	r24,r25							\n\
	breq	ZERO_GT_02		; = 0				\n\
	ldi	r24,1							\n\
	rjmp	ZERO_GT_03						\n\
ZERO_GT_02:								\n\
	clr	r24							\n\
ZERO_GT_03:								\n\
	clr	r25							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD 0=					*
 ************************************************/
#define	WORD_NAME	ZERO_EQ_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"0=\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	or	r24,r25							\n\
	breq	ZERO_EQ_00		;				\n\
	clr	r24							\n\
	clr	r25							\n\
	rjmp	ZERO_EQ_01						\n\
ZERO_EQ_00:								\n\
	ldi	r24,1							\n\
ZERO_EQ_01:								\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD NOT					*
 ************************************************/
#define	WORD_NAME	NOT

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_ZERO_EQ_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD AND					*
 ************************************************/
#define	WORD_NAME	AND

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	and	r24,r22							\n\
	and	r25,r23							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD OR					*
 ************************************************/
#define	WORD_NAME	OR

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	or	r24,r22							\n\
	or	r25,r23							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD XOR					*
 ************************************************/
#define	WORD_NAME	XOR

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME)  "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r23,r22							\n\
	DS_POP	r25,r24							\n\
	eor	r24,r22							\n\
	eor	r25,r23							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD


/************************************************
 * WORD ,	(n --- )			*
 ************************************************/
#define	WORD_NAME	COMMA_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\",\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
 .word CF_DEBUG_STACK_DUMP	\n\
	.word	CF_HERE							\n\
	.word	CF_STORE_						\n\
	.word	CF_LIT							\n\
	.word	2							\n\
	.word	CF_ALLOT						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD >R ( n --- ) ( R: --- n )		*
 ************************************************/
#define	WORD_NAME	TO_R

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\">R\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	push	r25							\n\
	push	r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD R> ( R: n --- ) ( --- n )		*
 ************************************************/
#define	WORD_NAME	FROM_R

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"R>\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	pop	r24							\n\
	pop	r25							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD R@ ( R: n --- n ) ( --- n )		*
 ************************************************/
#define	WORD_NAME	R_FETCH

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"R@\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	pop	r24							\n\
	pop	r25							\n\
	push	r25							\n\
	push	r24							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD @ ( a --- n )				*
 ************************************************/
#define	WORD_NAME	FETCH_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\"@\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	movw	r30,r24							\n\
	ld	r24,z+							\n\
	ld	r25,z							\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD C@ ( a --- c )				*
 ************************************************/
#define	WORD_NAME	CFETCH_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"C@\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	movw	r30,r24							\n\
	ld	r24,z							\n\
	DS_PUSH	__zero_reg__,r24					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ! ( n a ---  )				*
 ************************************************/
#define	WORD_NAME	STORE_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB						\n\
	.ascii	\"!\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	movw	r30,r24							\n\
	DS_POP	r25,r24							\n\
	st	z+,r24							\n\
	st	z,r25							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD C! ( c a ---  )				*
 ************************************************/
#define	WORD_NAME	CSTORE_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"C!\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	movw	r30,r24							\n\
	DS_POP	r25,r24							\n\
	st	z,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD +! ( n a ---  )				*
 ************************************************/
#define	WORD_NAME	P_STORE_

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	\"+!\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
	.word	CF_SWAP							\n\
	.word	CF_OVER							\n\
	.word	CF_FETCH_						\n\
	.word	CF_PLUS_						\n\
	.word	CF_SWAP							\n\
	.word	CF_STORE_						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD WORD ( char "<chars>ccc<char>"  -- c-addr )	*
 ************************************************/
#define	WORD_NAME	WORD

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	call	__WordGet						\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD COUNT ( c-addr1 -- c-addr2 u )		*
 ************************************************/
#define	WORD_NAME	COUNT

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r27,r26			; DS_POP X			\n\
	ld	r24,x+			; r24 <- (x+)			\n\
	DS_PUSH	r27,r26			; DS_PUSH X			\n\
	DS_PUSH	__zero_reg__,r24					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD QUERY (  --- )				*
 ************************************************/
#define	WORD_NAME	QUERY

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
QUERY_00:								\n\
	call	__Query							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD RP!					*
 *	return stack pointer initialize		*
 ************************************************/
#define	WORD_NAME	RPSTORE_

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	\"RP!\"        						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	ldi	r26,lo8(__save_rsp)					\n\
	ldi	r27,hi8(__save_rsp)					\n\
	ld	r24,x+							\n\
	ld	r25,x  							\n\
	cli								\n\
	out	__SP_L__,r24		; return stack pointer init	\n\
	out	__SP_H__,r25						\n\
	sei								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD RP@					*
 *	return stack pointer fetch		*
 ************************************************/
#define	WORD_NAME	RPFETCH_

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	\"RP@\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	in	r24,__SP_L__						\n\
	in	r25,__SP_H__						\n\
	DS_PUSH	r25,r24							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD SP!					*
 *	data stack pointer initialize		*
 ************************************************/
#define	WORD_NAME	SPSTORE_

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	\"SP!\"        						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	call	__GetDataStackLast					\n\
	movw	r28,r24			; set data stack pointer	\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD SP@					*
 *	data stack pointer 			*
 ************************************************/
#define	WORD_NAME	SPFETCH_

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	\"SP@\"        						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	movw	r24,r28			; data stack pointer		\n\
	DS_PUSH	r25,r24			; push				\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD DP!					*
 ************************************************/
#define	WORD_NAME	DPSTORE_

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	\"DP!\"        						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	ldi	r24,lo8(__qbfWordPad)	; dictionary on RAM pointer set	\n\
	ldi	r25,hi8(__qbfWordPad)					\n\
	ldi	r26,lo8(__dp)						\n\
	ldi	r27,hi8(__dp)						\n\
	st	x+,r24							\n\
	st	x,r25							\n\
	jmp	NEXT							\n"
SET_LAST_WORD



/************************************************
 * WORD MOVE ( src dst cnt --- )		*
 *	cell(16bit) move 			*
 ************************************************/
#define	WORD_NAME	MOVE

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24			; cnt				\n\
	DS_POP	r27,r26			; x = dst			\n\
	DS_POP	r31,r30			; z = src			\n\
	movw	r22,r24							\n\
	or	r22,r23							\n\
	breq	MOVE_01			; cnt = 0 exit			\n\
MOVE_00:								\n\
	ld	__tmp_reg__,z+						\n\
	st	x+,__tmp_reg__						\n\
	ld	__tmp_reg__,z+						\n\
	st	x+,__tmp_reg__						\n\
	sbiw	r24,1							\n\
	brne	MOVE_00							\n\
MOVE_01:								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD CMOVE ( src dst cnt --- )		*
 *	byte move	 			*
 ************************************************/
#define	WORD_NAME	CMOVE

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24			; cnt				\n\
	DS_POP	r27,r26			; x = dst			\n\
	DS_POP	r31,r30			; z = src			\n\
	movw	r22,r24							\n\
	or	r22,r23							\n\
	breq	CMOVE_01		; cnt = 0 exit			\n\
CMOVE_00:								\n\
	ld	__tmp_reg__,z+						\n\
	st	x+,__tmp_reg__						\n\
	sbiw	r24,1							\n\
	brne	CMOVE_00						\n\
CMOVE_01:								\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD DECIMAL					*
 ************************************************/
#define	WORD_NAME	DECIMAL

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	ldi	r26,lo8(__base)						\n\
	ldi	r27,hi8(__base)						\n\
	st	x+,10			; BASE = 10			\n\
	st	x,__zero_reg__						\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD HEX					*
 ************************************************/
#define	WORD_NAME	HEX

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	ldi	r26,lo8(__base)						\n\
	ldi	r27,hi8(__base)						\n\
	st	x+,16			; BASE = 16			\n\
	st	x,__zero_reg__						\n\
	jmp	NEXT							\n"
SET_LAST_WORD


/************************************************
 * WORD [					*
 ************************************************/
#define	WORD_NAME	LBRACKET_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	\"[\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r26,lo8(__state)					\n\
	ldi	r27,hi8(__state)					\n\
	st	x+,__zero_reg__						\n\
	st	x,__zero_reg__						\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD ]					*
 ************************************************/
#define	WORD_NAME	RBRACKET_

NF(WORD_NAME) ":							\n\
	.byte	1 + WORD_MSB + WORD_IMMIDIATE				\n\
	.ascii	\"]\"							\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	ldi	r26,lo8(__state)					\n\
	ldi	r27,hi8(__state)					\n\
	ldi	r24,0xC0		; STATE = 0xC0			\n\
	st	x+,r24							\n\
	st	x,__zero_reg__						\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD (NUMBER) ( a --- n f )			*
 ************************************************/
#define	WORD_NAME	P_NUMBER

NF(WORD_NAME) ":							\n\
	.byte	8 + WORD_MSB						\n\
	.ascii	\"(NUMBER)\"						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24							\n\
	call	__ToNumSingle						\n\
	DS_PUSH	r25,r24							\n\
	DS_PUSH	r23,r22							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD NUMBER ( a --- n )			*
 ************************************************/
#define	WORD_NAME	NUMBER

NF(WORD_NAME) ":							\n\
	.byte	6 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_P_NUMBER		; ( a --- n f )			\n\
	.word	CF_Q_BRANCH						\n\
	.word	NUMBER_01 - .						\n\
	.word	CF_EXIT							\n\
NUMBER_01:								\n\
	.word	CF_LIT			; true				\n\
	.word	1							\n\
	.word	CF_LIT			; error #0			\n\
	.word	0							\n\
	.word	CF_ABORT						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD QUIT (  --  )				*
 ************************************************/
#define	WORD_NAME	QUIT

NF(WORD_NAME) ":							\n\
	.byte	4 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_LBRACKET_						\n\
QUIT_00:								\n\
	.word	CF_RPSTORE_						\n\
	.word	CF_CR							\n\
	.word	CF_QUERY						\n\
	.word	CF_INTERPRET						\n\
	.word	CF_STATE						\n\
	.word	CF_FETCH_						\n\
	.word	CF_ZERO_EQ_						\n\
	.word	CF_Q_BRANCH						\n\
	.word	QUIT_02 - .						\n\
	.word	CF_PRINT_OK						\n\
QUIT_02:								\n\
	.word	CF_BRANCH						\n\
	.word	QUIT_00 - .						\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD ABORT (  --  )				*
 ************************************************/
#define	WORD_NAME	ABORT

NF(WORD_NAME) ":							\n\
	.byte	5 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_SPSTORE_						\n\
	.word	CF_DECIMAL						\n\
	.word	CF_QUIT							\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD INTERPRET (  --  )			*
 ************************************************/
#define	WORD_NAME	INTERPRET

NF(WORD_NAME) ":							\n\
	.byte	9 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	DOCOL							\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
INTERPRET_00:								\n\
	.word	CF_LIT				; BL			\n\
	.word	' '				;			\n\
	.word	CF_WORD				; WORD			\n\
	.word	CF_DUP				; DUP			\n\
	.word	CF_CFETCH_			; C@			\n\
	.word	CF_Q_BRANCH			; 0= then exit		\n\
	.word	INTERPRET_06 - .					\n\
	.word	CF_FIND				; FIND			\n\
	.word	CF_Q_DUP			; ?DUP			\n\
	.word	CF_Q_BRANCH			; IF			\n\
	.word	INTERPRET_05 - .					\n\
	.word	CF_ZERO_LT_			;   <0			\n\
	.word	CF_Q_BRANCH			;   IF EXECUTE		\n\
	.word	INTERPRET_03 - .					\n\
	.word	CF_STATE			;   STATE		\n\
	.word	CF_FETCH_			;   @			\n\
	.word	CF_Q_BRANCH			;   IF			\n\
	.word	INTERPRET_03 - .					\n\
	.word	CF_COMMA_			;     ,			\n\
	.word	CF_BRANCH						\n\
	.word	INTERPRET_04 - .		;   ELSE		\n\
INTERPRET_03:								\n\
	.word	CF_EXECUTE			;     EXECUTE		\n\
INTERPRET_04:								\n\
	.word	CF_Q_STACK			;   ?STACK		\n\
	.word	CF_BRANCH			;			\n\
	.word	INTERPRET_00 - .					\n\
INTERPRET_05:					; ELSE			\n\
; TODO COMPILE	.word	CF_HERE				;   HERE		\n\
	.word	CF_NUMBER			;   NUMBER		\n\
; TODO COMPILE	.word	CF_LITERAL			;   LITERAL		\n\
	.word	CF_Q_STACK			;   ?STACK		\n\
	.word	CF_BRANCH			; jmp INTERPRET_00	\n\
	.word	INTERPRET_00 - .					\n\
INTERPRET_06:					; 			\n\
	.word	CF_DROP							\n\
	.word	CF_EXIT							\n"
SET_LAST_WORD

/************************************************
 * WORD CR					*
 ************************************************/
#define	WORD_NAME	CR

NF(WORD_NAME) ":							\n\
	.byte	2 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	call	__Serial_println					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD PRINT_OK				*
 ************************************************/
#define	WORD_NAME	PRINT_OK

NF(WORD_NAME) ":							\n\
	.byte	8 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
;.if (WORD_TRACE==1)							\n\
;	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
;	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
;	call __WordNamePrint						\n\
;.endif									\n\
	call	__PrintOk						\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD WORDNAMEPRINT				*
 ************************************************/
#define	WORD_NAME	WORDNAMEPRINT

NF(WORD_NAME) ":							\n\
	.byte	13 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
	DS_POP	r25,r24							\n\
	call	__WordNamePrint						\n\
	jmp	NEXT							\n"
SET_LAST_WORD




/************************************************************************
 * ARDUINO CONTROL WORDS						*
 ************************************************************************/

/************************************************
 * WORD analogWrite ( value pin --- )		*
 ************************************************/
#define	WORD_NAME	analogWrite

NF(WORD_NAME) ":							\n\
	.byte	11 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24			; pin				\n\
	DS_POP	r23,r22			; value				\n\
	call	__wiring_analogWrite					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD digitalWrite ( value pin --- )		*
 ************************************************/
#define	WORD_NAME	digitalWrite

NF(WORD_NAME) ":							\n\
	.byte	12 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24			; pin				\n\
	DS_POP	r23,r22			; value				\n\
	call	__wiring_digitalWrite					\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD LED ( f --- )				*
 ************************************************/
#define	WORD_NAME	LED

NF(WORD_NAME) ":							\n\
	.byte	3 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24			; flag				\n\
	call	__led							\n\
	jmp	NEXT							\n"
SET_LAST_WORD

/************************************************
 * WORD setMotorSpeed ( p_l p_r --- )		*
 ************************************************/
#define	WORD_NAME	setMotorSpeed

NF(WORD_NAME) ":							\n\
	.byte	13 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	ldi r24,lo8(" NF(WORD_NAME) ")					\n\
	ldi r25,hi8(" NF(WORD_NAME) ")					\n\
	call __WordNamePrint						\n\
.endif									\n\
	DS_POP	r25,r24			; pin				\n\
	DS_POP	r23,r22			; value				\n\
	call	__setMotorSpeed						\n\
	jmp	NEXT							\n"
SET_LAST_WORD





/************************************************
 * WORD MOTOR_L ( value --- )			*
 *	ch	0:L, 1:R			*
 *	value	-255..255			*
 ************************************************/
#define	WORD_NAME	MOTOR_L

NF(WORD_NAME) ":							\n\
	.byte	7 + WORD_MSB						\n\
	.ascii	" WORDNAME "						\n\
	.p2align 1							\n\
	.word	DIC_LAST_WORD						\n"
CF(WORD_NAME) ":							\n\
	.word	" PF(WORD_NAME) "					\n"
PF(WORD_NAME) ":							\n\
.if (WORD_TRACE==1)							\n\
	.word	CF_LIT							\n\
	.word	" NF(WORD_NAME) "					\n\
	.word	CF_WORDNAMEPRINT					\n\
.endif									\n\
	.word	CF_DUP							\n\
	.word	CF_ZERO_LT_		; sign_flag			\n\
	.word	CF_SWAP							\n\
	.word	CF_ABS							\n\
	.word	CF_DUP							\n\
	.word	CF_LIT							\n\
	.word	255							\n\
	.word	CF_GT_							\n\
	.word	CF_Q_BRANCH  \n\
	.word	MOTOR_L_00  \n\
	.word	CF_DROP							\n\
	.word	CF_LIT							\n\
	.word	255							\n\
MOTOR_L_00:				; ( sign value )		\n"
SET_LAST_WORD










/********************************************************
 * CAUTION 						*
 *	__BUILTIN_LAST_WARD placed at the end of dic	*
 ********************************************************/
"\n\
__BUILTIN_LAST_WARD:							\n\
	.word	DIC_LAST_WORD						\n"


// user dictionary area 
"\
__dic:									\n\
	.skip 10240							\n\
"
);

}


