-- Configuration file for Ateles --
sim_name = 'ns_const_state'

-- Variables to be set for the simulation --

-- Polynomial degree used for approximation
degree = 2 
-- Isothermal coefficient
isen = 1.4
--Boltzman constant
boltz= 287.0
-- Density
rho  = 1.4
-- Pressure
p    = 1.0
--Mach number
mach = 0.1
-- Background velocity 
vel  = mach * math.sqrt(isen * p / rho ) 
-- logging, with higher level, the solver writes out 
-- more information regarding the settings.
logging = { level = 8 }

-- Control the Simulation times --
-- Set the simultion time and when information should write out.
-- The max defines the max. simulation time, the min. where to 
-- start the simulation and the interval, after how many iteraition 
-- information should be written out to the output file.
sim_control = {
  time_control = { 
    max = {iter=20000}, 
    min = 0.0,
    interval = { iter = 1000 },
  }
}
-- Check for Nans and unphysical values
check =  { interval = 1 }


-- Mesh configuration -- 
-- The mesh is here a line, with starting 
-- points origin and a length of cube_length
-- and the number of elements of element_count

-- lenght of bounding cube
cube_length = 2.0
mesh = {
  predefined = 'line',     
  origin = {
    (-1.0)*cube_length/2.0,
    (-1.0)*cube_length/2.0,
    (-1.0)*cube_length/2.0
  },
  length = cube_length,
  element_count = 1
}

-- Equation definitions --
-- We use here Navier-Stokes 2D 
-- therefore we need to set the 
-- additional viscous parameters, we 
-- are not using a geometry here, therefore
-- the material paramters can be set to 0.0,
-- for velocity we need to set 2 components, as 
-- we have a 2D test case.
equation = {
  name      = 'navier_stokes_2d',                 
  isen_coef = isen,
  r         = boltz,
  -- Viscous parameters
  therm_cond = 0.5,
  mu         = 2.0,
  ip_param   = 4.0,
  material = {
    characteristic = 0.0,
    relax_velocity = {0.0, 0.0},
    relax_temperature = 0.0
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
-- modg_2d results in a 2D simulation
-- the temporal table defines which time stepping
-- scheme is used here. In this test case we consider
-- the one step explite Euler scheme. 
-- The Cfl defines the timestep width, for 
-- viscous flows we als need to define the cfl_visc. 
scheme = {
  spatial =  {
    name = 'modg_2d',
    m = degree,                 
  },
  temporal = {
    name = 'explicitEuler',
    control = {
      name     = 'cfl',
      cfl      = 0.1,
      cfl_visc = 0.4,
    },
  }, 
}

-- Projection type --
-- We consider here fast polynomial 
-- transformation with an oversampling 
-- factor of 2, which means that we 
-- use two times more points for the 
-- approximation
projection = {
  kind = 'fpt', 
  factor = 2.0,
}

-- Define the inital conditions --
-- We need to set density, pressure and 
-- the velocity in x and y direction
initial_condition = { 
  density  = rho,
  pressure = p,
  velocityX = vel,
  velocityY = 0.0,
}

-- Tracking --
-- We track here a point (just origin is given)
-- and the quantities momentum, density and 
-- energy. The interval defines after how 
-- many iterations the quantity information 
-- should be writen out.
tracking = {
  label = 'track_const_state_l2p',
  folder = './',
  variable = {'momentum','density','energy'},
  shape = {
    kind = 'canoND', 
    object= { 
      origin ={0., 0., 0.} 
    } 
  },
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = {iter = 1000}
  },
  output = { 
    format = 'ascii', 
    ndofs = 1 
  }
}
