-- Configuration file for Ateles --
sim_name = 'shear-hat'

-- Variables to be set for the simulation --

-- Polynomial degree used for approximation
degree = 3
-- Isothermal coefficient
isen = 1.4
--Boltzman constant
boltz = 287.0
-- Density
rho = 1.4
-- Pressure
p = 1.0
--Mach number
mach = 0.1
-- Viscousity parameters
conducT = 0.5
mu = 2.0
c2 = 8./3.
-- Spatial function to describe the y-velocity profile
vel  = function (x,y,z)
  if x > 0 then
    return mach * math.sqrt(isen * p / rho ) * (1. - (x-0.5*dx)/dx)
  else
    return mach * math.sqrt(isen * p / rho ) * (1. + (x+0.5*dx)/dx)
  end
end

-- Control the Simulation times --
-- Set the simultion time and when information should write out.
-- The max defines the max. simulation time, the min. where to 
-- start the simulation and the interval, after how many iteraition 
-- information should be written out to the output file.
sim_control = {
  time_control = {
    max      = { iter = 400 },
    min      = 0.0,
    interval = { iter = 1 },
  }
}

-- Mesh configuration -- 
-- The mesh is here a line, with starting 
-- points origin and a length of cube_length
-- and the number of elements of element_count

-- Element size for the mesh
dx = 1.0e-4
mesh = {
    predefined = 'line',
    origin = { -dx, -0.5*dx, -0.5*dx },
    length = 2*dx,
    element_count = 2
}

-- Equation definitions --
-- We use here Navier-Stokes 3D 
-- therefore we need to set the 
-- additional viscous parameters, we 
-- are not using a geometry here, therefore
-- the material paramters can be set to 0.0,
-- for velocity we need to set 3 components, as 
-- we have a 3D test case.
equation = {
  name      = 'navier_stokes',
  isen_coef = isen,
  r         = boltz,
  -- Viscous parameters
  therm_cond = conducT,
  mu         = mu,
  ip_param   = c2*3*(degree+2)/(2*(degree+3)),
  material = {
    characteristic = 0.0,
    relax_velocity = {0.0, 0.0, 0.0},
    relax_temperature = 0.0
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
-- modg results in a 3D simulation
-- the temporal table defines which time stepping
-- scheme is used here. In this test case we consider
-- the four step explite Runge-Kutta scheme. 
-- The Cfl defines the timestep width, for 
-- viscous flows we als need to define the cfl_visc. 
scheme = {
  spatial =  {
    name = 'modg',
    m = degree
  },
  temporal = {
    name = 'explicitRungeKutta',
    steps = 4,
    control = {
      name     = 'cfl', 
      cfl      = 0.8,
      cfl_visc = 0.4
    }
  }
}
-- Projection type --
-- We consider here fast polynomial 
-- transformation
projection = {
  kind = 'fpt',
  factor = 1.0
}

-- Define the inital conditions --
-- We need to set density, pressure and 
-- the velocity in x, y and z direction
initial_condition = {
  density   = rho,
  pressure  = p,
  velocityX = 0.0,
  velocityY = vel,
  velocityZ = 0.0
}

-- Tracking --
-- We track here a point (just origin is given)
-- and the quantities momentum, density and 
-- energy. The interval defines after how 
-- many iterations the quantity information 
-- should be writen out.
tracking = {
  label = 'track_shearhat3D',
  folder = './',
  variable = {'momentum','density','energy'},
  shape = {
    kind = 'canoND', 
    object= { 
      origin ={0.01*dx, 0., 0.} 
    } 
  },
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = {iter = 10}
  },
  output = { 
    format = 'ascii',
    ndofs = 1, 
    use_get_point = true 
  }
}
