-- Configuration file for Ateles --
simulation_name = 'shear_tube'

-- Variables to be set for the simulation --

-- Polynomial degree used for approximation
degree = 1
-- Pressure
press = 1
-- Density
dens = 1.4
--Mach number
mach = 0.8
-- Control the Simulation times --
-- Set the simultion time and when information should write out.
-- The max defines the max. simulation time, the min. where to 
-- start the simulation and the interval, after how many iteraition 
-- information should be written out to the output file.
sim_control = {
  time_control = {
    min = 0,
    max = {iter = 15},
    interval = {iter = 1},
    check_iter = 1
  }
}

-- Mesh configuration -- 
-- The mesh is here a line, with starting 
-- points origin and a length of cube_length.
-- The refinement level is set to 3
-- and the number of elements of element_count
mesh = {
    predefined = 'cube',
    refinementLevel = 3,
    length = 0.125,
    origin = {0.0, -0.0625, -0.0625}
}

-- Equation definitions --
-- We use here Navier-Stokes 3D 
-- therefore we need to set the 
-- additional viscous parameters, we 
-- are not using a geometry here, therefore
-- the material paramters can be set to 0.0,
-- for velocity we need to set 3 components, as 
-- we have a 3D test case.
equation = {
  name       = 'navier_stokes',
  isen_coef  = 1.4,
  r          = 296,
  therm_cond = 5.92e-3,
  mu         = 4e-6, 
  ip_param   = 4, 
  material = {
    characteristic = 0.0,
    relax_velocity = {0.0, 0.0, 0.0},
    relax_temperature = 1.0
  }
}
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
-- modg results in a 3D simulation
-- the temporal table defines which time stepping
-- scheme is used here. In this test case we consider
-- the two step explite Runge-Kutta scheme. Where we 
-- consider a fixed time step.
-- The stabilization table provides which filter is used
-- to stabilize the simulation.
scheme = {
  spatial = {
    name = 'modg',
    m    = degree
  },
  temporal = {
    name = 'explicitSSPRungeKutta',
    steps = 2,
    control = {
      name = 'fixed',
      dt = 2.5e-4
    }
  },
  stabilization = {
    name = 'cons_positivity_preserv',
    eps  = 1.0e-8
  }
}

-- Projection type --
-- We consider here fast polynomial 
-- transformation and an oversampling 
-- factor of 3.0, resulting in 3 times 
-- more integration points for the approximation.
-- Instead of the default Chebyshev nodes we use here 
-- Lobatto points 
projection = {
  kind = 'fpt',
  factor = 3.0,
  lobattoPoints = true
}

-- Description of the Jet-Tube
jet_radius = mesh.length * 2^(-mesh.refinementLevel)
velAmpl = mach*math.sqrt(equation.isen_coef*press/dens)

velX = function(x,y,z)
  if (math.abs(y) < jet_radius and math.abs(z) < jet_radius) then
    return velAmpl
  else
    return 0.0
  end
end

-- Define the inital conditions --
-- We need to set density, pressure and 
-- the velocity in x, y and z direction
initial_condition = { 
  density = dens,
  pressure = press,
  velocityX = velX,
  velocityY = 0.0,
  velocityZ = 0.0,
}

-- Tracking --
-- We track here a point (just origin is given)
-- and the quantities momentum, density and 
-- energy. The interval defines after how 
-- many iterations the quantity information 
-- should be writen out.
tracking = {
  label = 'track_momentum',
  folder = './',
  variable = { 'momentum' },
  shape = {
    kind = 'canoND',
    object= { 
      origin ={ jet_radius/2, jet_radius*3/2, jet_radius*3/2 } 
    }
  },
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = {iter = 1}
  },
  output = { 
    format = 'ascii', 
    use_get_point=true 
  },
}
