! Copyright (c) 2011-2014 Jens Zudrop <j.zudrop@grs-sim.de>
! Copyright (c) 2011 Metin Cakircali <m.cakircali@grs-sim.de>
! Copyright (c) 2011-2016 Harald Klimach <harald.klimach@uni-siegen.de>
! Copyright (c) 2011-2012 Laura Didinger <l.didinger@grs-sim.de>
! Copyright (c) 2011 Simon Zimny <s.zimny@grs-sim.de>
! Copyright (c) 2012-2013 Melven Zoellner <yameta@freenet.de>
! Copyright (c) 2012 Jan Hueckelheim <j.hueckelheim@grs-sim.de>
! Copyright (c) 2013-2014 Verena Krupp <verena.krupp@uni-siegen.de>
! Copyright (c) 2013-2015, 2017-2019 Peter Vitt <peter.vitt2@uni-siegen.de>
! Copyright (c) 2014-2016 Nikhil Anand <nikhil.anand@uni-siegen.de>
! Copyright (c) 2016 Tobias Girresser <tobias.girresser@student.uni-siegen.de>
! Copyright (c) 2017 Daniel Petró <daniel.petro@student.uni-siegen.de>
!
! Permission to use, copy, modify, and distribute this software for any
! purpose with or without fee is hereby granted, provided that the above
! copyright notice and this permission notice appear in all copies.
!
! THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHORS DISCLAIM ALL WARRANTIES
! WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
! MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR
! ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
! WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
! ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
! OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
! **************************************************************************** !

!> author: Jens Zudrop
!! Module containing all the types and subroutines to specify the
!! scheme of a solver.
module atl_scheme_module
  use aotus_module,                 only: flu_State, aot_get_val
  use aot_table_module,             only: aot_table_open, aot_table_close

  use env_module,                   only: rk, labelLen
  use tem_aux_module,               only: tem_abort
  use tem_tools_module,             only: upper_to_lower
!NA!  use tem_construction_module,      only: tem_levelNeighbor_type
  use tem_stencil_module,           only: tem_stencilHeader_type, &
    &                                     init
  use tem_logging_module,           only: logUnit


  use ply_modg_basis_module,        only: scalProdDualLeg,     &
    &                                     scalProdDualLegDiff, &
    &                                     ply_modg_basis_type

  use atl_modg_scheme_module,       only: atl_modg_scheme_type, &
    &                                     atl_modg_scheme_init
  use atl_modg_2d_scheme_module,    only: atl_modg_2d_scheme_type,    &
    &                                     atl_modg_2d_scheme_init
  use atl_modg_1d_scheme_module,    only: atl_modg_1d_scheme_type,    &
    &                                     atl_modg_1d_scheme_init
  use atl_stabilization_module,     only: atl_stabilization_type, &
    &                                     atl_ini_stabilization

  implicit none

  private

  integer, parameter :: atl_modg_scheme_prp = 6
  integer, parameter :: atl_modg_2d_scheme_prp = 7
  integer, parameter :: atl_modg_1d_scheme_prp = 8

  !> Datatype to specify the timestepping method.
  type atl_local_timestep_type
    !> The local timestep.
    real(kind=rk) :: dt
  end type


  !> type to define a one dimensional stencil for reconstructions.
  type atl_oneDimStencil_type
    !> the 1D stencil in treelm coordinates.
    integer :: stencil
    !> the number of elements in the stencil, including the cell itself
    !! you reconstruct for.
    integer :: nElems
    !> relative position of the stencil elements to the current cell.
    !! Note, that this vector has length (nElems-1) since the current cell
    !! itself is not stored here.
    integer, allocatable :: elemPos(:)
    !> relative position of the stencil elements in negative direction to the current cell.
    !! Note, that this vector has length (nElems-1) since the current cell
    !! itself is not stored here. The entries start with the cell that is most far
    !! from the current cell away.
    integer, allocatable :: ngElemPos(:)
    !> for each element of the mesh we store the lowest and highest left shift
    !! that build correct stencils (i.e. correct means: not including
    !! any boundary element).
    !! The first dimension is the number of elements associated with this stencil.
    !! The second dimension is 2, the first is the lowest possible left shift index
    !! the second is the highest possible left shift index.
    integer,allocatable :: bnd(:,:)
  end type

  !> type specifying all informations about the stencil for the dimension
  !! by dimension reconstruction.
  type atl_dimbydimstencil_type
    !> the stencil in x direction
    type(atl_oneDimStencil_type) :: xStencil
    !> the stencil in y direction
    type(atl_oneDimStencil_type) :: yStencil
    !> the stencil in z direction
    type(atl_oneDimStencil_type) :: zStencil
  end type atl_dimbydimstencil_type


  !> type containing all the informations related to the scheme, e.g.:
  !! time and space discretization, scheme order, etc.
  type atl_scheme_type
    !> integer representing the current discretization scheme.
    integer             :: scheme

    !> the number of degrees of freedom for the selected scheme for a single cell
    !! and a single variable of the equation.
    !! For example we have: P1PM => nDofs=4, P2PM = 10). This number includes only the
    !! degrees of freedom which will be stored. We do not include the number of
    !! reconstructed degrees of freedom here!
    integer             :: nDoFs

    !> the number of reconstructed degrees of freedom for the selected scheme
    !! for a single cell and a single variable of the equation (including
    !! the reconstructed degrees of freedoms).
    integer             :: nDoFsRecons

    !> The number of dofs on the faces.
    integer             :: nFaceDofs

    !> variable to specify the space integration.
    ! ToDO VK: think we can delete this datatype, it was used in the weno scheme
    ! futher it is used flux/atl_hlleFlux_module to specify the number of surface points
    !type(space_quadrature_type) :: space_integration

    !> levelwise information of time discretization
    type(atl_local_timestep_type) :: time

    !! if you want to add another scheme you should add it here and give
    !! a unique code above atl_scheme_type%scheme! Please add a comment, too.
    !! usage is specified by atl_scheme_type%scheme.

    !> Parameters of the modal discontinuous Galerkin scheme if
    !! scheme is set to modg.
    type(atl_modg_scheme_type) :: modg

    !> Parameters of the modal discontinuous Galerkin scheme if
    !! scheme is set to modg 2d.
    type(atl_modg_2d_scheme_type) :: modg_2d

    !> Parameters of the modal discontinuous Galerkin scheme if
    !! scheme is set to modg 1d.
    type(atl_modg_1d_scheme_type) :: modg_1d

    !> Informations about the polynomial basis of a MODG scheme.
    type(ply_modg_basis_type) :: modg_basis

    !> The stabilization(s) for the scheme.
    !! Applied one after each other. Starting with index 1, then 2, ...
    type(atl_stabilization_type), allocatable :: stabilization(:)

    !> Precomputed Scalar Products
    real(kind=rk), allocatable :: dl_prod(:,:)
    real(kind=rk), allocatable :: dl_prodDiff(:,:)

    !> Temp Arrays needed for evaluation of physical fluxes
    real(kind=rk), allocatable :: temp_over(:,:,:)
    real(kind=rk), allocatable :: temp_modal(:,:,:)
    real(kind=rk), allocatable :: temp_nodal(:,:,:)

  end type

  public :: atl_scheme_type,          &
    &       atl_init_scheme,          &
    &       atl_dimbydimstencil_type, &
    &       atl_onedimstencil_type,   &
    &       atl_define_SchemeStencil, &
    &       atl_local_timestep_type,  &
    &       atl_modg_scheme_type,     &
    &       atl_modg_scheme_prp,      &
    &       atl_modg_2d_scheme_prp,   &
    &       atl_modg_1d_scheme_prp,   &
    &       atl_schemeID2ndim


contains


  !> subroutine to intialize a scheme as specified by a given lua script file.
  subroutine atl_init_scheme(me, conf, minlevel, maxlevel)
    ! --------------------------------------------------------------------------
    !> The global minimum level of the mesh
    integer, intent(in) :: minLevel
    !> The global maximum level of the mesh
    integer, intent(in) :: maxLevel
    !> the scheme you want to initialize.
    type(atl_scheme_type), intent(out) :: me(minlevel:maxlevel)
    !> flu binding to lua configuration file.
    type(flu_State), intent(in) :: conf
    ! --------------------------------------------------------------------------
    integer :: scheme_table, spatial_table
    character(len=labelLen) :: scheme_name
    character(len=labelLen) :: sname
    integer :: iError, ilevel
    type(atl_stabilization_type), allocatable :: stabilization(:)
    ! --------------------------------------------------------------------------

    ! open the scheme table
    call aot_table_open(L=conf, thandle=scheme_table, key='scheme')
    if(scheme_table.eq.0) then
      write(logUnit(1),*) 'ERROR in init_kernel_state: no scheme table in ' // &
        & 'lua configuration file found,stopping...'
      call tem_abort()
    end if

    ! Init the stabilzation (same for all the levels)
    call atl_ini_stabilization(conf = conf, parent_table = scheme_table, &
                              & filter = stabilization )
    do iLevel = minlevel, maxlevel
      allocate(me(iLevel)%stabilization(size(stabilization)))
      me(iLevel)%stabilization(:) = stabilization(:)
    end do

    ! open the spatial subtable
    call aot_table_open(L=conf, parent=scheme_table, &
      &                 thandle=spatial_table, key='spatial')

    ! get the name of the scheme
    call aot_get_val(L = conf, thandle = spatial_table, &
      &              key = 'name', &
      &              val = scheme_name, &
      &              ErrCode = iError)

    sname = upper_to_lower(scheme_name)
    sname = adjustl(sname)

    select case(trim(sname))
!HK: Unmaintained:
!HK!    case ('pnpm')
!HK!      me%scheme = pnpm_scheme_prp
!HK!
!HK!      call init_pnpm_scheme(me%pnpm, conf, spatial_table)
!HK!      call init_gauss_quadrature(me%space_integration, me%pnpm%pm, &
!HK!                                & 0.0_rk, 1.0_rk )
!HK!
!HK!      !> the number of degrees of freedom for the original field and the
!HK!      !! reconstructed field per conservative variable
!HK!      me%nDoFs = ((me%pnpm%pn+1) * (me%pnpm%pn+2) * (me%pnpm%pn+3)) &
!HK!        &           / 6
!HK!      me%nDoFsRecons &
!HK!        & = ((me%pnpm%pm+1) * (me%pnpm%pm+2) * (me%pnpm%pm+3)) / 6

    case('modg')
      me(:)%scheme = atl_modg_scheme_prp

      write(logUnit(1),*) 'Init MODG scheme ...'
      do ilevel = minlevel, maxlevel
        call atl_modg_scheme_init(me = me(ilevel)%modg, &
          &                       nDofs = me(ilevel)%nDofs, &
          &                       nFaceDofs = me(ilevel)%nFaceDofs, &
          &                       conf = conf, thandle = spatial_table, &
          &                       currentLevel = iLevel, maxLevel = maxLevel )

      !precompute and store the scalar product between ansatz and test function
      call compute_scalProd_DualLeg(me(iLevel)%dl_prod, me(iLevel)%dl_prodDiff,&
        &                             me(iLevel)%modg%maxpolyDegree)
      end do

    case('modg_2d')
      me(:)%scheme = atl_modg_2d_scheme_prp

      write(logUnit(1),*) 'Init 2D MODG scheme ...'
      do ilevel = minlevel, maxlevel
        call atl_modg_2d_scheme_init(me = me(ilevel)%modg_2d, &
          &                       nDofs = me(ilevel)%nDofs, &
          &                       nFaceDofs = me(ilevel)%nFaceDofs, &
          &                       conf = conf, thandle = spatial_table, &
          &                       currentLevel = iLevel, maxLevel = maxLevel )
      ! precompute and store the scalar product between ansatz and test function
      call compute_scalProd_DualLeg(me(iLevel)%dl_prod, me(iLevel)%dl_prodDiff,&
        &                             me(iLevel)%modg_2d%maxpolyDegree)
      end do


    case('modg_1d')
      me(:)%scheme = atl_modg_1d_scheme_prp

      write(logUnit(1),*) 'Init 1D MODG scheme ...'
      do ilevel = minlevel, maxlevel
        call atl_modg_1d_scheme_init(me = me(ilevel)%modg_1d, &
          &                       nDofs = me(ilevel)%nDofs, &
          &                       nFaceDofs = me(ilevel)%nFaceDofs, &
          &                       conf = conf, thandle = spatial_table, &
          &                       currentLevel = iLevel, maxLevel = maxLevel )
      end do


    case default
      write(logUnit(1),*) 'ERROR in init_kernel_state: unknown scheme name ' &
        &            // trim(scheme_name) // ' !'
      write(logUnit(1),*) 'Supported schemes are: '
      write(logUnit(1),*) '* modg'
      write(logUnit(1),*) '* modg_2d'
      write(logUnit(1),*) 'Stopping....'
      call tem_abort()
    end select

    call aot_table_close(L = conf, thandle = spatial_table)
    call aot_table_close(L = conf, thandle = scheme_table)

    ! Now, we init the timestepping scheme.
    do ilevel = minlevel, maxlevel
      call init_local_time_integration(me(ilevel)%time)
    end do



  end subroutine atl_init_scheme


  !> precompute the scalar products of the anstaz and test function
  subroutine compute_scalProd_DualLeg (dl_prod, dl_prodDiff, maxPolyDegree)
    !real(kind=rk), allocatable, intent(out), dimension (:,:) ::  dl_prod
    !real(kind=rk), allocatable, intent(out), dimension (:,:) :: dl_prodDiff
    real(kind=rk), allocatable, intent(out) ::  dl_prod    (:,:)
    real(kind=rk), allocatable, intent(out) :: dl_prodDiff (:,:)
    integer,intent(in) :: maxPolyDegree
    ! ----------------------------------------------------------------!
    integer :: iTest, iAns
    ! ----------------------------------------------------------------!

    allocate(dl_prod(2, maxPolyDegree+1))
    allocate(dl_prodDiff(2,maxPolyDegree+1))

    dl_prod = 0.0_rk
    do iTest=1, maxpolyDegree+1
      iAns = iTest-2
      if (iAns >= 1) then
        dl_prod(1,iTest) = scalProdDualLeg(iAns, iTest)
      end if
      dl_prod(2,iTest) = scalProdDualLeg(iTest, iTest)
    end do

    dl_prodDiff = 0.0_rk
    do iTest=2, maxpolyDegree+1
      iAns = iTest-5
      if (iAns >= 1) then
        dl_prodDiff(1,iTest) = scalProdDualLegDiff(iAns, iTest-1)
      end if
      dl_prodDiff(2,iTest) = scalProdDualLegDiff(iTest-1, iTest-1)
    end do

  end subroutine compute_scalProd_DualLeg

!> Subroutine do define a specific stencil for a certain scheme.
!!
!! The scheme is looked up from a given configuration file.
!! @todo HK: why is it looked up from there? Actually the spatial scheme
!!           definition has already been parsed once. Why is there no routine,
!!           which does that, and stores the resulting data in a spatial
!!           scheme data structure?
  subroutine atl_define_SchemeStencil(conf, me)
    ! --------------------------------------------------------------------------!
    ! --------------------------------------------------------------------------!
    !> flu binding to configuration file.
    type(flu_State) :: conf
    !> the neighbor list you want to init.
    type(tem_stencilHeader_type), intent(inout) :: me
    ! --------------------------------------------------------------------------!
    ! we dont need a special stencil for modg, so we set info
    ! for an empty stencil.
    call init( me = me, QQN = 0, QQ = 0, useAll = .true. )

  end subroutine atl_define_SchemeStencil

  !> summary: subroutine to map boundary informations to the stencil. This is
  !! necessary since there might be some stencils that are exceeding the fluid cell
  !! domain.
!NA!  subroutine mapStencilIndexBnd(neigh, stencil, nElems)
!NA!    ! --------------------------------------------------------------------------
!NA!    !> information about neighbouring elements. The position of boundary cells
!NA!    !! in nghElems will be set to new, arbitrary but valid positions.
!NA!    type( tem_levelNeighbor_type ), intent(inout) :: neigh
!NA!    !> the dim by dim stencil that is intialized.
!NA!    type(atl_dimbydimstencil_type),intent(inout) :: stencil
!NA!    !> The number of fluids, ghost and halo cells on this level (not including
!NA!    !! boundary cells).
!NA!    integer , intent(in) :: nElems
!NA!    ! --------------------------------------------------------------------------
!NA!    ! bounds of the elempos array
!NA!    integer :: lowerBound, upperBound
!NA!    ! --------------------------------------------------------------------------
!NA!
!NA!    ! do a mapping for the x stencil
!NA!    lowerBound = 1
!NA!    upperBound = 2*(stencil%xStencil%nElems-1)
!NA!    call map1DStencilIndexBnd(stencil%xStencil, &
!NA!         & neigh%nghElems(lowerBound:upperBound,:), nElems)
!NA!    ! do it for y
!NA!    lowerBound = lowerBound + 2*(stencil%xStencil%nElems-1)
!NA!    upperBound = upperBound + 2*(stencil%yStencil%nElems-1)
!NA!    call map1DStencilIndexBnd(stencil%yStencil, &
!NA!      & neigh%nghElems(lowerBound:upperBound,:), nElems)
!NA!    ! do it for z
!NA!    lowerBound = lowerBound + 2*(stencil%yStencil%nElems-1)
!NA!    upperBound = upperBound + 2*(stencil%zStencil%nElems-1)
!NA!    call map1DStencilIndexBnd(stencil%zStencil, &
!NA!      & neigh%nghElems(lowerBound:upperBound,:), nElems)
!NA!
!NA!  end subroutine

  !> summary: routine to init the timestepping scheme.
  subroutine init_local_time_integration(me)
    ! --------------------------------------------------------------------------
    !> the scheme you want to initialize.
    type(atl_local_timestep_type), intent(inout) :: me
    ! --------------------------------------------------------------------------
    ! --------------------------------------------------------------------------
    me%dt = 0.0_rk
  end subroutine


  function atl_schemeID2ndim(schemeID) result(ndim)
  ! ----------------------------------------------------------------------------
  integer,intent(in) :: schemeID
  integer :: ndim
  ! ----------------------------------------------------------------------------

  if(schemeID .eq. atl_modg_scheme_prp) then
    ndim = 3
  else if (schemeID .eq. atl_modg_2d_scheme_prp) then
    ndim = 2
  else
     ndim = 1
   end if
  end function

end module atl_scheme_module
