-- We'll create a square nelems x nelems mesh.
nelems = 10
length = 6.0

origin = {-3.0, -3.0}

-- A minimum level, by which all parts in the computational domain should at
-- least be resolved with. Default is 0.
minlevel = math.ceil(math.log(nelems+2, 2))
dx = length/nelems
eps = length * 2^-21

-- Location to write the mesh in.
-- Note the trailing path seperator, needed, if all mesh files should be in a
-- directory. This directory has to exist before running Seeder in this case!
folder = 'mesh/' --default: 'mesh_'

-- Some comment, you might want to put into the mesh file for later reference.
comment = 'Plane square mesh for 2D with boundary conditions.'

-- Debug output can be used to output prelimnary tree in restart format
-- and this restart file can be converted to vtk format by Harvester
NOdebug = { debugMode = true, debugMesh = 'dbg-mesh_' }

-- How detailed should the output be?
-- The higher the level, the more output you'll get from seeder.
logging = { level = 10 }

-- Bounding cube: the root node of the octree, defining the complete universe,
-- from which all elements are derived by recursive bisection.
-- The origin is the corner from which on the cube is spanned with the given
-- length in each direction.
bounding_cube = { origin = {origin[1]-dx, origin[2]-dx, -dx},
                  length = dx*(2^minlevel) }

smoothbounds = true
smoothlevels = true

-- *********************** Table of spatial objects *********************** --
-- Each spatial object is defined by an attribute and some geometric entity
-- attached to this attribute. Attributes might be defined multiple times.
-- Attributes are described by a kind (boundary, seed or refinement), a level
-- and maybe further kind specific values, like a label for the boundary.
-- Geometric objects might by right now:
-- - canoND (point, line, plane or box)
-- - STL
-- - Sphere
-- - Cylinder
--
-- Periodic boundaries are special, spatial objects of this kind can only make
-- use of geometric objects of the kind 'periodic'.
spatial_object = {
  { -- West boundary
    attribute = { 
      kind = 'boundary',
      label = 'west',
      level = minlevel
    },

    geometry = {
      kind = 'canoND',
      object = { 
        { origin = {origin[1]-eps, origin[2]-eps, -eps},
          vec = {
            {0, length+2*eps,        0},
            {0,            0, dx+2*eps}
          }
        }
      }
    }
  }, -- end of west boundary

  { -- East boundary
    attribute = { 
      kind = 'boundary',
      label = 'east',
      level = minlevel
    },

    geometry = {
      kind = 'canoND',
      object = { 
        { origin = {origin[1]+length+eps, origin[2]-eps, -eps},
          vec = {
            {0, length+2*eps,        0},
            {0,            0, dx+2*eps}
          }
        }
      }
    }
  }, -- end of east boundary

  { -- Seed
    attribute = { kind = 'seed' },
    geometry = {
      kind = 'canoND',
      object = { origin = {origin[1]+0.5*length,
                           origin[2]+0.5*length,
                                     0.5*dx}
      }
    }
  }, -- end of seed

  { -- Periodicity in Y
    attribute = {kind='periodic', level = minlevel},
    geometry = {
      kind = 'periodic',
      object = { 
        plane1 = { 
          origin = {origin[1]-eps, origin[2]-eps, -eps },
          vec = {{length+2*eps, 0,        0},
                 {           0, 0, dx+2*eps}  }
        },
        plane2 = { 
          origin = {origin[1]-eps, origin[2]+length+eps, -eps },
          vec = { {           0, 0, dx+2*eps},
                  {length+2*eps, 0,        0}}
        }
      }  
    }
  }, -- end of periodicity in Y

  { -- Periodicity in Z
    attribute = {kind='periodic', level = minlevel},
    geometry = {
      kind = 'periodic',
      object = { 
        plane1 = { 
          origin = {origin[1]-eps, origin[2]-eps, dx+eps },
          vec = {{length+2*eps,            0, 0},
                 {           0, length+2*eps, 0}}
        },
        plane2 = { 
          origin = {origin[1]-eps, origin[2]-eps, -eps },
          vec = {{           0, length+2*eps, 0},
                 {length+2*eps,            0, 0}}
        }
      }  
    }
  } -- end of periodicity in Z
} -- end of spatial objects
-- ************************************************************************ --
