-- Configuration file for Ateles --
require 'hyperfun'


-- global simulation options
simulation_name = 'jet_nslim01'
sim_control = {
              time_control  = {
                                min = 0.0,
                                max = {sim = 10.0, clock = 2*3600 - 60},
                                interval = { iter=100 },
                              },
              abort_criteria = { stop_file = 'stop' }
             }

weights = 'mesh/weights.lsb'
write_weights = 'mesh/weights'
check = { interval = 10 }

-- Mesh definitions --
mesh = 'mesh/'


-- Restart settings
restart = { 
            read = 'restart/jet_nslim01_lastHeader.lua',
            init_on_missing = true,
            write = 'restart/',
            time_control = {
                             align_trigger = true,
                             min = 0,
                             max = sim_control.time_control.max, 
                             interval = 0.01
                           }
          }

tracking = {
  label = 'indicators',
  folder = 'tracking/',
  variable = {'linindicator', 'inviscindicator'},
  shape = {kind = 'all'},
  time_control = {
    align_trigger = true,
    min = 0,
    max = sim_control.time_control.max, 
    interval = 0.01
  },
  output = {format = 'vtk'}
}

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Equation definitions --
equation = {
    name   = 'navier_stokes_2d',
    therm_cond = 2.555e-02,
    isen_coef = 1.4,
    r      = 296.0,
    mu     = 1.0e-6,
    ip_param = 4,
    linearization_indicator = 'error',
    linear_limit = 0.01,
    visc_limit   = 0.01,
    -- Parameters of the penalization
    porosity             = 1.0,
    viscous_permeability = 1.0,
    thermal_permeability = 1.0,
    material = {
                 characteristic = 0.0, --'characteristic',
                 relax_velocity = {0.0, 0.0}, --'relax_velocity',
                 relax_temperature = 0.0, --'relax_temperature'
    },
    penalization = {
        global = { kind = 'const',
                   const = {0.0, 0.0, 0.0, 0.0}
        }
    }

}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
                  name = 'modg_2d',            -- we use the modal discontinuous Galerkin scheme 
                  m = 15,        -- the maximal polynomial degree for each spatial direction
               }, 
    stabilization = {
                      {
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 16,
                      },
                      { name = 'cons_positivity_preserv',
                        eps = 1e-5
                      }
                    },
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitSSPRungeKutta', 
               steps = 2,
               control = {
                          name = 'cfl', 
                          cfl  = 0.7, 
                          cfl_visc  = 0.4
                         },
               },
}

-- The ambient medium
press = 1
dens = 1

-- the Mach number of the jet
mach = 0.6

-- the velocity amplitude in z direction
velAmpl = mach*math.sqrt(equation.isen_coef*press/dens) -- 0.1
densAmpl = dens


-- the radius of the jet
jet_radius = 0.1
jet_center = 1e-4
momentum_thickness = jet_radius/20

function velX_inlet(x,y,z,t)
  r = math.sqrt( (y-jet_center)*(y-jet_center) )
  return velAmpl * (1/2) * ( 1 +  tanh((jet_radius-r)/(2*momentum_thickness) ) )
end 

function dens_inlet(x,y,z,t)
  tmpVel = velX_inlet(x,y,z,t)/velAmpl
  return densAmpl / ( 1 + (equation.isen_coef-1)/2*mach*mach*tmpVel*(1-tmpVel)  )
end

-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = dens,
                      pressure = press,
                      velocityX = 0.0,
                      velocityY = 0.0,
                    }

-- ...the general projection table
projection = {
              kind = 'fpt',  
              factor = 1.0,
              lobattoPoints = true
             }


-- Boundary definitions
boundary_condition = {
                       { 
                       label = 'West',  
                       kind = 'inflow_normal',      
                       density = dens_inlet,
                       v_norm = velX_inlet,
                       v_tan = 0.0,
                       enforce_zero_grad = true
                       }
                       ,
                       { 
                       label = 'East',  
                       --kind = 'outflow',       
                       kind='primitives',
                       pressure = press,
                       density = dens,
                       velocityX = 0.0,
                       velocityY = 0.0
                       }
                       ,
                       { 
                       label = 'North', 
                       kind='primitives',
                       pressure = press,
                       density = dens,
                       velocityX = 0.0,
                       velocityY = 0.0
                       --kind = 'outflow',       
                       --pressure = press,
                       }
                       ,
                       { 
                       label = 'South', 
                       kind='primitives',
                       pressure = press,
                       density = dens,
                       velocityX = 0.0,
                       velocityY = 0.0
                       --kind = 'outflow',       
                       --pressure = press,
                       }
                       ,
                     }  

