-- Configuration file for Ateles --
require('seeder')

timestep_info = 1000
io_buffer_size = 128

-- ...the length of the cube
cubeLength = 2.0


-- global simulation options
simulation = {
              name          = 'kelvin_helm_inst',                -- the name of the simualtion
              time    =  {
                           useIterations = false,
                           min = 0.0,
                           max = 1.0e+03,
                         }
             }


-- Mesh definitions --
mesh = './mesh/'


-- Restart settings
restart = { 
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = './restart/',                                        
            -- temporal definition of restart write
            time = {   
                      useIterations = false,  
                      min = 0, 
                      max = simulation.time.max, 
                      interval = simulation.time.max/1e3,
                   },  
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Equation definitions --
equation = {
    name   = 'euler_2d',
    isen_coef = 2.0, 
    r      = 296.0,
}

-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg_2d',            -- we use the modal discontinuous Galerkin scheme 
               m = 63,                   -- the maximal polynomial degree for each spatial direction
               dealiasFactor = 1.0,      -- factor to remove aliases: 1.0 means no additional dealiasing
               blocksize = 32,
               }, 
    -- the stabilzation of the scheme
    stabilization = {
                       -- Spectral viscosity, i.e.:
                       -- u(x) = \sum_{k=0}^N \sigma_k \hat{u}_k L_k(x)
                       -- with
                       -- \sigma_k = exp(- \alpha (k/N)^order )
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 11,
                    },
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  --'explicitEuler',
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.6,  -- CourantÐFriedrichsÐLewy number
                         },
               },
}

check = {
          interval = 1,
        }

-- transition zone width 
width = 0.0002/minlevel
-- distance of the single sine mode pertubation
amplPert = 0.0
-- amplitudes of the initial conditions
vel =  0.15
dens = 0.5
press = 1.0
--press = 1.0 
-- wavenumber of the single mode pertubation
periods_upper=15
periods_lower=12
-- the phase shift between upper and lower sinusoidal wave
phase = 0.4
-- initial condition for the x component
function iniVelX(x,y,z)
  if( y > 0 ) then
    center = math.sin(periods_upper*math.pi*x)*amplPert + 0.5
    return vel*math.tanh((y-center)/width)
  else
    center = math.sin(periods_lower*math.pi*x + phase)*amplPert - 0.5
    return -vel*math.tanh((y-center)/width)
  end 
end
-- initial condition for the y component
ywidth = width*3000
function iniVelY(x,y,z)
  if( y > 0 ) then
    center = math.sin(periods_upper*math.pi*x)*amplPert + 0.5
    return (vel/1000.0)*math.exp(-math.abs(y-center)/ywidth)*math.sin(periods_upper*math.pi*x)
  else
    center = math.sin(periods_lower*math.pi*x + phase)*amplPert - 0.5
    return (-vel/1000.0)*math.exp(-math.abs(y-center)/ywidth)*math.sin(periods_lower*math.pi*x)
  end 
end
-- Initial condition for the density
function iniDens(x,y,z)
  if( y > 0 ) then
    center = math.sin(periods_upper*math.pi*x)*amplPert + 0.5
    return dens*(math.tanh((y-center)/width)+2) + 2.0
  else
    center = math.sin(periods_lower*math.pi*x + phase)*amplPert - 0.5
    return dens*(-math.tanh((y-center)/width)+2) + 2.0
  end 
end

-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = iniDens,
                      pressure = press,
                      velocityX = iniVelX,
                      velocityY = iniVelY,--{
                                          --  predefined = 'random',
                                          --  min = -1.0,
                                          --  max = 1.0,
                                          --},
                    }

-- ...the general projection table
projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }


-- Boundary definitions
boundary_condition = {
                       { 
                       label = 'slipTop', 
                       kind = 'slipwall',       
                       }
                       ,
                       { 
                       label = 'slipBottom', 
                       kind = 'slipwall',       
                       }
                       ,
                     }  
