-- Configuration file for Ateles --
--
-- This the very basic setup of a constant state with a 1D Euler simulation
-- for debugging the material with multiple levels.
-- The domain is of size = 1 and the material occupies half
-- of the domain. i.e in the interval (0.5,1), which is resolved coarsely
-- with just one elment.
-- The left half is built by four refined elements and contains the fluid.
-- We define an acoustic pulse that runs in this part of the domain
-- against the wall and is to be reflected there.
require 'seeder'


time_to_dump = 0.5

wall_x = 0.5
wall_thickness = 0.5

-- ... the smallness parameter of the penalization
penalization_eps = 0.1 --8.0/(scheme.spatial.m+1)
alpha_v = 1e-3
alpha_t = 1e-3

--Simulation
spatial_degree = 7
cfl = 0.1

--  parameters for IC
ini_vel_x = 0.0
ini_dens  = 1.4
ini_press = 1.0

-- equation parameters
isen  = 1.4
boltz = 1.0
sphere_diameter = 1.0
-- background state
back_press = ini_press/isen                   --equation.background.pressure
back_dens = ini_dens                          --equation.background.density
ini_temp = back_press/back_dens/boltz
c = math.sqrt((isen*back_press)/back_dens)    --speed of sound

--GaussPulse parameter
--Amplitude
amp_Dens = 0.001
amp_Press =0.001
--Center
axis_x = 0.25
--Width Control
width_con = 0.004	

-- global simulation options
sim_name = 'gp_nvrstk'

sim_control = {
  time_control = {
    max =  { sim = time_to_dump, iter=2 },
    interval = {iter=1},       
  }
}

-- Mesh definitions --
mesh = 'mesh_'

function p_prime(x,y,z)
d = ((x-axis_x)^2) --+ ((y-axis_y)^2) 
arg = (-d/width_con)* math.log(2)
res =  eps*math.exp(arg)
return res
end

function pulse_press (x,y,z)
res = back_press + p_prime(x,y,z)
return res
end

function pulse_dens (x,y,z)
res = back_dens + p_prime(x,y,z)
return res
end

function pulse_velx (x,y,z)
res =  p_prime(x,y,z)
return res
end



-- Now, define the initial conditions
initial_condition = { 
  density = pulse_dens,
  pressure = pulse_press,
  velocityX = pulse_velx,
  velocityY = 0.0,
  velocityZ = 0.0
}

obstacle_shape = { 
  kind = 'canoND', 
  object = {
    origin = { wall_x + eps, -eps, -eps },
    vec = {
      { wall_thickness - 2*eps,       0.0,       0.0 },
      {                    0.0,  dx+2*eps,       0.0 },
      {                    0.0,       0.0,  dx+2*eps },
    },
    segments = { 20, 20, 20 }
  }
}

variable = {
  {
    name = 'characteristic',
    ncomponents = 1,
    vartype = "st_fun",
    st_fun = 
    {
      { const = { 0.0 } },
      {
        const = {1.0},
        shape = obstacle_shape
      }
    }
  },

  {
    name = 'bk_press',
    ncomponents = 1,
    vartype = 'st_fun',
    st_fun = back_press
  },

  {
     name = 'pert_press',
     ncomponents = 1,
     vartype = 'operation',
     operation = {
                    kind = 'difference',
                    input_varname = {'pressure', 'bk_press'},
                 }
  }
}

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg',
    m    = spatial_degree
  },
  -- the temporal discretization scheme
  temporal = {
    --name = 'explicitRungeKutta',
    name = 'imexRungeKutta',
    steps = 4,
    control = {
                 name     = 'cfl',
                 cfl      = cfl,
                 cfl_visc = cfl
              }
  }
}

-- Equation definitions --
equation = {
  name       = 'euler',
  -- general fluid's parameter
  isen_coef  = isen,
  r          = boltz,
  -- viscous parameters
  -- Parameters of the penalization
  porosity             = penalization_eps,
  viscous_permeability = alpha_v*penalization_eps,
  thermal_permeability = alpha_t*penalization_eps,
  material = {
    characteristic    = 'characteristic',
    relax_velocity    = {0.0, 0.0, 0.0},
    relax_temperature = ini_temp
  }
}
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

projection = {
  kind = 'fpt',
  --factor = 2.0,
}

-- Boundary definitions
boundary_condition = {
  {
    label = 'east',
    kind = 'wall'
  },
  {
    label = 'west',
    kind = 'outflow',
    pressure = back_press,
  }
}
