-- Configuration file for Ateles --
-- Testcase to be used with the subres example from seeder.
-- A simple flow around a sphere with an inital velocity and fully periodic
-- boundary conditions.
-- For using the mesh from subres.lua:
-- * set the mesh accordingly
-- * set the characteristic variable to make use of the solidsphere color
--
-- Only the characteristic needs to be set, the other material values take
-- effect only where the characteristic is defined. Thus, they can be simple
-- constants.

sim_name = 'sphere'

-- Initial Mach number:
mach = 0.3

-- the order of the scheme:
spatial_order = 12

-- Geometry definition (only relevant when not using the mesh from seeder)
sphere_radius = 0.12
sphere_orig = {0.5, 0.5, 0.5}
function inside_sphere(x,y,z)
  rx = x - sphere_orig[1]
  ry = y - sphere_orig[2]
  rz = z - sphere_orig[3]
  radius = math.sqrt(rx^2 + ry^2 + rz^2)
  return (radius < sphere_radius)
end

-- Mesh, use the mesh created by subres.lua
-- mesh = 'mesh/'
-- otherwise use an internally defined mesh:
mesh = { predefined = 'cube',
         origin = {0.0, 0.0 , 0.0},
         length = 1,
         refinementLevel = 2
}

maxtime = 2.0

sim_control = {
  time_control = {
    max = { iter=1000 },
    interval = { iter = 1 },
  }
}

-- General settings
logging = { level = 10 }
check =  { interval = 1 }
io_buffer_size = 30

-- Equation definitions --
isen = 1.4
boltz= 1.0/1.4
rho  = 1.4
p    = 1.0
-- inital velocity
vel  = mach * math.sqrt(isen * p / rho )

equation = {
  name   = 'euler',
  numflux = 'hll',
  -- Physical parameters of the gas phase
  isen_coef            = isen,
  r                    = boltz,
  -- Parameters of the penalization
  porosity             = 1.0,
  viscous_permeability = 1e-16,
  thermal_permeability = 1e-8,
  -- The definition of an obstacle by penalization
  material = {
    characteristic = 'Xi', -- Replace by color from seeder mesh.
    relax_velocity = {0.0, 0.0, 0.0},
    relax_temperature = p / ( rho * boltz )
  }
}

-- Restart settings
restart = {
  write = 'res_sphere_',
  time_control = {
    min      = 0,
    max      = maxtime,
    interval = { iter = 10 },
  }
}

-- Scheme definitions --
scheme = {

  spatial =  {
    name = 'modg',
    modg_space = 'Q',
    m = spatial_order-1
  },
  temporal = {
    name = 'imexRungeKutta',
    steps = 4,
    control = {
      name     = 'cfl',
      cfl      = 0.8,
      cfl_visc = 0.6
    }
  },

  NOstabilization = {
    {
       name  = 'spectral_viscosity',
       alpha = 32,
       order = 15
    }
  }
}

-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)


-- Initial condition definitions --
function iniVel(x,y,z)
  if inside_sphere(x,y,z) then
    return 0.0
  else
    return vel
  end
end

initial_condition = {
  density   = rho,
  pressure  = p,
  velocityX = iniVel,
  velocityY = 0.0,
  velocityZ = 0.0
}

-- ...the general projection table
projection = {
  kind = 'fpt',
  factor = 2.0,
  blocksize = 32
}

function characteristic(x,y,z)
  if inside_sphere(x,y,z) then
    return 1.0
  else
    return 0.0
  end
end

eps = 0.5^8
variable = {
  {
     name = 'Xi',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = {
       { const = { 0.0 } },
       {
          fun   = characteristic,
          shape = {
            kind = 'canoND',
            object = {
              origin = {
                0.5-sphere_radius - eps,
                0.5-sphere_radius - eps,
                0.5-sphere_radius - eps
              },
              vec = {
                { 2*(sphere_radius+eps), 0.0, 0.0 },
                { 0.0, 2*(sphere_radius+eps), 0.0 },
                { 0.0, 0.0, 2*(sphere_radius+eps) }
              },
              segments = { 33, 33, 33 }
            }
          }

       }
     }
  }
}
