-- Configuration file for Ateles (Periodic Oscillator) --

-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Acoustic equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of standing waves for acoustic equation. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- To verify diffusion and dispersion relations this testcases allows you to modify the spatial harmonics by
-- varying the integer mode number in x and y direction by modifying the lua variables m and n.
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks).
-- To calculate a grid convergence behavior please modify the level variable. An increment of one will half the radius of your elements.

--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
-- Parameters to vary --
degree = 12
poly_space = 'Q'
-- ...the uniform refinement level for the periodic cube
level = 2

logging = { level = 10 }
-- Check for Nans and unphysical values
check =  {
           interval = 1,
         }

-- ...the general projection table
projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }

--...Configuration of simulation time
sim_control = { 
                time_control = { max = 0.03, -- {iter=100}, -- final Simulated time
                                 min = 0.0, 
                                 interval = {iter = 1} 
                                }
              }

--...Configuration for the restart file
restart = { 
            -- file to restart from
            read = './restart_press_velx_prim/simulation_lastHeader.lua',                        
--            -- folder to write restart data to
            write = 'restart_press_velx_prim/',
            -- temporal definition of restart write
            time_control = { max = sim_control.time_control.max,  -- final Simulated time
                             min = 0,
                             interval = {iter = 25 }, --sim_control.time_control.max/50.0 
                            }
          }
-- End  Parameters to vary --
--------------------------------------------------------------------------------
-- Definition of the test-case.

-- Mesh definitions --
--mesh = './mesh2/'
---- Mesh definitions --
mesh = {
        predefined = 'line_bounded', -- use the predefined line with boundaries
        origin = { -2.0 },           -- origin of the line
        length = 4.0,                -- length of all elements
        element_count = 5            -- number of elements
       }
---- ...the length of the cube
--cubeLength = 2.0
--mesh = { predefined = 'line',
--         origin = { 
--                    (-1.0)*cubeLength/2.0,
----                    (-1.0)*cubeLength/2.0,
----                    (-1.0)*cubeLength/2.0
--                  },
--         length = cubeLength,
--         refinementLevel = level
--       }


-- Equation definitions --
equation = {
             name   = 'linearEuler',
             isen_coef = 1.4,
             background = {
                 density = 1.225, 
                 velocityX = 100.0,
                 velocityY = 0.0,
                 velocityZ = 0.0,
                 pressure = 100000.0
                 }
           }

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
               m =  degree,                   -- the maximal polynomial degree for each spatial direction
               modg_space = poly_space
               }, 
    -- the temporal discretization scheme
    temporal = {
                 name = 'explicitRungeKutta', 
                 steps = 4,
              -- how to control the timestep
                 control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.95,     -- CourantÐFriedrichsÐLewy number
                         },
               },
             }


-- variables for gaussian pluse             
c = math.sqrt(equation.isen_coef* equation.background.pressure / equation.background.density)
ampl_density= equation.background.density/c 
ampl_pressure= equation.background.pressure/c 

function init_density(x,y,z)
if x<0 then
  return(0.0)
else
  return(1.0)
end
end

function ic_gauss_density(x,y,z)
  d= x*x+(y-0.5)^2+(z-0.5)^2 
  if d<0.6 then
    return( ampl_density * math.exp(-d/0.01*math.log(2)) )
  end
end

function ic_gauss_pressure(x,y,z)
  d= x*x+(y-0.5)^2+(z-0.5)^2 
  if d<0.6 then
    return( ampl_pressure * math.exp(-d/0.01*math.log(2)) )
  end
end

-- Initial Condition definitions --
initial_condition = { density = 0.0, --ic_gauss_density,
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                      pressure = ic_gauss_pressure, 
                    }

-- Boundary conditions
boundary_condition = {
                       {
                        label = 'east',
                        --kind = 'slipwall',
                        --kind = 'inflow',
                        kind = 'conservatives',
                        density = 0.0, 
                        v_x = 0.0, 
                        v_y = 0.0, 
                        v_z = 0.0, 
                        pressure = 0.0,
                      	}
                        ,
                       {
                        label = 'west',
                        --kind = 'slipwall',
                        kind = 'conservatives',
                        density = 0.0, 
                        v_x = 0.0, 
                        v_y = 0.0, 
                        v_z = 0.0, 
                        pressure = 0.0,
                        --kind = 'outflow',
                        --pressure = equation.background.pressure,
                        }
                  --      ,
                  --     {
                  --      label = 'North',
                  --      kind = 'slipwall',
                  --      v_normal = equation.background.velocityY,
                  --      }
                  --      ,
                  --     {
                  --      label = 'South',
                  --      kind = 'slipwall',
                  --      v_normal = equation.background.velocityY,
                  --      }
}
