-- Simple mesh configuration for multilevel check of sponges.
-- This setup provides a line of three coarse elements, where the
-- one in the middle is refined by one level.
-- The coarse elements have an edge length of 1.
-- There is an east boundary on the right and a west boundary on the
-- left, other directions are periodic.
-- ------------------------------------------------------------------------- --

-- Location to write the mesh in.
-- Note the trailing path seperator, needed, if all mesh files should be in a
-- directory. This directory has to exist before running Seeder in this case!
folder = 'mesh_' --default: 'mesh_'
eps = 2^-15

-- Some comment, you might want to put into the mesh file for later reference.
comment = 'Multilevel periodic line of elements'

-- Debug output can be used to output prelimnary tree in restart format
-- and this restart file can be converted to vtk format by Harvester
NOdebug = { debugMode = true, debugMesh = 'debug/' }

-- How detailed should the output be?
-- The higher the level, the more output you'll get from seeder.
logging = { level = 5 }

-- Bounding cube: the root node of the octree, defining the complete universe,
-- from which all elements are derived by recursive bisection.
-- The origin is the corner from which on the cube is spanned with the given
-- length in each direction.
bounding_cube = { origin = {-2.0, -1.0, -1.0},
                  length = 8.0
}

-- A minimum level, by which all parts in the computational domain should at
-- least be resolved with. Default is 0.
minlevel = 3

-- We require elements neighboring boundaries to be refined down to the level
-- of the boundaries themselves. However, this is checked on all levels, and
-- results in some kind of smoothing of the refinements towards the actual
-- boundaries.
-- If this is not desired, this can be deactivated by setting smoothbounds to
-- false.
-- WARNING: If this is set to false, you might end up with elements that only
--          have partial boundaries. To avoid this, you need to configure an
--          appropriate refinement box around the geometry, with the same level
--          as the boundary itself.
-- Default is to use smoothening towards the boundaries (true).
smoothbounds = false

-- To smooth between levels to restrict level jump in the domain to 1
-- Default is set to true
smoothlevels = false

-- *********************** Table of spatial objects *********************** --
-- Each spatial object is defined by an attribute and some geometric entity
-- attached to this attribute. Attributes might be defined multiple times.
-- Attributes are described by a kind (boundary, seed or refinement), a level
-- and maybe further kind specific values, like a label for the boundary.
-- Geometric objects might by right now:
-- - canoND (point, line, plane or box)
-- - STL
-- - Sphere
-- - Cylinder
--
-- Periodic boundaries are special, spatial objects of this kind can only make
-- use of geometric objects of the kind 'periodic'.
spatial_object = {
  { -- >>>>>>>>>> West boundary
    attribute = { 
      kind = 'boundary',
      label = 'west',
      level = minlevel
    },
                
    geometry = {
      kind = 'canoND',
      object = { 
        { origin = {-1.0-eps, -eps, -eps},
          vec    = {
            { 0.0, 1.0+2*eps,       0.0 },
            { 0.0,       0.0, 1.0+2*eps }
          }
        }
      }
    }
  }, -- <<<<<<<<<<<<<<< West Boundary

  { -- >>>>>>>>>> East boundary
    attribute = { 
      kind = 'boundary',
      label = 'east',
      level = minlevel
    },
                
    geometry = {
      kind = 'canoND',
      object = { 
        { origin = {2.0+eps, -eps, -eps},
          vec    = {
            { 0.0, 1.0+2*eps,       0.0 },
            { 0.0,       0.0, 1.0+2*eps }
          }
        }
      }
    }
  }, -- <<<<<<<<<<<<<<< East Boundary

  { -- >>>>>>>>>> Central refinement box
    attribute = { 
      kind = 'refinement',
      level = minlevel+1
    },
                
    geometry = {
      kind = 'canoND',
      object = { 
        { origin = {eps, -eps, -eps},
          vec    = {
            { 1.0-2*eps,       0.0,       0.0 },
            {       0.0, 1.0+2*eps,       0.0 },
            {       0.0,       0.0, 1.0+2*eps }
          }
        }
      }
    }
  }, -- <<<<<<<<<<<<<<< Central refinement box

  { -- >>>>>>>>>> SEED
    -- Defining a seed to identify the part of the computational domain in
    -- the universe cube.
    attribute = { kind = 'seed' },
    geometry = { -- single point definition with a canoND object.
                 kind = 'canoND',
                 object = { origin = {0.5, 0.5, 0.5} }
               }
  }, -- <<<<<<<<<<<<<<< SEED

  
  { -- >>>>>>>>>> Periodicity in Y
    attribute = { kind = 'periodic', level = minlevel },
    geometry = {
      kind = 'periodic',
      object = { 
        plane1 = { 
          origin = { -1.0-eps, -eps, -eps },
          vec = { {       0.0, 0.0, 1.0+2*eps },
                  { 3.0+2*eps, 0.0,       0.0 }
          }
        },
        plane2 = { 
          origin = { -1.0-eps, 1.0+eps, -eps },
          vec = { { 3.0+2*eps, 0.0,       0.0 },
                  {       0.0, 0.0, 1.0+2*eps } }
        }
      }  
    }
  }, -- <<<<<<<<<<<<<<< Periodicity in Y

  { -- >>>>>>>>>> Periodicity in Z
    attribute = { kind = 'periodic', level = minlevel },
    geometry = {
      kind = 'periodic',
      object = { 
        plane1 = { 
          origin = { -1.0-eps, -eps, -eps },
          vec = { {       0.0, 1.0+2*eps, 0.0 },
                  { 3.0+2*eps,       0.0, 0.0 }
          }
        },
        plane2 = { 
          origin = { -1.0-eps, -eps, 1.0+eps },
          vec = { { 3.0+2*eps,       0.0, 0.0 },
                  {       0.0, 1.0+2*eps, 0.0 } }
        }
      }  
    }
  } -- <<<<<<<<<<<<<<< Periodicity in Z

} -- end of spatial objects
-- ************************************************************************ --
