-- Setup of a simple shear profile for Navier-Stokes 2D --
--
-- This is a simple setup with an intial linear velocity profile in
-- y direction.
-- The velocity profile forms a kind of hat with a peak at the x-axis.
sim_name = 'shear-hat'

-- Polynomial degree used for approximation
degree = 3

-- Isothermal coefficient
isen = 1.4
--Boltzman constant
boltz = 287.0
-- Density
rho = 1.4
-- Pressure
p = 1.0
--Mach number
mach = 0.1
-- Viscousity parameters
conducT = 0.5
mu = 2.0
c2 = 8./3.

-- Spatial function to describe the y-velocity profile
vel  = function (x,y,z)
  if x > 0 then
    return mach * math.sqrt(isen * p / rho ) * (1. - (x-0.5*dx)/dx)
  else
    return mach * math.sqrt(isen * p / rho ) * (1. + (x+0.5*dx)/dx)
  end
end

-- Control the Simulation times --
-- Set the simultion time and when information should write out.
-- The max defines the max. simulation time, the min. where to
-- start the simulation and the interval, after how many iteraition
-- information should be written out to the output file.
sim_control = {
  time_control = {
    max      = { iter = 400 },
    min      = 0.0,
    interval = { iter = 1 }
  }
}

-- Mesh configuration --
-- We use just two elements in x direction here.
-- This mesh can be internally generated by the 'line', kind of mesh.
dx = 1.0e-4
mesh = {
  predefined = 'line',
  origin = { -dx, -0.5*dx, -0.5*dx },
  length = 2*dx,
  element_count = 2
}

-- Equation definitions --
-- To solve the 2D compressible Navier Stokes equations we need to
-- define the physical fluid parameters, and additionally an internal
-- penalization factor, which is used in the implementation of the viscous
-- fluxes.
equation = {
  name      = 'navier_stokes_2d',
  isen_coef = isen,
  r         = boltz,
  numflux   = 'hll',
  -- Viscous parameters
  therm_cond = conducT,
  mu         = mu,
  ip_param   = c2,
  material = {
    characteristic = 0.0,
    relax_velocity = {0.0, 0.0},
    relax_temperature = 0.0
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
-- modg_2d results in a 2D simulation
-- the temporal table defines which time stepping
-- scheme is used here. In this test case we consider
-- the four step explite Runge-Kutta scheme.
-- The Cfl defines the timestep width, for
-- viscous flows we als need to define the cfl_visc.
scheme = {
  spatial =  {
    name = 'modg_2d',
    m = degree
  },
  temporal = {
    name = 'explicitRungeKutta',
    steps = 4,
    control = {
      name     = 'cfl',
      cfl      = 0.8,
      cfl_visc = 0.4
    }
  }
}

-- Projection type --
-- We consider here fast polynomial
-- transformation
projection = {
  kind = 'fpt',
}

-- Define the inital conditions --
-- We need to set density, pressure and
-- the velocity in x and y direction
initial_condition = {
  density   = rho,
  pressure  = p,
  velocityX = 0.0,
  velocityY = vel
}

-- Tracking --
-- We track here a point (just origin is given)
-- and the quantities momentum, density and
-- energy. The interval defines after how
-- many iterations the quantity information
-- should be writen out.
tracking = {
  label = 'track_shearhat2D',
  folder = './',
  variable = {'momentum','density','energy'},
  shape = {
    kind = 'canoND',
    object= {
      origin ={0.01*dx, 0, 0}
    }
  },
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = {iter = 10}
  },
  output = {
    format = 'ascii',
    use_get_point = true
  }
}

