﻿using System;
using System.Collections.Generic;
using NaGet.Packages;
using NaGet.Packages.Install;
using NaGet.SubCommands;
using NaGet.SubCommands.SubTask;
using NaGet.Tasks;

namespace NaGet.SubCommands
{
	/// <summary>
	/// ソフトのアンインストール処理
	/// </summary>
	public class NaGetUninstall2 : NaGetTaskSet2
	{
		
		private IList<Uninstallation> uninsts;
		
		public NaGetUninstall2(PackageListsManager pkgMan, IList<InstalledPackage> pkgs)
		{
			uninsts = new List<Uninstallation>(pkgs.Count);
			foreach (InstalledPackage instPkg in pkgs) {
				uninsts.Add(new Uninstallation(instPkg));
			}
			
			// taskセットの初期化
			initSubTask();
			foreach (Uninstallation uninst in uninsts) {
				registSubTask(string.Format("アンインストール: {0}", uninst.ToString()),
				              new FunctionalSubTask<Uninstallation>(runUninstall, uninst));
			}
			registSubTask("インストール済みのソフトリスト更新",
			              new LocalUpdateSubTask(pkgMan));
		}
		
		public override void Run()
		{
			NotifyStarted();
			RaiseTaskSetEvent(TaskEventType.STARTED, string.Empty);
			
			try {
				while (hasMoreSubTask) {
					RaiseTaskSetEvent(TaskEventType.STARTED_SUBTASK, currentSubTaskName);
					currentSubTask.Run();
					RaiseTaskSetEvent(TaskEventType.COMPLETED_SUBTASK, currentSubTaskName);
					
					if (cancelCalled) {
						throw new TaskCanceledException("cancel is called");
					}
					
					NotifyGoToNextSubTask();
				}
			} catch (TaskCanceledException) {
				cancelCalled = true;
			} catch (Exception e) {
				RaiseTaskSetEvent(TaskEventType.ERROR, e.Message);
			}
			
			if (cancelCalled) {
				NotifyCancelled();
				RaiseTaskSetEvent(TaskEventType.CANCELED, "キャンセルされました");
			} else {
				NotifyCompleted();
				RaiseTaskSetEvent(TaskEventType.COMPLETED, string.Empty);
			}
		}
		
		private void runUninstall(Uninstallation uninst)
		{
			if (uninst.Installed) {
				try {
					uninst.OutputDataReceived += this.ReceivedOutputData;
					uninst.ErrorDataReceived += this.ReceivedErrorData;
					int exitCode = uninst.Uninstall();
					if (exitCode != 0) {
						RaiseTaskSetEvent(TaskEventType.WARNING, "アンインストールが正常に終えていない可能性があります。プロセスの終了コード:"+exitCode);
					}
				} catch (Exception e) {
					RaiseTaskSetEvent(TaskEventType.ERROR, e.Message);
					throw e;
				}
			} else {
				RaiseTaskSetEvent(TaskEventType.WARNING, string.Format("{0}は既にアンインストールされているか、ソフトの存在を確認できませんでした", uninst));
			}
		}
		
		public override bool Cancelable {
			get {
				return !cancelCalled && Running && isDuringUninstalling;
			}
		}
		
		private bool isDuringUninstalling {
			get {
				return currentSubTask is LocalUpdateSubTask;
			}
		}
	}
}
