﻿using System;
using System.Text;
using System.IO;
using System.Collections.Generic;

namespace NaGet.Packages
{
	/// <summary>
	/// 環境構築/移行を便利にするためのパッケージ名リストの表現クラス
	/// </summary>
	public class PackageCollectionFileData : System.Collections.ObjectModel.Collection<string>
	{
		public PackageCollectionFileData()
		{
		}
		
		/// <summary>
		/// パッケージ名をリストに入れる
		/// </summary>
		/// <param name="pkgs">パッケージ情報</param>
		public void loadPackages<T>(IEnumerable<T> pkgs) where T : Package
		{
			foreach (Package pkg in pkgs) {
				if (this.IndexOf(pkg.Name) < 0) {
					this.Add(pkg.Name);
				}
			}
		}
		
		/// <summary>
		/// パッケージ名をリストに入れる
		/// </summary>
		/// <param name="pkgs">パッケージ情報</param>
		public void loadPackages<T>(IEnumerator<T> e) where T : Package
		{
			while(e.MoveNext()) {
				T pkg = e.Current;
				if (this.IndexOf(pkg.Name) < 0) {
					this.Add(pkg.Name);
				}
			}
		}
		
		/// <summary>
		/// このリストからパッケージのリストを得る
		/// </summary>
		/// <returns>パッケージのリスト</returns>
		public void generatePackages(PackageListsManager pkgMan, out IList<Package> pkgs, out IList<string> invalidPkgNames)
		{
			pkgs = new List<Package>();
			invalidPkgNames = new List<string>();
			
			foreach (string pkgName in this) {
				Package pkg = pkgMan.availablePkgList.GetPackageForName(pkgName);
				if (pkg != null) {
					if ((pkgMan.installedPkgList.GetPackageForName(pkgName) == null)
					   && (pkgMan.systemInstalledPkgList.GetPackageForName(pkgName) == null)) {
						pkgs.Add(pkg);
					} else {
						// すでにインストールされている場合は何もしない
					}
				} else {
					invalidPkgNames.Add(pkgName);
				}
			}
		}
		
		#region ファイル入出力
		
		/// <summary>
		/// リストをストリームに保存する
		/// </summary>
		/// <param name="stream">保存先</param>
		public void saveAs(StreamWriter stream)
		{
			foreach (string name in this) {
				stream.WriteLine("{0}", name);
			}
		}
		
		/// <summary>
		/// リストをファイルに保存する
		/// </summary>
		/// <param name="filepath">保存先</param>
		public void saveAs(String filepath)
		{
			using (FileStream stream = new FileStream(filepath, FileMode.Create)) {
				using (StreamWriter sw = new StreamWriter(stream, Encoding.UTF8)) {
					saveAs(sw);
				}
			}
		}
		
		/// <summary>
		/// ストリームからリストを読み込む
		/// </summary>
		/// <param name="stream">読み取り元</param>
		public void load(StreamReader stream)
		{
			string line = null;
			
			while ((line = stream.ReadLine()) != null) {
				this.Add(line.Trim());
			}
		}

		/// <summary>
		/// ファイルからリストを読み込む
		/// </summary>
		/// <param name="stream">読み取り元</param>
		public void load(string filepath)
		{
			using (FileStream fs = new FileStream(filepath, FileMode.Open)) {
				using (StreamReader sr = new StreamReader(fs, Encoding.UTF8)) {
					load(sr);
				}
			}
		}
		
		#endregion
	}
}
