/*
 * Copyright (c) 2006 Atmark Techno, Inc.  All Rights Reserved.
 */

#include <ns9750/ioregs.h>
#include <ns9750/memcpy.h>
#include <target/herrno.h>
#include <target/io.h>
#include <target/scan.h>
#include <target/str.h>
#include <target/buffer.h>
#include <target/memcmp.h>
#include <target/setenv.h>
#include <target/atag.h>
#include <target/gunzip.h>
#include "linux.h"
#include "memmap.h"
#include <target/memzero.h>
#include <target/flash.h>
#include <target/mmu.h>
#include "board.h"

/* for setting the root device */
#define MINORBITS       8
#define MKDEV(ma,mi)    (((ma) << MINORBITS) | (mi))

#define RAMDISK_MAJOR   1	/* major for "RAM disk" */
#define RAMDISK0_MINOR	0	/* b 1 0 == /dev/ram0 */

#define ROM_FLASH_MAJOR	31	/* major for "ROM/flash memory card" */
#define FLASH0_MINOR	16	/* b 31 16 == /dev/flash0 */
#define FLASH1_MINOR	17	/* b 31 17 == /dev/flash1 */

#define IDE0_MAJOR      3
#define IDE1_MAJOR      22

static int major = RAMDISK_MAJOR;
static int minor = RAMDISK0_MINOR;

#define COMMAND_LINE_SIZE 1024

extern unsigned long bytes_out;

#define GPIO_STATUS_1_FLASHBOOT  0x10000

#define SECTOR_SIZE     0x200
#define INODE_SIZE      0x80

#define read4_little_endian(cp) ((unsigned long)(cp[0]) + \
                                 ((unsigned long)(cp[1]) << 8) + \
                                 ((unsigned long)(cp[2]) << 16) + \
                                 ((unsigned long)(cp[3]) << 24))

extern int ide_detect_devices (void);
extern int ide_startup_devices (void);
extern int ide_read_sectors (int dev, unsigned long lba,
			     unsigned char *buf, int count);

typedef struct {
  unsigned char boot_ind;
  unsigned char head;
  unsigned char sector;
  unsigned char cyl;
  unsigned char sys_ind;
  unsigned char end_head;
  unsigned char end_sector;
  unsigned char end_cylinder;
  unsigned char start4[4];
  unsigned char size4[4];
} partition;

typedef struct {
  unsigned long  s_inodes_count;
  unsigned long  s_blocks_count;
  unsigned long  s_r_blocks_count;
  unsigned long  s_free_blocks_count;
  unsigned long  s_free_inodes_count;
  unsigned long  s_first_data_block;
  unsigned long  s_log_block_size;
  unsigned long  s_log_frag_size;
  unsigned long  s_blocks_per_group;
  unsigned long  s_frags_per_group;
  unsigned long  s_inodes_per_group;
  unsigned long  s_mtime;
  unsigned long  s_wtime;
  unsigned short s_mnt_count;
  unsigned short s_max_mnt_count;
  unsigned short s_magic;
  unsigned short s_state;
  unsigned short s_errors;
  unsigned short s_minor_rev_level;
  unsigned long  s_lastcheck;
  unsigned long  s_checkinterval;
  unsigned long  s_creator_os;
  unsigned long  s_rev_level;
  unsigned short s_def_resuid;
  unsigned short s_def_resgid;
  unsigned long  s_first_ino;
  unsigned short s_inode_size;
  unsigned short s_block_group_nr;
  unsigned long  s_feature_compat;
  unsigned long  s_feature_incompat;
  unsigned long  s_feature_ro_compat;
  unsigned char  s_uuid[16];
  char           s_volume_name[16];
  char           s_last_mounted[64];
  unsigned long  s_algorithm_usage_bitmap;
  unsigned char  s_prealloc_blocks;
  unsigned char  s_prealloc_dir_blocks;
  unsigned short s_padding1;
  unsigned long  s_reserved[204];
} ext2_super_block;

typedef struct {
  unsigned long  bg_block_bitmap;
  unsigned long  bg_inode_bitmap;
  unsigned long  bg_inode_table;
  unsigned short bg_free_blocks_count;
  unsigned short bg_free_inodes_count;
  unsigned short bg_used_dirs_count;
  unsigned short bg_pad;
  unsigned long  bg_reserved[3];
} ext2_group_desc;

typedef struct {
  unsigned short i_mode;
  unsigned short i_uid;
  unsigned long  i_size;
  unsigned long  i_atime;
  unsigned long  i_ctime;
  unsigned long  i_mtime;
  unsigned long  i_dtime;
  unsigned short i_gid;
  unsigned short i_links_count;
  unsigned long  i_blocks;
  unsigned long  i_flags;
  union {
    struct {
      unsigned long l_i_reserved1;
    } linux1;
    struct {
      unsigned long h_i_translator;
    } hurd1;
    struct {
      unsigned long m_i_reserved1;
    } masix1;
  } osd1;
  unsigned long  i_block[15];
  unsigned long  i_generation;
  unsigned long  i_file_acl;
  unsigned long  i_dir_acl;
  unsigned long  i_faddr;
  union {
    struct {
      unsigned char  l_i_frag;
      unsigned char  l_i_fsize;
      unsigned short i_pad1;
      unsigned short l_i_uid_high;
      unsigned short l_i_gid_high;
      unsigned long  l_i_reserved2;
    } linux2;
    struct {
      unsigned char  h_i_frag;
      unsigned char  h_i_fsize;
      unsigned short h_i_mode_high;
      unsigned short h_i_uid_high;
      unsigned short h_i_gid_high;
      unsigned long  h_i_author;
    } hurd2;
    struct {
      unsigned char  m_i_frag;
      unsigned char  m_i_fsize;
      unsigned short m_pad1;
      unsigned long  m_i_reserved2[2];
    } masix2;
  } osd2;
} ext2_inode;

typedef struct {
  unsigned long  inode;
  unsigned short rec_len;
  unsigned short name_len;
  char           name[255];
} ext2_dir_entry;

static int linux_cmdfunc(int argc, char *argv[])
{
	struct tag *tag = (struct tag *) LINUX_PARAM_ADDRESS;

	/* zero param block */
	memzero (tag, LINUX_PARAM_SIZE);

	/* set up core tag */
	tag->hdr.tag = ATAG_CORE;
	tag->hdr.size = tag_size(tag_core);
	tag->u.core.flags = 0;
	tag->u.core.pagesize = 0x1000;
	tag->u.core.rootdev = MKDEV(major, minor);

	/* 32 MB of SDRAM at 0xc0000000 */
	tag = tag_next(tag);
	tag->hdr.tag = ATAG_MEM;
	tag->hdr.size = tag_size(tag_mem32);
	tag->u.mem.size = DRAM1_SIZE;
	tag->u.mem.start = DRAM1_START;

	if (major == RAMDISK_MAJOR) {
	  /* an initial ramdisk image in flash at 0x00700000 */
	  tag = tag_next(tag);
	  tag->hdr.tag = ATAG_INITRD2;
	  tag->hdr.size = tag_size(tag_initrd);
	  tag->u.initrd.start = INITRD_LOAD_ADDRESS;
	  tag->u.initrd.size  = bytes_out;
	}

	/* the command line arguments */
	if (argc > 1) {
		tag = tag_next(tag);
		tag->hdr.tag = ATAG_CMDLINE;
		tag->hdr.size = (COMMAND_LINE_SIZE + 3 +
			 sizeof(struct tag_header)) >> 2;

		{
			const unsigned char *src;
			unsigned char *dst;
			dst = tag->u.cmdline.cmdline;
			memzero (dst, COMMAND_LINE_SIZE);
			while (--argc > 0) {
				src = *++argv;
				hprintf ("Doing %s\n", src);
				while (*src)
					*dst++ = *src++;
				*dst++ = ' ';
			}
			*--dst = '\0';
		}
	}

	tag = tag_next(tag);
	tag->hdr.tag = 0;
	tag->hdr.size = 0;

	/* branch to kernel image */
	__asm__ volatile (
	"	mov	r4, #0x00000000\n"	/* start of DRAM */
	"	add	r4, r4, #0x00028000\n"	/* kernel offset */
	"	mov	r0, #0\n"		/* kernel sanity check */
	"	mov	r1, #256\n"		/* NS9750 arch. number */
	"       orr     r1, r1, #217\n"
	"	mov	r2, #0\n"
	"	mov	r3, #0\n"
	"	mov	pc, r4"			/* go there! */
	);

	/* never get here */
	return 0;
}

static int copy_file (unsigned long  count, unsigned long table[],
			unsigned long start, unsigned long sectors_per_block,
			unsigned long *pfile_size, addr_t *pload_address)
{
  int           i;
  unsigned long file_data_sector, copy_size;
  unsigned char buf[SECTOR_SIZE * 8];
  
  hprintf (".");
  
  for (i = 0; i < count && *pfile_size > 0; i++) {
    file_data_sector = start + sectors_per_block * table[i];
    if (ide_read_sectors (0, file_data_sector, buf, sectors_per_block)) {
      return -1;
    }
    copy_size = (*pfile_size > SECTOR_SIZE * sectors_per_block) ?
      (SECTOR_SIZE * sectors_per_block) : *pfile_size;
    memcpy ((unsigned char *)*pload_address, buf, copy_size);
    *pfile_size -= copy_size;
    *pload_address += copy_size;
  }
  
  return 0;
}

#if !defined (CONSOLE)
char console[] = {"console=null"};
#else
char console[] = {"console=ttyAM0,115200"};
#endif

static int boot_cmdfunc(int argc, char *argv[])
{
	int gunzip_kernel = 1;

	unsigned char    buf[SECTOR_SIZE * 8], buf2[SECTOR_SIZE * 8];
	partition        part[4];
	int              i, j, gzimage = 0, ext = 0;
	char             filename[16];
	unsigned long    start, size, super_block_sector, sectors_per_block,
	  blocks_per_group, inodes_per_group, group_desc_sector,
	  first_inode_table, inode_table_sector, dir_entry_table_sector,
	  inode, group, file_size, file_block[15], file_data_sector,
	  addr_per_block;
	ext2_super_block *psb;
	ext2_group_desc  *pgd;
	ext2_inode       *pinode;
	ext2_dir_entry   *pde = 0;
	addr_t           load_address;
	int              o_argc;

	if (argc > 1) {
	  o_argc = argc;
	}
	else {
	  o_argc = get_option_count ();
	}
	
	{
	char *o_argv[o_argc + 3];
	if (argc > 1) {
	  for (i = 0; i < argc; i++) {
	    o_argv[i] = argv[i];
	  }
	}
	else {
	  if (o_argc <= 1) {
	    o_argv[o_argc++] = console;
	  }
	  else {
	    get_options(o_argv);
	  }
	}

	major = RAMDISK_MAJOR;
	minor = RAMDISK0_MINOR;
	if (!argc) {
	  if (!(*get_bbu_reg_addr(NS_BBU_GPIO_STATUS_1) &
		GPIO_STATUS_1_FLASHBOOT)) {
	    if (ide_detect_devices ()) {
	      return 0;
	    }
	    led_on();
	    if (ide_startup_devices ()) {
	      return 0;
	    }
	    if (ide_read_sectors (0, 0, buf, 1)) {
	      return 0;
	    }

	    memcpy (part, buf + 0x1be, sizeof(partition) * 4);
	    for (i = 0; i < 4; i++) {
	      if (part[i].sys_ind != 0x83) {
		continue;
	      }
	      start = read4_little_endian (part[i].start4);
	      size = read4_little_endian (part[i].size4);
	      hprintf ("/dev/hda%d: start=0x%x, size=0x%x\n",
		       i + 1, start, size);
	      super_block_sector = start + 2;
	      if (ide_read_sectors (0, super_block_sector, buf, 1)) {
		hprintf ("Super block read error.\n");
		continue;
	      }
	      psb = (ext2_super_block *)buf;
	      if (psb->s_magic != 0xef53) {
		hprintf ("Isn't ext2.\n");
		continue;
	      }
	      if (psb->s_inode_size != INODE_SIZE) {
		hprintf ("Bad inode size.\n");
		continue;
	      }
	      if (psb->s_log_block_size > 2) {
		hprintf ("Bad block size.\n");
		continue;
	      }
	      sectors_per_block = 2;
	      for (j = 0; j < psb->s_log_block_size; j++) {
		sectors_per_block *= 2;
	      }
	      blocks_per_group = psb->s_blocks_per_group;
	      inodes_per_group = psb->s_inodes_per_group;
	      group_desc_sector =
		start + sectors_per_block * (1 + psb->s_first_data_block);
	      if (ide_read_sectors (0, group_desc_sector, buf, 1)) {
		hprintf ("Group desc read error.\n");
		continue;
	      }
	      pgd = (ext2_group_desc *)buf;
	      first_inode_table = pgd->bg_inode_table;
	      inode_table_sector =
		start + sectors_per_block * first_inode_table;
	      if (ide_read_sectors (0, inode_table_sector, buf, 1)) {
		hprintf ("Inode table read error.\n");
		continue;
	      }
	      for (j = 0; j < SECTOR_SIZE; j += INODE_SIZE) {
		pinode = (ext2_inode *)(buf + j);
		if ((pinode->i_mode & 0xf000) == 0x4000) {
		  break;
		}
	      }
	      if (j >= SECTOR_SIZE) {
		hprintf ("Can't find root.\n");
		continue;
	      }
	      dir_entry_table_sector =
		start + sectors_per_block * pinode->i_block[0];
	      if (ide_read_sectors
		  (0, dir_entry_table_sector, buf, sectors_per_block)) {
		hprintf ("Root directory entry read error.\n");
		continue;
	      }
	      for (pde = (ext2_dir_entry *)buf;
		   (addr_t)pde - (addr_t)buf <=
		     SECTOR_SIZE * sectors_per_block - 12;
		   pde = (ext2_dir_entry *)(((addr_t)pde) + pde->rec_len)) {
		if (!pde->rec_len) {
		  break;
		}
		if (pde->name_len == 4) {
		  if (!memcmp (pde->name, "boot", 4)) {
		    break;
		  }
		}
	      }
	      if ((addr_t)pde - (addr_t)buf >
		  SECTOR_SIZE * sectors_per_block - 12 ||
		  !pde->rec_len) {
		hprintf ("Can't find /boot\n");
		continue;
	      }
	      group = (pde->inode - 1) / inodes_per_group;
	      inode = (pde->inode - 1) % inodes_per_group;
	      inode_table_sector = start + sectors_per_block *
		(blocks_per_group * group + first_inode_table) +
		inode / (SECTOR_SIZE / INODE_SIZE);
	      if (ide_read_sectors (0, inode_table_sector, buf, 1)) {
		hprintf ("Inode table read error.\n");
		continue;
	      }
	      pinode = (ext2_inode *)
		(buf + INODE_SIZE * (inode % (SECTOR_SIZE / INODE_SIZE)));
	      if ((pinode->i_mode & 0xf000) != 0x4000) {
		hprintf ("/boot is not directory.\n");
		continue;
	      }
	      dir_entry_table_sector =
		start + sectors_per_block * pinode->i_block[0];
	      if (ide_read_sectors
		  (0, dir_entry_table_sector, buf, sectors_per_block)) {
		hprintf ("/boot directory entry read error.\n");
		continue;
	      }
	      for (pde = (ext2_dir_entry *)buf;
		   (addr_t)pde - (addr_t)buf <=
		     SECTOR_SIZE * sectors_per_block - 16;
		   pde = (ext2_dir_entry *)(((addr_t)pde) + pde->rec_len)) {
		if (!pde->rec_len) {
		  break;
		}
		gzimage = 0;
		ext = 0;
		if (pde->name_len >= 5) {
		  if (!memcmp(pde->name, "Image", 5) ||
		      !memcmp(pde->name, "linux", 5)) {
		    if (pde->name_len == 5) {
		      break;
		    }
		    if (pde->name_len >= 9) {
		      if (!memcmp(pde->name + 5, ".bin", 4)) {
			ext = 4;
			if (pde->name_len == 9) {
			  break;
			}
		      }
		    }
		    if (pde->name_len == 5 + ext + 3) {
		      if (!memcmp(pde->name + 5 + ext, ".gz", 3)) {
			gzimage = 1;
			break;
		      }
		    }
		  }
		}
	      }
	      if ((addr_t)pde - (addr_t)buf >
		  SECTOR_SIZE * sectors_per_block - 16 ||
		  !pde->rec_len) {
		hprintf ("Can't find /boot/Image(or linux)(.bin)(.gz)\n");
		continue;
	      }
	      memcpy (filename, pde->name, pde->name_len);
	      filename[pde->name_len] = '\0';
	      group = (pde->inode - 1) / inodes_per_group;
	      inode = (pde->inode - 1) % inodes_per_group;
	      inode_table_sector = start + sectors_per_block *
		(blocks_per_group * group + first_inode_table) +
		inode / (SECTOR_SIZE / INODE_SIZE);
	      if (ide_read_sectors (0, inode_table_sector, buf, 1)) {
		hprintf ("Inode table read error.\n");
		continue;
	      }
	      pinode = (ext2_inode *)
		(buf + INODE_SIZE * (inode % (SECTOR_SIZE / INODE_SIZE)));
	      if ((pinode->i_mode & 0xf000) != 0x8000) {
		hprintf ("%s is not file.\n", filename);
		continue;
	      }
	      file_size = pinode->i_size;
	      if (file_size <= 0) {
		hprintf ("%s size is zero.\n", filename);
		continue;
	      }
	      if (file_size > INITRD_LOAD_ADDRESS - LINUX_LOAD_ADDRESS) {
		hprintf ("%s size is too large.\n", filename);
		continue;
	      }
	      hprintf ("%s is found.\n", filename);
	      for (j = 0; j < 15; j++) {
		file_block[j] = pinode->i_block[j];
	      }
	      load_address =
		(!gzimage) ? LINUX_LOAD_ADDRESS : INITRD_LOAD_ADDRESS;
	      hprintf ("Copying        kernel");
	      if (copy_file (12, file_block, start,
			     sectors_per_block, &file_size, &load_address)) {
		hprintf ("%s data read error.\n", filename);
		continue;
	      }
	      if (file_size <= 0) {
		gunzip_kernel = 0;
		break;
	      }
	      file_data_sector =
		start + sectors_per_block * file_block[12];
	      if (ide_read_sectors
		  (0, file_data_sector, buf, sectors_per_block)) {
		hprintf ("%s data read error.\n", filename);
		continue;
	      }
	      addr_per_block =
		(SECTOR_SIZE * sectors_per_block) / sizeof(unsigned long);
	      if (copy_file (addr_per_block, (unsigned long *)buf, start,
			       sectors_per_block, &file_size, &load_address)) {
		hprintf ("%s data read error.\n", filename);
		continue;
	      }
	      if (file_size <= 0) {
		gunzip_kernel = 0;
		break;
	      }
	      file_data_sector =
		start + sectors_per_block * file_block[13];
	      if (ide_read_sectors
		  (0, file_data_sector, buf2, sectors_per_block)) {
		hprintf ("%s data read error.\n", filename);
		continue;
	      }
	      for (j = 0; j < addr_per_block && file_size > 0; j++) {
		file_data_sector =
		  start + sectors_per_block * *(((unsigned long *)buf2) + j);
		if (ide_read_sectors
		    (0, file_data_sector, buf, sectors_per_block)) {
		  hprintf ("%s data read error.\n", filename);
		  break;
		}
		if (copy_file (addr_per_block, (unsigned long *)buf, start,
				 sectors_per_block, &file_size,
				 &load_address)) {
		  hprintf ("%s data read error.\n", filename);
		  break;
		}
	      }
	      if (file_size <= 0) {
		gunzip_kernel = 0;
		break;
	      }
	      hprintf ("%s size is too large.\n", filename);
	      continue;
	    }

	    if (gunzip_kernel) {
	      return 0;
	    }
	    hprintf ("done.\n");
	    if (gzimage) {
	      boost_on (BOOST_LINUX_MODE);
	      gunzip_object
		(" kernel", INITRD_LOAD_ADDRESS, LINUX_LOAD_ADDRESS);
	      boost_off ();
	    }
	    major = IDE0_MAJOR;
	    minor = 1 + i;
	  }
	}
	argc = o_argc;
	argv = o_argv;

	if (argc < 1)
		return -H_EUSAGE;

	led_on();
	
	boost_on (BOOST_LINUX_MODE);
	if (gunzip_kernel) {
	  gunzip_object (" kernel", LINUX_SRC_ADDRESS, LINUX_LOAD_ADDRESS);
	}
	if (major == RAMDISK_MAJOR) {
	  gunzip_object ("ramdisk", INITRD_SRC_ADDRESS, INITRD_LOAD_ADDRESS);
	}
	boost_off ();

	linux_cmdfunc (argc, argv);

	}

	return 0;
}

const command_t linux_command =
	{ "linux", "<linux options>", "start Linux", &linux_cmdfunc };
const command_t boot_command =
	{ "boot", "", "boot default Linux kernel and ramdisk", &boot_cmdfunc };

