/*
 * Copyright (C) 2003-2005 Atmark Techno, Inc.  All Rights Reserved.
 */

#include "flash_region.h"
#include "memregions.h"
#include <target/flash.h>
#include "mode.h"

struct FlashRegion {
        enum FlashRegionID id;
        unsigned int start;
        unsigned int size;
        char * blocking;
};

struct FlashRegion flash_regions_8MiB[] = {
        { .id       = FlashRegionIdAll,
          .start    = FLASH_8MiB_ALL_START,
          .size     = FLASH_8MiB_ALL_SIZE,
          .blocking = "8x8K/l,126x64K/l,8x8K/l",
        },
        { .id       = FlashRegionIdFpga,
          .start    = FLASH_8MiB_FPGA_START,
          .size     = FLASH_8MiB_FPGA_SIZE,
          .blocking = "8x64K/l",
        },
        { .id       = FlashRegionIdBootloader,
          .start    = FLASH_8MiB_BOOTLOADER_START,
          .size     = FLASH_8MiB_BOOTLOADER_SIZE,
          .blocking = "2x64K/l",
        },
        { .id       = FlashRegionIdImage,
          .start    = FLASH_8MiB_IMAGE_START,
          .size     = FLASH_8MiB_IMAGE_SIZE,
          .blocking = "109x64K",
        },
        { .id       = FlashRegionIdKernel,
          .start    = FLASH_8MiB_KERNEL_START,
          .size     = FLASH_8MiB_KERNEL_SIZE,
          .blocking = "48x64K",
        },
        { .id       = FlashRegionIdUser,
          .start    = FLASH_8MiB_USER_START,
          .size     = FLASH_8MiB_USER_SIZE,
          .blocking = "61x64K",
        },
        { .id       = FlashRegionIdConfig,
          .start    = FLASH_8MiB_CONFIG_START,
          .size     = FLASH_8MiB_CONFIG_SIZE,
          .blocking = "8x8K",
        },
        { .id       = FlashRegionIdFree1,
          .start    = FLASH_8MiB_FREE1_START,
          .size     = FLASH_8MiB_FREE1_SIZE,
          .blocking = "8x8K",
        },
        { .id       = FlashRegionIdFree2,
          .start    = FLASH_8MiB_FREE2_START,
          .size     = FLASH_8MiB_FREE2_SIZE,
          .blocking = "7x64K",
        },
};

struct FlashRegion flash_regions_4MiB[] = {
        { .id       = FlashRegionIdAll,
          .start    = FLASH_4MiB_ALL_START,
          .size     = FLASH_4MiB_ALL_SIZE,
          .blocking = "63x64K/l,8x8K/l",
        },
        { .id       = FlashRegionIdFpga,
          .start    = FLASH_4MiB_FPGA_START,
          .size     = FLASH_4MiB_FPGA_SIZE,
          .blocking = "8x64K/l",
        },
        { .id       = FlashRegionIdBootloader,
          .start    = FLASH_4MiB_BOOTLOADER_START,
          .size     = FLASH_4MiB_BOOTLOADER_SIZE,
          .blocking = "2x64K/l",
        },
        { .id       = FlashRegionIdImage,
          .start    = FLASH_4MiB_IMAGE_START,
          .size     = FLASH_4MiB_IMAGE_SIZE,
          .blocking = "53x64K",
        },
        { .id       = FlashRegionIdKernel,
          .start    = FLASH_4MiB_KERNEL_START,
          .size     = FLASH_4MiB_KERNEL_SIZE,
          .blocking = "23x64K",
        },
        { .id       = FlashRegionIdUser,
          .start    = FLASH_4MiB_USER_START,
          .size     = FLASH_4MiB_USER_SIZE,
          .blocking = "30x64K",
        },
        { .id       = FlashRegionIdConfig,
          .start    = FLASH_4MiB_CONFIG_START,
          .size     = FLASH_4MiB_CONFIG_SIZE,
          .blocking = "8x8K",
        },
        { .id       = FlashRegionIdFree1,
          .start    = 0,
          .size     = 0,
          .blocking = "0x0K",
        },
        { .id       = FlashRegionIdFree2,
          .start    = 0,
          .size     = 0,
          .blocking = "0x0K",
        },
};

struct FlashRegion flash_regions_spi[] = {
        { .id       = FlashRegionIdAll,
          .start    = FLASH_SPI_ALL_START,
          .size     = FLASH_SPI_ALL_SIZE,
          .blocking = "128x64K/l",
        },
        { .id       = FlashRegionIdFpga,
          .start    = FLASH_SPI_FPGA_START,
          .size     = FLASH_SPI_FPGA_SIZE,
          .blocking = "16x64K/l",
        },
        { .id       = FlashRegionIdBootloader,
          .start    = FLASH_8MiB_BOOTLOADER_START,
          .size     = FLASH_8MiB_BOOTLOADER_SIZE,
          .blocking = "2x64K/l",
        },
        { .id       = FlashRegionIdImage,
          .start    = FLASH_8MiB_IMAGE_START,
          .size     = FLASH_8MiB_IMAGE_SIZE,
          .blocking = "109x64K",
        },
        { .id       = FlashRegionIdKernel,
          .start    = FLASH_8MiB_KERNEL_START,
          .size     = FLASH_8MiB_KERNEL_SIZE,
          .blocking = "48x64K",
        },
        { .id       = FlashRegionIdUser,
          .start    = FLASH_8MiB_USER_START,
          .size     = FLASH_8MiB_USER_SIZE,
          .blocking = "61x64K",
        },
        { .id       = FlashRegionIdConfig,
          .start    = FLASH_SPI_CONFIG_START,
          .size     = FLASH_SPI_CONFIG_SIZE,
          .blocking = "1x64K",
        },
        { .id       = FlashRegionIdFree1,
          .start    = 0,
          .size     = 0,
          .blocking = "0x0K",
        },
        { .id       = FlashRegionIdFree2,
          .start    = 0,
          .size     = 0,
          .blocking = "0x0K",
        },
};

static int is_flash_8MiB(void)
{
        return ((flash_get_size(FLASH_START) == FLASH_8MiB) && (!is_compatible_mode()));
}

/* is there any header file I can include? -- yashi 2006/05/17 */
#define NULL ((void *)0)

unsigned int flash_region_get_start_address(enum FlashRegionID id)
{
	unsigned int ret = 0xffffffff;

        if (id > FlashRegionIdMax)
                return ret;

	if (is_flash_spi()) {
		ret = flash_regions_spi[id].start;
	} else {
		if (is_flash_8MiB())
			ret = flash_regions_8MiB[id].start;
		else
			ret = flash_regions_4MiB[id].start;
	}

	return ret;
}

unsigned int flash_region_get_end_address(enum FlashRegionID id)
{
	unsigned int ret = 0;

        if (id > FlashRegionIdMax)
                return ret;

	return flash_region_get_start_address(id) + flash_region_get_size(id) - 1;
}

int flash_region_get_size(enum FlashRegionID id)
{
	int ret = -1;

        if (id > FlashRegionIdMax)
                return ret;

	if (is_flash_spi()) {
		ret = flash_regions_spi[id].size;
	} else {
		if (is_flash_8MiB())
			ret = flash_regions_8MiB[id].size;
		else
			ret = flash_regions_4MiB[id].size;
	}

	return ret;
}

char * flash_region_get_name(enum FlashRegionID id)
{
	char *ret = NULL;

        if (id > FlashRegionIdMax)
		return ret;

	switch(id) {
	case FlashRegionIdAll:
		ret = "all"; break;
	case FlashRegionIdFpga:
		ret = "fpga"; break;
	case FlashRegionIdBootloader:
		ret = "bootloader"; break;
	case FlashRegionIdImage:
		ret = "image"; break;
	case FlashRegionIdKernel:
		ret = "kernel"; break;
	case FlashRegionIdUser:
		ret = "user"; break;
	case FlashRegionIdConfig:
		ret = "config"; break;
	case FlashRegionIdFree1:
		ret = "free1"; break;
	case FlashRegionIdFree2:
		ret = "free2"; break;
	default:
		ret = "unknown"; break;
	}

	return ret;
}

char * flash_region_get_blocking(enum FlashRegionID id)
{
	char *ret = NULL;

        if (id > FlashRegionIdMax)
		return ret;

	if (is_flash_spi()) {
		ret = flash_regions_spi[id].blocking;
	} else {
		if (is_flash_8MiB())
			ret = flash_regions_8MiB[id].blocking;
		else
			ret = flash_regions_4MiB[id].blocking;
	}

	return ret;
}
