/*
 * Copyright (c) 2003 Atmark Techno, Inc.  All Rights Reserved.
 *
 * Command set for Vanilla ConnectME test
 */

#include "version.h"

#include <target/command.h>
#include <target/herrno.h>
#include <target/io.h>
#include <target/scan.h>

#include "bootmode.h"

#define NR_TEST 3
#define NR_PORT 8

#define TIMER_BASE_ADDR 0xffb00018
#define TSTAT_BASE_ADDR 0xffb0001c
#define PORTC_BASE_ADDR 0xffb00028
#define PORTA_BASE_ADDR 0xffb00020
#define INTRE_BASE_ADDR 0xffb00030

#define TIMER (*((volatile unsigned int *)TIMER_BASE_ADDR))
#define TSTAT (*((volatile unsigned int *)TSTAT_BASE_ADDR))
#define PORTA (*((volatile unsigned int *)PORTA_BASE_ADDR))
#define PORTC (*((volatile unsigned int *)PORTC_BASE_ADDR))
#define INTRE (*((volatile unsigned int *)INTRE_BASE_ADDR))

#define TIMER1       (1 << 5)
#define TIMER2       (1 << 4)

#define TIMER_TE     (1 << 31)
#define TIMER_TIE    (1 << 30)
#define TIMER_TIRO   (1 << 29)
#define TIMER_TPRE   (1 << 28)
#define TIMER_TCLK   (1 << 27)
#define TIMER_ITC(x) (x << 0)

#define TSTAT_TIP    (1 << 30)

#define PORT0 0
#define PORT1 1
#define PORT2 2
#define PORT3 3
#define PORT4 4
#define PORT5 5
#define PORT6 6
#define PORT7 7

#define DCD (1<<PORT0)
#define CTS (1<<PORT1)
#define DSR (1<<PORT2)
#define RXD (1<<PORT3)
#define RTS (1<<PORT5)
#define DTR (1<<PORT6)
#define TXD (1<<PORT7)

#define CMODE(port_num) (0x01000000 << (port_num))
#define CDIR(port_num)  (0x00010000 << (port_num)) /* 0: IN   1: OUT */
#define CSF(port_num)   (0x00000100 << (port_num))
#define CDATA(port_num) (0x00000001 << (port_num))

#define AMODE(port_num) (0x01000000 << (port_num))
#define ADIR(port_num)  (0x00010000 << (port_num)) /* 0: IN   1: OUT */
#define ADATA(port_num) (0x00000001 << (port_num)) /* */
        
        
static void make_noise(void)
{
        word_t val;
        val = PORTC;

        val &= ~(CMODE(5) | CSF(5) | CDATA(5));
        val |= CDIR(5);

        PORTC = val;
}

        static void stop_noise(void)
{
        word_t val;
        val = PORTC;

        val &= ~(CDIR(5));

        PORTC = val;
}

static void __udelay(void)
{
        int i;
        for (i=0; i<2; i++);
}

static void udelay(int usec)
{
        int i;
        for (i=0; i<usec; i++)
                __udelay();
}

static void __mdelay(void)
{
        int i;
        for (i=0; i<1000; i++)
                __udelay();
}

static void mdelay(int msec)
{
        int i;
        for (i=0; i<msec; i++)
                __mdelay();
}

static void init_gpio(void)
{
        PORTA = 0x00e00000;
}

static void set_gpio(int flags)
{
        init_gpio();

        PORTA |= flags;
}

static int get_gpio(void)
{
        return PORTA;
}

static int check_gpio(int flags)
{
        word_t val;

        val = PORTA & (CTS | DSR | DCD);
        return val == flags ? 0 : -1;
}

#define XTALE    (18432000)
#define TIMEOUT  (1)
#define ITC(x)   (( (XTALE/4096) *TIMEOUT) - 1)
//#define ITC(x)  (0x7ffffff)

static void enable_timer(void)
{
        TIMER |= (TIMER_TE);
        TSTAT &= ~(TSTAT_TIP);
        INTRE |= (TIMER2);
}

static void disable_timer(void)
{
        TIMER &= ~(TIMER_TE);
        INTRE &= ~(TIMER2);
}

static void init_timer(void)
{
        static int initialized = 0;

        if (initialized) {
                return;
        }

        initialized = 1;

        TIMER =  (TIMER_TIE | TIMER_TIRO | TIMER_TPRE | ITC(TIMEOUT) );
}

/**************************************************/


static int do_serial_test(void)
{
        int i, j, ret = 0;
        init_timer();
        for (j=0; j<NR_TEST; j++) {
                for (i=0; i<256; i++) {
/*                 for (i=0x30; i<0x3a; i++) { */
                        enable_timer();
                        hputchar(i);
                        if (hgetchar() != i) {
                                disable_timer();
                                ret = -H_EIO;
                                goto end;
                        }
                        disable_timer();
                }
        }
end:
        return ret;
}

static int do_gpio_test(void)
{
        int i;

        for (i=0; i<NR_TEST; i++) {
                int res;
                set_gpio(RTS|DTR);
                udelay(100);
                res = check_gpio(CTS|DSR|DCD);
                if (res)
                        return -H_EIO;

                set_gpio(DTR);
                udelay(100);
                res = check_gpio(DSR|DCD);
                if (res)
                        return -H_EIO;

                set_gpio(RTS);
                udelay(100);
                res = check_gpio(CTS|DCD);
                if (res)
                        return -H_EIO;

                set_gpio(0);
                udelay(100);
                res = check_gpio(CTS|DSR);
                if (res)
                        return -H_EIO;
        }

        return 0;
}

static int do_test_done(void)
{
        make_noise();
        return 0;
}

void do_failed(void)
{
        int i;
        __asm__(".global _do_failed");
        for (i=0; i<60; i++) {
                make_noise();
                mdelay(250);
                stop_noise();
                mdelay(250);
        }
}

static int do_mfgi_test(void)
{
        /* wait for a bit */
        mdelay(10);

        if (is_mfgi_hi())
                return 0;
        else
                return -1; /* error */
}


static int test_cmdfunc(int argc, char *argv[])
{
        int res;

        res = do_serial_test();
        if (res)
                goto error;

        res = do_gpio_test();
        if (res)
                goto error;

        res = do_mfgi_test();
        if (res)
                goto error;

        do_test_done();
        return 0;

error:
        do_failed();
        return 1;
}

const command_t test_command = {
        "test", 0, "run test", &test_cmdfunc };

static int test_serial_cmdfunc (int argc, char *argv[]) { return do_serial_test(); }
static int test_gpio_cmdfunc   (int argc, char *argv[]) { return do_gpio_test(); }
static int test_mfgi_cmdfunc   (int argc, char *argv[]) { return do_mfgi_test(); }
static int test_done_cmdfunc   (int argc, char *argv[]) { return do_test_done(); }
static int test_failed_cmdfunc (int argc, char *argv[]) { do_failed(); return 0; }
static int stop_noise_cmdfunc  (int argc, char *argv[]) { stop_noise(); return 0; }

static int set_gpio_cmdfunc (int argc, char *argv[])
{
        word_t port;

        if (argc != 2)
                return -H_EUSAGE;

        argv++;
        if (scan(*argv, &port))
                return -1; /* error */

        set_gpio(port);
        return 0;
}

static int get_gpio_cmdfunc (int argc, char *argv[])
{
        hprintf("%x\n", get_gpio());
        return 0;
}

const command_t test_serial_command = {
        "test_serial", 0, "run serial test", &test_serial_cmdfunc };
const command_t test_gpio_command = {
        "test_gpio", 0, "run gpio test", &test_gpio_cmdfunc };
const command_t test_mfgi_command = {
        "test_mfgi", 0, "run mfgi test", &test_mfgi_cmdfunc };
const command_t test_done_command = {
        "test_done", 0, "emulate test is finished", &test_done_cmdfunc };
const command_t test_failed_command = {
        "test_failed", 0, "emulate test is failed", &test_failed_cmdfunc };
const command_t stop_noise_command = {
        "stop_noise", 0, "stop noise", &stop_noise_cmdfunc };
const command_t set_gpio_command = {
        "set_gpio", "<pin>", "set gpio pin high", &set_gpio_cmdfunc };
const command_t get_gpio_command = {
        "get_gpio", 0, "get gpio status", &get_gpio_cmdfunc };


static int mdelay_cmdfunc(int argc, char *argv[])
{
        word_t val;

        if (argc != 2)
                return -H_EUSAGE;

        argv++;
        if (scan(*argv, &val))
                return -1; /* error */

        mdelay(val);

        return 0;
}

const command_t mdelay_command =
{ "mdelay", "<msec>", "delay msec milisecond", &mdelay_cmdfunc };
