(ns atena.gui.HistoryPanel
  (:import (java.awt BorderLayout GridLayout GridBagLayout Insets
                     Font Color Dimension FlowLayout)
           (java.awt.event  ActionListener)
           (javax.swing JPanel JLabel JTextField JButton JCheckBox JOptionPane JScrollPane RowFilter)
           (javax.swing.border TitledBorder LineBorder)
           (javax.swing.event DocumentListener)
           (javax.swing.table DefaultTableCellRenderer TableColumn
                              TableCellRenderer DefaultTableModel
                              DefaultTableColumnModel TableRowSorter)
           (java.util.regex Pattern Matcher)
           (java.util Calendar)
           (java.io File)
           (atena.gui HistoryTable)
           )
  (:require [atena.utils.gridbaglayout]
            [atena.gui.Utility]
            [clojure.contrib.string :as string])
  (:use [clojure.contrib.duck-streams :only (reader read-lines writer write-lines with-out-writer)]
        [clojure.set])
  (:gen-class
   :extends javax.swing.JPanel
   :init init
   :post-init after-ctor
   :state state
   :methods [[addAddButtonListener [java.awt.event.ActionListener] void]
             [addSaveButtonListener [java.awt.event.ActionListener] void]
             [addDeleteButtonListener [java.awt.event.ActionListener] void]
             [getMournings [] Object]
             [getHistories [] Object]
             [uuid [] String]
             [uuid [String] void]
             [mourningsPath [] String]
             [mourningsPath [String] void]
             [historiesFile [] java.io.File]
             [historiesFile [java.io.File] void]
             [sender [] String]
             [sender [String] void]
             [readMournings [] Object]
             [readHistories [] Object]
             [save [] void]
             [refresh [] void]
            ]
   ))


(defn -init []
  [[]
   (ref {:uuid          (ref nil)
         :mourningsPath (ref "")
         :historiesFile (ref nil)
         :sender        (ref nil)
         :addButton     (JButton. "追加")
         :deleteButton  (JButton. "削除")
         :saveButton    (JButton. "保存")
         :table         (atena.gui.HistoryTable.)
        })])

(atena.gui.Utility/defaddlistener -addAddButtonListener this addButton)
(atena.gui.Utility/defaddlistener -addSaveButtonListener this saveButton)
(atena.gui.Utility/defaddlistener -addDeleteButtonListener this deleteButton)

(atena.gui.Utility/defield mourningsPath this)
(atena.gui.Utility/defield historiesFile this)
(atena.gui.Utility/defield uuid this)
(atena.gui.Utility/defield sender this)
  
(defn -getMournings
  [this]
  (let [{:keys [table]} @(.state this)
        model           (. table getModel)]
    (. model getMournings)))

(defn -getHistories
  [this]
  (let [{:keys [table]} @(.state this)
        model           (. table getModel)]
    (. model getHistories)))

(defn readMourningsFile
  [filePath]
  (let [file (java.io.File. filePath)]
    (if (not (. file exists))
        (. file createNewFile))
    (reduce (fn [dic line]
              (let [fields (vec (string/split #"," line))
                    k (keyword (fields 0))
                    v (subvec fields 1)]
                (assoc dic k v)))
            {}
            (read-lines file))))

(defn readHistoriesFile
  [file]
  (if (= nil file)
      {}
      (do
        (if (not (. file exists))
            (. file createNewFile))
        (reduce (fn [dic line]
                  (let [fields (vec (string/split #"," line))
                        k      (keyword (fields 0))
                        l      (reduce (fn [d h]
                                         (let [f (vec (string/split #" " h))
                                               y (f 0)
                                               v (subvec f 1)]
                                           (assoc d y v)))
                                       {}
                                       (subvec fields 1))]
                    (assoc dic k l)))
                {}
                (read-lines file)))))
  
(defn -readMournings
  [this]
  (readMourningsFile (. this mourningsPath)))
  
(defn -readHistories
  [this]
  (readHistoriesFile (. this historiesFile)))
  
(defn writeMourningsFile
  [filePath
   data]
  (write-lines (writer (str filePath))
               (map #(let [uuid  (subs (str (key %1)) 1)
                           years (string/join "," (val %1))]
                       (str uuid "," years))
                    data)))

(defn writeHistoriesFile
  [filePath
   data]
  (defn history
    "[扱うデータ] key: year, val: [sent-date recv-date]"
    [entry]
    (string/join " " (flatten entry)))

  (defn histories
    "[扱うデータ] {history0, history1, ...}"
    [dic]
    (string/join "," (map history dic)))

  (defn person
    "[扱うデータ] key: uuid, val: histories"
    [entry]
    (let [uuid  (subs (str (key entry)) 1)
          hstrs (histories (val entry))]
      (str uuid "," hstrs)))

  (let [lines (map person data)]
    (write-lines (writer (str filePath)) lines)))

(defn -save
  [this]
  (let [mPath (. this mourningsPath)
        mHash (readMourningsFile mPath)
        mData (assoc mHash (keyword (. this uuid)) (vec (. this getMournings)))
        hFile (. this historiesFile)
        hHash (readHistoriesFile hFile)
        hData (assoc hHash (keyword (. this uuid)) (. this getHistories))
       ]
    (writeMourningsFile mPath mData)
    (writeHistoriesFile hFile hData)))

  
(defn -refresh
  [this]
  (let [key-uuid (keyword (. this uuid))
        mournings ((. this readMournings) key-uuid)
        histories ((. this readHistories) key-uuid)
        mset      (reduce #(conj %1 %2) #{} mournings)
        hset      (reduce #(conj %1 (first %2)) #{} histories)
        years     (union mset hset)
        data      (map #(let [h (if (= nil histories)
                                    [nil nil]
                                    (let [h1 (histories %1)]
                                      (if (= nil h1)
                                          [nil nil]
                                          h1)))]
                          (vec (list %1 (contains? mset %1) (h 0) (h 1))))
                       years)
        {:keys [table]} @(.state this)
        model   (. table getModel)
       ]
    (. model setData data)
    )
  )
  
(defn -setMourningsByUuid
  [this
   uuid-string]
  (let [{:keys [table]} @(.state this)
        model   (. table getModel)
        alldata (. this readMournings)
        k       (keyword uuid-string)
        target  (alldata k)]
    (. model setMournings target)
    (. this uuid uuid-string)))
  
(defn -after-ctor [this]
  (let [{:keys [filePath
                addButton
                deleteButton
                saveButton
                table]} @(.state this)
        model           (. table getModel)
        panel           this]

    (doto panel
      (.setLayout (GridBagLayout.))
      (.setBorder (TitledBorder. (LineBorder. Color/GRAY 2)
                                 "履歴"
                                 TitledBorder/LEFT
                                 TitledBorder/TOP
                                 (Font. Font/MONOSPACED Font/BOLD 12)))
      (atena.utils.gridbaglayout/grid-bag-layout
        :gridx 0, :gridy 0
        :anchor :WEST
        (doto (JPanel. (FlowLayout. FlowLayout/LEFT))
          (.add addButton)
          (.add deleteButton)
          (.add saveButton)
        )
        :insets (Insets. 5 5 5 5)
        :gridx 0, :gridy 1
        ;:fill :HORIZONTAL
        ;:weightx 1.0
        :weighty 1.0
        (doto (JScrollPane. table)
          (.setMinimumSize (Dimension. 286 300))
        )
      )
    )

    ;追加ボタン
    (doto addButton
      (.addActionListener
        (proxy [ActionListener] []
          (actionPerformed [evt]
            (. model addRow
              (to-array [(.. Calendar getInstance (get Calendar/YEAR))
              　         false
                         false
                         false
                         ]))))))
    ;削除ボタン
    (doto deleteButton
      (.addActionListener
        (atena.gui.Utility/make-confirm-ActionListener
          "削除確認"
          "選択されている履歴情報を削除しますか？"
          (fn []
            (let [indexes (map (fn [index]
                                 (. table convertRowIndexToModel index))
                               (. table getSelectedRows))]
              (doseq [index (reverse indexes)]
                (. model removeRow index)))))))
    ;保存ボタン
    (doto saveButton
      (.addActionListener
        (proxy [ActionListener][]
          (actionPerformed [evt]
            (. panel save)))))
  )
)

