// AXL3 - An XLL Library
// Wrapper class library for Microsft Excel's xll add-in which written by C++.
//
// Copyright(c) 2009-2009, yukimi_sake@users.sourceforge.jp. All rights reserved.

//////// Implements whitch corespond to Excel4/Excel12 call back functions. ////////

#pragma once

#include <string>

namespace xl{
	class xlVar;
	class xlRange;

	class xlAPI{
		friend class xlBase;
		friend class xlVar;
		friend class xlRange;

	private:
		xlAPI(){};
		bool m_excel12;
		union{
			void* m_xlproc;
			XLPROC4 m_xlproc4;
			XLPROC12 m_xlproc12;
		};

		template <class P> int BoolToOper(const int num,  P* oper){
			oper->xltype = xltypeInt;
			oper->val.w = num;
			return 1;
		}

		template <class P> int CharToOper(const char* chr, P* oper){
			int len = strlen(chr);
			char* str = new char[len+1];
			str[0] = (BYTE)len;
			memcpy_s(str+1, len+1, chr, len); 
			oper->xltype = xltypeStr;
			oper->val.str = str;
			return (BYTE)1;
		}

		template <> int CharToOper<xloper12>(const char* chr, xloper12* oper){
			int len = MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED, chr, -1, 0, 0);
			wchar_t* str = new wchar_t[len];
			MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED, chr,-1, str+1, len-1);
			str[0] = (WORD)len-1;
			oper->xltype = xltypeStr;
			oper->val.str = str;
			return 1;
		}

		template <class P> int IntToOper(const int num, P* oper){
			oper->xltype = xltypeInt;
			oper->val.w = num;
			return 1;
		}

		template <class P> int ErrToOper(const int num, P* oper){
			oper->xltype = xltypeErr;
			oper->val.err = num;
			return 1;
		}

		template <class P> int IntToMissing(P* oper){
			oper->xltype = xltypeMissing;
			oper->val.num = 0;
			return 1;
		}

		template <class P> int NumToOper(const double num, P* oper){
			oper->xltype = xltypeNum;
			oper->val.num = num;
			return 1;
		}

		template <class P, class F, int N> int PerformCall(const int iFunc, void* pRet,const int nArg, P** rgxloper){
			int ret;
			if (pRet){
				ret = ((F)m_xlproc)(iFunc, nArg, rgxloper, (P*)(((xlBase*)pRet)->m_xloper));
				*(((xlBase*)pRet)->m_refcount) = F_CALL_XLFREE; // call xlFree flag.
			}else{
				ret = ((F)m_xlproc)(iFunc, nArg, rgxloper, 0);
			}
			return ret;
		}

		template <class P, int N> int GetArgs(P* rgxloper[N], const int nArg, ...){
			va_list pArg;
			if ((nArg < 0) || (nArg > N)) return xlretInvCount;
			va_start(pArg, nArg);
			for (int i = 0; i < nArg ; i++){
				rgxloper[i] = (P*)(va_arg(pArg, xlVar*)->m_xloper);
			}
			return 0;
		}

		template <class P, class F, int N> int CallWithTemplate(const int iFunc, void* pRet, const char* cTemplate, va_list pArg){
			BYTE rgflags[N];
			P* rgxloper[N];
			int ret;
			int nArg = cTemplate ? strlen(cTemplate) : 0;
			if ((nArg < 0) || (nArg > N)){va_end(pArg); return xlretInvCount;}
			for (int i= 0; i < nArg ; i++){
				switch(cTemplate[i]){
					case '_':  // TypeMissing
						rgflags[i] = IntToMissing(rgxloper[i] = new P);
						break;
					case 'B':  // Bool
						rgflags[i] = BoolToOper(va_arg(pArg, bool), rgxloper[i] = new P);
						break;
					case 'C':  // Char*
						rgflags[i] = CharToOper(va_arg(pArg, char*), rgxloper[i] = new P);
						break;
					case 'E':  // ErrorNo
						rgflags[i] = ErrToOper(va_arg(pArg, int), rgxloper[i] = new P);
						break;
					case 'I':  // Integer (for command)
						rgflags[i] = IntToOper(va_arg(pArg, int), rgxloper[i] = new P);
						break;
					case 'N':  // Number(Float)
						rgflags[i] = NumToOper(va_arg(pArg, double), rgxloper[i] = new P);
						break;
					case 'P':  // xlBase* or xlVar* or xlRange*
						rgxloper[i] = (P*)(va_arg(pArg, xlBase*)->m_xloper);
						rgflags[i] = 0;
						break;
					case 'S':  // std::string
						rgflags[i] = CharToOper(va_arg(pArg, std::string).c_str(), rgxloper[i] = new P);
						break;
					default:   // Invalid template char
						return  xlretInvXloper;
				}
			}

			ret = PerformCall<P,F,N>(iFunc, pRet, nArg, &rgxloper[0]);

			for (int i = 0; i < nArg; i++){
				if (rgflags[i] && !(rgxloper[i]->xltype & (xlbitXLFree | xlbitDLLFree))){
					if (rgxloper[i]->xltype & xltypeStr){ delete [] rgxloper[i]->val.str; }
					delete rgxloper[i];
				}
			}
			return ret;
		}

	public:
		BOOL Initialized(){
			return (BOOL)m_xlproc;
		}

		void InitXlFunc(){
			if (m_xlproc12 || m_xlproc4) return; 
			HMODULE hmod = GetModuleHandle(NULL);
			XLPROC12 xlproc12 = (XLPROC12)GetProcAddress(hmod, "MdCallBack12");
			XLPROC4  xlproc4  = (XLPROC4 )GetProcAddress(hmod, "MdCallBack");
			if(xlproc12){
				m_xlproc12 = xlproc12;
				m_excel12= true;
				xlBase::m_excel12 = true;
				xlBase::m_xlproc = xlproc12;
			}else{
				m_xlproc4 = xlproc4;
				m_excel12 = false;
				xlBase::m_excel12 = false;
				xlBase::m_xlproc = xlproc4;
			}
		}

		int Excel4v(int xlfn, xloper4* operRes, int count, LPXLOPER opers[]){
			if ((count < 0) || (count > 30)) return xlretInvCount;
			return (m_xlproc4)(xlfn, count, &opers[0], operRes);
		}

		int Excel12v(int xlfn, xloper12* operRes, int count, LPXLOPER12 opers[]){
			if ((count < 0) || (count > 255)) return xlretInvCount;
			return (m_xlproc12)(xlfn, count, &opers[0], operRes);
		}

		template <class T> int Callv(int iFunc, void* pRet, int nArg, T* pArg[]){
			int ret;
			if(m_excel12){
				xloper12* rgxloper12[255];
				if ((nArg < 0) || (nArg > 255)) return xlretInvCount;
				for (int i= 0; i < nArg ; i++){ rgxloper12[i] = pArg[i]->m_xloper12; }
				ret = (m_xlproc12)(iFunc, nArg, &rgxloper12[0], pRet? ((T*)pRet)->m_xloper12 : 0);
			}else{
				xloper4* rgxloper4[30];
				if ((nArg < 0) || (nArg > 30)) return xlretInvCount;
				for (int i= 0; i < nArg ; i++){ rgxloper4[i] = pArg[i]->m_xloper4; }
				ret = (m_xlproc4)(iFunc, nArg, &rgxloper4[0], pRet? ((T*)pRet)->m_xloper4 : 0);
			}
			if (pRet) *(((xlBase*)pRet)->m_refcount) = F_CALL_XLFREE; // call xlFree flag.
			return ret;
		}

		int Call(int iFunc, xlVar* pRet, int nArg, ... ){
			int ret;
			va_list pArg;
			va_start(pArg, nArg);
			if(m_excel12){
				xloper12* rgxloper12[255];
				ret = GetArgs<xloper12, 255>(rgxloper12, nArg, pArg);
				if(ret == 0) ret = PerformCall<xloper12, XLPROC12, 255>(iFunc, pRet, nArg, &rgxloper12[0]);
			}else{
				xloper4* rgxloper4[30];
				ret = GetArgs<xloper4, 30>(rgxloper4, nArg, pArg);
				if(ret == 0) ret = PerformCall<xloper4, XLPROC4, 30>(iFunc, pRet, nArg, &rgxloper4[0]);
			}
			va_end(pArg);
			return ret;
		}

		int Call(int iFunc, void* pRet, const char* tTemplate, ... ){
			int ret;
			va_list pArg;
			va_start(pArg, tTemplate);
			if(m_excel12){
				ret = CallWithTemplate<xloper12, XLPROC12, 255>(iFunc, pRet, tTemplate, pArg);
			}else{
				ret = CallWithTemplate<xloper4, XLPROC4, 30>(iFunc, pRet, tTemplate, pArg);
			}
			va_end(pArg);
			return ret;
		}

		int XlFree(int nArg, ...){
			int ret;
			va_list pArg;
			va_start(pArg, nArg);
			if(m_excel12){
				xloper12* rgxloper12[255];
				ret = GetArgs<xloper12, 255>(rgxloper12, nArg, pArg);
				if(ret == 0) return (m_xlproc12)(xlFree, nArg, &rgxloper12[0], 0);
			}else{
				xloper4* rgxloper4[30];
				ret = GetArgs<xloper4, 30>(rgxloper4, nArg, pArg);
				if(ret == 0) return (m_xlproc4)(xlFree, nArg, &rgxloper4[0], 0);
			}
			va_end(pArg);
		}

		bool IsExcel12(){ return m_excel12; }

		static xlAPI& Instance(){ static xlAPI x; return x; }
	};

	__declspec(selectany) xl::xlAPI API = xl::xlAPI::Instance();
}

