//
//  CMRBrowser-Delegate.m
//  BathyScaphe
//
//  Updated by Tsutomu Sawada on 07/09/18.
//  Copyright 2005-2010 BathyScaphe Project. All rights reserved.
//  encoding="UTF-8"
//

#import "CMRBrowser_p.h"
#import "BoardManager.h"
#import "missing.h"
#import "BSNobiNobiToolbarItem.h"
#import "BSQuickLookPanelController.h"
#import "DatabaseManager.h"
#import "BSLabelManager.h"

extern NSString *const ThreadsListDownloaderShouldRetryUpdateNotification;

@implementation CMRBrowser(Delegate)
BOOL isOptionKeyDown(void)
{
    NSUInteger flag_ = [[NSApp currentEvent] modifierFlags];
    if (flag_ & NSAlternateKeyMask) {
        return YES;
    } else {
        return NO;
    }
}

#pragma mark NSControl Delegate (SearchField)
// Available in RainbowJerk and later.
// 検索フィールドで return などを押したとき、フォーカスをスレッド一覧に移動させる
- (void)controlTextDidEndEditing:(NSNotification *)aNotification
{
    if ([aNotification object] == [self searchField]) {
        [[self window] makeFirstResponder:[self threadsListTable]];
    }
}

#pragma mark KFSplitView Delegate
- (BOOL)splitView:(id)sender canCollapseSubview:(NSView *)subview
{
    return (subview == bottomSubview);
}

- (void)splitView:(id)sender didDoubleClickInDivider:(NSInteger)index
{
    [self collapseOrExpandSplitView:sender];
    [self synchronizeLayoutSwitcher];
}

- (void)splitView:(id)sender didFinishDragInDivider:(NSInteger)index
{
    [self synchronizeLayoutSwitcher];
}

- (void)splitViewDidCollapseSubview:(NSNotification *)notification
{
    [[self indexingNavigator] setHidden:YES];
    [[self numberOfMessagesField] setHidden:YES];
    [[self document] setShowsThreadDocument:NO];
}

- (void)splitViewDidExpandSubview:(NSNotification *)notification
{
    [[self indexingNavigator] setHidden:NO];
    [[self numberOfMessagesField] setHidden:NO];
    [[self document] setShowsThreadDocument:YES];
}

- (void)splitView:(id)sender resizeSubviewsWithOldSize:(NSSize)oldSize
{
    // It's our responsibility to set the frame rectangles of
    // all uncollapsed subviews.
    NSInteger i, numSubviews, numDividers;
    CGFloat heightTotal, splitViewWidth, splitViewHeight, newSubviewHeight;
    CGFloat curYAxisPos, dividerThickness, scaleFactor, availableSpace;
    CGFloat minimumFirstSubviewHeight;
    
    CGFloat widthTotal, curXAxisPos, minimumFirstSubviewWidth, newSubviewWidth;
    
    id subview, subviews;

    // setup
    subviews = [sender subviews];
    numSubviews = [subviews count];
    numDividers = numSubviews - 1;
    splitViewWidth = [sender frame].size.width;
    splitViewHeight = [sender frame].size.height;
    dividerThickness = [sender dividerThickness];

    minimumFirstSubviewHeight = 90;
    minimumFirstSubviewWidth = 120;

    if ([sender isVertical]) {
        widthTotal = 0;
        for (i = 1; i < numSubviews; i++)
        {
            subview = [subviews objectAtIndex:i];
            if (![sender isSubviewCollapsed:subview]) {
                widthTotal += [subview frame].size.width;
            }
        }

        availableSpace = splitViewWidth - minimumFirstSubviewWidth - numDividers*dividerThickness;
        if (widthTotal > availableSpace) {
            if (availableSpace < 0) {
                scaleFactor = 0;
            } else {
                scaleFactor = availableSpace / widthTotal;
            }
        } else {
            scaleFactor = 1;
        }
        
        curXAxisPos = splitViewWidth;
        for (i = numSubviews - 1; i > 0; i--) {
            subview = [subviews objectAtIndex:i];
            if (![sender isSubviewCollapsed:subview]) {
                newSubviewWidth = floor([subview frame].size.width*scaleFactor);
                curXAxisPos -= newSubviewWidth;
                [subview setFrame:NSMakeRect(curXAxisPos, 0, newSubviewWidth, splitViewHeight)];
            }
            
            curXAxisPos -= dividerThickness;
        }
        
        subview = [subviews objectAtIndex:0];
        [subview setFrame:NSMakeRect(0, 0, curXAxisPos, splitViewHeight)];
        if ([subview isKindOfClass:[NSScrollView class]]) {
            CGFloat tableViewWidth = [[(NSScrollView *)subview documentView] frame].size.width;
            CGFloat scrollViewWidth = [[(NSScrollView *)subview contentView] frame].size.width;
            [(NSScrollView *)subview setHasHorizontalScroller:(tableViewWidth > scrollViewWidth)];
        }
        
    } else {
        // tabulate the total space taken up by uncollapsed subviews other than the first
        heightTotal = 0;
        for (i = 1; i < numSubviews; i++)
        {
            subview = [subviews objectAtIndex:i];
            if (![sender isSubviewCollapsed:subview]) {
                heightTotal += [subview frame].size.height;
            }
        }

        // if the uncollapsed subviews (not counting the first) take up too much space then
        // we have to scale them
        availableSpace = splitViewHeight - minimumFirstSubviewHeight - numDividers*dividerThickness;
        if (heightTotal > availableSpace) {
            if (availableSpace < 0) {
                scaleFactor = 0;
            } else {
                scaleFactor = availableSpace / heightTotal;
            }
        } else {
            scaleFactor = 1;
        }

        // we walk up the Y-axis, setting subview frames as we go
        curYAxisPos = splitViewHeight;
        for (i = numSubviews - 1; i >0; i--) {
            subview = [subviews objectAtIndex:i];
            if (![sender isSubviewCollapsed:subview]) {
                // expanded subviews need to have their origin set correctly and
                // their size scaled.

                newSubviewHeight = floor([subview frame].size.height*scaleFactor);
                curYAxisPos -= newSubviewHeight;
                [subview setFrame:NSMakeRect(0, curYAxisPos, splitViewWidth, newSubviewHeight)];
            }

            // account for the divider taking up space
            curYAxisPos -= dividerThickness;
        }

        // the first subview subview's height is whatever's left over
        subview = [subviews objectAtIndex:0];
        [subview setFrame:NSMakeRect(0, 0, splitViewWidth, curYAxisPos)];
    }

    // if we wanted error checking, we could call adjustSubviews.  It would
    // only change something if we messed up and didn't really tile the split view correctly.

    //[sender adjustSubviews];
}

#pragma mark NSOutlineView Delegate
- (void)outlineViewSelectionDidChange:(NSNotification *)notification
{
    NSInteger                 rowIndex_;
    NSOutlineView       *brdListTable_;
    NSDictionary        *item_;
    
    brdListTable_ = [notification object];

    UTILAssertNotificationName(
        notification,
        NSOutlineViewSelectionDidChangeNotification);
    UTILAssertNotificationObject(
        notification,
        [self boardListTable]);
    
    rowIndex_ = [brdListTable_ selectedRow];
    
    if ([brdListTable_ numberOfSelectedRows] > 1) return;
    if (rowIndex_ < 0) return;
    if (rowIndex_ >= [brdListTable_ numberOfRows]) return;

    item_ = [brdListTable_ itemAtRow:rowIndex_];

    if (!item_) return;
    if (![item_ hasURL] && ![BoardListItem isFavoriteItem:item_] && ![BoardListItem isSmartItem:item_]) return;

    [self showThreadsListForBoard:item_];
}

- (void)outlineView:(NSOutlineView *)olv willDisplayCell:(NSCell *)cell forTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
    if ([[tableColumn identifier] isEqualToString:BoardPlistNameKey]) {
        [cell setImage:[item icon]];
    }
}

- (void)outlineView:(NSOutlineView *)outlineView willDisplayOutlineCell:(id)cell forTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
    if (floor(NSAppKitVersionNumber) > NSAppKitVersionNumber10_5) {
        // アウトライン・ビューのトライアングルを、ハイライト時は白で描くために
        if ([outlineView selectedRow] == [outlineView rowForItem:item]) {
            [cell setBackgroundStyle:NSBackgroundStyleDark];
        } else {
            [cell setBackgroundStyle:NSBackgroundStyleLight];
        }
    }
}

#pragma mark Type-To-Select Support
- (NSIndexSet *)outlineView:(BSBoardListView *)boardListView findForString:(NSString *)aString
{
    SmartBoardList       *source;
    BoardListItem   *matchedItem;
    NSInteger             index;

    source = (SmartBoardList *)[boardListView dataSource];
    
    matchedItem = [source itemWithNameHavingPrefix:aString];

    if (!matchedItem) {
        return nil;
    }
        
    index = [self searchRowForItemInDeep:matchedItem inView:boardListView];
    if (-1 == index) return nil;
    return [NSIndexSet indexSetWithIndex:index];
}

#pragma mark NSTableView Delegate
- (void)tableView:(NSTableView *)tableView didClickTableColumn:(NSTableColumn *)tableColumn
{
    static BOOL hasOptionClicked = NO;
    BoardManager *bm = [BoardManager defaultManager];
    NSString *boardName = [[self currentThreadsList] boardName];

    // Sort:
    // カラムヘッダをクリックしたとき、まず
    // -[NSObject(NSTableDataSource) tableView:sortDescriptorsDidChange:] が送られ、
    // その後で -[NSObject(NSTableViewDelegate) tableView:didClickTableColumn:] が送られる。

    // Sort:
    // Mac OS標準的ソート変更 (Finderのリスト表示参照)
    // ソートの向きは各カラムごとに保存されており、
    // ハイライトされているカラムヘッダがクリックされた時以外は、
    // 保存されている向きでソートされる。
    // 既にハイライトされているヘッダをクリックした場合は
    // 昇順／降順の切り替えと見なす。

    // Sort:
    // option キーを押しながらヘッダをクリックした場合は、変更後の設定を CMRPref に保存する（グローバルな設定の変更）。
    // ただし、option キーを押しながらクリックした場合、sortDescriptorDidChange: は呼ばれない。
    // それどころか、カラムのハイライトも更新されない。
    // 仕方がないので、option キーを押しながらクリックされた場合は、
    // ここでダミーのクリックイベントをもう一度発生させ、通常のカラムヘッダクリックをシミュレートする。
    // ダミーイベントによってもう一度 -tableView:didClickTableColumn: が発生するので、
    // そこで必要な処理を行なう。
    if (isOptionKeyDown()) {
        NSEvent *dummyEvent = [NSApp currentEvent];
        hasOptionClicked = YES;
        // このへん、Thousand のコード（THTableHeaderView.m）を参考にした
        NSEvent *downEvent = [NSEvent mouseEventWithType:NSLeftMouseDown
                                                location:[dummyEvent locationInWindow]
                                           modifierFlags:0
                                               timestamp:[dummyEvent timestamp]
                                            windowNumber:[dummyEvent windowNumber]
                                                 context:[dummyEvent context]
                                             eventNumber:[dummyEvent eventNumber]+1
                                              clickCount:1
                                                pressure:1.0];
        NSEvent *upEvent = [NSEvent mouseEventWithType:NSLeftMouseUp
                                              location:[dummyEvent locationInWindow]
                                         modifierFlags:0
                                             timestamp:[dummyEvent timestamp]
                                          windowNumber:[dummyEvent windowNumber]
                                               context:[dummyEvent context]
                                           eventNumber:[dummyEvent eventNumber]+2
                                            clickCount:1
                                              pressure:1.0];
        [NSApp postEvent:upEvent atStart:NO];
        [NSApp postEvent:downEvent atStart:YES];

        return;
    }

    // 設定の保存
    [bm setSortDescriptors:[tableView sortDescriptors] forBoard:boardName];

    if (hasOptionClicked) {
        [CMRPref setThreadsListSortDescriptors:[tableView sortDescriptors]];
        hasOptionClicked = NO;
    }

    NSInteger selected = [tableView selectedRow];
    if (selected != -1) {
        CMRAutoscrollCondition prefMask = [CMRPref threadsListAutoscrollMask];
        if (prefMask & CMRAutoscrollWhenTLSort) {
            [tableView scrollRowToVisible:selected];
        } else {
            [tableView scrollRowToVisible:0];
        }
    } else {
        [tableView scrollRowToVisible:0];
    }

    UTILDebugWrite(@"Catch tableView:didClickTableColumn:");
}

- (void)saveBrowserListColumnState:(NSTableView *)targetTableView
{
    [CMRPref setThreadsListTableColumnState:[targetTableView columnState]];
    if ([[self splitView] isVertical]) {
        NSScrollView *scrollView = [targetTableView enclosingScrollView];
        if (!scrollView) {
            return;
        }
        CGFloat tableViewWidth = [targetTableView frame].size.width;
        CGFloat scrollViewWidth = [[scrollView contentView] frame].size.width;
        [scrollView setHasHorizontalScroller:(tableViewWidth > scrollViewWidth)];        
    }
}

- (void)tableViewColumnDidMove:(NSNotification *)aNotification
{
    [self saveBrowserListColumnState:[aNotification object]];
}

- (void)tableViewColumnDidResize:(NSNotification *)aNotification
{
    [self saveBrowserListColumnState:[aNotification object]];
}

- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
    BSQuickLookPanelController *qlc = [BSQuickLookPanelController sharedInstance];
    NSTableView *tableView = [aNotification object];
    if ([qlc isLooking]) {
        [[tableView dataSource] tableView:self quickLookAtRowIndexes:[tableView selectedRowIndexes] keepLook:YES];
    }
}

#pragma mark RBSplitView Delegate
- (void)splitView:(RBSplitView *)sender wasResizedFrom:(CGFloat)oldDimension to:(CGFloat)newDimension
{
    [sender adjustSubviewsExcepting:[self boardListSubView]];
}

- (void)splitView:(RBSplitView*)sender changedFrameOfSubview:(RBSplitSubview*)subview from:(NSRect)fromRect to:(NSRect)toRect
{
    if (subview == [self boardListSubView]) {
        NSNotification *notification = [NSNotification notificationWithName:BSNobiNobiToolbarItemShouldAdjustWidthNotification
                                                                     object:self
                                                                   userInfo:nil];
        [[NSNotificationQueue defaultQueue] enqueueNotification:notification
                                                   postingStyle:NSPostWhenIdle
                                                   coalesceMask:(NSNotificationCoalescingOnName|NSNotificationCoalescingOnSender)
                                                       forModes:nil];
    }
}

// 掲示板リストとスレッド一覧の境界から掲示板リストに5ピクセル内側の領域までは、リサイズ可能領域とする
- (NSUInteger)splitView:(RBSplitView *)sender dividerForPoint:(NSPoint)point inSubview:(RBSplitSubview *)subview
{
    if (subview == [self boardListSubView]) {
		NSRect lead = [subview frame];
        CGFloat width = lead.size.width;
        lead.origin.x += (width - 5);
        lead.size.width = 5;
		if ([sender mouse:point inRect:lead]) {
			return 0;
		}
    }
    return NSNotFound;
}

// 上のメソッドで拡張したリサイズ可能領域内でマウスカーソルを適切に変更する
- (NSRect)splitView:(RBSplitView *)sender cursorRect:(NSRect)rect forDivider:(NSUInteger)divider
{
    if (divider == 0) {
        rect.origin.x -= 5;
        rect.size.width += 5;
    }
    return rect;
}
@end


@implementation CMRBrowser(NotificationPrivate)
- (void)registerToNotificationCenter
{
    NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
    DatabaseManager *db = [DatabaseManager defaultManager];
    [nc addObserver:self
           selector:@selector(boardManagerUserListDidChange:)
               name:CMRBBSManagerUserListDidChangeNotification
             object:[BoardManager defaultManager]];
    [nc addObserver:self
           selector:@selector(threadsListDownloaderShouldRetryUpdate:)
               name:ThreadsListDownloaderShouldRetryUpdateNotification
             object:nil];
    [nc addObserver:self
           selector:@selector(threadDocumentDidToggleDatOchiStatus:)
               name:CMRAbstractThreadDocumentDidToggleDatOchiNotification
             object:nil];
    [nc addObserver:self
           selector:@selector(threadDocumentDidToggleLabel:)
               name:CMRAbstractThreadDocumentDidToggleLabelNotification
             object:nil];
    [nc addObserver:self
           selector:@selector(databaseWillUpdateThreadItem:)
               name:DatabaseWillUpdateThreadItemNotification
             object:db];
    [nc addObserver:self
           selector:@selector(databaseWillDeleteThreadItems:)
               name:DatabaseWillDeleteThreadItemsNotification
             object:db];

    [nc addObserver:self
           selector:@selector(adjustLazily:)
               name:BSNobiNobiToolbarItemShouldAdjustWidthNotification
             object:self];
    [nc addObserver:self
           selector:@selector(labelDisplayNamesUpdated:)
               name:BSLabelManagerDidUpdateDisplayNamesNotification
             object:nil];

    [super registerToNotificationCenter];
}

- (void)removeFromNotificationCenter
{
    NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
    DatabaseManager *db = [DatabaseManager defaultManager];

    [nc removeObserver:self name:BSLabelManagerDidUpdateDisplayNamesNotification object:nil];
    [nc removeObserver:self name:BSNobiNobiToolbarItemShouldAdjustWidthNotification object:self];

    [nc removeObserver:self name:DatabaseWillDeleteThreadItemsNotification object:db]; 
    [nc removeObserver:self name:DatabaseWillUpdateThreadItemNotification object:db];
    [nc removeObserver:self name:CMRAbstractThreadDocumentDidToggleLabelNotification object:nil];
    [nc removeObserver:self name:CMRAbstractThreadDocumentDidToggleDatOchiNotification object:nil];
    [nc removeObserver:self name:ThreadsListDownloaderShouldRetryUpdateNotification object:nil];
    [nc removeObserver:self name:CMRBBSManagerUserListDidChangeNotification object:[BoardManager defaultManager]];

    [super removeFromNotificationCenter];
}

- (void)adjustLazily:(NSNotification *)notification
{
    RBSplitSubview *view = [self boardListSubView];
    id hoge = [[self toolbarDelegate] itemForItemIdentifier:@"Boards List Space"];
    [hoge adjustWidth:[view dimension]];
    // 進行状況表示も位置調整
    NSRect currentRect = [[self statusMessageField] frame];
    CGFloat maxX = currentRect.origin.x + currentRect.size.width;
    CGFloat newOriginX = [view dimension] + 8;
    CGFloat newWidth = maxX - newOriginX;
    currentRect.origin.x = newOriginX;
    currentRect.size.width = newWidth;
    [[self statusMessageField] setFrame:currentRect];
}

- (void)boardManagerUserListDidChange:(NSNotification *)notification
{
    UTILAssertNotificationName(notification, CMRBBSManagerUserListDidChangeNotification);
    UTILAssertNotificationObject(notification, [BoardManager defaultManager]);

    [[self boardListTable] reloadData];
    id item = [[self currentThreadsList] boardListItem];
    [self reselectBoard:item];
}

- (void)appDefaultsLayoutSettingsUpdated:(NSNotification *)notification
{
    UTILAssertNotificationName(notification, AppDefaultsLayoutSettingsUpdatedNotification);
    UTILAssertNotificationObject(notification, CMRPref);
    
    [BSDBThreadList resetDataSourceTemplates];
    [BSDBThreadList resetDataSourceTemplateForDateColumn];

    [self updateThreadsListTableWithNeedingDisplay:YES];
    [self updateBoardListViewWithNeedingDisplay:YES];
    [self updateThreadEnergyColumn];
    
    if ([[self superclass] instancesRespondToSelector:_cmd]) {
        [super appDefaultsLayoutSettingsUpdated:notification];
    }
}

- (void)labelDisplayNamesUpdated:(NSNotification *)notification
{
    UTILAssertNotificationName(notification, BSLabelManagerDidUpdateDisplayNamesNotification);

    NSInteger column = [[self threadsListTable] columnWithIdentifier:BSThreadLabelKey];
    if (column != -1) {
        NSRect rect = [[self threadsListTable] rectOfColumn:column];
        [[self threadsListTable] setNeedsDisplayInRect:rect];
    }

    [self setupBrowserContextualMenuLabelNames];

    if ([[self superclass] instancesRespondToSelector:_cmd]) {
        [super appDefaultsLayoutSettingsUpdated:notification];
    }
}

- (void)cleanUpItemsToBeRemoved:(NSArray *)files
{
    BSTitleRulerView *ruler = (BSTitleRulerView *)[[self scrollView] horizontalRulerView];
    [ruler setTitleStr:NSLocalizedString(@"titleRuler default title", @"Startup Message")];
    [ruler setPathStr:nil];

    if ([[self superclass] instancesRespondToSelector:_cmd]) {
        [super cleanUpItemsToBeRemoved:files];
    }
}

- (void)threadsListDidChange:(NSNotification *)notification
{
    UTILAssertNotificationName(notification, CMRThreadsListDidChangeNotification);

    [[self threadsListTable] reloadData];
    [self synchronizeWindowTitleWithDocumentName];
    [self reselectThreadIfNeeded];
    UTILNotifyName(CMRBrowserThListUpdateDelegateTaskDidFinishNotification);
}

- (void)threadsListWantsPartialReload:(NSNotification *)notification
{
    UTILAssertNotificationName(notification, BSDBThreadListWantsPartialReloadNotification);
    id indexes = [[notification userInfo] objectForKey:@"Indexes"];

    if (indexes == [NSNull null]) {
        [[self threadsListTable] reloadData];
    } else {
        NSUInteger    index;
        NSInteger             size = [indexes lastIndex]+1;
        NSRange         e = NSMakeRange(0, size);

        while ([indexes getIndexes:&index maxCount:1 inIndexRange:&e] > 0) {
            NSRect rect = [[self threadsListTable] rectOfRow:index];
            if (!NSEqualRects(rect, NSZeroRect)) {
                [[self threadsListTable] setNeedsDisplayInRect:rect];
            }
        }
    }

    [self synchronizeWindowTitleWithDocumentName];

    if ([self keepPaths]) {
        [self setKeepPaths:nil];
        [self setKeepCondition:CMRAutoscrollNone];
    }
}

- (void)threadsListDownloaderShouldRetryUpdate:(NSNotification *)notification
{
    [self reloadThreadsList:nil];
}

- (void)threadDocumentDidToggleDatOchiStatus:(NSNotification *)aNotification
{
    NSString *path = [[aNotification userInfo] objectForKey:@"path"];
    [[self currentThreadsList] toggleDatOchiThreadItemWithPath:path];
}

- (void)threadDocumentDidToggleLabel:(NSNotification *)aNotification
{
    NSString *path = [[aNotification userInfo] objectForKey:@"path"];
    NSUInteger code = [[[aNotification userInfo] objectForKey:@"code"] unsignedIntegerValue];
    [[self currentThreadsList] setLabel:code forThreadItemWithPath:path];
}

- (void)databaseWillUpdateThreadItem:(NSNotification *)aNotification
{
    [self storeKeepPath:CMRAutoscrollWhenThreadUpdate];
}

- (void)databaseWillDeleteThreadItems:(NSNotification *)aNotification
{
    [self storeKeepPath:CMRAutoscrollWhenThreadDelete];
}

// Added in InnocentStarter.
- (void)sleepDidEnd:(NSNotification *)aNotification
{
    if (![CMRPref isOnlineMode]) return;
    NSTimeInterval delay = [CMRPref delayForAutoReloadAtWaking];

    if ([CMRPref autoReloadViewerWhenWake] && [self shouldShowContents] && [self threadAttributes]) {
        [self performSelector:@selector(reloadThread:) withObject:nil afterDelay:delay];
    }

    if ([CMRPref autoReloadListWhenWake] && [BoardListItem isBoardItem:[[self currentThreadsList] boardListItem]]) {
        [self performSelector:@selector(reloadThreadsList:) withObject:nil afterDelay:delay];
    }
}

- (void)reselectThreadIfNeeded
{
    CMRAutoscrollCondition type = [self keepCondition];
    if ([self keepPaths]) {
        CMRAutoscrollCondition mask = [CMRPref threadsListAutoscrollMask];
        [self selectRowIndexesWithThreadPaths:[self keepPaths] byExtendingSelection:NO scrollToVisible:(mask & type)];
        [self setKeepPaths:nil];
        [self setKeepCondition:CMRAutoscrollNone];
    } else {
        // 3ペインで、掲示板を切り替えた場合に、表示中のスレッドを再選択するために
        if (type == CMRAutoscrollNone && [self shouldShowContents] && [self path]) {
            [self selectRowWithCurrentThread:NO];
            return;
        }
        if (type != CMRAutoscrollWhenThreadUpdate && type != CMRAutoscrollWhenThreadDelete) {
            [[self threadsListTable] scrollRowToVisible:0];
        }
    }
}
@end
