//
//  HMPlugInPaneViewController.m
//  BathyScaphe
//
//  Created by masakih on 12/08/04.
//  Copyright 2012 BathyScaphe Project. All rights reserved.
//  encoding="UTF-8"
//

#import "BSPSPlugInPaneViewController.h"

#import "BSLinkPreviewSelector.h"
#import "BSLPSPreviewerItem.h"

static NSString *const BSLPSItemPastboardType = @"PSPItemPastboardType";
static NSString *const BSLPSRowIndexType = @"PSPRowIndexType";

@interface BSPSPlugInPaneViewController ()

@property (retain, nonatomic) NSArray *items;
@property (assign, nonatomic) NSView *currentItemPreferenceView;

@property (readonly) BSLPSPreviewerItem *selection;
@end

@implementation BSPSPlugInPaneViewController
@synthesize itemsController = _itemsController;
@synthesize items = _items;
@synthesize currentItemPreferenceView = _currentItemPreferenceView;
@synthesize defaultPreferenceView = _defaultPreferenceView;
@synthesize itemPreferencePlaceholder = _itemPreferencePlaceholder;
@synthesize itemsListView = _itemsListView;

- (id)init
{
	self = [super initWithNibName:NSStringFromClass([self class]) bundle:[NSBundle bundleForClass:[self class]]];
		
	return self;
}

- (void)awakeFromNib
{	
	[self.itemsListView registerForDraggedTypes:[NSArray arrayWithObject:BSLPSItemPastboardType]];
	[self.itemsController addObserver:self
						   forKeyPath:@"selection"
							  options:0
							  context:self.itemsController];
	
	[self.itemsController bind:NSContentArrayBinding
					  toObject:[BSLinkPreviewSelector sharedInstance]
				   withKeyPath:@"items.previewerItems"
					   options:nil];
}

- (void)setRepresentedObject:(id)representedObject
{
	[super setRepresentedObject:representedObject];
	self.items = representedObject;
}

- (BSLPSPreviewerItem *)selection
{
	return [[self.itemsController selectedObjects] lastObject];
}

- (void)setCurrentItemPreferenceView:(NSView *)currentItemPreferenceView
{
	[self.currentItemPreferenceView removeFromSuperview];
	NSRect placeholderFrame = [self.itemPreferencePlaceholder frame];
	NSRect itemViewFrame = [currentItemPreferenceView frame];
	itemViewFrame.origin.x = 0;
	itemViewFrame.origin.y = NSHeight(placeholderFrame) - NSHeight(itemViewFrame);
	[currentItemPreferenceView setFrame:itemViewFrame];
	[self.itemPreferencePlaceholder addSubview:currentItemPreferenceView];
	_currentItemPreferenceView = currentItemPreferenceView;
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context
{	
	if(context == self.itemsController) {
		id preview = self.selection.previewer;
		NSView *preferenceView = nil;
		if([preview respondsToSelector:@selector(preferenceView)]) {
			preferenceView = [preview preferenceView];
		}
		if(!preferenceView) {
			preferenceView = self.defaultPreferenceView;
		}
		self.currentItemPreferenceView = preferenceView;
		
		return;
	}
	
	[super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
}

- (IBAction)toggleAPlugin:(id)sender
{
	BSLPSPreviewerItem *obj = [self selection];
	if(!obj) return;
	
	if([obj.previewer respondsToSelector:@selector(togglePreviewPanel:)]) {
		[obj.previewer performSelector:@selector(togglePreviewPanel:) withObject:self];
	}
}
- (IBAction)openPreferences:(id)sender
{
	BSLPSPreviewerItem *obj = [self selection];
	if(!obj) return;
	
	if([obj.previewer respondsToSelector:@selector(showPreviewerPreferences:)]) {
		[obj.previewer performSelector:@selector(showPreviewerPreferences:) withObject:self];
	}
}

- (IBAction)addPlugIn:(id)sender
{
	NSOpenPanel *panel = [NSOpenPanel openPanel];
	[panel setResolvesAliases:YES];
	[panel setAllowsMultipleSelection:YES];
	[panel setAllowedFileTypes:[NSArray arrayWithObjects:@"bundle", @"plugin", nil]];
	
	[panel setMessage:@"Message"];
	
	__block BOOL hasError = NO;
	[panel beginSheetModalForWindow:[self.view window]
				  completionHandler:^(NSInteger result) {
					  if(hasError) return;
					  if(result == NSFileHandlingPanelCancelButton) return;
					  
					  for(NSURL *fileURL in [panel URLs]) {
						  @try {
							  [[BSLinkPreviewSelector sharedInstance] addItemFromURL:fileURL];
						  }
						  @catch (id ex) {
							  NSString *message = PSLocalizedString(@"Selected plug-in has been deleted already.", @"");
							  NSString *info = PSLocalizedString(@"You must restart BatyScaphe, if you want to install this plug-in.",@"");
							  NSAlert *alert = [NSAlert alertWithMessageText:message
															   defaultButton:@"OK"
															 alternateButton:nil
																 otherButton:nil
												   informativeTextWithFormat:info];
							  [panel orderOut:nil];
							  [alert beginSheetModalForWindow:[self.view window]
												modalDelegate:self
											   didEndSelector:@selector(alertDidEnd:returnCode:contextInfo:)
												  contextInfo:NULL];
							  
							  hasError = YES;
						  }
					  }
				  }];
}
- (void)alertDidEnd:(NSAlert *)alert returnCode:(NSInteger)returnCode contextInfo:(void *)contextInfo
{
	//
}

- (IBAction)removePlugIn:(id)sender
{
	BSLPSPreviewerItem *obj = [self selection];
	BSLinkPreviewSelector *previewerSelector = [BSLinkPreviewSelector sharedInstance];
	[previewerSelector removeItem:obj];
}

- (BOOL)acceptsFirstResponder
{
	return YES;
}
- (BOOL)becomeFirstResponder
{
	[[[self view] window] makeFirstResponder:self.itemsListView];
	return YES;
}
#pragma mark## NSTableView Delegate ##

- (BOOL)tableView:(NSTableView *)tableView writeRowsWithIndexes:(NSIndexSet *)rowIndexes toPasteboard:(NSPasteboard*)pboard
{
	if([rowIndexes count] != 1) return NO;
	
	NSUInteger index = [rowIndexes firstIndex];
	
	[pboard declareTypes:[NSArray arrayWithObjects:BSLPSItemPastboardType, BSLPSRowIndexType, nil] owner:nil];
	[pboard setData:[NSKeyedArchiver archivedDataWithRootObject:[self.items objectAtIndex:index]]
			forType:BSLPSItemPastboardType];
	[pboard setPropertyList:[NSNumber numberWithUnsignedInteger:index] forType:BSLPSRowIndexType];
	
	return YES;
}

- (NSDragOperation)tableView:(NSTableView*)targetTableView
				validateDrop:(id <NSDraggingInfo>)info
				 proposedRow:(NSInteger)row
	   proposedDropOperation:(NSTableViewDropOperation)dropOperation
{
	NSPasteboard *pboard = [info draggingPasteboard];
	if(![[pboard types] containsObject:BSLPSItemPastboardType]) {
		return NSDragOperationNone;
	}
	
	if(dropOperation == NSTableViewDropOn) {
        [targetTableView setDropRow:row dropOperation:NSTableViewDropAbove];
	}
	
	NSUInteger originalRow = [[pboard propertyListForType:BSLPSRowIndexType] unsignedIntegerValue];
	if(row == originalRow || row == originalRow + 1) {
		return NSDragOperationNone;
	}
	
	return NSDragOperationMove;
}

- (BOOL)tableView:(NSTableView*)tableView
	   acceptDrop:(id <NSDraggingInfo>)info
			  row:(NSInteger)row
	dropOperation:(NSTableViewDropOperation)dropOperation
{
	NSPasteboard *pboard = [info draggingPasteboard];
	if(![[pboard types] containsObject:BSLPSItemPastboardType]) {
		return NO;
	}
	
	if(row < 0) row = 0;
	
	NSUInteger originalRow = [[pboard propertyListForType:BSLPSRowIndexType] unsignedIntegerValue];
	
	NSData *itemData = [pboard dataForType:BSLPSItemPastboardType];
	BSLPSPreviewerItem *item = [NSKeyedUnarchiver unarchiveObjectWithData:itemData];
	if(![item isKindOfClass:[BSLPSPreviewerItem class]]) {
		return NO;
	}
	
	[[BSLinkPreviewSelector sharedInstance] moveItem:item toIndex:row];
		
	return YES;
}

@end
