/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.cat.build.stepcounter.task;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.cat.build.stepcounter.task.valueobject.StepCounterCsvProcessInput;

/**
 * Apache Antタスク [StepCounterCsv]のクラス。
 *
 * StepCounterCsvのタスクです。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.cat.build.stepcounter.task]にStepCounterCsvBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;stepcountercsv&quot; classname=&quot;benten.cat.build.stepcounter.task.StepCounterCsvTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class StepCounterCsvTask extends Task {
    /**
     * StepCounterCsvのタスクです。
     */
    protected StepCounterCsvProcessInput fInput = new StepCounterCsvProcessInput();

    /**
     * フィールド [targetDir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetDirProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[targetDir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * StepCounterが出力したCSVファイルが格納されている\nディレクトリを指定します。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetDir(final String arg) {
        fInput.setTargetDir(arg);
        fIsFieldTargetDirProcessed = true;
    }

    /**
     * Antタスクの[targetDir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * StepCounterが出力したCSVファイルが格納されている\nディレクトリを指定します。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetDir() {
        return fInput.getTargetDir();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("StepCounterCsvTask begin.");

        // 項目番号[1]、アトリビュート[targetDir]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetDirProcessed == false) {
            throw new BuildException("必須アトリビュート[targetDir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- targetDir:[" + getTargetDir() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、StepCounterCsvProcessインタフェースを実装して benten.cat.build.stepcounter.taskパッケージに StepCounterCsvProcessImplクラスを作成することにより解決できる場合があります。
            final StepCounterCsvProcess proc = new StepCounterCsvProcessImpl();
            if (proc.execute(fInput) != StepCounterCsvBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
