/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.engine.omegat;

import java.io.File;
import java.io.IOException;

import junit.framework.TestCase;
import benten.core.io.Files;

public class DefaultTmEngineTest extends TestCase {

	private File createTmxFile(final String encoding) throws IOException {

		final StringBuilder sb = new StringBuilder();
		sb.append("<?xml version=\"1.1\" encoding=\"" + encoding + "\"?>");
		sb.append("<tmx version=\"1.4\">");
		sb.append("  <header creationtool=\"blancoTmx\" creationtoolversion=\"0.0.9\"");
		sb.append("    segtype=\"block\" o-tmf=\"XLIFF\" adminlang=\"EN-US\" srclang=\"EN-US\" datatype=\"unknown\"/>");
		sb.append("  <body>");
		sb.append("    <tu>");
		sb.append("      <tuv xml:lang=\"EN-US\"><seg>aaa</seg></tuv>");
		sb.append("      <tuv xml:lang=\"JA\"><seg>あああ</seg></tuv>");
		sb.append("    </tu>");
		sb.append("    <tu>");
		sb.append("      <tuv xml:lang=\"EN-US\"><seg>bbb</seg></tuv>");
		sb.append("      <tuv xml:lang=\"JA\"><seg>いいい</seg></tuv>");
		sb.append("    </tu>");
		sb.append("    <tu>");
		sb.append("      <tuv xml:lang=\"EN-US\"><seg>bbb x.</seg></tuv>");
		sb.append("      <tuv xml:lang=\"JA\"><seg>いいい x.</seg></tuv>");
		sb.append("    </tu>");
		sb.append("  </body>");
		sb.append("</tmx>");

		final File tmxFile = File.createTempFile("benten", ".tmx");
		for (final File file : tmxFile.getParentFile().listFiles()) {
			if (file != tmxFile && file.getName().endsWith(".tmx")) {
				file.delete();
			}
		}
		tmxFile.deleteOnExit();
		Files.writeStringToFile(tmxFile, sb.toString(), encoding);
		return tmxFile;
	}

	public void testFuzzySearch() throws IOException {

		final DefaultTmEngine target = new DefaultTmEngine();

		final File tmxFileUTF8 = createTmxFile("UTF-8");
		try {
			target.loadTmx(null);
			fail();
		} catch (final NullPointerException e) {
			assertTrue(true);
		}
		try {
			target.loadTmx(tmxFileUTF8);
			fail();
		} catch (final IllegalArgumentException e) {
			assertTrue(true);
		}
		try {
			target.fuzzySearch(null);
			assertTrue(true);
		} catch (final NullPointerException e) {
			fail();
		}

		target.loadTmx(tmxFileUTF8.getParentFile());

		assertEquals(0, target.fuzzySearch("a").size());
		assertEquals(1, target.fuzzySearch("aaa").size());
		assertEquals("aaa", target.fuzzySearch("aaa").get(0).getSource());
		assertEquals("あああ", target.fuzzySearch("aaa").get(0).getTarget());
		assertNotNull(target.fuzzySearch("aaa").get(0).getOrigin());

		assertEquals(2, target.fuzzySearch("bbb").size());
		assertEquals("bbb", target.fuzzySearch("bbb").get(0).getSource());
		assertEquals("いいい", target.fuzzySearch("bbb").get(0).getTarget());
		assertNotNull(target.fuzzySearch("bbb").get(0).getOrigin());

		assertEquals("bbb x.", target.fuzzySearch("bbb").get(1).getSource());
		assertEquals("いいい x.", target.fuzzySearch("bbb").get(1).getTarget());
		assertNotNull(target.fuzzySearch("bbb").get(1).getOrigin());

		target.unload();

		assertEquals(0, target.fuzzySearch("a").size());
		assertEquals(0, target.fuzzySearch("aaa").size());

		tmxFileUTF8.delete();
		final File tmxFileUTF16 = createTmxFile("UTF-16");
		target.loadTmx(tmxFileUTF16.getParentFile());

		assertEquals(0, target.fuzzySearch("a").size());
		assertEquals(1, target.fuzzySearch("aaa").size());
		assertEquals("aaa", target.fuzzySearch("aaa").get(0).getSource());
		assertEquals("あああ", target.fuzzySearch("aaa").get(0).getTarget());
		assertNotNull(target.fuzzySearch("aaa").get(0).getOrigin());

		assertEquals(2, target.fuzzySearch("bbb").size());
		assertEquals("bbb", target.fuzzySearch("bbb").get(0).getSource());
		assertEquals("いいい", target.fuzzySearch("bbb").get(0).getTarget());
		assertNotNull(target.fuzzySearch("bbb").get(0).getOrigin());

		assertEquals("bbb x.", target.fuzzySearch("bbb").get(1).getSource());
		assertEquals("いいい x.", target.fuzzySearch("bbb").get(1).getTarget());
		assertNotNull(target.fuzzySearch("bbb").get(1).getOrigin());

		target.unload();

		assertEquals(0, target.fuzzySearch("a").size());
		assertEquals(0, target.fuzzySearch("aaa").size());
	}
}
