/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core;

import java.io.File;
import java.io.IOException;
import java.net.URL;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;

/**
 * Benten 非 UI プラグインの基底クラス。
 *
 * <UL>
 * <LI>Benten で使用する非 UI 系プラグインの基底クラスです。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public abstract class AbstractBentenPlugin extends Plugin {

	/**
	 * デバッグするかどうか。
	 *
	 * <UL>
	 * <LI>Benten 開発者向けのデバッグ・フラグです。
	 * <LI>Java 起動オプションで -Dbenten.debug=true を指定することにより有効となります。
	 * </UL>
	 */
	public static final boolean IS_DEBUG = Boolean.valueOf(System.getProperty("benten.debug")); //$NON-NLS-1$

	/**
	 * プラグイン id の取得。
	 *
	 * <UL>
	 * <LI>Benten プロジェクトではプラグイン id はパッケージ名と同一です。
	 * </UL>
	 *
	 * @return プラグイン id
	 */
	public final String getPluginId() {
		return getClass().getPackage().getName();
	}

	/**
	 * 指定されたメッセージの Eclipse ログ出力。
	 *
	 * @param message ログ出力するメッセージ
	 */
	public void log(final String message) {
		printLog(getPluginId(), message);
	}

	/**
	 * 指定された例外の Eclipse ログ出力。
	 *
	 * @param e ログ出力する例外
	 */
	public void log(final Throwable e) {
		printLog(getPluginId(), e);
	}

	/**
	 * 指定されたメッセージの Eclipse ログ出力。
	 *
	 * @param pluginId プラグイン id
	 * @param message ログ出力するメッセージ
	 */
	public void printLog(final String pluginId, final String message) {
		final String s = message == null ? "" : message.toString(); //$NON-NLS-1$
		getLog().log(new Status(IStatus.INFO, pluginId, s));
		if (IS_DEBUG) {
			System.out.println(message);
		}
	}

	/**
	 * 指定された例外の Eclipse ログ出力。
	 *
	 * @param pluginId プラグイン id
	 * @param e ログ出力する例外
	 */
	public void printLog(final String pluginId, final Throwable e) {
		getLog().log(new Status(IStatus.ERROR, pluginId, -1, e.toString(), e));
		if (IS_DEBUG) {
			e.printStackTrace();
		}
	}

	/**
	 * このプラグイン配下のファイルを取得。
	 *
	 * <UL>
	 * <LI>指定する path の先頭は "/" である必要があります。
	 * </UL>
	 *
	 * @param path パス
	 * @return ファイル。存在しない場合やアクセスできない場合は null。
	 */
	public File getFile(final String path) {
		try {
			final URL entry = getBundle().getEntry(path);
			return new File(FileLocator.resolve(entry).getPath());
		} catch (final IOException e) {
			log(e);
			return null;
		}
	}
}
