/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.io;

import java.io.File;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;

/**
 * Eclipse リソースを保持するように拡張されたファイル・クラス。
 * 
 * @author KASHIHARA Shinji
 */
@SuppressWarnings("serial")
public class ResourceFile extends File {

	/** リソース */
	private final IResource resource;

	/**
	 * コンストラクター。
	 * @param resource リソース
	 */
	public ResourceFile(final IResource resource) {
		super(new File(resource.getLocation().toOSString()), ""); //$NON-NLS-1$
		this.resource = resource;
	}

	/**
	 * 直下のファイルをリソース・ファイルのリストとして取得。
	 * @return リソース・ファイルのリスト。コンストラクタで渡されたリソースが IContainer でない場合は null。
	 */
	@Override
	public ResourceFile[] listFiles() {
		if (resource instanceof IContainer) {
			final IContainer folder = (IContainer) resource;
			IResource[] memmbers;
			try {
				memmbers = folder.members();
			} catch (final CoreException e) {
				throw new IllegalArgumentException(e);
			}
			final ResourceFile[] pFiles = new ResourceFile[memmbers.length];
			for (int i = 0; i < memmbers.length; i++) {
				pFiles[i] = new ResourceFile(memmbers[i]);
			}
			return pFiles;
		}
		return null;
	}

	/**
	 * IResource を取得。
	 * @return IResource
	 */
	public IResource getResource() {
		return resource;
	}

	/**
	 * ワークスペース相対パス文字列を取得。<br>
	 * ディレクトリー区切り文字は常に / で、先頭の / はありません。
	 * @return パス文字列
	 */
	public String getRelativePath() {
		return relativePath(resource);
	}

	/**
	 * このオブジェクトの文字列表現を取得。<br>
	 * このメソッドの戻り値は {@link #getRelativePath} と同じです。
	 * @return getRelativePath() 文字列
	 */
	@Override
	public String toString() {
		return getRelativePath();
	}

	/**
	 * ワークスペース相対パス文字列を取得。<br>
	 * {@link #getRelativePath} の static 版です。
	 * @param resource リソース
	 * @return パス文字列
	 */
	public static String relativePath(final IResource resource) {
		return resource.getFullPath().toOSString().replace("\\", "/").replaceFirst("^/", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}
}
