/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.model;

import java.util.Arrays;

import junit.framework.TestCase;

public class SentencePartitionListTest extends TestCase {

	public void testSize() {
		try {
			new SentencePartitionList(null);
			fail();
		} catch (final NullPointerException e) {
			assertTrue(true);
		}
		assertEquals(1, new SentencePartitionList("").size());
		assertEquals(1, new SentencePartitionList("test").size());
		assertEquals(1, new SentencePartitionList(" test ").size());

		assertEquals(1, new SentencePartitionList("test.text").size());
		assertEquals(2, new SentencePartitionList("test. text").size());
		assertEquals(2, new SentencePartitionList(" test. text ").size());
		assertEquals(2, new SentencePartitionList(" test.  text ").size());
		assertEquals(2, new SentencePartitionList(" test.\rtext ").size());
		assertEquals(2, new SentencePartitionList(" test.\r\ntext ").size());
		assertEquals(2, new SentencePartitionList(" test.\r\n\ttext ").size());
		assertEquals(3, new SentencePartitionList(" test text. test text. test text.").size());

		assertEquals(2, new SentencePartitionList("test.&nbsp;text").size());
		assertEquals(2, new SentencePartitionList("test. &nbsp;text").size());
		assertEquals(2, new SentencePartitionList("test. &nbsp; text").size());

		assertEquals(1, new SentencePartitionList("test?text").size());
		assertEquals(2, new SentencePartitionList("test? text").size());
		assertEquals(2, new SentencePartitionList("test?  text").size());

		assertEquals(1, new SentencePartitionList("test!text").size());
		assertEquals(2, new SentencePartitionList("test! text").size());
		assertEquals(2, new SentencePartitionList("test!  text").size());

		assertEquals(1, new SentencePartitionList("test:text").size());
		assertEquals(2, new SentencePartitionList("test: text").size());
		assertEquals(2, new SentencePartitionList(" test: text ").size());
		assertEquals(2, new SentencePartitionList(" test:  text ").size());
		assertEquals(2, new SentencePartitionList(" test:\rtext ").size());
		assertEquals(2, new SentencePartitionList(" test:\r\ntext ").size());
		assertEquals(2, new SentencePartitionList(" test:\r\n\ttext ").size());
		assertEquals(3, new SentencePartitionList(" test text: test text: test text:").size());

		assertEquals(2, new SentencePartitionList("test. text.").size());
		assertEquals(2, new SentencePartitionList("<test>. text.").size());
		assertEquals(2, new SentencePartitionList("<test. text.>").size());
		assertEquals(2, new SentencePartitionList("(test). text.").size());
		assertEquals(2, new SentencePartitionList("(test. text.)").size());

		assertEquals(1, new SentencePartitionList("<a>test</a> text.").size());
		assertEquals(2, new SentencePartitionList("<a>test. text.</a>").size());
		assertEquals(2, new SentencePartitionList("<a>test</a>: abc").size());
		assertEquals(1, new SentencePartitionList("<a href=\"a\">test</a> text.").size());
		assertEquals(2, new SentencePartitionList("<a href=\"a\">test. text.</a>").size());
		assertEquals(2, new SentencePartitionList("<a href=\"a\">test</a>: abc").size());

		assertEquals(1, new SentencePartitionList("<i>test.</i> text.").size());
		assertEquals(1, new SentencePartitionList("<font size=\"1\">test.</font> text.").size());
	}

	public void testSizeJa() throws Exception {
		assertEquals(1, new SentencePartitionList("あ").size());
		assertEquals(1, new SentencePartitionList("。あ").size());
		assertEquals(1, new SentencePartitionList("あい。").size());
		assertEquals(2, new SentencePartitionList("あいう。えおか。").size());
		assertEquals(2, new SentencePartitionList("。あいう。えおか。").size());
		assertEquals(3, new SentencePartitionList("あ。い。う。").size());
		assertEquals(2, new SentencePartitionList("This is a pen. あれはノートですか？").size());
		assertEquals(1, new SentencePartitionList("あ。。。").size());
	}

	public void testJoin() {

		// 末尾が句読点でない
		assertEquals("aaa bbbccc.", join("aaa bbb", "ccc."));
		assertEquals("aaa bbb-ccc.", join("aaa bbb-", "ccc."));
		assertEquals("あああ。いいい。", join("あああ。", "いいい。"));

		// 末尾が句読点
		assertEquals("aaa bbb. ccc.", join("aaa bbb.", "ccc."));
		assertEquals("aaa bbb! ccc.", join("aaa bbb!", "ccc."));
		assertEquals("aaa bbb? ccc.", join("aaa bbb?", "ccc."));

		// 末尾が句読点 かつ 次要素先頭が空白や改行
		assertEquals("aaa bbb. ccc.", join("aaa bbb.", " ccc."));
		assertEquals("aaa bbb.\nccc.", join("aaa bbb.", "\nccc."));

		// 末尾が句読点 かつ 次要素先頭が特定文字
		assertEquals("aaa bbb.<a hoge>", join("aaa bbb.", "<a hoge>"));
		assertEquals("aaa bbb.) hoge", join("aaa bbb.", ") hoge"));
		assertEquals("aaa bbb.] hoge", join("aaa bbb.", "] hoge"));
		assertEquals("aaa bbb.} hoge", join("aaa bbb.", "} hoge"));

		// 複合パターン
		assertEquals("aaa bbb. あああ。ddd?", join("aaa bbb.", "あああ。", "ddd?"));
	}

	private String join(final String... text) {
		final SentencePartitionList list = new SentencePartitionList();
		list.addAll(Arrays.asList(text));
		return list.join();
	}
}
