/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.core;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import benten.core.BentenConstants;
import benten.core.io.Files;
import benten.twa.filter.core.valueobject.BentenExportHtmlProcessInput;
import benten.twa.filter.engine.ecma376.BentenTwaFilterEcma376DocxEngine;
import benten.twa.filter.engine.ecma376.BentenTwaFilterEcma376PptxEngine;
import benten.twa.filter.engine.ecma376.BentenTwaFilterEcma376XlsxEngine;
import benten.twa.filter.engine.html.BentenTwaFilterHtmlEngine;
import benten.twa.filter.engine.odf.BentenTwaFilterOdfEngine;
import benten.twa.filter.engine.po.BentenTwaFilterPoEngine;
import benten.twa.filter.engine.properties.BentenTwaFilterJavaPropertyResourceBundleEngine;
import benten.twa.filter.engine.text.BentenTwaFilterTextEngine;
import benten.twa.filter.messages.BentenExportHtmlMessages;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.xliff.BlancoXliffParser;
import blanco.xliff.valueobject.BlancoXliff;

/**
 * 翻訳成果のエクスポート
 *
 * <pre>
 * 翻訳成果物をプロジェクトからエクスポートします。
 *   1.  XLIFF とインポートされた翻訳対象物をもちいて、翻訳成果物をエクスポートします。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳対象物－翻訳中間形式変換機能: HTMLエクスポート機能」に対応します。
 *
 * @author KASHIHARA Shinji
 * @author IGA Tosiki
 */
public class BentenExportHtmlProcessImpl extends AbstractTraverseDir implements BentenExportHtmlProcess {
	/**
	 * HTMLエクスポート機能のためのメッセージ。
	 */
	protected static final BentenExportHtmlMessages fMsg = new BentenExportHtmlMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenExportHtmlProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * 処理クラスのリスト。
	 */
	protected List<BentenTwaFilterEngine> fTwaEngineList = new ArrayList<BentenTwaFilterEngine>();

	/**
	 * コンストラクタ。
	 */
	public BentenExportHtmlProcessImpl() {
		fTwaEngineList.add(new BentenTwaFilterHtmlEngine());
		fTwaEngineList.add(new BentenTwaFilterTextEngine());
		fTwaEngineList.add(new BentenTwaFilterJavaPropertyResourceBundleEngine());
		fTwaEngineList.add(new BentenTwaFilterEcma376DocxEngine());
		fTwaEngineList.add(new BentenTwaFilterEcma376XlsxEngine());
		fTwaEngineList.add(new BentenTwaFilterEcma376PptxEngine());
		fTwaEngineList.add(new BentenTwaFilterPoEngine());
		fTwaEngineList.add(new BentenTwaFilterOdfEngine());
	}

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenExportHtmlProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenExportHtmlProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenExportHtmlProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirSourcexliff = new File(fInput.getSourcexliffdir());
		final File dirSourcehtml = new File(fInput.getSourcehtmldir());
		if (dirSourcexliff.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getSourcexliffdir()));
		}
		if (dirSourcexliff.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getSourcexliffdir()));
		}
		if (dirSourcehtml.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE006(fInput.getSourcehtmldir()));
		}
		if (dirSourcehtml.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE007(fInput.getSourcehtmldir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenExportHtmlProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) throws IOException {
				// 処理進捗が 2 度呼ばれます。
				fCounter += 2;
			}

			@Override
			public void processFilterdFile(final File file, final String baseDir) throws IOException {
				fCounter++;
			}

			/**
			 * カウンタ数の取得。
			 * @return カウンタ数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(input);
		inner.processDir(dirSourcehtml);

		beginTask(inner.getCounter());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(dirSourcehtml);

		if (progress(fMsg.getCoreP004(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		// 処理できるファイルかどうかを判定。
		for (BentenTwaFilterEngine processor : fTwaEngineList) {
			if (processor.canProcess(file)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenExportHtmlProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		final String relativePath = Files.relativePath(new File(fInput.getSourcehtmldir()), file);
		final File toDir = new File(fInput.getTargetdir(), relativePath).getParentFile();
		final File xliffDir = new File(fInput.getSourcexliffdir(), relativePath).getParentFile();

		writeHtmlFromXliff(file, xliffDir, toDir);
	}

	@Override
	public void processFilterdFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenExportHtmlProcessImpl#processFilterdFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP013(file.getName()))) {
			return;
		}

		final String relativePath = Files.relativePath(new File(fInput.getSourcehtmldir()), file);
		final File outputFile = new File(fInput.getTargetdir(), relativePath);
		Files.copyFile(file, outputFile);

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}

	/**
	 * HTML と XLIFF を入力して、HTML を出力します。
	 * @param sourceFile HTML ファイル
	 * @param xliffDir XLIFF ディレクトリー
	 * @param toDir 出力ディレクトリー
	 * @throws IOException 入出力例外が発生した場合
	 */
	protected void writeHtmlFromXliff(final File sourceFile, final File xliffDir, final File toDir) throws IOException {
		final String xliffFileName = sourceFile.getName() + BentenConstants.FILE_EXT_XLIFF;
		final File inXliffFile = new File(xliffDir, xliffFileName);
		final File outHtmlFile = (new File(toDir, sourceFile.getName()));

		if (progress(fMsg.getCoreP012(sourceFile.getName()))) {
			return;
		}

		if (inXliffFile.exists() == false || inXliffFile.isFile() == false) {
			// XLIFF ファイルがないので、しかたなく、HTML ファイルをそのままコピーします。
			Files.copyFile(sourceFile, outHtmlFile);
			getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
			return;
		}

		byte[] bytes = null;

		final BlancoXliff xliff = new BlancoXliffParser().parse(inXliffFile);
		if (xliff.getFileList().size() == 0) {
			// XLIFF の内容が取得できないケースは、原則としてありえないものと考えます。
			throw new IllegalArgumentException("An unexpected situation has occurred. XLIFF format is invalid."); //$NON-NLS-1$
		}

		for (BentenTwaFilterEngine processor : fTwaEngineList) {
			if (processor.canProcess(sourceFile)) {
				bytes = processor.mergeXliff(sourceFile, xliff, fResultInfo);
				break;
			}
		}

		// 該当ディレクトリーが無い場合には、ディレクトリー作成。
		// すでに存在する場合は、それはそれで良い。
		if (toDir.exists() == false) {
			if (toDir.mkdirs() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE008(toDir.getAbsolutePath()));
			}
		} else {
			if (toDir.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE009(fInput.getTargetdir()));
			}
		}

		Files.writeByteArrayToFile(outHtmlFile, bytes);

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}
}
