/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.tmx.dialogs;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import benten.core.model.BentenXliff;
import benten.twa.BentenTwaConstants;
import benten.twa.tmx.messages.BentenExtractTmxFragmentMessages;
import benten.twa.tmx.status.BentenExtractTmxFragmentStatus;
import benten.twa.ui.dialogs.AbstractTmWorkflowDialog;
import benten.ui.UiStatus;
import benten.ui.UiStatusException;
import benten.ui.fields.ResourceFolderTextField;
import benten.ui.preference.BentenProjectProperty;
import benten.ui.preference.BentenProjectProperty.ProjectProperty;
import benten.ui.views.UiToolkit;

/**
 * TMX フラグメントの抽出
 *
 * <pre>
 * XLIFF フラグメントから、TMX フラグメントを抽出します。
 *   1.  XLIFF から翻訳単位を抽出して TMX を作成します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳メモリー断片抽出・マージ機能: TMXフラグメント抽出機能」に対応します。
 *
 * @author IGA Tosiki
 */
public class ExtractTmxFragmentDialog extends AbstractTmWorkflowDialog {

	/**
	 * TMXフラグメント抽出機能のためのメッセージ。
	 */
	protected static final BentenExtractTmxFragmentMessages fMsg = new BentenExtractTmxFragmentMessages();

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	protected final UiToolkit toolkit = new UiToolkit();

	/** コントロール・コンテナー */
	private final ExtractTmxFragmentControls controls = new ExtractTmxFragmentControls();

	/** コントロール・コンテナー・クラス */
	public static class ExtractTmxFragmentControls {

		/** ターゲット・ディレクトリー・テキスト */
		public ResourceFolderTextField targetDirText;

		/** 出力ディレクトリー・テキスト */
		public ResourceFolderTextField toDirText;

		/** state チェックボックス */
		public Button stateCheck;

		/** state コンボ */
		public Combo stateCombo;

		/** 日付包含チェックボックス */
		public Button includeDateCheck;
	}

	/**
	 * コンストラクター。
	 * @param parentShell 親シェル
	 * @param selection 選択
	 */
	public ExtractTmxFragmentDialog(final Shell parentShell, final IStructuredSelection selection) {
		super(parentShell, selection, fMsg.getWindowTitle());
	}

	@Override
	public void createControls(final Composite composite) {
		composite.setLayout(new GridLayout(3, false));

		controls.targetDirText = new ResourceFolderTextField(composite, fMsg.getDialogCtrlLabelFromXliffDir());
		controls.targetDirText.addModifyListener(this);
		if (getSelectionResource() != null) {
			controls.targetDirText.setText(getSelectionResource().getProject().getName() + "/" //$NON-NLS-1$
					+ BentenTwaConstants.DEFAULT_XLIFF_FRAGMENT_DIR);
		}

		controls.toDirText = new ResourceFolderTextField(composite, fMsg.getDialogCtrlLabelToTmxDir());
		controls.toDirText.addModifyListener(this);
		if (getSelectionResource() != null) {
			controls.toDirText.setText(getSelectionResource().getProject().getName() + "/" //$NON-NLS-1$
					+ BentenTwaConstants.DEFAULT_TMX_FRAGMENT_DIR);
		}

		final Group extractMethodGroup = new Group(composite, SWT.NULL);
		extractMethodGroup.setText(fMsg.getDialogCtrlGroupExtractMethod());
		final GridData extractMethodGrid = new GridData(GridData.FILL_HORIZONTAL);
		extractMethodGrid.horizontalSpan = 2;
		extractMethodGroup.setLayoutData(extractMethodGrid);
		extractMethodGroup.setLayout(new GridLayout(2, false));

		controls.stateCheck = new Button(extractMethodGroup, SWT.CHECK);
		controls.stateCheck.setText(fMsg.getDialogCtrlButtonStateCheck());
		controls.stateCheck.setSelection(true);
		controls.stateCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean checked = ((Button) e.widget).getSelection();
				controls.stateCombo.setEnabled(checked);
			}
		});

		controls.stateCombo = new Combo(extractMethodGroup, SWT.READ_ONLY);
		controls.stateCombo.setItems(BentenXliff.getStateItems());
		controls.stateCombo.setLayoutData(new GridData());

		final Group outputFileGroup = new Group(composite, SWT.NULL);
		outputFileGroup.setText(fMsg.getDialogCtrlGroupFilename());
		final GridData outputFileNameOptionGrid = new GridData(GridData.FILL_HORIZONTAL);
		outputFileNameOptionGrid.horizontalSpan = 2;
		outputFileGroup.setLayoutData(outputFileNameOptionGrid);
		outputFileGroup.setLayout(new GridLayout(1, false));

		controls.includeDateCheck = new Button(outputFileGroup, SWT.CHECK);
		controls.includeDateCheck.setText(fMsg.getDialogCtrlButtonIncludeDateCheck());
		controls.includeDateCheck.setSelection(true);
		new Label(composite, SWT.NONE); // ダミー。

		createTmPreferenceArea(composite);
	}

	@Override
	public IStatus getResultStatus() {
		final BentenExtractTmxFragmentStatus status = new BentenExtractTmxFragmentStatus();
		status.setTargetDir(controls.targetDirText.getFile());
		status.setToDir(controls.toDirText.getFile());
		if (controls.stateCheck.getSelection()) {
			status.setState(controls.stateCombo.getText());
		}
		status.setIncludeDate(controls.includeDateCheck.getSelection());
		status.setProject(controls.toDirText.getResource());

		final IPreferenceStore projectStore = BentenProjectProperty.getStore(status.getProject());
		status.setTransTargetId(projectStore.getString(ProjectProperty.TRANS_TARGET_ID.name()));
		status.setTransSourceLang(projectStore.getString(ProjectProperty.TRANS_SOURCE_LANG.name()));

		if (status.getTransTargetId() == null || status.getTransTargetId().trim().length() == 0) {
			// TRANS_TARGET_ID が空白の場合には「noname」をセットします。
			status.setTransTargetId("noname"); //$NON-NLS-1$
		}

		return status;
	}

	@Override
	public IStatus validate() throws UiStatusException {
		validatePath(controls.targetDirText);
		validatePath(controls.toDirText);
		return UiStatus.OK_STATUS;
	}
}
