/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.xliff.core;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;

import benten.core.BentenConstants;
import benten.core.model.BentenXliff;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import benten.twa.xliff.core.valueobject.BentenMergeXliffProcessInput;
import benten.twa.xliff.messages.BentenMergeXliffMessages;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * XLIFF フラグメントのマージ
 *
 * <pre>
 * XLIFF フラグメントを、もとになった XLIFF にマージします。
 *   1.  翻訳単位の id をもとに、XLIFF をマージします。
 * </pre>
 * 
 * <UL>
 * <LI>マージは trans-unit の id をもとに、同じものに対する上書きをおこないます。
 * <LI>同じ id が見つかった場合には、単に trans-unit の上書きをおこないます。state 同士の優先順位などは判断していません。
 * </UL>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳中間形式断片抽出・マージ機能: XLIFFフラグメント・マージ機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class BentenMergeXliffProcessImpl extends AbstractTraverseDir implements BentenMergeXliffProcess {
	/**
	 * XLIFF マージ機能のためのメッセージ。
	 */
	protected static final BentenMergeXliffMessages fMsg = new BentenMergeXliffMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenMergeXliffProcessInput fInput;

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenMergeXliffProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenMergeXliffProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenMergeXliffProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File fileSourcedir = new File(input.getSourcedir());
		if (fileSourcedir.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE021(fInput.getSourcedir()));
		}
		if (fileSourcedir.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE022(fInput.getSourcedir()));
		}

		final File fileTargetdir = new File(input.getTargetdir());
		if (fileTargetdir.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE023(fInput.getTargetdir()));
		}
		if (fileTargetdir.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE024(fInput.getTargetdir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenMergeXliffProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) throws IOException {
				final FileCounter2 inner2 = new FileCounter2();
				inner2.execute(new File(fInput.getTargetdir()));
				fCounter += inner2.getCounter();
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
				// こちらはカウントしません。
			}

			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(input);
		inner.processDir(new File(input.getSourcedir()));
		beginTask(inner.getCounter());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(new File(input.getSourcedir()));

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	/**
	 * 処理の内部処理でアイテムが処理されるたびに進捗報告としてコールバック。
	 *
	 * @param argProgressMessage 現在処理しているアイテムに関するメッセージ。
	 * @return 処理をそのまま継続する場合は false。処理中断をリクエストしたい場合は true。
	 */
	public boolean progressInner(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	public void processFile(final File fromXliffFile, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenMergeXliffProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(fromXliffFile.getName()))) {
			return;
		}

		final BentenXliff inXliff = BentenXliff.loadInstance(fromXliffFile);

		class MyTraverseDir extends AbstractTraverseDir {
			BentenXliff fXliff = null;

			boolean fIsSaved = false;

			public void execute(final BentenXliff toXliff, final File dirTo) throws IOException {
				fXliff = toXliff;
				processDir(dirTo);
			}

			@Override
			protected boolean canProcess(final File file) {
				return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
			}

			/**
			 * 保管がおこなわれたかどうか。
			 * @return 保管が行われていれば true。
			 */
			public boolean isSaved() {
				return fIsSaved;
			}

			@Override
			protected void processFile(final File toXliffFile, final String baseDir) throws IOException {
				final BentenXliff toXliff = BentenXliff.loadInstance(toXliffFile);
				int transUnitCount = 0;
				final int transUnitTotalCount = fXliff.getAllTransUnitList().size();
				for (final BlancoXliffTransUnit inUnit : fXliff.getAllTransUnitList()) {
					if (progressInner(fMsg.getCoreP101(toXliffFile.getName(), BigDecimal.valueOf(++transUnitCount),
							BigDecimal.valueOf(transUnitTotalCount)))) {
						break;
					}

					final BlancoXliffTransUnit toUnit = toXliff.getTransUnit(inUnit.getId());

					if (toUnit == null) {
						// 該当する id は見つかりませんでした。
						continue;
					}

					if (inUnit.getSource() == null) {
						// source が null だと処理できません。スキップします。
						continue;
					}
					if (inUnit.getSource().equals(toUnit.getSource()) == false) {
						// id は一致するのに、source が一致しない場合、上書きせずにスキップします。
						continue;
					}

					// id が一致して、source も一致しました。

					fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);

					// 単に trans-unit の内容を上書きします。
					// state 同士の優先順位などは判断していません。
					// また、上書きの結果、内容が変わったかどうかチェックしていないので、単位数には、見つかった trans-unit の数だけ報告が上がります。

					toUnit.setTranslate(inUnit.getTranslate());
					toUnit.setContextGroupList(inUnit.getContextGroupList());
					toUnit.setTarget(inUnit.getTarget());
					toUnit.setNoteList(inUnit.getNoteList());
				}

				try {
					if (toXliff.save()) {
						// XLIFF に実際に保管が実行されました。
						fIsSaved = true;
					}
				} catch (final IOException e) {
					throw new IllegalArgumentException(fMsg.getCoreE013(), e);
				}

				if (fIsSaved) {
					if (progress(fMsg.getCoreP012())) {
						return;
					}
				} else {
					if (progress(fMsg.getCoreP013())) {
						return;
					}
				}
			}
		}
		final MyTraverseDir traverseDir = new MyTraverseDir();
		traverseDir.execute(inXliff, new File(fInput.getTargetdir()));
		if (traverseDir.isSaved()) {
			getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
		} else {
			getResultInfo().setSkipCount(getResultInfo().getSkipCount() + 1);
		}
	}
}

/**
 * ファイル数を数えるためだけのためのクラス。
 */
class FileCounter2 extends AbstractTraverseDir {
	private int fCounter = 0;

	/**
	 * 走査の実行。
	 * @param target ターゲット
	 * @throws IOException 入出力例外が発生した場合
	 */
	public void execute(final File target) throws IOException {
		processDir(target);
	}

	/**
	 * カウンターの取得。
	 * @return カウンター
	 */
	public int getCounter() {
		return fCounter;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	protected void processFile(final File outputFile, final String baseDir) throws IOException {
		fCounter++;
	}
}
