/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.xliff.dialogs;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import benten.core.BentenConstants;
import benten.core.model.BentenXliff;
import benten.twa.BentenTwaConstants;
import benten.twa.ui.dialogs.AbstractWorkflowDialog;
import benten.twa.xliff.messages.BentenExtractXliffFragmentMessages;
import benten.twa.xliff.status.BentenExtractXliffFragmentStatus;
import benten.ui.UiStatus;
import benten.ui.UiStatusException;
import benten.ui.fields.ResourceFileTextField;
import benten.ui.fields.ResourceFolderTextField;

/**
 * XLIFF フラグメント抽出ダイアログです。
 *
 * XLIFF フラグメントの抽出
 *
 * <pre>
 * XLIFF から XLIFF フラグメントを抽出します。
 *   1.  XLIFF から所定の条件に合う翻訳単位を抽出して XLIFF フラグメントを作成します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳中間形式断片抽出・マージ機能: XLIFFフラグメント抽出機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class ExtractXliffFragmentDialog extends AbstractWorkflowDialog {

	/**
	 * XLIFF フラグメント抽出機能のためのメッセージ。
	 */
	protected static final BentenExtractXliffFragmentMessages fMsg = new BentenExtractXliffFragmentMessages();

	/** コントロール・コンテナー */
	private final ExtractXliffFragmentControls controls = new ExtractXliffFragmentControls();

	/** コントロール・コンテナー・クラス */
	public static class ExtractXliffFragmentControls {

		/** 入力 XLIFF ディレクトリー・テキスト */
		public ResourceFileTextField fromXliffFileText;

		/** 出力ディレクトリー・テキスト */
		public ResourceFolderTextField toDirText;

		/** state チェックボックス */
		public Button stateCheck;

		/** state コンボ */
		public Combo stateCombo;

		/** カウント・チェックボックス */
		public Button countCheck;

		/** カウント・テキスト */
		public Text countText;

		/** 日付包含チェックボックス */
		public Button includeDateCheck;
	}

	/**
	 * コンストラクター。
	 * @param parentShell 親シェル
	 * @param selection 選択
	 */
	public ExtractXliffFragmentDialog(final Shell parentShell, final IStructuredSelection selection) {
		super(parentShell, selection, fMsg.getWindowTitle());
	}

	@Override
	public void createControls(final Composite composite) {
		composite.setLayout(new GridLayout(3, false));

		controls.fromXliffFileText = new ResourceFileTextField(composite, fMsg.getDialogCtrlLabelFromFile(),
				BentenConstants.FILE_EXT_XLIFF);
		controls.fromXliffFileText.addModifyListener(this);
		if (getSelectionResource() != null) {
			controls.fromXliffFileText.setText(getSelectionResource().getFullPath().toString());
		}

		controls.toDirText = new ResourceFolderTextField(composite, fMsg.getDialogCtrlLabelToDir());
		controls.toDirText.addModifyListener(this);
		if (getSelectionResource() != null) {
			controls.toDirText.setText(getSelectionResource().getProject().getName() + "/" //$NON-NLS-1$
					+ BentenTwaConstants.DEFAULT_XLIFF_FRAGMENT_DIR);
		}

		final Group group = new Group(composite, SWT.NULL);
		group.setText(fMsg.getDialogCtrlGroupExtractMethod());
		final GridData howToExtractGrid = new GridData(GridData.FILL_HORIZONTAL);
		howToExtractGrid.horizontalSpan = 2;
		group.setLayoutData(howToExtractGrid);
		group.setLayout(new GridLayout(2, false));

		controls.stateCheck = new Button(group, SWT.CHECK);
		controls.stateCheck.setText(fMsg.getDialogCtrlButtonStateCheck());
		controls.stateCheck.setSelection(false);
		controls.stateCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean checked = ((Button) e.widget).getSelection();
				controls.stateCombo.setEnabled(checked);
			}
		});

		controls.stateCombo = new Combo(group, SWT.READ_ONLY);
		controls.stateCombo.setItems(BentenXliff.getStateItems());
		controls.stateCombo.setLayoutData(new GridData());

		controls.countCheck = new Button(group, SWT.CHECK);
		controls.countCheck.setText(fMsg.getDialogCtrlButtonCountCheck());
		controls.countCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean checked = ((Button) e.widget).getSelection();
				controls.countText.setEnabled(checked);
			}
		});

		controls.countText = new Text(group, SWT.BORDER);
		controls.countText.setLayoutData(new GridData());
		controls.countText.setEnabled(false);
		controls.countText.addModifyListener(this);

		final Label spacer = new Label(composite, SWT.NULL);
		spacer.setText(""); //$NON-NLS-1$

		final Group group1 = new Group(composite, SWT.NULL);
		group1.setText(fMsg.getDialogCtrlGroupFilename());
		final GridData outputFileNameOptionGrid = new GridData(GridData.FILL_HORIZONTAL);
		outputFileNameOptionGrid.horizontalSpan = 2;
		group1.setLayoutData(outputFileNameOptionGrid);
		group1.setLayout(new GridLayout(1, false));

		controls.includeDateCheck = new Button(group1, SWT.CHECK);
		controls.includeDateCheck.setText(fMsg.getDialogCtrlButtonIncludeDateCheck());
		controls.includeDateCheck.setSelection(true);
	}

	@Override
	public IStatus getResultStatus() {
		final BentenExtractXliffFragmentStatus status = new BentenExtractXliffFragmentStatus();
		status.setFromXliffFile(controls.fromXliffFileText.getFile());
		status.setToDir(controls.toDirText.getFile());
		if (controls.stateCheck.getSelection()) {
			status.setState(controls.stateCombo.getText());
		}
		if (controls.countCheck.getSelection()) {
			status.setSizeLimit(Integer.parseInt(controls.countText.getText()));
		}
		status.setIncludeDate(controls.includeDateCheck.getSelection());
		status.setProject(controls.toDirText.getResource());
		return status;
	}

	@Override
	public IStatus validate() throws UiStatusException {
		validatePath(controls.fromXliffFileText);
		validatePath(controls.toDirText);
		validateFromTo(controls.fromXliffFileText, controls.toDirText);

		if (controls.countCheck != null && controls.countCheck.getSelection()) {
			try {
				final int count = Integer.parseInt(controls.countText.getText());
				if (count < 1) {
					throw new UiStatusException(fMsg.getDialogMsg001());
				}
			} catch (final RuntimeException e) {
				throw new UiStatusException(fMsg.getDialogMsg002());
			}
		}
		return UiStatus.OK_STATUS;
	}
}
