/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.ui;

import java.io.File;
import java.io.IOException;
import java.net.URL;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import benten.core.CorePlugin;
import benten.ui.messages.AbstractBentenUiPluginMessages;

/**
 * Benten UI プラグインの基底クラス。
 *
 * <UL>
 * <LI>Benten で使用する UI 系プラグインの基底クラスです。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public abstract class AbstractBentenUiPlugin extends AbstractUIPlugin {

	/**
	 * Benten UI プラグインの基底クラスのためのメッセージ。
	 */
	protected static final AbstractBentenUiPluginMessages fMsg = new AbstractBentenUiPluginMessages();

	/**
	 * プラグイン id の取得。
	 *
	 * <UL>
	 * <LI>Benten プロジェクトではプラグイン id はパッケージ名と同一です。
	 * </UL>
	 *
	 * @return プラグイン id
	 */
	public final String getPluginId() {
		return getClass().getPackage().getName();
	}

	/**
	 * 指定されたメッセージの Eclipse ログ出力。
	 *
	 * @param message ログ出力するメッセージ
	 */
	public void log(final String message) {
		CorePlugin.getDefault().printLog(getPluginId(), message);
	}

	/**
	 * 指定された例外の Eclipse ログ出力。
	 *
	 * @param e ログ出力する例外
	 */
	public void log(final Throwable e) {
		CorePlugin.getDefault().printLog(getPluginId(), e);
	}

	/**
	 * エラー・ダイアログを表示。
	 * @param e 例外
	 */
	public void showErrorDialog(final Throwable e) {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				final UiMultiStatus s = new UiMultiStatus(IStatus.ERROR, e.getMessage());
				s.add(new UiStatus(IStatus.ERROR, e.toString()));
				for (final StackTraceElement ste : e.getStackTrace()) {
					s.add(new UiStatus(IStatus.ERROR, ste.toString()));
				}
				UiPlugin.openDialog(fMsg.getMsg001(), s);
			}
		});
	}

	/**
	 * このプラグイン配下のファイルを取得。
	 *
	 * <UL>
	 * <LI>指定する path の先頭は "/" である必要があります。
	 * </UL>
	 *
	 * @param path パス
	 * @return ファイル。存在しない場合やアクセスできない場合は null。
	 */
	public File getFile(final String path) {
		try {
			final URL entry = getBundle().getEntry(path);
			return new File(FileLocator.resolve(entry).getPath());
		} catch (final IOException e) {
			log(e);
			return null;
		}
	}

	/**
	 * イメージの取得。
	 *
	 * <UL>
	 * <LI>一度取得したイメージは Eclipse のイメージ・レジストリーで管理され、適切に破棄されます。
	 * </UL>
	 *
	 * @param path パス
	 * @return イメージ。存在しない場合は null。
	 */
	public Image getImage(final String path) {
		Image image = getImageRegistry().get(path);
		if (image == null) {
			final ImageDescriptor descriptor = imageDescriptorFromPlugin(getPluginId(), path);
			if (descriptor != null) {
				getImageRegistry().put(path, image = descriptor.createImage());
			}
		}
		return image;
	}
}
