#include "StdAfx.h"
#include "BITSDef.h"
#include "BITSUtil.h"
#include "MiscUtil.h"


__checkReturn bool BITSUtil::VerifyJobDisplayName(__in LPCWSTR lpszJobDisplayName)
{
	if (NULL == lpszJobDisplayName) {
		ATLASSERT(false);
		return false;
	}

	CString strJobDisplayName = lpszJobDisplayName;
	if (BITS_MAX_DISPLAY_NAME_LEN < strJobDisplayName.GetLength()) {
		return false;
	}

	return true;
}


bool BITSUtil::VerifyJobType(__in BG_JOB_TYPE jobType)
{
	if (jobType < BG_JOB_TYPE_DOWNLOAD || BG_JOB_TYPE_UPLOAD_REPLY < jobType) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyJobPriority(__in BG_JOB_PRIORITY jobPriority)
{
	if (jobPriority < BG_JOB_PRIORITY_FOREGROUND || BG_JOB_PRIORITY_LOW < jobPriority) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyJobDescription(__in LPCWSTR lpszJobDescription)
{
	if (NULL == lpszJobDescription) {
		ATLASSERT(false);
		return false;
	}

	CString strJobDescription = lpszJobDescription;
	if (BITS_MAX_DESCRIPTION_LEN < strJobDescription.GetLength()) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyRemoteFileName(__in LPCWSTR lpszFileName)
{
	if (NULL == lpszFileName) {
		ATLASSERT(false);
		return false;
	}

	CUrl url;
	BOOL bResult = url.CrackUrl(lpszFileName);
	if (FALSE == bResult) {
		return false;
	}

	ATL_URL_SCHEME scheme = url.GetScheme();
	if (ATL_URL_SCHEME_HTTP != scheme && ATL_URL_SCHEME_HTTPS != scheme && ATL_URL_SCHEME_FILE != scheme) {
		return false;
	}
	
	return true;
}


__checkReturn bool BITSUtil::VerifyLocalFileName(__in LPCWSTR lpszFileName)
{
	if (NULL == lpszFileName) {
		ATLASSERT(false);
		return false;
	}

	if (FALSE == ::PathIsUNCServerShare(lpszFileName)) {
		if (TRUE == ::PathIsRelative(lpszFileName)) {
			return false;
		}
		if (TRUE == ::PathIsRoot(lpszFileName)) {
			return false;
		}
		if (TRUE == ::PathIsDirectory(lpszFileName)) {
			return false;
		}
	}

	if (false == IsValidFileCharacter(lpszFileName)) {
		return false;
	}

	return IsWritableDrive(lpszFileName);
}


__checkReturn bool BITSUtil::VerifyJobProxyUsage(__in BG_JOB_PROXY_USAGE jobProxyUsage)
{
	if (jobProxyUsage < BG_JOB_PROXY_USAGE_PRECONFIG || BG_JOB_PROXY_USAGE_AUTODETECT < jobProxyUsage) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyAuthTarget(__in BG_AUTH_TARGET authTarget)
{
	if (authTarget < BG_AUTH_TARGET_SERVER || BG_AUTH_TARGET_PROXY < authTarget) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyAuthScheme(__in BG_AUTH_SCHEME authScheme)
{
	if (authScheme < BG_AUTH_SCHEME_BASIC || BG_AUTH_SCHEME_PASSPORT < authScheme) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyHttpRedirectPolicy(__in DWORD dwPolicy)
{
	if (BG_HTTP_REDIRECT_POLICY_ALLOW_SILENT != dwPolicy && BG_HTTP_REDIRECT_POLICY_ALLOW_REPORT != dwPolicy &&
		BG_HTTP_REDIRECT_POLICY_DISALLOW != dwPolicy) {
			return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyCertStoreLocation(__in BG_CERT_STORE_LOCATION certStoreLocation)
{
	if (certStoreLocation < BG_CERT_STORE_LOCATION_CURRENT_USER ||
		BG_CERT_STORE_LOCATION_LOCAL_MACHINE_ENTERPRISE < certStoreLocation) {
			return false;
	}

	return true;
}


__checkReturn bool BITSUtil::VerifyJobHelperTokenFlags(__in DWORD dwFlags)
{
	if (0 == dwFlags || (BG_TOKEN_LOCAL_FILE & dwFlags) || (BG_TOKEN_NETWORK & dwFlags)) {
		return true;
	}

	return false;
}


__checkReturn bool BITSUtil::IsValidFileCharacter(__in LPCWSTR lpszFileName)
{
	if (NULL == lpszFileName) {
		ATLASSERT(false);
		return false;
	}

	LPCWSTR lpszSkipedRootPath = ::PathSkipRoot(lpszFileName);
	CString strSkipedRootPath = lpszSkipedRootPath;
	if (-1 != strSkipedRootPath.FindOneOf(L":/*\"?<>")) {
		return false;
	}

	return true;
}


__checkReturn HRESULT BITSUtil::AddJobFile(__in const CComPtr<IBackgroundCopyJob>& spJob,__in LPCWSTR lpszRemoteUrl,
										   __in LPCWSTR lpszLocalName)
{
	if (!spJob || NULL == lpszRemoteUrl || NULL == lpszLocalName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->AddFile(lpszRemoteUrl, lpszLocalName);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}

__checkReturn HRESULT BITSUtil::AddJobFileSet(__in const CComPtr<IBackgroundCopyJob>& spJob, __in ULONG cFileCount,
											  __in BG_FILE_INFO* paFileSet)
{
	if (!spJob || NULL == paFileSet) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->AddFileSet(cFileCount, paFileSet);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::AddJobFileWithRanges(__in const CComPtr<IBackgroundCopyJob3>& spJob3,
													 __in LPCWSTR lpszRemoteUrl, __in LPCWSTR lpszLocalName,
													 __in DWORD dwRangeCount, __in BG_FILE_RANGE Range[])
{
	if (!spJob3 || NULL == lpszRemoteUrl || NULL == lpszLocalName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob3->AddFileWithRanges(lpszRemoteUrl, lpszLocalName, dwRangeCount, Range);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn bool BITSUtil::IsWritableDrive(__in LPCWSTR lpszFileName)
{
	if (NULL == lpszFileName) {
		ATLASSERT(false);
		return false;
	}

	int nDriveNumber = ::PathGetDriveNumber(lpszFileName);
	if (-1 == nDriveNumber) {
		ATLASSERT(false);
		return false;
	}

	WCHAR szRootPath[4] = {0};
	::PathBuildRoot(szRootPath, nDriveNumber);
	UINT uDriveType = ::GetDriveType(szRootPath);
	if (DRIVE_UNKNOWN == uDriveType || DRIVE_NO_ROOT_DIR == uDriveType || DRIVE_CDROM == uDriveType) {
		return false;
	}

	return true;
}


__checkReturn HRESULT BITSUtil::GetJobDisplayName(__in const CComPtr<IBackgroundCopyJob>& spJob, __out CString* pstrJobDisplayName)
{
	if (!spJob || NULL == pstrJobDisplayName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	pstrJobDisplayName->Empty();

	CString strJobDisplayName;
	LPWSTR lpszJobDisplayName = NULL;
	HRESULT hr = spJob->GetDisplayName(&lpszJobDisplayName);
	if (SUCCEEDED(hr) && NULL != lpszJobDisplayName) {
		strJobDisplayName = lpszJobDisplayName;
		::CoTaskMemFree(lpszJobDisplayName);
		lpszJobDisplayName = NULL;
	} else {
		ATLASSERT(false);
		return hr;
	}

	*pstrJobDisplayName = strJobDisplayName;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobDescription(__in const CComPtr<IBackgroundCopyJob>& spJob, __out CString* pstrJobDescription)
{
	if (!spJob || NULL == pstrJobDescription) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	pstrJobDescription->Empty();

	CString strJobDescription;
	LPWSTR lpszJobDescription = NULL;
	HRESULT hr = spJob->GetDescription(&lpszJobDescription);
	if (SUCCEEDED(hr) && NULL != lpszJobDescription) {
		strJobDescription = lpszJobDescription;
		::CoTaskMemFree(lpszJobDescription);
		lpszJobDescription = NULL;
	} else {
		ATLASSERT(false);
		return hr;
	}

	*pstrJobDescription = strJobDescription;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobId(__in const CComPtr<IBackgroundCopyJob>& spJob, __out GUID* pguid)
{
	if (!spJob || NULL == pguid) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetId(pguid);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobId(__in const CComPtr<IBackgroundCopyJob>& spJob, __out CString* pstrJobId)
{
	if (!spJob || NULL == pstrJobId) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	GUID guid = GUID_NULL;
	HRESULT hr = GetJobId(spJob, &guid);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}

	OLECHAR szBuff[256] = {0};
	int nResult = ::StringFromGUID2(guid, reinterpret_cast<LPOLESTR>(&szBuff), _countof(szBuff) - 1);
	if (0 == nResult) {
		ATLASSERT(false);
		return E_UNEXPECTED;
	}

	*pstrJobId = szBuff;
	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobType(__in const CComPtr<IBackgroundCopyJob>& spJob, __out BG_JOB_TYPE* pJobType)
{
	if (!spJob || NULL == pJobType) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetType(pJobType);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobPriority(__in const CComPtr<IBackgroundCopyJob>& spJob, __out BG_JOB_PRIORITY* pJobPriority)
{
	if (!spJob || NULL == pJobPriority) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetPriority(pJobPriority);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobState(__in const CComPtr<IBackgroundCopyJob>& spJob, __out BG_JOB_STATE* pJobState)
{
	if (!spJob || NULL == pJobState) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetState(pJobState);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobOwner(__in const CComPtr<IBackgroundCopyJob>& spJob, __out CString* pstrJobOwner)
{
	if (!spJob || NULL == pstrJobOwner) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	CString strJobOwner;
	LPWSTR lpszOwner = NULL;
	HRESULT hr = spJob->GetOwner(&lpszOwner);
	if (SUCCEEDED(hr) && NULL != lpszOwner) {
		strJobOwner = MiscUtil::GetAccountNameFromSid(lpszOwner);
		ATLASSERT(!strJobOwner.IsEmpty());
		::CoTaskMemFree(lpszOwner);
		lpszOwner = NULL;
	} else {
		ATLASSERT(false);
		return hr;
	}

	*pstrJobOwner = strJobOwner;
	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobOwner(__in const CComPtr<IBackgroundCopyJob>& spJob, __out CSid* pJobOwnerSid)
{
	if (!spJob || NULL == pJobOwnerSid) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	CString strJobOwner;
	LPWSTR lpszOwner = NULL;
	HRESULT hr = spJob->GetOwner(&lpszOwner);
	if (SUCCEEDED(hr) && NULL != lpszOwner) {
		bool bResult = MiscUtil::ConvertStringSidToCSid(lpszOwner, pJobOwnerSid);

		::CoTaskMemFree(lpszOwner);
		lpszOwner = NULL;

		if (false == bResult) {
			ATLASSERT(false);
			return E_UNEXPECTED;
		}
	} else {
		ATLASSERT(false);
		return hr;
	}

	return hr;
}

__checkReturn HRESULT BITSUtil::GetJobTimes(__in const CComPtr<IBackgroundCopyJob>& spJob, __in BG_JOB_TIMES* pJobTimes)
{
	if (!spJob || NULL == pJobTimes) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetTimes(pJobTimes);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobProgress(__in const CComPtr<IBackgroundCopyJob>& spJob,
											   __out BG_JOB_PROGRESS* pJobProgress)
{
	if (!spJob || NULL == pJobProgress) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetProgress(pJobProgress);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobErrorCount(__in const CComPtr<IBackgroundCopyJob>& spJob,
												 __out ULONG* puErrorCount)
{
	if (!spJob || NULL == puErrorCount) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetErrorCount(puErrorCount);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobMinimumRetryDelay(__in const CComPtr<IBackgroundCopyJob>& spJob,
														__out ULONG* puRetryDelay)
{
	if (!spJob || NULL == puRetryDelay) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetMinimumRetryDelay(puRetryDelay);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobNoProgressTimeout(__in const CComPtr<IBackgroundCopyJob>& spJob,
														__out ULONG* puRetryPeriod)
{
	if (!spJob || NULL == puRetryPeriod) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetNoProgressTimeout(puRetryPeriod);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobNotifyInterface(__in const CComPtr<IBackgroundCopyJob>& spJob,
													  __out CComPtr<IUnknown>* pspNotifyInterface)
{
	if (!spJob || NULL == pspNotifyInterface) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetNotifyInterface(&(*pspNotifyInterface));
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobNotifyFlags(__in const CComPtr<IBackgroundCopyJob>& spJob,
												  __out ULONG* pNotifyFlags)
{
	if (!spJob || NULL == pNotifyFlags) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetNotifyFlags(pNotifyFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobMinimumRetryDelay(__in const CComPtr<IBackgroundCopyJob>& spJob,
														__out ULONG uRetryDelay)
{
	if (!spJob) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ATLASSERT(MINIMUM_RETRY_DELAY <= uRetryDelay);

	HRESULT hr = spJob->SetMinimumRetryDelay(uRetryDelay);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobNoProgressTimeout(__in const CComPtr<IBackgroundCopyJob>& spJob,
														__out ULONG uRetryPeriod)
{
	if (!spJob) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->SetNoProgressTimeout(uRetryPeriod);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::TakeJobOwnership(__in const CComPtr<IBackgroundCopyJob>& spJob)
{
	if (!spJob) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->TakeOwnership();
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::IsJobOwner(__in const CComPtr<IBackgroundCopyJob>& spJob, __out bool* pbOwner)
{
	if (!spJob || NULL == pbOwner) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	*pbOwner = false;
	CSid jobOwnerSid;
	HRESULT hr = BITSUtil::GetJobOwner(spJob, &jobOwnerSid);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}

	CSid processUserSid;
	bool bResult = MiscUtil::GetCurrentProcessUserSid(&processUserSid);
	if (false == bResult) {
		ATLASSERT(false);
		return E_UNEXPECTED;
	}

	if (jobOwnerSid != processUserSid) {
		*pbOwner = false;
		return S_OK;
	}

	*pbOwner = true;
	return S_OK;
}


HRESULT BITSUtil::GetBITSErrorDescription(__in const CComPtr<IBackgroundCopyManager>& spBITSManager,
										__in HRESULT hResult, __out CString* pstrErrorDescription)
{
	if (!spBITSManager) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	LPWSTR lpszErrorDescription = NULL;
	HRESULT hr = spBITSManager->GetErrorDescription(hResult, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), 
		&lpszErrorDescription);
	if (FAILED(hr) || NULL == lpszErrorDescription) {
		ATLASSERT(false);
		return hr;
	}

	*pstrErrorDescription = lpszErrorDescription;
	::CoTaskMemFree(lpszErrorDescription);
	lpszErrorDescription = NULL;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobProxySettings(__in const CComPtr<IBackgroundCopyJob>& spJob, 
													__out BG_JOB_PROXY_USAGE* pProxyUsage, __out CString* pstrProxyList, 
													__out CString* pstrProxyBypassList)
{
	if (!spJob || NULL == pProxyUsage || NULL == pstrProxyList || NULL == pstrProxyBypassList) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	pstrProxyList->Empty();
	pstrProxyBypassList->Empty();

	LPWSTR lpszProxyList = NULL;
	LPWSTR lpszProxyBypassList = NULL;

	HRESULT hr = spJob->GetProxySettings(pProxyUsage, &lpszProxyList, &lpszProxyBypassList);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}

	if (NULL != lpszProxyList) {
		*pstrProxyList			= lpszProxyList;
		::CoTaskMemFree(lpszProxyList);
		lpszProxyList = NULL;	
	}

	if (NULL != lpszProxyBypassList) {
		*pstrProxyBypassList	= lpszProxyBypassList;
		::CoTaskMemFree(lpszProxyBypassList);
		lpszProxyBypassList = NULL;
	}

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobProxySettings(__in const CComPtr<IBackgroundCopyJob>& spJob, 
													__in BG_JOB_PROXY_USAGE proxyUsage, __in LPCWSTR lpszProxyList,
													__in LPCWSTR lpszProxyBypassList)
{
	if (!spJob || NULL == lpszProxyList || NULL == lpszProxyBypassList) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = E_FAIL;
	if (BG_JOB_PROXY_USAGE_OVERRIDE == proxyUsage) {
		hr = spJob->SetProxySettings(proxyUsage, lpszProxyList, lpszProxyBypassList);
		ATLASSERT(SUCCEEDED(hr));
	} else {
		hr = spJob->SetProxySettings(proxyUsage, NULL, NULL);
		ATLASSERT(SUCCEEDED(hr));
	}
	
	return hr;
}


__checkReturn bool BITSUtil::IsJobEditableState(__in const CComPtr<IBackgroundCopyJob>& spJob)
{
	BG_JOB_STATE jobState = BG_JOB_STATE_ERROR;
	HRESULT hr = BITSUtil::GetJobState(spJob, &jobState);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return false;
	}

	// LԂɂȂƃWuҏWłȂ
	if (BG_JOB_STATE_ACKNOWLEDGED == jobState || BG_JOB_STATE_CANCELLED == jobState) {
		return false;
	}

	return true;
}


__checkReturn HRESULT BITSUtil::GetJobLocalFileName(__in const CComPtr<IBackgroundCopyFile>& spFile,
													__out CString* pstrLocalFilename)
{
	if (!spFile || NULL == pstrLocalFilename) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}
	pstrLocalFilename->Empty();

	LPWSTR lpszLocalFilename = NULL;
	HRESULT hr = spFile->GetLocalName(&lpszLocalFilename);
	if (FAILED(hr)) {
		return hr;
	}
	*pstrLocalFilename = lpszLocalFilename;

	::CoTaskMemFree(lpszLocalFilename);
	lpszLocalFilename = NULL;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobRemotelFileName(__in const CComPtr<IBackgroundCopyFile>& spFile,
													  __out CString* pstrRemoteFilename)
{
	if (!spFile || NULL == pstrRemoteFilename) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}
	pstrRemoteFilename->Empty();

	LPWSTR lpszRemoteFilename = NULL;
	HRESULT hr = spFile->GetRemoteName(&lpszRemoteFilename);
	if (FAILED(hr)) {
		return hr;
	}
	*pstrRemoteFilename = lpszRemoteFilename;

	::CoTaskMemFree(lpszRemoteFilename);
	lpszRemoteFilename = NULL;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobErrorIF(__in const CComPtr<IBackgroundCopyJob>& spJob, __out CComPtr<IBackgroundCopyError>* pspError)
{
	if (!spJob || NULL == pspError || (*pspError)) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob->GetError(&(*pspError));
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobError(__in const CComPtr<IBackgroundCopyError>& spError, __out BG_ERROR_CONTEXT* pErrorContext,
											__out HRESULT* phrError)
{
	if (!spError || NULL == pErrorContext || NULL == phrError) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spError->GetError(pErrorContext, phrError);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobErrorContextDescription(__in const CComPtr<IBackgroundCopyError>& spError,
															  __out CString* pstrErrorContextDescription)
{
	if (!spError || NULL == pstrErrorContextDescription) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}
	pstrErrorContextDescription->Empty();

	LPWSTR lpszErrorContextDescription = NULL;
	HRESULT hr = spError->GetErrorContextDescription(MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), &lpszErrorContextDescription);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}
	
	*pstrErrorContextDescription = lpszErrorContextDescription;

	::CoTaskMemFree(lpszErrorContextDescription);
	lpszErrorContextDescription = NULL;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobErrorDescription(__in const CComPtr<IBackgroundCopyError>& spError,
													   __out CString* pstrErrorDescription)
{
	if (!spError || NULL == pstrErrorDescription) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}
	pstrErrorDescription->Empty();

	LPWSTR lpszErrorDescription = NULL;
	HRESULT hr = spError->GetErrorDescription(MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), &lpszErrorDescription);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}

	*pstrErrorDescription = lpszErrorDescription;

	::CoTaskMemFree(lpszErrorDescription);
	lpszErrorDescription = NULL;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobErrorProtocol(__in const CComPtr<IBackgroundCopyError>& spError,
													__out CString* pstrErrorProtocol)
{
	if (!spError || NULL == pstrErrorProtocol) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}
	pstrErrorProtocol->Empty();

	LPWSTR lpszProtocol = NULL;
	HRESULT hr = spError->GetProtocol(&lpszProtocol);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}

	*pstrErrorProtocol = lpszProtocol;

	::CoTaskMemFree(lpszProtocol);
	lpszProtocol = NULL;

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobErrorFile(__in const CComPtr<IBackgroundCopyError>& spError,
												__out CComPtr<IBackgroundCopyFile>* pspFile)
{
	if (!spError || NULL == pspFile) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spError->GetFile(&(*pspFile));
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn bool BITSUtil::IsBITS15Supported(__in const CComPtr<IBackgroundCopyJob>& spJob)
{
	if (!spJob) {
		ATLASSERT(false);
		return false;
	}

	CComQIPtr<IBackgroundCopyJob2> spJob2 = spJob;
	if (!spJob2) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::IsBITS20Supported(__in const CComPtr<IBackgroundCopyJob>& spJob)
{
	if (!spJob) {
		ATLASSERT(false);
		return false;
	}

	CComQIPtr<IBackgroundCopyJob3> spJob3 = spJob;
	if (!spJob3) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::IsBITS25Supported(__in const CComPtr<IBackgroundCopyJob>& spJob)
{
	if (!spJob) {
		ATLASSERT(false);
		return false;
	}

	CComQIPtr<IBackgroundCopyJob4> spJob4 = spJob;
	if (!spJob4) {
		return false;
	}

	return true;
}


__checkReturn bool BITSUtil::IsBITS40Supported( __in const CComPtr<IBackgroundCopyJob>& spJob )
{
	if (!spJob) {
		ATLASSERT(false);
		return false;
	}

	CComQIPtr<IBitsTokenOptions> spTokenOption = spJob;
	if (!spTokenOption) {
		return false;
	}

	return true;
}


__checkReturn HRESULT BITSUtil::SetJobCredentials(__in const CComPtr<IBackgroundCopyJob2>& spJob2,
												  __in BG_AUTH_CREDENTIALS* pCredentials)
{
	if (!spJob2 || NULL == pCredentials) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob2->SetCredentials(pCredentials);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::RemoveJobCredentials(__in const CComPtr<IBackgroundCopyJob2>& spJob2,
													 __in BG_AUTH_TARGET authTarget, __in BG_AUTH_SCHEME authScheme)
{
	if (!spJob2 || false == VerifyAuthTarget(authTarget) || false == VerifyAuthScheme(authScheme)) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob2->RemoveCredentials(authTarget, authScheme);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobReplyFileName(__in const CComPtr<IBackgroundCopyJob2>& spJob2, 
													__out CString* pstrReplyFileName)
{
	if (!spJob2 || NULL == pstrReplyFileName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	pstrReplyFileName->Empty();

	LPWSTR lpszReplyFileName = NULL;
	HRESULT hr = spJob2->GetReplyFileName(&lpszReplyFileName);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}

	if (NULL != lpszReplyFileName) {
		*pstrReplyFileName = lpszReplyFileName;
		::CoTaskMemFree(lpszReplyFileName);
		lpszReplyFileName = NULL;
	}

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobReplyFileName(__in const CComPtr<IBackgroundCopyJob2>& spJob2, 
													__in LPCWSTR lpszReplyFileName)
{
	if (!spJob2 || NULL == lpszReplyFileName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob2->SetReplyFileName(lpszReplyFileName);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobReplyProgress(__in const CComPtr<IBackgroundCopyJob2>& spJob2, 
													__out BG_JOB_REPLY_PROGRESS* pReplyProgress)
{
	if (!spJob2 || NULL == pReplyProgress) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob2->GetReplyProgress(pReplyProgress);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobNotifyCmdLine(__in const CComPtr<IBackgroundCopyJob2>& spJob2,
													__out CString* pstrNotifyCommandPath, __out CString* pstrNotifyCommandArg)
{
	if (!spJob2 || NULL == pstrNotifyCommandPath || NULL == pstrNotifyCommandArg) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}
	pstrNotifyCommandPath->Empty();
	pstrNotifyCommandArg->Empty();

	LPWSTR lpszNotifyCommandPath	= NULL;
	LPWSTR lpszNotifyCommandArg		= NULL;
	HRESULT hr = spJob2->GetNotifyCmdLine(&lpszNotifyCommandPath, &lpszNotifyCommandArg);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}

	if (NULL != lpszNotifyCommandPath) {
		*pstrNotifyCommandPath = lpszNotifyCommandPath;
		::CoTaskMemFree(lpszNotifyCommandPath);
		lpszNotifyCommandPath = NULL;
	}

	if (NULL != lpszNotifyCommandArg) {
		*pstrNotifyCommandArg = lpszNotifyCommandArg;
		::CoTaskMemFree(lpszNotifyCommandArg);
		lpszNotifyCommandArg = NULL;
	}

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobNotifyCmdLine(__in const CComPtr<IBackgroundCopyJob2>& spJob2,
													__in LPCWSTR lpszNotifyCommandPath, __in LPCWSTR lpszNotifyCommandArg)
{
	if (!spJob2 || NULL == lpszNotifyCommandPath || NULL == lpszNotifyCommandArg) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob2->SetNotifyCmdLine(lpszNotifyCommandPath, lpszNotifyCommandArg);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobFileACLFlags(__in const CComPtr<IBackgroundCopyJob3>& spJob3,
												   __out DWORD* pdwACLFlags)
{
	if (!spJob3 || NULL == pdwACLFlags) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob3->GetFileACLFlags(pdwACLFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobFileACLFlags(__in const CComPtr<IBackgroundCopyJob3>& spJob3,
												   __in DWORD dwACLFlags)
{
	if (!spJob3) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob3->SetFileACLFlags(dwACLFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobHttpOptions(__in const CComPtr<IBackgroundCopyJobHttpOptions>& spHttpOptions,
												  __out DWORD* pdwFlags)
{
	if (!spHttpOptions || NULL == pdwFlags) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spHttpOptions->GetSecurityFlags(pdwFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobHttpOptions(__in const CComPtr<IBackgroundCopyJobHttpOptions>& spHttpOptions,
												  __in DWORD dwFlags)
{
	if (!spHttpOptions) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spHttpOptions->SetSecurityFlags(dwFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobPeerCachingFlags(__in const CComPtr<IBackgroundCopyJob4>& spJob4,
													   __out DWORD* pdwFlags)
{
	if (!spJob4 || NULL == pdwFlags) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob4->GetPeerCachingFlags(pdwFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobPeerCachingFlags(__in const CComPtr<IBackgroundCopyJob4>& spJob4,
													   __in DWORD dwFlags)
{
	if (!spJob4) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob4->SetPeerCachingFlags(dwFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobOwnerIntegrityLevel(__in const CComPtr<IBackgroundCopyJob4>& spJob4,
														  __out ULONG* puLevel)
{
	if (!spJob4 || NULL == puLevel) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob4->GetOwnerIntegrityLevel(puLevel);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobOwnerElevationState(__in const CComPtr<IBackgroundCopyJob4>& spJob4,
														  __out BOOL* pbElevated)
{
	if (!spJob4 || NULL == pbElevated) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob4->GetOwnerElevationState(pbElevated);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobMaximumDownloadTime(__in const CComPtr<IBackgroundCopyJob4>& spJob4,
														  __out ULONG* puTimeout)
{
	if (!spJob4 || NULL == puTimeout) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob4->GetMaximumDownloadTime(puTimeout);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobMaximumDownloadTime(__in const CComPtr<IBackgroundCopyJob4>& spJob4,
														  __in ULONG uTimeout)
{
	if (!spJob4) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spJob4->SetMaximumDownloadTime(uTimeout);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobCustomHeaders(__in const CComPtr<IBackgroundCopyJobHttpOptions>& spHttpOptions,
													__out CString* pstrCustomHeader)
{
	if (!spHttpOptions || NULL == pstrCustomHeader) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	pstrCustomHeader->Empty();

	LPWSTR lpszCustomHeader = NULL;
	HRESULT hr = spHttpOptions->GetCustomHeaders(&lpszCustomHeader);
	if (FAILED(hr)) {
		// [U[قȂƎsȂǁAsP[Xɋ߂Ƃ̂ASSERT͏oȂ
		// ATLASSERT(false);
		return hr;
	}

	if (NULL != lpszCustomHeader) {
		*pstrCustomHeader = lpszCustomHeader;
		::CoTaskMemFree(lpszCustomHeader);
		lpszCustomHeader = NULL;
	}
	
	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobCustomHeaders(__in const CComPtr<IBackgroundCopyJobHttpOptions>& spHttpOptions,
													__in LPCWSTR lpszCustomHeader)
{
	if (!spHttpOptions) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}
	
	if (NULL != lpszCustomHeader && L'\0' == *lpszCustomHeader) {
		lpszCustomHeader = NULL;
	}

	HRESULT hr = spHttpOptions->SetCustomHeaders(lpszCustomHeader);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobClientCertificateByName(__in const CComPtr<IBackgroundCopyJobHttpOptions>& spHttpOptions,
															  __in BG_CERT_STORE_LOCATION certStoreLocation,
															  __in LPCWSTR lpszStoreName,
															  __in LPCWSTR lpszSubjectName)
{
	if (!spHttpOptions || NULL == lpszStoreName || NULL == lpszSubjectName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spHttpOptions->SetClientCertificateByName(certStoreLocation, lpszStoreName, lpszSubjectName);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobClientCertificateById(__in const CComPtr<IBackgroundCopyJobHttpOptions>& spHttpOptions,
															__in BG_CERT_STORE_LOCATION certStoreLocation,
															__in LPCWSTR lpszStoreName,
															__in byte* pCertHashBlob)
{
	if (!spHttpOptions || NULL == lpszStoreName || NULL == pCertHashBlob) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spHttpOptions->SetClientCertificateByID(certStoreLocation, lpszStoreName, pCertHashBlob);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::RemoveJobClientCertificate(__in const CComPtr<IBackgroundCopyJobHttpOptions>& spHttpOptions)
{
	if (!spHttpOptions) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spHttpOptions->RemoveClientCertificate();
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobFileRange(__in const CComPtr<IBackgroundCopyFile2>& spFile2, 
												__out DWORD* pdwRangeCount, __out BG_FILE_RANGE** ppFileRange)
{
	if (!spFile2 || NULL == pdwRangeCount || NULL == ppFileRange) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spFile2->GetFileRanges(pdwRangeCount, ppFileRange);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobFileProgress(__in const CComPtr<IBackgroundCopyFile>& spFile,
												   __out BG_FILE_PROGRESS* pProgress)
{
	if (!spFile || NULL == pProgress) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spFile->GetProgress(pProgress);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobFileTemporaryName(__in const CComPtr<IBackgroundCopyFile3>& spFile3,
														__out CString* pstrTemporaryName)
{
	if (!spFile3 || NULL == pstrTemporaryName) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}
	pstrTemporaryName->Empty();

	LPWSTR lpszTemporaryName = 0;
	HRESULT hr = spFile3->GetTemporaryName(&lpszTemporaryName);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}
	if (NULL != lpszTemporaryName) {
		*pstrTemporaryName = lpszTemporaryName;
		::CoTaskMemFree(lpszTemporaryName);
		lpszTemporaryName = NULL;
	}

	return hr;
}


__checkReturn HRESULT BITSUtil::IsJobFileDownloadedFromPeer(__in const CComPtr<IBackgroundCopyFile3>& spFile3,
															__out BOOL* pbIsDownloadedFromPeer)
{
	if (!spFile3 || NULL == pbIsDownloadedFromPeer) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spFile3->IsDownloadedFromPeer(pbIsDownloadedFromPeer);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobFileValidationState(__in const CComPtr<IBackgroundCopyFile3>& spFile3,
														  __out BOOL* pbValidationState)
{
	if (!spFile3 || NULL == pbValidationState) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spFile3->GetValidationState(pbValidationState);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobFileValidationState(__in const CComPtr<IBackgroundCopyFile3>& spFile3,
														  __in BOOL bValidationState)
{
	if (!spFile3) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spFile3->SetValidationState(bValidationState);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobHelperTokenFlags(__in const CComPtr<IBitsTokenOptions>& spTokenOptions,
													   __out DWORD* pdwFlags)
{
	if (!spTokenOptions || NULL == pdwFlags) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spTokenOptions->GetHelperTokenFlags(pdwFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobHelperTokenFlags(__in const CComPtr<IBitsTokenOptions>& spTokenOptions,
													   __in DWORD dwFlags)
{
	if (!spTokenOptions || false == VerifyJobHelperTokenFlags(dwFlags)) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spTokenOptions->SetHelperTokenFlags(dwFlags);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobHelperTokenSid(__in const CComPtr<IBitsTokenOptions>& spTokenOptions,
													 __out CString* pstrSid)
{
	if (!spTokenOptions || NULL == pstrSid) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}
	pstrSid->Empty();

	LPWSTR lpszSid = NULL;
	HRESULT hr = spTokenOptions->GetHelperTokenSid(&lpszSid);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return hr;
	}
	
	if (NULL != lpszSid) {
		*pstrSid = lpszSid;
		::CoTaskMemFree(lpszSid);
		lpszSid = NULL;
	}

	return hr;
}


__checkReturn HRESULT BITSUtil::SetJobHelperToken(__in const CComPtr<IBitsTokenOptions>& spTokenOptions)
{
	if (!spTokenOptions) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spTokenOptions->SetHelperToken();
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::ClearJobHelperToken(__in const CComPtr<IBitsTokenOptions>& spTokenOptions)
{
	if (!spTokenOptions) {
		ATLVERIFY(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spTokenOptions->ClearHelperToken();
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}


__checkReturn HRESULT BITSUtil::GetJobPeerDownloadStats(__in const CComPtr<IBackgroundCopyFile4>& spFile4,
														__out PUINT64 pFromOrigin, __out PUINT64 pFromPeers)
{
	if (!spFile4 || NULL == pFromOrigin || NULL == pFromPeers) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr = spFile4->GetPeerDownloadStats(pFromOrigin, pFromPeers);
	ATLASSERT(SUCCEEDED(hr));

	return hr;
}

