
/* ------------------------
Container.js
-- コンテナオブジェクトで直接的にMathML要素と対応するもの

*MathFrac - mfrac - 分数
*MathSqrt - msqrt - 平方根
*MathRoot - mroot - 累乗根
*MathFenced - mfenced - 括弧
*MathSup - msup - 累乗
*MathSub - msub - 添字
*/

/* ====================
** 分数（<mfrac>）
*/
function MathFrac(par) {
    this.__super__(par);
    this.separator = new Separator();
    this.obj.appendChild(this.separator.obj);
    this._layout();
}

BUtil.inherit(MathFrac, BExpContainer);

BUtil.setProperty(MathFrac, {
    objType: 'frac',
    rowCount: 2,
    mathmlTag: 'mfrac',
    texCommand: 'frac'
});

ContainerButtons.submit(2, 'frac', function(wrap){return new MathFrac(wrap);});

// 表示部品削除
MathFrac.prototype.removeObj = function() {
    this.separator.removeObj();
    this._removeObj();
}

// --- private
MathFrac.prototype._layout = function() {
    // 計算して保存
    var nume = this.rows[0].getPreferredSize();
    var denomi = this.rows[1].getPreferredSize();
    this.width = (nume.width > denomi.width) ? nume.width : denomi.width;
    this.width += 4;
    this.top = nume.height + 2;
    this.bottom = denomi.height + 3;
    this.height = this.top + this.bottom;
    // 適用
    this._setSize();
    BUtil.obj.setPos(this.rows[0].obj, (this.width - nume.width) / 2, 0);
    BUtil.obj.setPos(this.rows[1].obj, (this.width - denomi.width) / 2, (nume.height + 5));
    BUtil.obj.setPos(this.separator.obj, 0, (nume.height + 2));
}

/* ====================
** 平方根（<msqrt>）
*/
function MathSqrt(par) {
    this.__super__(par);
    this.separator = new Separator();
    this.obj.appendChild(this.separator.obj);
    this.img = document.createElement('img');
    this.img.src = Const.imgDir + 'sqrt.png';
    this.img.className = 'containerParts';
    this.obj.appendChild(this.img);
    this._layout();
}

BUtil.inherit(MathSqrt, BExpContainer);

BUtil.setProperty(MathSqrt, {
    objType: 'sqrt',
    rowCount: 1,
    mathmlTag: 'msqrt',
    texCommand: 'sqrt'
});

ContainerButtons.submit(21, 'sqrt', function(wrap){return new MathSqrt(wrap);});

// 表示部品削除
MathSqrt.prototype.removeObj = function() {
    this.separator.removeObj();
    this.obj.removeChild(this.img);
    this._removeObj();
}

// ---- private
MathSqrt.prototype._layout = function() {
    this._setLayout(4, 0, 2, 14);
    //this._calcPosition(4, 2, 14);
    //this._setSize();
    //BUtil.obj.setPos(this.content.obj, 14, 4);
    BUtil.obj.setSize(this.separator.obj, this.width - 10, 1);
    BUtil.obj.setPos(this.separator.obj, 10, 0);
    BUtil.obj.setSize(this.img, 10, this.height);
}

/* ====================
** 累乗根（<mroot>）
*/
function MathRoot(par) {
    this.__super__(par,[2, par.level]);
    this.content = this.rows[1];
    this.separator = new Separator();
    this.obj.appendChild(this.separator.obj);
    this.img = document.createElement('img');
    this.img.src = Const.imgDir + 'sqrt.png';
    this.img.className = 'containerParts';
    this.obj.appendChild(this.img);
    this._layout();
}

BUtil.inherit(MathRoot, BExpContainer);

BUtil.setProperty(MathRoot, {
    objType: 'root',
    rowCount: 2,
    mathmlTag: 'mroot',
    texCommand: 'sqrt'
});

ContainerButtons.submit(22, 'root', function(wrap){return new MathRoot(wrap);});

// MathMLコード書き出し
MathRoot.prototype.outputMathML = function(expand) {
    return this._tagMML(this.rows[1].outputMathML(expand) + this.rows[0].outputMathML(expand));
}

// LaTeXコード書き出し
MathRoot.prototype.outputLatex = function() {
    return this._commandTex('[' + this.rows[0].outputLatex(true) + ']'+ this.rows[1].outputLatex());
}

// 表示部品削除
MathRoot.prototype.removeObj = function() {
    this.separator.removeObj();
    this.obj.removeChild(this.img);
    this._removeObj();
}

// ---- private
MathRoot.prototype._layout = function() {
    var proot = this.rows[0].getPreferredSize();
    var prow = this.rows[1].getPreferredSize();
    this.top = (proot.height > (prow.top + 4)) ? proot.height : (prow.top + 4);
    this.bottom = prow.bottom + 2;
    this.height = this.top + this.bottom;
    this.width = prow.width + proot.width + 14;
    this._setSize();
    BUtil.obj.setPos(this.content.obj, proot.width + 14, this.top - prow.top);
    BUtil.obj.setSize(this.separator.obj, prow.width + 4, 1);
    BUtil.obj.setPos(this.separator.obj, proot.width + 10, this.top - prow.top - 4);
    BUtil.obj.setSize(this.img, 10, prow.height + 6);
    BUtil.obj.setPos(this.img, proot.width, this.top - prow.top - 4);
}

/* ====================
** 括弧（<mfenced>）
*/
function MathFenced(par, kind) {
    this.__super__(par);
    // 開き括弧画像
    this.imgOpen = document.createElement('img');
    this.imgOpen.src = Const.imgDir + 'openParenthesis.png';
    this.imgOpen.className = 'containerParts';
    this.obj.appendChild(this.imgOpen);
    // 閉じ括弧画像
    this.imgClose = document.createElement('img');
    this.imgClose.src = Const.imgDir + 'closeParenthesis.png';
    this.imgClose.className = 'containerParts';
    this.obj.appendChild(this.imgClose);
    this._layout();
}

BUtil.inherit(MathFenced, BExpContainer);

BUtil.setProperty(MathFenced, {
    objType: 'fenced',
    rowCount: 1,
    mathmlTag: 'mfenced'
});

ContainerButtons.submit(1, 'fenced', function(wrap){return new MathFenced(wrap);});

// MathMLコード書き出し
MathFenced.prototype.outputMathML = function(expand) {
    var out = '<mfenced>' + "\n";
    out += this.content.outputMathML(expand);
    out += '</mfenced>' + "\n";
    return out;
}

// LaTeXコード書き出し
MathFenced.prototype.outputLatex = function() {
    var out = Const.backslash + 'left(';
    out += this.content.outputLatex(true);
    out += Const.backslash + 'right)';
    return out;
}

// 表示部品削除
MathFenced.prototype.removeObj = function() {
    this.obj.removeChild(this.imgOpen);
    this.obj.removeChild(this.imgClose);
    this._removeObj();
}

// ---------- private
// コンテナ内配置
MathFenced.prototype._layout = function() {
    this._setLayout(2, 12, 2, 12);
    //this._calcPosition(2, 2, 24);

    //BUtil.obj.setSize(this.obj, this.width, this.height);
    //BUtil.obj.setPos(this.content.obj, 12, 2);
    BUtil.obj.setSize(this.imgOpen, 10, this.height);
    BUtil.obj.setPos(this.imgOpen, 0, 0);
    BUtil.obj.setSize(this.imgClose, 10, this.height);
    BUtil.obj.setPos(this.imgClose, this.width - 10, 0);
}


/* ====================
** 累乗（<msup>）
*/
function MathSup(par) {
    this.__super__(par, [2]);
    this._layout();
}

BUtil.inherit(MathSup, BExpContainer);

BUtil.setProperty(MathSup, {
		  objType: 'sup',
		  rowCount: 1,
		  mathmlTag: 'msup',
		  isNeedPrev: true
});

ContainerButtons.submit(11, 'sup', function(wrap){return new MathSup(wrap);});

// MathMLコード書き出し
MathSup.prototype.outputMathML = function(expand, out) {
    if(out == '') out = '<mo></mo>';
    out += this.content.outputMathML(expand);
    return this._tagMML(out);
}

// LaTeXコード書き出し
MathSup.prototype.outputLatex = function() {
    return '^' + this.content.outputLatex();
}

// ---- private
MathSup.prototype._layout = function() {
    // 計算して保存
    var prefer = this.content.getPreferredSize();
    this.top = prefer.height + 4;
    this.bottom = 0;
    this.height = this.top + this.bottom;
    this.width = prefer.width + 4;
    // 適用
    this._setSize();
    BUtil.obj.setPos(this.content.obj, 2, 2);    
}

/* ====================
** 添字（<msub>）
*/
function MathSub(par) {
    this.construct(par, [2]);
    this._layout();
}

BUtil.inherit(MathSub, BExpContainer);

BUtil.setProperty(MathSub, {
		  objType: 'sub',
		  rowCount: 1,
		  mathmlTag: 'msub',
		  isNeedPrev: true
});

ContainerButtons.submit(12, 'sub', function(wrap){return new MathSub(wrap);});

// MathMLコード書き出し
MathSub.prototype.outputMathML = function(expand, out) {
    if(out == '') out = '<mo></mo>';
    out += this.content.outputMathML(expand);
    return this._tagMML(out);
}

// LaTeXコード書き出し
MathSub.prototype.outputLatex = function() {
    return '_' + this.content.outputLatex();
}

// ----- private
MathSub.prototype._layout = function() {
    // 計算して保存
    var prefer = this.content.getPreferredSize();
    this.top = 0;
    this.bottom = prefer.height + 4;
    this.height = this.top + this.bottom;
    this.width = prefer.width + 4;
    // 適用
    this._setSize();
    BUtil.obj.setPos(this.content.obj, 2, 2);
}

