/*
 BrEdiMa
 http://bredima.sourceforge.jp/
 Copyright(c) 2005-2008 NAKANO Yasuhito / U.E.C. Murao Lab.
 This product is distributed under the MIT license.
     http://www.opensource.org/licenses/mit-license.php
*/

/**
 * @class メインクラス
 */
Bredima = function(dom, type) {
    this.serial = Bredima.serial++;
    this.dom = dom;

    // 初期設定から設定をコピー
    this.config = {};
    for(var key in Bredima.config.def) {
	this.config[key] = Bredima.config.def[key];
    }

    if(type == 'float') {
	this.ignition = dom;
	this.dom = document.createElement('div');
	this.dom.style.width = '580px';
	this.dom.style.position = 'absolute';
	this.dom.style.display = 'none';
	var body = document.getElementsByTagName('body')[0];
	body.appendChild(this.dom);
	var self = this;
	Bredima.util.addListener(this.ignition, 'click',
				 function() {
				     var offset = Bredima.util.getOffset(self.ignition, body);
				     self.dom.style.left = Math.min(offset.left, body.clientWidth - 590) + 'px';
				     self.dom.style.top = (offset.top + 24) + 'px';
				     if(self.isIgnited) {
					 self.frame.toggleVisibility();
				     }
				     else {
					 self.isIgnited = true;
					 self.dom.style.display = 'block';
					 self.write();
				     }
				 });
    }
}

Bredima.version = '0.7.4.20080120';

/* ==============================
 外向きクラス関数
*/

/**
 * 全フレーム共通の設定を行う
 */
Bredima.setConfig = function(key, val) {
    Bredima.config.set(key, val);
}

/**
 * 指定したtextarea/inputに文字列を挿入するユーティリティ関数
 * @param {Object} dom DOMオブジェクト
 * @param {String} str 挿入する文字列
*/
Bredima.insertTo = function(dom, str) {
    dom.focus();
    if(dom.selectionStart != null) {
	var start = dom.selectionStart;
	var end = dom.selectionEnd;
	dom.value = dom.value.substring(0, start) +
	    str + dom.value.substring(end);
	dom.setSelectionRange(start, end + str.length);
    }
    else {
	var range = document.selection.createRange();
	range.text = str;
	range.select();
    }
}

/* ---------------------------------
内向き
*/
Bredima.serial = 0;

/* ==============================
 外向きインスタンス関数
*/
/**
 * インスタンスの設定を変更
 */
Bredima.prototype.setConfig = function(key, val) {
    for(var v in this.config) {
	if(v == key) {
	    this.config[key] = val;
	}
    }
}

/**
 * MathML出力
 */
Bredima.prototype.toMML = function(expand) { return this.exp.root.toMML(expand); }

/**
 * LaTeX出力
 */
Bredima.prototype.toLatex = function() { return this.exp.root.toLatex(); }

/**
 * JSON出力
 */
Bredima.prototype.toJSON = function() {
    var rd = this.exp.root.toRundown();
    if(rd.toJSON)
	return rd.toJSON();
    else
	return JSON.stringify(rd);
}

/**
 * 指定された領域にフレーム作成
 */
Bredima.prototype.write = function(dom) {
    if(dom) this.dom = dom;
    this.init = new Bredima.Init();
    this.locale = new Bredima.Locale(this);
    this.history = new Bredima.History();
    this.frame = new Bredima.Frame(this, this.dom);
    this.exp = new Bredima.ExpCtrl(this);
    this.menu = new Bredima.Menu(this);
    this.init.observer.attach(this);
    this.init.exec();
}

Bredima.prototype.setExpression = function(json) {
    this.exp.init(JSON.parse(json));
}

/**
 * 「挿入」ボタンを押した際のイベントハンドラ
 * 利用する場合は別の関数を上書き代入
 */
Bredima.prototype.onsubmit = function() {}

Bredima.prototype.onchange = function() {}
/* ---------------------------------
内向き
*/
/**
 * （アクセサ）Bredimaの設定を返す
 */
Bredima.prototype.getConfig = function(key) { return this.config[key];}

/**
 * 初期化終了時に呼ばれる
 */
Bredima.prototype.onInitialized = function() {
    var rundown = (this.config.json) ? JSON.parse(this.config.json) : null;
    this.frame.finalize();
    this.exp.init(rundown, 'first');
    //this.frame.appear();
}
/*
    json2.js
    2007-11-06

    Public Domain

    No warranty expressed or implied. Use at your own risk.

    See http://www.JSON.org/js.html

    This file creates a global JSON object containing two methods:

        JSON.stringify(value, whitelist)
            value       any JavaScript value, usually an object or array.

            whitelist   an optional that determines how object values are
                        stringified.

            This method produces a JSON text from a JavaScript value.
            There are three possible ways to stringify an object, depending
            on the optional whitelist parameter.

            If an object has a toJSON method, then the toJSON() method will be
            called. The value returned from the toJSON method will be
            stringified.

            Otherwise, if the optional whitelist parameter is an array, then
            the elements of the array will be used to select members of the
            object for stringification.

            Otherwise, if there is no whitelist parameter, then all of the
            members of the object will be stringified.

            Values that do not have JSON representaions, such as undefined or
            functions, will not be serialized. Such values in objects will be
            dropped, in arrays will be replaced with null. JSON.stringify()
            returns undefined. Dates will be stringified as quoted ISO dates.

            Example:

            var text = JSON.stringify(['e', {pluribus: 'unum'}]);
            // text is '["e",{"pluribus":"unum"}]'

        JSON.parse(text, filter)
            This method parses a JSON text to produce an object or
            array. It can throw a SyntaxError exception.

            The optional filter parameter is a function that can filter and
            transform the results. It receives each of the keys and values, and
            its return value is used instead of the original value. If it
            returns what it received, then structure is not modified. If it
            returns undefined then the member is deleted.

            Example:

            // Parse the text. If a key contains the string 'date' then
            // convert the value to a date.

            myData = JSON.parse(text, function (key, value) {
                return key.indexOf('date') >= 0 ? new Date(value) : value;
            });

    This is a reference implementation. You are free to copy, modify, or
    redistribute.

    Use your own copy. It is extremely unwise to load third party
    code into your pages.
*/

/*jslint evil: true */
/*extern JSON */

if (!this.JSON) {

    JSON = function () {

        function f(n) {    // Format integers to have at least two digits.
            return n < 10 ? '0' + n : n;
        }

        Date.prototype.toJSON = function () {

// Eventually, this method will be based on the date.toISOString method.

            return this.getUTCFullYear()   + '-' +
                 f(this.getUTCMonth() + 1) + '-' +
                 f(this.getUTCDate())      + 'T' +
                 f(this.getUTCHours())     + ':' +
                 f(this.getUTCMinutes())   + ':' +
                 f(this.getUTCSeconds())   + 'Z';
        };


        var m = {    // table of character substitutions
            '\b': '\\b',
            '\t': '\\t',
            '\n': '\\n',
            '\f': '\\f',
            '\r': '\\r',
            '"' : '\\"',
            '\\': '\\\\'
        };

        function stringify(value, whitelist) {
            var a,          // The array holding the partial texts.
                i,          // The loop counter.
                k,          // The member key.
                l,          // Length.
                r = /["\\\x00-\x1f\x7f-\x9f]/g,
                v;          // The member value.

            switch (typeof value) {
            case 'string':

// If the string contains no control characters, no quote characters, and no
// backslash characters, then we can safely slap some quotes around it.
// Otherwise we must also replace the offending characters with safe sequences.

                return r.test(value) ?
                    '"' + value.replace(r, function (a) {
                        var c = m[a];
                        if (c) {
                            return c;
                        }
                        c = a.charCodeAt();
                        return '\\u00' + Math.floor(c / 16).toString(16) +
                                                   (c % 16).toString(16);
                    }) + '"' :
                    '"' + value + '"';

            case 'number':

// JSON numbers must be finite. Encode non-finite numbers as null.

                return isFinite(value) ? String(value) : 'null';

            case 'boolean':
            case 'null':
                return String(value);

            case 'object':

// Due to a specification blunder in ECMAScript,
// typeof null is 'object', so watch out for that case.

                if (!value) {
                    return 'null';
                }

// If the object has a toJSON method, call it, and stringify the result.

                if (typeof value.toJSON === 'function') {
                    return stringify(value.toJSON());
                }
                a = [];
                if (typeof value.length === 'number' &&
                        !(value.propertyIsEnumerable('length'))) {

// The object is an array. Stringify every element. Use null as a placeholder
// for non-JSON values.

                    l = value.length;
                    for (i = 0; i < l; i += 1) {
                        a.push(stringify(value[i], whitelist) || 'null');
                    }

// Join all of the elements together and wrap them in brackets.

                    return '[' + a.join(',') + ']';
                }
                if (whitelist) {

// If a whitelist (array of keys) is provided, use it to select the components
// of the object.

                    l = whitelist.length;
                    for (i = 0; i < l; i += 1) {
                        k = whitelist[i];
                        if (typeof k === 'string') {
                            v = stringify(value[k], whitelist);
                            if (v) {
                                a.push(stringify(k) + ':' + v);
                            }
                        }
                    }
                } else {

// Otherwise, iterate through all of the keys in the object.

                    for (k in value) {
                        if (typeof k === 'string') {
                            v = stringify(value[k], whitelist);
                            if (v) {
                                a.push(stringify(k) + ':' + v);
                            }
                        }
                    }
                }

// Join all of the member texts together and wrap them in braces.

                return '{' + a.join(',') + '}';
            }
        }

        return {
            stringify: stringify,
            parse: function (text, filter) {
                var j;

                function walk(k, v) {
                    var i, n;
                    if (v && typeof v === 'object') {
                        for (i in v) {
                            if (Object.prototype.hasOwnProperty.apply(v, [i])) {
                                n = walk(i, v[i]);
                                if (n !== undefined) {
                                    v[i] = n;
                                }
                            }
                        }
                    }
                    return filter(k, v);
                }


// Parsing happens in three stages. In the first stage, we run the text against
// regular expressions that look for non-JSON patterns. We are especially
// concerned with '()' and 'new' because they can cause invocation, and '='
// because it can cause mutation. But just to be safe, we want to reject all
// unexpected forms.

// We split the first stage into 4 regexp operations in order to work around
// crippling inefficiencies in IE's and Safari's regexp engines. First we
// replace all backslash pairs with '@' (a non-JSON character). Second, we
// replace all simple value tokens with ']' characters. Third, we delete all
// open brackets that follow a colon or comma or that begin the text. Finally,
// we look to see that the remaining characters are only whitespace or ']' or
// ',' or ':' or '{' or '}'. If that is so, then the text is safe for eval.

                if (/^[\],:{}\s]*$/.test(text.replace(/\\./g, '@').
replace(/"[^"\\\n\r]*"|true|false|null|-?\d+(?:\.\d*)?(:?[eE][+\-]?\d+)?/g, ']').
replace(/(?:^|:|,)(?:\s*\[)+/g, ''))) {

// In the second stage we use the eval function to compile the text into a
// JavaScript structure. The '{' operator is subject to a syntactic ambiguity
// in JavaScript: it can begin a block or an object literal. We wrap the text
// in parens to eliminate the ambiguity.

                    j = eval('(' + text + ')');

// In the optional third stage, we recursively walk the new structure, passing
// each name/value pair to a filter function for possible transformation.

                    return typeof filter === 'function' ? walk('', j) : j;
                }

// If the text is not JSON parseable, then a SyntaxError is thrown.

                throw new SyntaxError('parseJSON');
            }
        };
    }();
}

/**
 * @class ロケール（言語+地域情報）
 * @constructor
 */
Bredima.Locale = function(bredima) {
    this.locale = bredima.getConfig('locale');
    this.text = Bredima.Locale.text[this.locale];
}

Bredima.Locale.text = {
    ja: {
	preview: 'プレビュー',
	live_preview: '自動プレビュー',
	show_mathml: 'MathMLソース表示',
	show_latex: 'LaTeXソース表示'
    },
    en: {
	preview: 'Preview',
	live_preview: 'Live preview',
	show_mathml: 'Show MathML source',
	show_latex: 'Show LaTeX source'
    }
};



/**
 * 定数定義
 * keys- キーコード, chars- 文字コード
 */
Bredima.consts = {

    keys: {
	bs: 8,
	left: 37,
	up: 38,
	right: 39,
	down: 40,
	enter: 13
    },
    texSize: ['', 'large', 'normal']
}

Bredima.config = {};

Bredima.config.def = {
    'uri_mimetex' : 'mimetex.cgi',
    'uri_img' : 'img/',
    'isMML' : false,
    'locale' : 'ja',
    'json' : '',
    'showJson' : '',
    'isGM' : false,
    'use_button': false
};

/**
 * 全Bredima共通設定を返す
 */
Bredima.config.get = function(key) {return Bredima.config.def[key];}

/**
 * 全Bredima共通設定を書き換える
 */
Bredima.config.set = function(key, val) {
    for(var v in Bredima.config.def) {
	if(v == key)
	    Bredima.config.def[key] = val;
    }
}

/**
 * @class 全体で使い回すクラス関数群
 */
Bredima.util = function(){};

/* ---------------------
オブジェクト構築関係
 */
/**
 * 継承
 * 継承されていないクラスではthis.__super__(foo)
 * 継承されるクラスではthis.superClass.apply(this, foo)
 * __super__がサブクラスで上書きされて自分自身への再帰になってしまうため。
 * @param {Object} sub サブクラス
 * @param {Object} sp スーパークラス
 */
Bredima.util.inherit = function(sub, sp) {
    function tmp() {};
    tmp.prototype = sp.prototype;
    sub.prototype = new tmp();
    sub.prototype.__super__ = sp;
    sub.superClass = sp;
    sub.prototype.constructor = sub;
}

/**
 * クラスのプロトタイプ属性を設定
 */
Bredima.util.setProperty = function(c, props) {
    for(var key in props) {
	c.prototype[key] = props[key];
    }
}

Bredima.util.applySuper = function(name, ist, args) {
     return ist.constructor.superClass.prototype[name].apply(ist, (args) ? args : []);
    //IEは空でもオブジェクトを渡さないとエラー
}

/* ---------------------
DOM関係
 */
/**
 * イベントの伝播を止める。ブラウザ依存なので依存性排除のためのラッパ。
 */
Bredima.util.stopEvent = function(evt) {
    if(evt && evt.preventDefault) {
	evt.preventDefault();
    }
    else if(window.event) {
	window.event.returnValue = false;
    }
    
    if(evt && evt.stopPropagation) {
	evt.stopPropagation();
    }
    else if(window.event) {
	window.event.cancelBubble = true;
    }
}

/**
 * 指定オブジェクトからの位置を算出
 * @param {Object} obj 算出したいオブジェクト
 * @param {Object} parent 基準にしたいオブジェクト
 */
Bredima.util.getOffset = function(obj, parent) {
    var left = 0;
    var top = 0;
    while(obj != parent) {
	left += obj.offsetLeft;
	top += obj.offsetTop;
	obj = obj.offsetParent;
    }
    return {left: left, top: top};
}

/**
 * 状況に合わせてイベントリスナを設定
 * @param {Object} elem イベントリスナを設定するDOMオブジェクト
 * @param {String} name イベントハンドラ名 onを抜いた形
 * @param {Function} func イベント発生時に呼び出す関数
 */
Bredima.util.addListener = function(elem, name, func) {
    if(Bredima.config.get('isGM'))
	elem.addEventListener(name, func, false);
    else
	elem['on' + name] = func;
}

/**
 * 設定したイベントリスナを解除
 * @param {Object} elem イベントリスナを解除するDOMオブジェクト
 * @param {String} name 解除するイベントハンドラ名 onを抜いた形
 * @param {Function} func 設定時に指定したものと同じ関数
 */
Bredima.util.removeListener = function(elem, name, func) {
    if(Bredima.config.get('isGM'))
	elem.removeEventListener(name, func, false);
    else
	elem['on' + name] = '';
}

/* ---------------------
スタイルシート関係
 */
/**
 * 幅と高さのスタイル設定
 * @param {DOMObject} obj 設定をするDOMオブジェクト
 * @param {Number} width 幅
 * @param {Number} height 高さ
 */
Bredima.util.setSize = function(obj, width, height) {
    obj.style.width = width + 'px';
    obj.style.height = height + 'px';
}

/**
 * 透過率を設定
 * @param {Object} dom 設定先のDOMオブジェクト
 * @param {Number} num 透過率 0-1 0で全部透過
 */
Bredima.util.setOpacity = function(dom, num) {
    dom.style.opacity = num;
    dom.style.filter = 'alpha(opacity=' + Math.floor(num * 100) + ')';
}

/* ---------------------
その他
 */
/**
 * 引数に合わせてタグを付けて返す
 * @param {String} tag タグ名
 * @param {String} str タグに挟まれる文字列
 * @param {String} attr タグ内に挿入される、属性を記述する文字列
 */
Bredima.util.addTag = function(tag, str, attr) {
    var out = '<' + tag + ((attr) ? (' ' + attr) : '') + '>';
    out += str;
    out += '</' + tag + '>' + "\n";
    return out;
}

/**
 * mimetexで画像を取得するURIを返す
 * @param {String} str tex文
 * @param {Number_String} size 取得画像のサイズ 1/2, あるいはmimetexでサポートされている文字サイズ
 */
Bredima.util.getTexURI = function(str, size) {
    if(size) {
	size = (typeof(size) == 'number') ? Bredima.consts.texSize[size] : size;
	size = '\\' + size + ' ';
    }
    return encodeURI(Bredima.config.get('uri_mimetex') + '?' + size + str);
}

Bredima.note = function(str) {
    document.getElementById('debugFrame').innerHTML += '[' + str + '] ';
}

/**
 * 単純な数値遷移処理
 * @param {Object} param パラメタを集めた連想配列
 * start: 開始値, end: 終了値, step: アニメーションコマ数
 * func: アニメーション処理関数, finalize: 終了処理関数
 */
Bredima.util.animation = function(param) {
    var number = param.start;
    var count = 0;
    var cv = (param.end - param.start) / param.step;
    var timer = setInterval(function() {
				param.func(number);
				if(count++ == param.step) {
				    clearInterval(timer);
				    if(param.finalize) param.finalize();
				}
				if(count == param.step)
				    number = param.end;
				else
				    number = param.start + cv * count;
			    }, 30);
}

/**
 * ちょっと凝った数値遷移処理
 * @param {Object} param パラメタを集めた連想配列
 * start: 開始値, end: 終了値, step: アニメーションコマ数
 * func: アニメーション処理関数, finalize: 終了処理関数
 */
Bredima.util.smoothAnimation = function(param) {
    var number = param.start;
    var count = 0;
    var cv = (param.end - param.start) / (param.step - 2); // constant velocity
    var timer = setInterval(function() {
				param.func(number);
				if(count++ == param.step) {
				    clearInterval(timer);
				    if(param.finalize) param.finalize();
				}
				if(count == param.step)
				    number = param.end;
				else if(count < 3)
				    number = param.start + (cv * 2 / 14 * ((count - 1) * 5 + 3));
				else if(count > param.step - 3)
				number = param.end - (cv * 2 / 14 * ((param.step - 1 - count) * 5 + 3));
				else
				    number = param.start + (cv * (count - 1));
			    }, 30);
}

/* =============================
 スタイルシート登録
 */
Bredima.style = {};

Bredima.style.str = '';

/**
 * スタイルを登録
 * @param {Array*} 登録するスタイルを可変長引数で指定
 */
Bredima.style.add = function() {
    for( var i = 0; i < arguments.length; i++) {
	var s = arguments[i];
	var a = new Array();
	for(var n = 0; n < s.length - 1; n++) {
	    a[n] = 'div.BrEdiMa_Frame ' + s[n];
	}
	Bredima.style.str += a.join(',') + '{' + s[s.length - 1] + '} ';
    }
}

/**
 * スタイルシートを書き出し
 */
Bredima.style.addStylesheet = function() {
    var head = document.getElementsByTagName('head')[0];
    if(!head) return;
    var style = document.createElement('style');
    if(typeof(style.styleSheet) == 'object') { // IE
	style.type = 'text/css';
	style.styleSheet.cssText = Bredima.style.str;
    }
    else { // それ以外
	style = document.createElement('link');
	style.rel = 'stylesheet';
	style.href = 'data:text/css,' + escape(Bredima.style.str);
    }
    head.appendChild(style);
}


/**
 @fileoverview
 1つのファイルにするまでもないクラスをまとめたファイル
*/

/**
 * @class 選択範囲の記録
 * @constructor
 */
Bredima.Range = function(start, end) {
    this.start = start;
    this.end = end;
}

/**
 * 選択範囲が同一のものか判定
 * @param {Object} r2 比較する選択範囲オブジェクト
 */
Bredima.Range.prototype.equals = function(r2) {
    return ((this.start == r2.start) && (this.end == r2.end));
}

/* ============================
*/
/**
 * @class Observerパターン
 * @constructor
 */
Bredima.Observer = function() {
    this.observers = new Array;
}

/**
 * 通知を行うオブジェクト/関数を追加
 * @param {Object} arg 通知先のオブジェクト、あるいは実行する関数
 */
Bredima.Observer.prototype.attach = function(arg) {
    this.observers.push(arg);
}

/**
 * 通知先を削除
 * @param {Object} arg 通知先リストから削除するオブジェクト/関数
 */
Bredima.Observer.prototype.detach = function(arg) {
    for(var i = 0; i < this.observers.length; i++) {
	if(arg == this.observers[i]) {
	    this.observers.splice(i, 1);
	    return true;
	}
    }
    return false;
}

/**
 * 登録した関数へ通知
 * @param {String} arg 通知文字列
 */
Bredima.Observer.prototype.notify = function(evt, prop) {
    for(var i = 0; i < this.observers.length; i++) {
	if(this.observers[i]['on' + evt]) this.observers[i]['on' + evt](prop);
    }
}

/**
 * @class 初期化管理クラス
 * @constructor
 */
Bredima.Init = function() {
    this.observer = new Bredima.Observer(); // 初期化終了を各インスタンスに伝えるObserver
}

Bredima.Init.isDone = false;
Bredima.Init.classes = new Array;
Bredima.Init.finished = 0;
Bredima.Init.initializers = new Array;

/**
 * 初期化が必要なクラスを登録
 */
Bredima.Init.attachInit = function(c) { // 初期化開始を各クラスに伝えるObserver
    Bredima.Init.classes.push(c);
}

/**
 * 呼ばれたクラスの初期化完了を連絡
 */
Bredima.Init.ackFinish = function() {
    if(++Bredima.Init.finished == Bredima.Init.classes.length) {
	// 初期化終了
	Bredima.Init.isDone = true;
	for(var i = 0; i < Bredima.Init.initializers.length; i++)
	    Bredima.Init.initializers[i].observer.notify('Initialized');
    }
}

/**
 * 初期化実行
 */
Bredima.Init.prototype.exec = function() {
    if(Bredima.Init.isDone)
	this.observer.notify('Initialized');
    else {
	Bredima.Init.initializers.push(this);
	if(Bredima.Init.initializers.length == 1) {
	    for(var i = 0; i < Bredima.Init.classes.length; i++) {
		Bredima.Init.classes[i].initializeOnLoad();
	    }
	}
    }
}
/**
 * @class 変更履歴管理
 * @constructor
 */
Bredima.History = function() {
    this.records = new Array;
    this.ptr = 0;
    this.p2 = 0; // 各record内での位置
    this.length = 0;
    this.isEnable = {undo: false, redo: false};
    this.observer = new Bredima.Observer();
}
// 履歴の最大保存個数 同一input内の履歴はまとめて1個
Bredima.History.maxLength = 10;

/**
 * 履歴を追加
 */
Bredima.History.prototype.add = function(record) {
    if(this.p2 > 0) {
	this.records[this.ptr].chop(this.p2);
	this.ptr++;
    }
    this.records[this.ptr++] = record;
    if(this.ptr >= Bredima.History.maxLength) {
	this.records.shift();
	this.ptr--;
    }
    this.length = this.ptr;
    this.p2 = 0;
    this.ackRecordArise();
}

/**
 * historyに追加前だけど履歴が追加されたことをhistoryとその受信者に連絡
 */
Bredima.History.prototype.ackRecordArise = function() {
    this._turn('undo', true);
    this._turn('redo', false);
    this.observer.notify('HistoryChange');
}

/**
 * Undo
 */
Bredima.History.prototype.undo = function() {
    if(this.p2 == 0) {
	if(this.ptr == 0) return false;
	this.p2 = this.records[--this.ptr].getLength() - 1;
    }
    this.records[this.ptr].revert(--this.p2);
    this._turn('redo', true);
    if((this.p2 == 0) && (this.ptr == 0)) this._turn('undo', false);
    this.observer.notify('HistoryChange');
    return true;
}

/**
 * Redo
 */
Bredima.History.prototype.redo = function() {
    if(this.ptr == this.length) return false;
    this.records[this.ptr].revert(++this.p2);
    this._turn('undo', true);
    if(this.p2 == this.records[this.ptr].getLength() - 1) {
	this.ptr++;
	this.p2 = 0;
	if(this.ptr == this.length) this._turn('redo', false);
    }
    this.observer.notify('HistoryChange');
    return true;
}

/**
 * Undo/Redoの実行可能属性を切り替え
 *@private
 */
Bredima.History.prototype._turn = function(name, b) {
    if(this.isEnable[name] != b) {
	this.observer.notify('HistoryEnableChange', (b ? 'enable' : 'disable') + '_' + name);
	this.isEnable[name] = b;
    }
}

/* ========================
*/
/**
 * @class rowfactor増減についての記録の実体
 * @constructor
 * @param {Object} row 状態を記録するrow
 */
Bredima.History.RowRecord = function(row) {
    this.row = row;
    this.records = new Array;
}

/**
 * input要素についての履歴ではないのでfalseを返す
 */
Bredima.History.RowRecord.prototype.isInput = function() { return false; }

/**
 * 変更を記録
 * オブジェクト変更操作の前と後で1回ずつ呼ぶ
 */
Bredima.History.RowRecord.prototype.commit = function(opt) {
    var current = this.row.container.exp.getCurrent();
    this.records.push({
			  rundown: this.row.toRundown(),
			  position: new Bredima.History.TreePosition(current),
			  selection: ((opt == 'selected') ? '' : current.getSelection())
		      });
    return true;
}

/**
 * 記録を終了
 */
Bredima.History.RowRecord.prototype.close = function() {
    this.position = new Bredima.History.TreePosition(this.row);
    this.row = '';
    return true;
}

/**
 * 記録している状態へ戻す
 */
Bredima.History.RowRecord.prototype.revert = function(order) {
    var row = this.position.trace();
    var record = this.records[order];
    row.initFromRundown(record.rundown);

    // 再配置
    row.repositionAll();
    // 選択でrootへの再配置を兼ねる
    record.position.trace().selectRange(record.selection);
}

/**
 * 記録している状態の個数を返す
 * 通常は2
 */
Bredima.History.RowRecord.prototype.getLength = function() {
    return this.records.length;
}

/* ========================
*/
/**
 * @class input要素内での文字列操作の記録
 * @constructor
 * @param {Object} input 状態を記録するinput
 */
Bredima.History.InputRecord = function(input) {
    this.input = input;
    this.records = new Array;
    this.cache();
    this.records.push({value: this.value, selection: this.selection});
    this.isTemporary = false;
    this.isCached = false;
}

/**
 * input要素についての履歴なのでtrueを返す
 */
Bredima.History.InputRecord.prototype.isInput = function() { return true; }

/**
 * 変更が行われた判定するために現在の状態を一時記録
 */
Bredima.History.InputRecord.prototype.cache = function() {
    if(!this.isCached) {
	this.isCached = true;
	this.value = this.input.toString();
	this.selection = this.input.getSelection();
    }
}

/**
 * cacheと比較して変更されていれば記録
 * @param {String} type 記録の種類
 * temporary: このcommitで変更が記録された場合その記録は1つ前の履歴を上書きする可能性がある
 * noselection: commit時にselectionを取得しない focusがない場所のcommit時に使用
 */
Bredima.History.InputRecord.prototype.commit = function(type) {
    this.isCached = false;
    var sel = (type == 'noselection') ? this.selection : this.input.getSelection();
    if(this.value != this.input.toString()) {
	if((type == 'temporary') && (this.isTemporary)){
	    this.records.pop();
	}
	this.records.push({ value: this.input.toString(), selection: sel });
	this.isTemporary = (type == 'temporary');
	return true;
    }
    else if(!this.selection.equals(sel)) {
	this.isTemporary = false;
    }
    return false;
}

/**
 * 記録を終了
 */
Bredima.History.InputRecord.prototype.close = function() {
    this.position = new Bredima.History.TreePosition(this.input);
    this.input = '';
    return ((this.records.length > 1) &&
	    !((this.records.length == 2) && (this.records[0].value == this.records[1].value)));
}

/**
 * 記録している状態へ戻す
 */
Bredima.History.InputRecord.prototype.revert = function(order) {
    var input = this.position.trace();
    input.init(this.records[order]);
}

/**
 * Undo後に変更が合った場合に不要になる部分を切り捨てる
 */
Bredima.History.InputRecord.prototype.chop = function(order) {
    this.records.splice(order + 1, this.records.length);
}

/**
 * 記録している状態の個数を返す
 */
Bredima.History.InputRecord.prototype.getLength = function() {
    return this.records.length;
}

/* ========================
*/
/**
 * @class 数式ツリー内での場所の記録
 * @constructor
 * @param {Object} row 記録を行う要素
 */
Bredima.History.TreePosition = function(elem) {
    this.tree = new Array;

    if(elem.constructor.classID == 'row') {
	this.tree.push(elem.getOrder());
	elem = elem.container;
    }
    while(elem.constructor.classID != 'exproot') {
	this.tree.push(elem.getOrder());
	this.tree.push(elem.row.getOrder());
	elem = elem.row.container;
    }
    this.root = elem;
}

/**
 * 記録していた要素を返す
 */
Bredima.History.TreePosition.prototype.trace = function() {
    var point = this.root;
    var i;
    for(i = this.tree.length - 1; i > 0; i-=2) {
	point = point.getRow(this.tree[i]);
	point = point.getRowfactor(this.tree[i-1]);
    }
    if(i == 0) point = point.getRow(this.tree[i]);
    return point;
}

/**
 * @class ポップアップメニュー管理
 * @constructor
 */
Bredima.Menu = function(bredima) {
    this.bredima = bredima;
    var c;
    var m;
    this.menus = {};
    var self = this;
    for(var i = 0; i < this.table.length; i++) {
	c = this.table[i];
	m = new Bredima.Panel.Menu();
	for(var n = 0; n < c.content.length; n++){
	    m.addMenuItem(new Bredima.Parts.Image(c.content[n]));
	}
	this.menus[c.key] = m;
	m.hide();
	bredima.frame.add(m);
	m.observer.attach(this);
    }
    bredima.frame.eventObserver.attach(this);
}

Bredima.Menu.prototype.table = [
    {key: 'fenced', content: ['btn_fenced.png', 'btn_brace.png', 'btn_bracket.png']},
    {key: 'permutation', content: ['btn_permutation.png', 'btn_combination.png', 'btn_homogeneous.png']}
];

Bredima.Menu.prototype.addMenu = function(menu, name){
    this.menus[name] = menu;
    menu.hide();
    this.bredima.frame.add(menu);
    menu.observer.attach(this);
}

/**
 * ポップアップメニューの表示/非表示を切り替え
 * @param {String} name 名前
 * @param {Object} user 使用元/イベントハンドラ呼び出し先
 * @param {Object} position 表示位置
 */
Bredima.Menu.prototype.toggle = function(name, user, position) {
    if(!this.opened) {
	this._open(name, user, position);
    }
    else if((this.name == name) && (this.user == user)) {
	this._close();
    }
    else {
	this._close();
	this._open(name, user, position);
    }
}	

/**
 * メニューを開く
 * @private
 */
Bredima.Menu.prototype._open = function(name, user, position) {
    this.name = name;
    this.user = user;
    this.opened = this.menus[name];
    var offset = Bredima.util.getOffset(user.getDom(), this.bredima.frame.getDom());
    this.opened.setPos(offset.left + position.left, offset.top + position.top);
    this.opened.show();
}

/**
 * メニューを閉じる
 * @private
 */
Bredima.Menu.prototype._close = function() {
    this.opened.hide();
    this.opened = '';
}

/**
 * メニューパネルのクリックを受信
 * @private
 */
Bredima.Menu.prototype.onMenuClick = function(arg) {
    this.user.menuClickPerformed(arg);
    this._close();
}

Bredima.Menu.prototype.onFrameClick = function() {
    if(this.opened) this._close();
}

/**
 * @class 数式全体を管理するオブジェクト　数式のrootに存在
 * @constructor
 */
Bredima.ExpCtrl = function(bredima){
    this.bredima = bredima;
    this.root = new Bredima.Exp.Root(this);
    this.obj = this.root.getDom();
    this.focus = ''; // 現在DOM的にフォーカスがあるオブジェクト
    this.current = ''; // 現在選択されていると見なしているオブジェクト
    this.history = this.bredima.history;
    this.history.observer.attach(this);
    this.observer = new Bredima.Observer();

    var self = this;
    Bredima.util.addListener(this.obj, 'keydown',
			     function(evt) {
				 if(self.focus) {
				     //self.record.commit();  // 前のキーのkeyupの前にkeydownが来ると困るのでやめ
				     if(!self.isProcessing) self.record.cache();
				     self.handleControlCharacter(evt);
				 }
			     });
    Bredima.util.addListener(this.obj, 'keypress',
			     function(evt) {
				 if(self.focus) {
				     var handled = self.handleControlCharacter(evt);
				     if(!handled) self.handleNormalCharacter(evt);
				 }
			     });
    Bredima.util.addListener(this.obj, 'keyup',
			     function(evt) {
				 if(self.focus) {
				     if(self.isProcessing) {
					 self.endRowRec();
					 self._startRec();
				     }
				     else {
					 // 前のcommitと連続している可能性あり
					 var change = self.record.commit('temporary');
					 self.record.cache();
					 if(change) self.history.ackRecordArise();
				     }
				     
				     // 再配置
				     self.focus.repositionToRoot();
				 }
			     });
}

/**
 * 起動時の初期化
 */

Bredima.ExpCtrl.prototype.init = function(rundown, first) {
    if(first != 'first')
	this.startRowRec(this.root.row);

    if(rundown) this.root.row.initFromRundown(rundown);

    this.root.repositionAll();
    this.root.setCursorTo('head');

    if(first != 'first') {
	this.current = this.root.row.getRowfactor(0).getIncomingInput();
	this.endRowRec();
	this._startRec();
    }
}

/* ====================
 オブジェクト挿入処理
 */
/**
 * 数式内のフォーカスがある位置にオブジェクトを挿入
 * 数式内にフォーカスがない状態での処理
 * @param {Function} func コンストラクタ実行関数
 */
Bredima.ExpCtrl.prototype.insElement = function(func) {
    this.startRowRec();

    this._insert(func);

    this.endRowRec();
    this._startRec();
}

/**
 * オブジェクト挿入の中身
 * 1つの履歴の中でオブジェクト挿入をしたい場合はこっち
 * @private
 */
Bredima.ExpCtrl.prototype._insert = function(func) {
    var origin = this.current;
    var ret = origin.split();
    var newelem = func({parent: origin.row, value: ret.selectedValue});
    origin.row.add(newelem, origin.getOrder() + 1);
    
    var ipt;
    if((ipt = newelem.getIncomingInput())) {
	ipt.selectAll();
    }
    else {
	ipt = ret.addedInput.getIncomingInput();
	ipt.setCursorTo('head');
    }
    this.current = ipt;

    // 再配置
    origin.repositionAll();
    newelem.repositionAll();
    ret.addedInput.repositionAll();
    newelem.repositionToRoot();
}

/* ====================
 数式内のフォーカス管理
 */

/**
 * 数式内のフォーカスを設定
 * @param {Object} フォーカスを置くBredima.Exp.Inputインスタンス
 */
Bredima.ExpCtrl.prototype.setFocus = function(input) {
    if(this.focus === input) return; // ウィンドウがアクティブになったときにも飛んでくるっぽいので対処
    this.focus = input;
    if(!this.isProcessing) {
	var prev = this.current;
	this.current = input;
	// 前のinputがhidden表示になるかもしれないので再配置
	if((prev) && (prev != input)) prev.repositionToRoot();

	this._startRec();

	// hiddenからの復帰があるので再配置
	input.repositionToRoot();
    }
}

/**
 * 設定したinput要素からフォーカスが離れたことを連絡
 */
Bredima.ExpCtrl.prototype.leaveFocus = function() {
    this.focus = '';
    if(!this.isProcessing) {
	this._endRec('noselection');
    }
}
    
/**
 * アクセサ - 現在対象となっているinput要素を返す
 */
Bredima.ExpCtrl.prototype.getCurrent = function() {return this.current; }

/* ====================
 数式内のカーソル処理
 */
/**
 * カーソルの横移動
 * @param {Object} rf 横移動元のRowfactor要素
 * @param {String} dir 移動方向 back/fwd
 */
Bredima.ExpCtrl.prototype.slideCursorFrom = function(rf, dir) {
    while(1) {
	var order = rf.getOrder();
	var c = (dir == 'back');
	var loc = (c ? 'tail' : 'head');
	if((c && order == 0) || (!c && order == rf.row.getLength() - 1)) {
	    // カーソルがrowをはみ出す場合
	    if(rf.row.container.constructor.classID == 'exproot')
		return false;
	    else if(rf.row.container.slideCursorFrom(rf.row, dir)) // コンテナオブジェクト内で済んだ場合
	    return true;
	    else
		rf = rf.row.container;
	}
	else{
	    // row内で済む場合
	    var neighbor = rf.row.getRowfactor(order + (c ? -1 : 1))
	    if(neighbor.setCursorTo(loc))
		return true;
	    else
		rf = neighbor;
	}
    }
    return false;
}

/**
 * カーソルの縦移動
 * @param {Object} rf 移動元のRowfactor要素
 * @param {String} dir 移動方向 up/down
 */
Bredima.ExpCtrl.prototype.liftCursorFrom = function(rf, dir) {
    while(1) {
	if(rf.row.container.constructor.classID == 'exproot')
	    return false;
	else if(rf.row.container.liftCursorFrom(rf, dir))
	    return true;
	else
	    rf = rf.row.container;
    }
}

/* ====================
 履歴管理
 */

/**
 * input内の記録を開始
 */
Bredima.ExpCtrl.prototype._startRec = function() {
    this.record = new Bredima.History.InputRecord(this.current);
    this.record.cache();
}

/**
 * input内の記録を終了
 */
Bredima.ExpCtrl.prototype._endRec = function(option) {
    this.record.commit(option);
    if(this.record.close()) this.history.add(this.record);
}

/**
 * rowの記録を開始
 */
Bredima.ExpCtrl.prototype.startRowRec = function(row) {
    this.isProcessing = true;
    this.record = new Bredima.History.RowRecord((row) ? row : this.current.row);
    this.record.commit();
}

/**
 * rowの記録を終了
 */
Bredima.ExpCtrl.prototype.endRowRec = function(nofocus) {
    this.record.commit();
    this.record.close();
    this.history.add(this.record);
    if(nofocus == 'nofocus') this.current.obj.blur();
    var self = this;
    setTimeout(function(){self.isProcessing = false;}, 0);
}

/* ====================
 その他
 */

/**
 * 履歴の更新を受信して自身の更新とする
 * @private
 */
Bredima.ExpCtrl.prototype.onHistoryChange = function() {
    this.observer.notify('ExpressionChange');
    this.bredima.onchange();
}


/* ================================
 キーイベント処理
*/

/**
 * 制御キーの処理
 * ブラウザ依存の関係でkeydownとkeypressの両方で呼ばれる
 * inputオブジェクトが使用
 * @param {Object} input 処理元のinputオブジェクト
 * @param {Event} evt inputが受け取ったイベントオブジェクト
 */
Bredima.ExpCtrl.prototype.handleControlCharacter = function(evt) {
    evt = (evt) ? evt : ((window.event) ? event : null);
    if(!evt) return false;
    // keydownで既にハンドリング済だった場合
    if(evt.type == 'keypress' && this.isHandledInKeydown) {
	if(this.isPreventOnKeydown) Bredima.util.stopEvent(evt);
	this.isHandledInKeydown = false;
	return true;
    }
    this.isPreventOnKeydown = false;
    // 処理対象のキーなら処理
    if(Bredima.ExpCtrl._ctrlCharHandler[evt.keyCode]) {
	var h = Bredima.ExpCtrl._ctrlCharHandler[evt.keyCode];
	if(h.addCnd(evt)) {
	    this.isHandledInKeydown = (evt.type == 'keydown');
	    if(h.position != '') var selpos = this.focus.getSelection();
	    if((h.position == '')
		|| ((h.position == 'head') && (selpos.end == 0))
	       || ((h.position == 'tail') && (selpos.start == this.focus.obj.value.length))) {
		Bredima.util.stopEvent(evt);
		this.isPreventOnKeydown = true;
		h.func.call(this, this.focus);
	    }
	    return true;
	}
    }
    return false;
}

/*
 制御キーのキー別処理内容
*/
Bredima.ExpCtrl._ctrlCharHandler = new Array;

// BackSpace;
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.bs] = {
    addCnd : function(){ return true;},
    position : 'head',
    func : function(input) {
	var order = input.getOrder() - 1;
	if(order >= 0) {
	    if(!this.isProcessing) {
		this._endRec();
		this.startRowRec();
	    }
	    input.row.remove(input.row.getRowfactor(order));
	    if(order > 0) input.mergeWithPrevInput();
	}
    }
};

// ←
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.left] = {
    // Operaで%が誤爆するのでshiftKeyの判定で弾く
    addCnd : function(evt) { return (!evt.shiftKey); }, 
    position : 'head',
    func : function(input) {
	this.slideCursorFrom(input, 'back');
    }
};

// →
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.right] = {
    // 「'」が誤爆するのでOperaのみkeypressで判定
    addCnd : function(evt) { return ((evt.which == undefined) // IEを逃がす
				  || (evt.charCode != undefined) // Mozilla, Safariを逃がす
				  || ((evt.type == 'keypress') && (evt.which != Bredima.consts.keys.right)));},
    position: 'tail',
    func: function(input) {
	this.slideCursorFrom(input, 'fwd');
    }
};

// ↑
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.up] = {
    // Operaで&が誤爆するのでshiftKeyの判定で弾く
    addCnd: function(evt) { return (!evt.shiftKey); },
    position: '',
    func: function(input) {
	this.liftCursorFrom(input, 'up');
    }
};

// ↓
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.down] = {
    // Operaで「)」が誤爆するのでshiftKeyの判定で弾く
    addCnd: function(evt) { return (!evt.shiftKey); },
    position: '',
    func: function(input) {
	this.liftCursorFrom(input, 'down');
    }
};

/**
 * 通常の文字キーの処理
 * inputのkeypressハンドラから呼び出し
 * @param {Object} input 処理元のinputオブジェクト
 * @param {Event} evt inputが受け取ったイベントオブジェクト
 */
Bredima.ExpCtrl.prototype.handleNormalCharacter = function(evt) {
    evt = (evt) ? evt : ((window.event) ? event: null);
    if(!evt) return false;
    var code = (evt.charCode) ? evt.charCode : evt.keyCode;
    //1文字ハンドリング
    var classid = Bredima.ExpCtrl._normalChar[code];
    if(classid) {
	Bredima.util.stopEvent(evt);
	if(classid != 'noaction') {
	    var func = Bredima.Exp.getClassById(classid);
	    if(!this.isProcessing) {
		// inputの履歴を閉じてrowの履歴を開く
		this._endRec();
		this.startRowRec();
	    }
	    this._insert(function(rd){return new func(rd);});
	}
	return true;
    }
    // 文字列ハンドリング
    else {
	var index = 0;
	var selpos = '';
	// 入力文字と文字列の末尾が一致
	while((index = Bredima.ExpCtrl._strSuffix.indexOf(String.fromCharCode(code), index)) >= 0) {
	    var k = Bredima.ExpCtrl._replaceStrings[index];
	    if(!selpos) selpos = this.current.getSelection();
	    var str = this.current.toString();
	    var start = selpos.end - k.str.length + 1;
	    // 全部一致
	    if(str.substring(start, selpos.end) == k.strShort) {
		Bredima.util.stopEvent(evt);
		this.current.init({
				      value: str.substring(0, start) + str.substring(selpos.end, str.length),
				      selection: new Bredima.Range(start, start)
				  });
		// 機能文字列を削除した状態で履歴に記録
		this._endRec('temporary');
		this.startRowRec();
		if(k.op == 'string')
		    this._insert(function(rd){return new Bredima.Token.String(rd, k.str)});
		else
		    this._insert(function(rd){return new (Bredima.Exp.getClassById(k.str))(rd)});
		return true;
	    }
	    index++;
	}
    }
    return false;
}

// 1文字ハンドリング一覧
Bredima.ExpCtrl._normalCharFeatures = [
    {code: 13, make: 'noaction'}, // 殺すだけ
    {code: '(', make: 'fenced'},
    {code: ')', make: 'fenced'},
    {code: '^', make: 'sup'},
    {code: '_', make: 'sub'}
];

Bredima.ExpCtrl._normalChar = new Array;
// 処理時に扱いやすいように整形
(function() {
     for(var i = 0; i < Bredima.ExpCtrl._normalCharFeatures.length; i++) {
	 var k = Bredima.ExpCtrl._normalCharFeatures[i];
	 Bredima.ExpCtrl._normalChar[(typeof k.code == 'number') ? k.code : k.code.charCodeAt(0)] =
	     k.make;
     }
 })();

// 文字列ハンドリング一覧
Bredima.ExpCtrl._replaceStrings = [
    {str: 'sin', op: 'string'},
    {str: 'cos', op: 'string'},
    {str: 'tan', op: 'string'},
    {str: 'log', op: 'string'},
    {str: 'frac', op: 'frac'},
    {str: 'sqrt', op: 'sqrt'},
    {str: 'lim', op: 'lim'}
];

Bredima.ExpCtrl._strSuffix = '';
// 処理時に扱いやすいように整形
(function() {
     for(var i = 0; i < Bredima.ExpCtrl._replaceStrings.length; i++) {
	 var k = Bredima.ExpCtrl._replaceStrings[i];
	 Bredima.ExpCtrl._strSuffix += k.str.charAt(k.str.length - 1);
	 k.strShort = k.str.substring(0, k.str.length - 1);
     }
 })();

/**
 * @class コンテナオブジェクト内で使用する小さいパーツ
 * @constructor
 */
Bredima.Parts = function() {
    this.setClassName('parts');
}

Bredima.util.setProperty(Bredima.Parts, { margin: 0 });

/**
 * 内部で管理しているDOMオブジェクト取得
 */
Bredima.Parts.prototype.getDom = function() {return this.obj; }

/**
 * パーツの位置を設定
 */
Bredima.Parts.prototype.setPos = function(left, top) {
    this.obj.style.left = left + 'px';
    this.obj.style.top = top + 'px';
}

/**
 * サイズを変更
 * @param {Number} width 幅
 * @param {Number} height 高さ
 */
Bredima.Parts.prototype.setSize = function(width, height) {
    Bredima.util.setSize(this.obj, width, height);
}

Bredima.Parts.prototype.addListener = function(name, func) {
    Bredima.util.addListener(this.obj, name, func);
}

/**
 * DOMオブジェクトのクラス名を設定
 * @param {String} list クラス名 StringのArrayでも可
 */
Bredima.Parts.prototype.setClassName = function(list) {
    this.className = (typeof list == 'string') ? new Array(list) : list.slice(0);
    this.obj.className = this.className.join(' ');
}

/**
 * DOMオブジェクトのクラス名を追加で設定
 * 必ず先にsetClassNameを呼んでおくこと
 * @param {String} list クラス名 StringのArrayでも可
 */
Bredima.Parts.prototype.addClassName = function(list) {
    this.className = this.className.concat(list);
    this.obj.className = this.className.join(' ');
}

/**
 * 一時的に付加するクラス名を設定
 * @param {String} list クラス名 StringのArrayでも可
 */
Bredima.Parts.prototype.setClassNameSuffix = function(list) {
    this.suffix = list;
    var list2 = this.className.concat(list);
    this.obj.className = list2.join(' ');
}

/**
 * 一時的に付加していたクラス名を破棄
 */
Bredima.Parts.prototype.removeClassNameSuffix = function() {
    if(this.suffix) {
	this.suffix = '';
	this.obj.className = this.className.join(' ');
    }
}

/**
 * （継承先クラスで使用）自分の再配置
 * 自分のサイズをフィールド値に合わせて表示に反映
 */
/*
Bredima.Parts.prototype._resize = function() {
    Bredima.util.setSize(this.obj, this.width, this.height);
}
*/
/**
 * 表示に必要なサイズを返す
 */
Bredima.Parts.prototype.getPreferredSize = function() {
    return {
	top: (this.top != null) ? (this.top + this.margin) : (this.height / 2 + this.margin),
	bottom: (this.bottom != null) ? (this.bottom + this.margin) : (this.height / 2 + this.margin),
	height: this.height + this.margin * 2,
	width: this.width + this.margin * 2
    };
}

Bredima.style.add(
    ['.exproot .parts', 'position: absolute;']
);
/**
 * @class シンボルを表示する画像か、文字を格納するspanを提供するパーツ
 * @constructor
 * @extends Bredima.Parts
 * @param {Object} bredima bredimaオブジェクト MathMLの判定に使用
 * @param {String} sid 表示するシンボルのID だいたいはMathML名
 * @param {Number} level 表示サイズのレベル
 * @param {String} fixsize 表示サイズをTokenの高さに揃える場合はfixsizeと渡す
 */
Bredima.Parts.Symbol = function(bredima, sid, level, fixsize) {
    if(!Bredima.Parts.Symbol.all[sid]) {
	// 存在しないコードが渡ってきたら刎ねる
	//Bredima.note(sid); // デバッグ用
	return false;
    }
    this.symbolID = sid;
    this.symbol = Bredima.Parts.Symbol.all[this.symbolID];
    this.level = level;
    this.mode = (bredima.getConfig('isMML')) ? 'text' : 'img';
    if(this.mode == 'text') {
	this.obj = document.createElement('span');
	this.obj.appendChild(document.createTextNode(this.symbol.code));
	this.width = Bredima.Token.width[this.level];
	this.height = Bredima.Token.height[this.level];
    }
    else {
	this.obj = document.createElement('img');
	this.obj.src = Bredima.util.getTexURI('\\' + this.symbol.tex, level);
	var s = Bredima.Token.height[this.level];
	var img = this.symbol.img[level];
	this.width = (fixsize) ? s : img.width;
	this.height = (fixsize) ? s : img.height;
	if(fixsize) {
	    this.obj.style.marginTop = img.margin.top + 'px';
	    this.obj.style.marginRight = img.margin.right + 'px';
	    this.obj.style.marginBottom = img.margin.bottom + 'px';
	    this.obj.style.marginLeft = img.margin.left + 'px';
	}
    }
    this.setClassName('parts');
}

Bredima.util.inherit(Bredima.Parts.Symbol, Bredima.Parts);

/**
 * MathML表記を返す
 * @param {Boolean} expand 実体参照でなく実体を渡すか
 */
Bredima.Parts.Symbol.prototype.toMML = function(expand) {
    return (expand) ? this.symbol.code : 
	'&' + ((this.symbol.mathml) ? this.symbol.mathml : this.symbolID) + ';';
}

/**
 * LaTeX表記を返す
 */
Bredima.Parts.Symbol.prototype.toLatex = function() {
    return '\\' + this.symbol.tex;
}

/**
 * シンボルの種類を返す
 */
Bredima.Parts.Symbol.prototype.getGroup = function() {return this.symbol.group;}


Bredima.Parts.Symbol.list = {
    // 演算子
    operator : {
    sdot: {code: '\u22c5', tex: 'cdot', img: ['',
        {width: 8, height: 2, margin: {top: 10, right: 7, bottom: 10, left: 7}},
        {width: 8, height: 2, margin: {top: 8, right: 5, bottom: 8, left: 5}}]},
    times: {code: '\u00d7', tex: 'times', img: ['',
        {width: 8, height: 9, margin: {top: 7, right: 7, bottom: 6, left: 7}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    divide: {code: '\u00f7', tex: 'div', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    PlusMinus: {code: '\u00b1', tex: 'pm', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    MinusPlus: {code: '\u2213', tex: 'mp', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    NotEqual: {code: '\u2260', tex: 'neq', img: ['',
        {width: 16, height: 23, margin: {top: 0, right: 3, bottom: 0, left: 3}},
        {width: 16, height: 19, margin: {top: 0, right: 1, bottom: 0, left: 1}}]},
    lE: {code: '\u2266', tex: 'leq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    gE: {code: '\u2267', tex: 'geq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    sub: {code: '\u2282', tex: 'subset', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    sup: {code: '\u2283', tex: 'supset', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    sube: {code: '\u2286', tex: 'subseteq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    supe: {code: '\u2287', tex: 'supseteq', img: ['',
        {width: 16, height: 15, margin: {top: 4, right: 3, bottom: 3, left: 3}},
        {width: 16, height: 15, margin: {top: 2, right: 1, bottom: 1, left: 1}}]},
    isin: {code: '\u2208', tex: 'in', img: ['',
        {width: 8, height: 11, margin: {top: 6, right: 7, bottom: 5, left: 7}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    ni: {code: '\u220b', tex: 'ni', img: ['',
        {width: 8, height: 11, margin: {top: 6, right: 7, bottom: 5, left: 7}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    cup: {code: '\u222a', tex: 'cup', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    cap: {code: '\u2229', tex: 'cap', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    larr: {code: '\u2190', tex: 'leftarrow', img: ['',
        {width: 16, height: 9, margin: {top: 7, right: 3, bottom: 6, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    rarr: {code: '\u2192', tex: 'rightarrow', img: ['',
        {width: 16, height: 9, margin: {top: 7, right: 3, bottom: 6, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    lArr: {code: '\u21d0', tex: 'Leftarrow', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    rArr: {code: '\u21d2', tex: 'Rightarrow', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    hArr: {code: '\u21d4', tex: 'Leftrightarrow', img: ['',
        {width: 16, height: 11, margin: {top: 6, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    compfn: {code: '\u2218', tex: 'circ', img: ['',
        {width: 8, height: 7, margin: {top: 8, right: 7, bottom: 7, left: 7}},
        {width: 8, height: 6, margin: {top: 6, right: 5, bottom: 6, left: 5}}]},
    ctdot: {code: '\u22ef', tex: 'cdots', img: ['',
        {width: 24, height: 2, margin: {top: 10, right: 0, bottom: 10, left: 0}},
        {width: 24, height: 2, margin: {top: 8, right: 0, bottom: 8, left: 0}}]},
    equiv: {code: '\u2261', tex: 'equiv', img: ['',
        {width: 16, height: 9, margin: {top: 7, right: 3, bottom: 6, left: 3}},
        {width: 16, height: 9, margin: {top: 5, right: 1, bottom: 4, left: 1}}]},
    perp: {code: '\u22a5', tex: 'perp', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]}
    },
    // 識別子
    identifier : {
    alpha: {code: '\u03b1', tex: 'alpha', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 8, height: 7, margin: {top: 6, right: 5, bottom: 5, left: 5}}]},
    beta: {code: '\u03b2', tex: 'beta', img: ['',
        {width: 16, height: 17, margin: {top: 3, right: 3, bottom: 2, left: 3}},
        {width: 16, height: 14, margin: {top: 2, right: 1, bottom: 2, left: 1}}]},
    delta: {code: '\u03b4', tex: 'delta', img: ['',
        {width: 8, height: 14, margin: {top: 4, right: 7, bottom: 4, left: 7}},
        {width: 8, height: 11, margin: {top: 4, right: 5, bottom: 3, left: 5}}]},
    epsiv: {code: '\u025b', tex: 'varepsilon', img: ['',
        {width: 8, height: 8, margin: {top: 7, right: 7, bottom: 7, left: 7}},
        {width: 8, height: 7, margin: {top: 6, right: 5, bottom: 5, left: 5}}]},
    gamma: {code: '\u03b3', tex: 'gamma', img: ['',
        {width: 16, height: 12, margin: {top: 5, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    omega: {code: '\u03c9', tex: 'omega', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    phi: {code: '\u03c6', tex: 'phi', img: ['',
        {width: 16, height: 17, margin: {top: 3, right: 3, bottom: 2, left: 3}},
        {width: 8, height: 14, margin: {top: 2, right: 5, bottom: 2, left: 5}}]},
    phiv: {code: '\u03d5', tex: 'varphi', img: ['',
        {width: 16, height: 12, margin: {top: 5, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 10, margin: {top: 4, right: 1, bottom: 4, left: 1}}]},
    theta: {code: '\u03b8', tex: 'theta', img: ['',
        {width: 8, height: 13, margin: {top: 5, right: 7, bottom: 4, left: 7}},
        {width: 8, height: 11, margin: {top: 4, right: 5, bottom: 3, left: 5}}]},
    Delta: {code: '\u0394', tex: 'Delta', img: ['',
        {width: 16, height: 12, margin: {top: 5, right: 3, bottom: 5, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]},
    pi: {code: '\u03c0', tex: 'pi', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    infin: {code: '\u221e', tex: 'infty', img: ['',
        {width: 16, height: 8, margin: {top: 7, right: 3, bottom: 7, left: 3}},
        {width: 16, height: 7, margin: {top: 6, right: 1, bottom: 5, left: 1}}]},
    bigtriangleup: {code: '\u25b3', tex: 'bigtriangleup', img: ['',
        {width: 16, height: 13, margin: {top: 5, right: 3, bottom: 4, left: 3}},
        {width: 16, height: 11, margin: {top: 4, right: 1, bottom: 3, left: 1}}]}
    },
    // その他パーツ
    other: {
    OverBar: {code: '\u00af', tex: ''},
    sum: {code: '\u2211', tex: 'sum', img: ['',
        {width: 24, height: 25, margin: {top: 0, right: 0, bottom: 0, left: 0}},
        {width: 16, height: 16, margin: {top: 1, right: 1, bottom: 1, left: 1}}]},
	mathint: {code: '\u222b', tex: 'int', mathml: 'int', img: ['',
        {width: 16, height: 40, margin: {top: 0, right: 6, bottom: 0, left: 7}},
        {width: 16, height: 18, margin: {top: 0, right: 1, bottom: 0, left: 1}}]}
	// intが予約語なので避ける
    }
};

Bredima.Parts.Symbol.all = {}; // 初期化時に設定

Bredima.Parts.Symbol.initializeOnLoad = function() {
    // all配列作成
    for(var group in Bredima.Parts.Symbol.list) {
	for(var id in Bredima.Parts.Symbol.list[group]) {
	    Bredima.Parts.Symbol.all[id] = Bredima.Parts.Symbol.list[group][id];
	    Bredima.Parts.Symbol.all[id].group = group;
	}
    }
    Bredima.Init.ackFinish();
}
Bredima.Init.attachInit(Bredima.Parts.Symbol);

/**
 * @class 横ライン部品
 * @constructor
 * @extends Bredima.Parts
 */
Bredima.Parts.Separator = function() {
    this.obj = document.createElement('div');
    this.__super__();
    this.addClassName('separator');
}

Bredima.util.inherit(Bredima.Parts.Separator, Bredima.Parts);

Bredima.style.add(
    ['.separator', 'background-color: black; height: 1px; font-size: 0; width: 100%;']
);
/* ----------------------------------------------
*/
/**
 * @class 画像部品
 * @constructor
 * @extends Bredima.Parts
 */
Bredima.Parts.Image = function(uri) {
    this.obj = new Image;
    this.__super__();
    this.setURI(uri);
    var self = this;
/*
    this.obj.onload = function() {
	self.width = self.obj.width;
	self.height = self.obj.height;
    }
*/
}

Bredima.util.inherit(Bredima.Parts.Image, Bredima.Parts);

/**
 * 画像のURIを変更
 * @param {String} ファイルのパス
 * @param {Boolean} imgディレクトリ以外を利用する場合は何か渡す
 */
Bredima.Parts.Image.prototype.setURI = function(uri, cnd) {
    var prefix = (cnd) ? '' : Bredima.config.get('uri_img');
    this.obj.src = prefix + uri;
}

/* ----------------------------------------------
*/
/**
 * @class 文字を格納したspan
 * @constructor
 * @extends Bredima.Parts
 * @param {String} str 文字
 * @param {Number} level 文字の大きさ
 */
Bredima.Parts.String = function(str, level) {
    this.obj = document.createElement('span');
    this.__super__();
    this.obj.appendChild(document.createTextNode(str));
    this.width = str.length * Bredima.Token.width[level];
    this.height = Bredima.Token.height[level];
}

Bredima.util.inherit(Bredima.Parts.String, Bredima.Parts);

Bredima.Parts.String.prototype.setString = function(str) {
    this.obj.replaceChild(document.createTextNode(str), this.obj.firstChild);
}

/* ----------------------------------------------
*/
/**
 * ボタン
 * @constructor
 * @extends Bredima.Parts
 * @param {Object} parts 中身に入れるパーツ
 * @param {Function} func イベントハンドラ
 */
Bredima.Parts.Button = function(parts, func) {
    this.obj = document.createElement('button');
    this.__super__();
    this.obj.setAttribute('type', 'button');
    //this.obj.type = 'button'; // IEで転けるので使わない
    this.parts = parts;
    this.addListener('click', func);
    this.obj.appendChild(this.parts.getDom());
}

Bredima.util.inherit(Bredima.Parts.Button, Bredima.Parts);

/**
 * ボタンの有効/無効を変更
 * @param {Boolean} b 無効にする場合はtrue
 */
Bredima.Parts.Button.prototype.setDisabled = function(b) {
    this.obj.disabled = b;
    Bredima.util.setOpacity(this.parts.getDom(), (b) ? 0.4 : 1);
}

/* ----------------------------------------------
*/
/**
 * @class ポップアップメニュー表示ボタン
 * @constructor
 * @extends Bredima.Parts
 * @param {Object} bredima bredimaインスタンス
 * @param {String} name 表示するメニュー名
 */
Bredima.Parts.Popup = function(bredima, name, user) {
    this.obj = document.createElement('div');
    this.__super__();
    this.img = new Bredima.Parts.Image('btn_popup.png');
    this.obj.appendChild(this.img.getDom());
    this.width = 8;
    this.height = 8;
    this.user = user;
    var self = this;

    this.addListener('click', function(evt) {
			 bredima.menu.toggle(name, self, {left: 0, top: 8});
			 Bredima.util.stopEvent(evt);
		     });
}

Bredima.util.inherit(Bredima.Parts.Popup, Bredima.Parts);

/**
 * メニュー内の項目がクリックされた際の処理
 * @private
 */
Bredima.Parts.Popup.prototype.menuClickPerformed = function(number) {
    this.user.menuClickPerformed(number);
}   

Bredima.Parts.MenuButton = function(bredima, name) {
    this.obj = document.createElement('span');
    arguments.callee.superClass.apply(this, arguments);
    this.addClassName('menubutton');
    this.img = new Bredima.Parts.Image('btn_' + name + '.png');
    this.obj.appendChild(this.img.getDom());
    var self = this;

    this.addListener('click', function(evt) {
			 bredima.menu.toggle(name, self, {left: 0, top: 28});
			 Bredima.util.stopEvent(evt);
		     });
}

Bredima.util.inherit(Bredima.Parts.MenuButton, Bredima.Parts);

Bredima.Parts.MenuButton.prototype.menuClickPerformed = function(arg) {
}

Bredima.style.add(
    ['.menubutton', 'border: 1px solid gray; padding: 1px 4px; font-size: 28px; background-color: #eee;']
);

/**
 * @class パネル部品（divとパーツ管理）
 * @constructor
 * @extends Bredima.Parts
 */
Bredima.Panel = function(classname) {
    this.obj = document.createElement('div');
    //if(classname) this.obj.className = classname;
    this.isVisible = true;
    //this.observer = new Bredima.Observer();
}

Bredima.util.inherit(Bredima.Panel, Bredima.Parts);

/**
 * 自身にパーツを登録
 * 登録したものは直接自分のフィールドに入る
 * @param {String} key フィールドの名前
 * @param {Object} child パーツ
 */
Bredima.Panel.prototype.add = function(child, key) {
    if(key) this[key] = child;
    this.obj.appendChild(child.getDom());
}

Bredima.Panel.prototype.addDom = function(dom, key) {
    if(typeof dom == 'string') dom = document.createElement(dom);
    if(key) this[key] = dom;
    this.obj.appendChild(dom);
}

/**
 * 表示する
 */
Bredima.Panel.prototype.show = function(arg) {
    this.obj.style.display = 'block';
    this.isVisible = true;
}
    
/**
 * 隠す
 */
Bredima.Panel.prototype.hide = function(arg) {
    this.obj.style.display = 'none';
    this.isVisible = false;
}

/**
 * 表示/非表示の切り替え
 */
Bredima.Panel.prototype.toggleVisibility = function(arg) {
    if(this.isVisible) {
	this.hide(arg);
	return 'hide';
    }
    else {
	this.show(arg);
	return 'show';
    }
}

/**
 * 表示状態を返す
 */
Bredima.Panel.prototype.getVisibility = function() {return this.isVisible; }


/**
 * @class モーダルダイアログ
 * @constructor
 * @extends Bredima.Panel
 */
Bredima.Panel.Dialog = function() {
    arguments.callee.superClass.apply(this, arguments);
    this.setClassName('dialog');

    Bredima.Panel.prototype.hide.apply(this);

    this.addDom('div', 'back');
    this.back.className = 'dialog_padding dialog_back';

    var table = document.createElement('table');
    table.className = 'dialog_padding';
    var td = table.insertRow(0).insertCell(0);
    td.className = 'dialog_outerBox';    
    this.frame = new Bredima.Panel();
    this.frame.setClassName('dialog_frame');
    td.appendChild(this.frame.getDom());
    this.addDom(table);

    var self = this;
    Bredima.util.addListener(table, 'click', function() {self.hide();});

    this.frame.addListener('click', function(evt) {Bredima.util.stopEvent(evt);});
    Bredima.util.setOpacity(this.back, 0);
    Bredima.util.setOpacity(this.frame.getDom(), 0);
}

Bredima.util.inherit(Bredima.Panel.Dialog, Bredima.Panel);

Bredima.Panel.Dialog.prototype.show = function() {
    var self = this;
    Bredima.Panel.prototype.show.apply(this);
    Bredima.util.animation({
			       start: 0, end: 1, step: 4,
			       func: function(num) {
				   Bredima.util.setOpacity(self.back, num * .7);
				   Bredima.util.setOpacity(self.frame.getDom(), num);
			       }
			   });
}

Bredima.Panel.Dialog.prototype.hide = function() {
    var self = this;
    Bredima.Panel.prototype.show.apply(this);
    Bredima.util.animation({
			       start: 1, end: 0, step: 4,
			       func: function(num) {
				   Bredima.util.setOpacity(self.back, num * .7);
				   Bredima.util.setOpacity(self.frame.getDom(), num);
			       },
			       finalize: function() { Bredima.Panel.prototype.hide.apply(self);}
			   });
}

/**
 * （継承先で使用）ダイアログのフレーム内にパーツを追加
 */
Bredima.Panel.Dialog.prototype._addFrame = function(elem) {
    this.frame.add(elem);
}

Bredima.style.add(
    ['.dialog', 'position: absolute; left: 0; top: 0; width: 100%; height: 100%; z-index: 5'],
    ['.dialog .dialog_padding', 'position: absolute; width: 100%; height: 100%;'],
    ['.dialog .dialog_back', 'background-color: #eee2dd; opacity: .7; filter: alpha(opacity=70);'],
    ['.dialog .dialog_outerBox', 'vertical-align: middle;'],
    ['.dialog .dialog_frame', 'margin-left: auto; margin-right: auto; padding: 10px; background-color: white; width: 60%;']
);

/**
 * @classタブ部品っぽいもの
 * @constructor
 * @param {Array} labels 表示するラベルの一覧 中身はStringのみ
 */
Bredima.Panel.Tab = function(labels) {
    this.__super__();
    this.setClassName('tab');
    this.observer = new Bredima.Observer();
    this.labels = new Array;
    this.selected = 0;
    var label;
    var self = this;
    for(var i = 0; i < labels.length; i++) {
	label = document.createElement('span');
	label.appendChild(document.createTextNode(labels[i]));
	Bredima.util.addListener(label, 'click', 
				 (function(num) {return function(evt) {
						     self._click(num, evt);
						 };})(i));
	this.obj.appendChild(label);
	this.labels.push(label);
    }
}

Bredima.util.inherit(Bredima.Panel.Tab, Bredima.Panel);

Bredima.Panel.Tab.prototype.getSelectedNum = function() {
    return this.selected;
}

/**
 * 指定したタブをハイライト
 * @param {Number} num タブの番号
 */
Bredima.Panel.Tab.prototype.hilite = function(num) {
    for(var i = 0; i < this.labels.length; i++) {
	this.labels[i].className = (i == num) ? 'selected' : '';
    }
}

/**
 * タブがクリックされたときのイベントハンドラ
 * @private
 */
Bredima.Panel.Tab.prototype._click = function(num, evt) {
    this.hilite(num);
    Bredima.util.stopEvent(evt);
    this.selected = num;
    this.observer.notify('TabClick', num);
}


Bredima.style.add(
    ['.tab', 'border-bottom: 1px solid #855;'],
    ['.tab span', 'border: 1px solid #855; border-bottom: 0; background-color: linen; padding: 0 6px; margin: 0px 5px; cursor: default;'],
    ['.tab span.selected', 'background-color: #855; color: white;']
);

/**
 * ポップアップメニュー
 * @constructor
 * @extends Bredima.Panel
 */
Bredima.Panel.Menu = function() {
    arguments.callee.superClass.apply(this, arguments);
    this.setClassName('menu');
    this.observer = new Bredima.Observer();
    this.serial = 0;
}

Bredima.util.inherit(Bredima.Panel.Menu, Bredima.Panel);

/**
 * メニュー項目を追加
 * @param {Object} elem 追加するParts
 */
Bredima.Panel.Menu.prototype.addMenuItem = function(elem) {
    var self = this;
    var p = document.createElement('p');
    p.className = 'menu_item';
    Bredima.util.addListener(p, 'click',
			     (function(num) {
				  return function(evt) {
				      Bredima.util.stopEvent(evt);
				      self.observer.notify('MenuClick', num); };
			      })(this.serial++));
    p.appendChild(elem.getDom());
    this.addDom(p);
}

Bredima.style.add(
    ['.menu', 'position: absolute; z-index: 1; border: 1px solid gray; background-color:silver; text-align: center;'],
    ['.menu .menu_item', 'margin: 1px 0; padding: 0; background-color: white; line-height: 0;']
);

/* ================================
*/
/**
 * 2次元配列ポップアップメニュー
 * @constructor
 * @extends Bredima.Panel
 */
Bredima.Panel.MatrixMenu = function() {
    arguments.callee.superClass.apply(this, arguments);
    this.setClassName('menu');
    this.observer = new Bredima.Observer();
    this.serial = 0;
    this.addDom(document.createElement('table'), 'table');
    this.tr = this.table.insertRow(0);
}

Bredima.util.inherit(Bredima.Panel.MatrixMenu, Bredima.Panel);

/**
 * メニューに項目を追加
 * @param {Object} elem 追加するパーツ
 * @param {Any} ret 項目クリック時にメニュー使用者に渡される項目
 */
Bredima.Panel.MatrixMenu.prototype.addMenuItem = function(elem, ret) {
    var td = this.tr.insertCell(-1);
    var self = this;
    td.className = 'menu_item';
    Bredima.util.addListener(td, 'click',
			     (function(ret) {
				  return function(evt) {
				      Bredima.util.stopEvent(evt);
				      self.observer.notify('MenuClick', ret);
				  }
			      })({serial: this.serial++, attr: ret}));
    td.appendChild(elem.getDom());
}

/**
 * メニューに改行を追加
 */
Bredima.Panel.MatrixMenu.prototype.newLine = function() {
    this.tr = this.table.insertRow(-1);
}


/**
 * @class フレーム管理
 * @constructor
 * @extends Bredima.Panel
 */
Bredima.Frame = function(bredima, dom) {
    this.bredima = bredima;
    this.obj = dom;
    this.obj.className = 'BrEdiMa_Frame';
    this.isVisible = true;

    // loading中の目隠し
/*    this.addDom('div', 'curtain');
    this.curtain.className = 'curtain';
    var note = document.createElement('p');
    note.appendChild(document.createTextNode('loading'));
    this.curtain.appendChild(note);
*/
    this.eventObserver = new Bredima.Observer();
    var self = this;
    this.addListener('click', function() {self.eventObserver.notify('FrameClick');});

    // About表示関係
    this.add(new Bredima.Frame.About(bredima), 'about');
    var logo = new Bredima.Parts.Image('logo.png');
    logo.addClassName('bredima_logo');
    logo.addListener('click', function() { bredima.frame.about.show(); });
    this.add(logo);

    this.add(new Bredima.Frame.Source(bredima), 'source');
    this.add(new Bredima.Frame.Preview(bredima), 'preview');

    if(bredima.getConfig('use_button'))
	this.add(new Bredima.Frame.CtrlButtons(bredima), 'buttons');
}

Bredima.util.inherit(Bredima.Frame, Bredima.Panel);

/**
 * 他のクラスの初期化が終わってから行う処理
 */
Bredima.Frame.prototype.finalize = function() {
    this.add(new Bredima.Frame.Toolbar(this.bredima), 'toolbar');
    this.add(this.bredima.exp.root);

    // 数式更新のタイミングでプレビューのアップデートをnotify
    this.bredima.exp.observer.attach(this.preview);
}

/**
 * ロード中の目隠しを取り去る
 */
Bredima.Frame.prototype.appear = function() {
    var self = this;
    Bredima.util.animation({
			       start: 1, end: 0, step: 4,
			       func: function(num) {
				   Bredima.util.setOpacity(self.curtain, num);
			       },
			       finalize: function(){self.curtain.style.display = 'none';}
			   });
}
    
Bredima.style.add(
    [' ', 'position: relative; border: 1px solid #dcc; background-color: white; color: black; font-size: 14px; line-height: 1; padding: 2px; height: 200px; text-align: left;'],
    ['.curtain', 'position: absolute; width: 100%; height: 100%; left: 0; top: 0; background-color: #f2eeee; z-index: 10; text-align: center; vertical-align: bottom;'],
    ['.curtain p', 'position: absolute; bottom: 30%; left: 60%;'],
    ['.bredima_logo', 'position: absolute; top: 3px; right: 3px;']
);


Bredima.Frame.CtrlButtons = function(bredima) {
    this.__super__();

    this.setClassName('ctrlbuttons')

    var btn = new Bredima.Parts.Button(new Bredima.Parts.String('挿入'), 
		       function() {
			   bredima.frame.hide();
			   bredima.onsubmit();
		       });
    this.add(btn);
}

Bredima.util.inherit(Bredima.Frame.CtrlButtons, Bredima.Panel);

Bredima.style.add(
    ['.ctrlbuttons', 'position: absolute; right: 2px; bottom: 2px; width: 15%; text-align: center;']
);


/**
 * @class コンテナオブジェクト挿入ボタン管理オブジェクト
 * @constructor
 * @extends Bredima.Panel
 */
Bredima.Frame.Toolbar = function(bredima) {
    this.__super__(bredima);
    this.setClassName('toolbar');

    // 操作ボタン
    this.add(new Bredima.Parts.Button(
		 new Bredima.Parts.Image('btn_undo.png'),
		 function() {bredima.history.undo();}),
	     'undo');
    this.add(new Bredima.Parts.Button(
		 new Bredima.Parts.Image('btn_redo.png'),
		 function() {bredima.history.redo();}),
	     'redo');
    this.undo.setDisabled(true);
    this.redo.setDisabled(true);
    this.add(new Bredima.Parts.Button(
		 new Bredima.Parts.Image('btn_source.png'),
		 function() {bredima.frame.source.show(); }
	     ));
    
    this.addDom(document.createTextNode(' '));

    // 使用頻度が高そうなコンテナオブジェクト
    for(var i = 0; i < this.order.length; i++) {
	var curr = Bredima.Frame.Toolbar.expandContainerNotation(this.order[i], bredima);
	this.add(new Bredima.Parts.Button(curr.img, curr.func2));
    }

    // ポップアップメニュー
    this.add(new Bredima.Frame.Toolbar.Sub(bredima));
    this.add(new Bredima.Frame.Toolbar.Symbol(bredima, 'operator'));
    this.add(new Bredima.Frame.Toolbar.Symbol(bredima, 'identifier'));

    bredima.history.observer.attach(this);
}

Bredima.util.inherit(Bredima.Frame.Toolbar, Bredima.Panel);

/**
 * コンテナオブジェクト省略記法からボタン画像と挿入関数を作って返すクラス関数
 */
Bredima.Frame.Toolbar.expandContainerNotation = function(arg, bredima) {
    var classfunc; var img; var func;
    if(typeof arg == 'string') {
	classfunc = Bredima.Exp.getClassById(arg)
	img = arg;
	func = function(rd){return new classfunc(rd); };
    }
    else {
	classfunc = Bredima.Exp.getClassById(arg.id);
	img = (arg.img) ? arg.img : arg.id;
	func = function(rd){return new classfunc(rd, arg.arg)};
    }
    return {
	img: new Bredima.Parts.Image('btn_' + img + '.png'),
	func: func,
	func2: (bredima) ? function() {bredima.exp.insElement(func)} : null
    }
}

// やっぱり集中管理できた方が楽そうなのでここでボタンの列びを指定
Bredima.Frame.Toolbar.prototype.order = ['frac', {id: 'fenced', arg: '0'}, 'sqrt', 'sup', 'sub'];

/**
 * 履歴に変化があった場合の処理
 * @private
 */
Bredima.Frame.Toolbar.prototype.onHistoryEnableChange = function(name) {
    var btn = ((name.indexOf('undo') >= 0) ? this.undo : 
	       ((name.indexOf('redo') >= 0) ? this.redo : ''));
    if(btn) btn.setDisabled((name.indexOf('disable') >= 0) ? true : false);
}


Bredima.style.add(
    ['.toolbar button', 'font-size: 12px; padding: 1px 2px;']
);

/* ================================
*/
/**
 * @class 使用頻度が低そうなコンテナオブジェクトをまとめたポップアップメニュー
 * @constructor
 * @extends Bredima.Parts.MenuButton
 */
Bredima.Frame.Toolbar.Sub = function(bredima) {
    this.bredima = bredima;
    var menu = new Bredima.Panel.MatrixMenu();
    for(var row = 0; row < this.order.length; row++) {
	for(var col = 0; col < this.order[row].length; col++) {
	    var curr = Bredima.Frame.Toolbar.expandContainerNotation(this.order[row][col]);
	    menu.addMenuItem(curr.img, curr.func);
	}
	menu.newLine();
    }
    bredima.menu.addMenu(menu, 'container');
    this.__super__(bredima, 'container');
}

Bredima.util.inherit(Bredima.Frame.Toolbar.Sub, Bredima.Parts.MenuButton);

Bredima.Frame.Toolbar.Sub.prototype.order = [
    ['root', 'over', 'vector'],
    ['sum', 'integral', 'intinterval'],
    [{id: 'permutation', arg: '0'}, {id: 'permutation', arg: 1, img: 'combination'}, {id: 'permutation', arg: 2, img: 'homogeneous'}]
];

/**
 * メニューがクリックされた時の処理
 */
Bredima.Frame.Toolbar.Sub.prototype.menuClickPerformed = function(arg) {
    this.bredima.exp.insElement(arg.attr);
}

/* ================================
*/
/**
 * @class シンボル挿入ボタン
 * @constructor
 * @extends Bredima.Parts.MenuButton
 * @param {String} id シンボルの種類 operator/identifier
 */
Bredima.Frame.Toolbar.Symbol = function(bredima, id) {
    this.bredima = bredima;
    var menu = new Bredima.Panel.MatrixMenu();
    for(var row = 0; row < this.order[id].length; row++) {
	for(var col = 0; col < this.order[id][row].length; col++) {
	    var curr = this.order[id][row][col];
	    menu.addMenuItem(
		new Bredima.Parts.Symbol(bredima, curr, 1, 'fixsize'),
		(function(name) {
		     return function(rd) {return new Bredima.Token.Symbol(rd, name);};
		 })(curr));
	}
	menu.newLine();
    }
    bredima.menu.addMenu(menu, id);
    this.__super__(bredima, id);
}

Bredima.util.inherit(Bredima.Frame.Toolbar.Symbol, Bredima.Parts.MenuButton);

Bredima.Frame.Toolbar.Symbol.prototype.order = {
    operator: [
	['sdot', 'times', 'divide', 'PlusMinus', 'MinusPlus', 'NotEqual', 'lE', 'gE'],
	['sub', 'sup', 'sube', 'supe', 'isin', 'ni', 'cup', 'cap'],
	['larr', 'rarr', 'lArr', 'rArr', 'hArr'],
	['compfn', 'ctdot', 'equiv', 'perp']
    ],
    identifier: [
	['alpha', 'beta', 'delta', 'epsiv'],
	['gamma', 'omega', 'phi', 'phiv'],
	['theta', 'Delta', 'pi'],
	['infin', 'bigtriangleup']
    ]
};

/**
 * メニューがクリックされた時の処理
 */
Bredima.Frame.Toolbar.Symbol.prototype.menuClickPerformed = function(arg) {
    this.bredima.exp.insElement(arg.attr);
}

/**
 * @class ソース表示部品
 * @constructor
 * @extends Bredima.Panel.Dialog
 */
Bredima.Frame.Source = function(bredima) {
    this.__super__();
    this.frame.addClassName('source_frame');

    this.getter = [
	function(){return bredima.toMML()},
	function(){return bredima.toLatex()},
	function(){return bredima.toJSON()}
    ];
    var label = ['MathML', 'LaTeX'];
    if(bredima.getConfig('showJson') == true) label.push('JSON');
    this.tab = new Bredima.Panel.Tab(label);
    this.tab.observer.attach(this);
    this.tab.hilite(0);
    this.frame.add(this.tab);

    // 実際に表示する場所
    this.field = document.createElement('textarea');
    this.field.className = 'source_field';
    this.frame.addDom(this.field);
}

Bredima.util.inherit(Bredima.Frame.Source, Bredima.Panel.Dialog);

Bredima.Frame.Source.prototype.show = function() {
    Bredima.util.applySuper('show', this);
    this.field.value = this.getter[this.tab.getSelectedNum()]();
}

/**
 * ソース種類切り替えタブのクリックを受信
 * @private
 */
Bredima.Frame.Source.prototype.onTabClick = function(num) {
    this.field.value = this.getter[num]();
}

Bredima.style.add(
    ['.source_field', 'width: 100%; height: 10em;']
);

/**
 * @class プレビュー部品
 * @constructor
 * @extends Bredima.Panel
 * @param {Object} bredima bredima
 */
Bredima.Frame.Preview = function(bredima) {
    this.__super__();
    this.bredima = bredima;
    this.isMML = bredima.getConfig('isMML');
    this.timer = ''; // 自動プレビューのためのタイマー
    var self = this;
    this.setClassName('preview');

    //ボタン表示の場合は表示領域を明け渡す
    if(bredima.getConfig('use_button'))
	this.obj.style.width = '85%';

    // タブ部品で表示状態の切り替え
    this.add(new Bredima.Panel.Tab(['Preview']), 'tab');
    this.tab.observer.attach(this);

    // 非表示時に隠すものはframe内に
    var frame = new Bredima.Panel();
    frame.setClassName('preview_frame');
    frame.hide();

    var toolbar = new Bredima.Panel();
    toolbar.setClassName('preview_toolbar');

    // Updateボタン
    toolbar.add(new Bredima.Parts.Button(new Bredima.Parts.Image('btn_reload.png'),
					    function() {self.preview(); }), 'button');

    // 自動更新チェック
    var t = document.createElement('input');
    t.type = 'checkbox'; // DOMツリーに組み込む前でないとIEでエラーが出る
    toolbar.addDom(t, 'check');
    toolbar.addDom(document.createTextNode('Automatic'));

    frame.add(toolbar, 'toolbar');

    // 表示領域
    frame.add(new Bredima.Panel(), 'display');
    frame.display.setClassName('preview_display');
    if(!this.isMML) {
	this.image = document.createElement('img');
	frame.display.addDom(this.image, 'img');
    }

    this.add(frame, 'frame');
}

Bredima.util.inherit(Bredima.Frame.Preview, Bredima.Panel);

/**
 * 実際にプレビューを領域内に表示
 */
Bredima.Frame.Preview.prototype.preview = function() {
    if(this.isMML)
	this.frame.display.innerHTML = this.bredima.toMML('expand');
    else
	this.image.src = Bredima.util.getTexURI(this.bredima.toLatex(), 'Large');
}

/**
 * タブを押した際の処理
 * @private
 */
Bredima.Frame.Preview.prototype.onTabClick = function() {
    var self = this;
    if(this.frame.getVisibility()) {
	clearTimeout(this.timer);
	Bredima.util.smoothAnimation( {
				    start: 70, end: 1, step: 6,
				    func: function(num) { self.frame.getDom().style.height = Math.floor(num) + 'px';},
				    finalize: function() {
					self.frame.hide();
					self.tab.hilite(-1);
				    }
				});
    }
    else {
	this.frame.getDom().style.height = '1px';
	this.frame.show();
	this.preview();
	Bredima.util.smoothAnimation({
				   start: 1, end: 70, step: 6,
				   func: function(num) {self.frame.getDom().style.height = Math.floor(num) + 'px';}
			       });
    }
}

/**
 * 数式の変更を受信した際の処理
 * 受信後0.5秒後にプレビュー更新
 * @private
 */
Bredima.Frame.Preview.prototype.onExpressionChange = function() {
    if(this.frame.getVisibility() && this.frame.toolbar.check.checked) {
	var self = this;
	clearTimeout(this.timer);
	this.timer = setTimeout(function() {self.preview();}, 500);
    }
}

Bredima.style.add(
    ['.preview', 'position: absolute; bottom: 2px; width: 99%; z-index: 1;'],
    ['.preview .preview_frame', 'background-color: #eee; padding: 5px; position: relative; overflow: hidden;'],
    ['.preview .preview_display', 'background-color: white; padding: 10px;'],
    ['.preview .preview_toolbar', 'position: absolute; right: 5px; top: 0px; padding: 4px 10px; background-color: #eee;']
);

Bredima.Frame.About = function(bredima) {
    this.__super__();
    this.frame.addClassName('about');

    this.frame.getDom().innerHTML = '<p>BrEdiMa</p><p>Version ' + Bredima.version + '</p><p><a href="http://bredima.sourceforge.jp/" target="_blank">http://bredima.sourceforge.jp</a></p>';

    //this.frame.getDom().onclick = '';
}

Bredima.util.inherit(Bredima.Frame.About, Bredima.Panel.Dialog);


/**
 * @class 数式内に乗る全要素の共通抽象クラス
 * @constructor
 * @extends Bredima.Parts
 */
Bredima.Exp = function(param) {
    if(this.domName) this.obj = document.createElement(this.domName);
    this.setClassName(['expelem']);
    if(this.constructor.classID != this.constructor.objType) this.addClassName(this.constructor.objType);
    this.addClassName(this.constructor.classID);
}

Bredima.util.inherit(Bredima.Exp, Bredima.Parts);

// 数式内に乗る要素の標準DOM要素はdiv。違う場合はプロパティ上書き。
Bredima.util.setProperty(Bredima.Exp, {domName: 'div'});

/* ====================
 Expインタフェース
   -----------------------------
 オブジェクト情報入出力関係
*/
/**
 * オブジェクトの文字列表記を返す 通常は概要オブジェクトを文字列化したもの
 */
Bredima.Exp.prototype.toString = function() {
    return JSON.stringify(this.toRundown);
}

/**
 * 概要オブジェクトを返す
 * 標準ではオブジェクトのID+_toRundownContent
 */
Bredima.Exp.prototype.toRundown = function() {
    return new Array().concat(this.constructor.classID, this._toRundownContent());
}

/**
 * 概要オブジェクトの中身を返す
 * 標準ではtoStringそのまま
 */
Bredima.Exp.prototype._toRundownContent = function() {
    return this.toString();
}

/**
 * MathML文字列を返す タグを付与するだけで実際の操作は_toMML()
 */
Bredima.Exp.prototype.toMML = function(expand, param) {
    if(this.MML) return Bredima.util.addTag(this.MML, this._toMMLContent(expand, param));
}
    
/**
 * 仮想関数 - MathML文字列中のタグの中身を返す
 * 標準ではtoStringそのまま
 */
Bredima.Exp.prototype._toMMLContent = function() {
    return this.toString();
}

/**
 * LaTeX文字列を返す コマンドを付与するだけで実際の操作は_toLatexContent()
 */
Bredima.Exp.prototype.toLatex = function() {
    if(this.tex) return '\\' + this.tex + this._toLatexContent();
}

/**
 * 仮想関数 - LaTeX文字列の中の先頭のコマンドを除いた部分を返す
 */
Bredima.Exp.prototype._toLatexContent = function() { Bredima.util.error(); }

/* -------------------------------
 レイアウト関係
*/
/**
 * 自分の配下にある全オブジェクトの再配置
 */
Bredima.Exp.prototype.repositionAll = function() {
    // 自分の配下に別のオブジェクトを持つ場合はオーバーライドする
    this._reposition();
}

/**
 * （仮想関数）Rootに向かって再配置
 */
Bredima.Exp.prototype.repositionToRoot = function() { Bredima.util.error(); }

/**
 * （仮想関数）自分の再配置
 */
Bredima.Exp.prototype._reposition = function() {
}

/**
 * （継承先クラスで使用）自分の再配置
 * 自分のサイズをフィールド値に合わせて表示に反映
 */
Bredima.Exp.prototype._resize = function() {
    if(this.top != null) this.height = this.top + this.bottom;
    Bredima.util.setSize(this.obj, this.width, this.height);
}

/* ====================
 クラス関数
 */

Bredima.Exp.classes = new Array;

/**
 * 数式内で利用するクラスを登録
 * @params {Function} func コンストラクタ関数
 * @params {String} type クラスの分類
 * @params {String} id クラスのID（オプション）
 */
Bredima.Exp.register = function(func, type, id) {
    func.classID = (id) ? id : type;
    func.objType = type;
    Bredima.Exp.classes.push(func);
}

/**
 * IDに該当するクラスを返す
 * @params {String} id ID
 */
Bredima.Exp.getClassById = function(id) {
    for(var i = 0; i < Bredima.Exp.classes.length; i++) {
	if(Bredima.Exp.classes[i].classID == id) {
	    return Bredima.Exp.classes[i];
	}
    }
    return null;
}

Bredima.style.add(
    ['.expelem', 'position: absolute;']
);

Bredima.Exp.Root = function(exp) {
    this.exp = exp;
    this.__super__();
    this.obj.className = 'exproot';

    this.row = new Bredima.Exp.Row({parent: this, level: 1});
    this.obj.appendChild(this.row.getDom());
}

Bredima.util.inherit(Bredima.Exp.Root, Bredima.Exp);

Bredima.Exp.register(Bredima.Exp.Root, 'container', 'exproot');

Bredima.Exp.Root.prototype.toRundown = function() {
    return this.row.toRundown();
}

Bredima.Exp.Root.prototype.toMML = function(expand) {
    var out = '<math xmlns="http://www.w3.org/1998/Math/MathML" display="block">' + "\n";
    out += this.row.toMML(expand);
    out += '</math>' + "\n";
    return out;
}

Bredima.Exp.Root.prototype.toLatex = function() {
    return this.row.toLatex('noFence');
}

Bredima.Exp.Root.prototype.repositionAll = function() {
    this.row.repositionAll();
}

Bredima.Exp.Root.prototype.repositionToRoot = function() {
}

Bredima.Exp.Root.prototype.orderOf = function() {
    return 0;
}

Bredima.Exp.Root.prototype.getRow = function() {
    return this.row;
}

Bredima.Exp.Root.prototype.setCursorTo = function(loc) {
    this.row.setCursorToRow(loc);
}

Bredima.style.add(
    ['.exproot', 'position: relative;']
);

/**
 * @class mrow管理オブジェクト
 * @constructor
 * @extends Bredima.Exp
 * @param {Object} container このmrowを含むコンテナオブジェクトインスタンス
 * @param {Undef} def 初期値　文字列か概要オブジェクト
 * @param {Number} level mrow表示サイズのレベル(1/2)
 */
Bredima.Exp.Row = function(param) {
    this.container = param.parent;
    this.level = param.level;
    this.__super__(param);
    this.rowfactors = new Array;

    if(param.rundown)
	this.initFromRundown(param.rundown);
    else
	this.add(new Bredima.Token.Input({parent: this, value: param.value}));
}

Bredima.util.inherit(Bredima.Exp.Row, Bredima.Exp);

Bredima.util.setProperty(Bredima.Exp.Row, {
    MML: 'mrow',
    margin: 2
});

Bredima.Exp.register(Bredima.Exp.Row, 'row');

/* ====================
 Exp.Rowで定義
 */
/**
 * 概要オブジェクトより初期化
 * @param {Array} rundown 概要オブジェクト
 */
Bredima.Exp.Row.prototype.initFromRundown = function(rundown) {
    var t, rowfactor;
    for(var i = this.rowfactors.length - 1; i >= 0; i--) {
	this.remove(this.rowfactors[i]);
    }
    for(var i = 0; i < rundown.length; i++) {
	t = rundown[i];
	if(typeof t == 'number' || typeof t == 'string') {
	    this.add(new Bredima.Token.Input({parent: this, value: t}));
	}
	else if(typeof t == 'object'){
	    var func = Bredima.Exp.getClassById(t[0]);
	    if(func && (func.objType == 'container' || func.objType == 'token')) {
		this.add(new func({parent: this, rundown: t.slice(1)}));
	    }
	}
    }
}

/* ----------------------------
 配列要素操作関係
 */
/**
 * rowfactors配列に渡されたオブジェクトを追加する だけ 内部側からの呼び出し専用
 * @param {Object} rowfactor 配置するオブジェクト
 */
Bredima.Exp.Row.prototype.add = function(rowfactor, order) {
    if(!order)
	this.rowfactors.push(rowfactor);
    else {
	var at = this.rowfactors.splice(0, order);
	this.rowfactors = at.concat(rowfactor, this.rowfactors);
    }
    this.obj.appendChild(rowfactor.getDom());
}

/**
 * mrow内の指定したオブジェクトを削除する だけ 内部側からの呼び出し専用
 * @param {Object} rowfactor 削除するオブジェクト
 */
Bredima.Exp.Row.prototype.remove = function(rowfactor) {
    this.obj.removeChild(rowfactor.getDom());
    this.rowfactors.splice(this.orderOf(rowfactor), 1);
}

/**
 * オブジェクトが配置されている位置を返す
 * @param {Object} rowfactor 位置を調べたいオブジェクト
 */
Bredima.Exp.Row.prototype.orderOf = function(rowfactor) {
    for(var i = 0; i < this.rowfactors.length; i++) {
	if(this.rowfactors[i] == rowfactor) return i;
    }
    return null;
}

/**
 * row内の配列の要素数を返す
 */
Bredima.Exp.Row.prototype.getLength = function() { return this.rowfactors.length; }

/**
 * row内の指定位置の要素を返す
 * @param {Number} order 位置
 */
Bredima.Exp.Row.prototype.getRowfactor = function(order) { return this.rowfactors[order]; }

/* -----------------------------
 他
 */
/**
 * mrowオブジェクトのheadかtailにカーソルを置く
 * @param {String} loc カーソルを置く位置 head / tail
 */
Bredima.Exp.Row.prototype.setCursorToRow = function(loc) {
    this.rowfactors[((loc == 'head') ? 0 : (this.rowfactors.length - 1))].setCursorTo(loc);
}

Bredima.Exp.Row.prototype.getOrder = function() { return this.container.orderOf(this);}

Bredima.Exp.Row.prototype.getLevel = function() { return this.level; }

/* ====================
 Expの実装
   -----------------------------
 オブジェクト情報入出力関係
 */

Bredima.Exp.Row.prototype.toRundown = function() {
    var ret = new Array;
    for(var i = 0; i < this.rowfactors.length; i++) {
	ret.push(this.rowfactors[i].toRundown());
    }
    return ret;
}

Bredima.Exp.Row.prototype._toMMLContent = function(expand) {
    var out = new Array;
    var rf;
    var mml;
    for(var i = 0; i < this.rowfactors.length; i++) {
	rf = this.rowfactors[i];
	mml = rf.toMML(expand, (rf.isMMLNeedPrev) ? out.pop() : null);
	if(mml != '') out = out.concat(mml);
    }
    return out.join('');
}

Bredima.Exp.Row.prototype.toLatex = function(nofence) {
    var out = new Array;
    for(var i = 0; i < this.rowfactors.length; i++)
	out.push(this.rowfactors[i].toLatex());
    if(nofence)
	return out.join(' ');
    else
	return '{' + out.join(' ') + '}';
}

/* -----------------------------
 レイアウト関係
 */

Bredima.Exp.Row.prototype.repositionAll = function() {
    for(var i = 0; i < this.rowfactors.length; i++) {
	this.rowfactors[i].repositionAll();
    }
    this._reposition();
}

Bredima.Exp.Row.prototype.repositionToRoot = function() {
    this._reposition();
    this.container.repositionToRoot();
}

Bredima.Exp.Row.prototype._reposition = function() {
    var tmpsize = new Array;
    this.width = 0;
    this.top = 0;
    this.bottom = 0;
    for(var i = 0; i < this.rowfactors.length; i++) {
	var s = this.rowfactors[i].getPreferredSize();
	this.top = Math.max(this.top, s.top);
	this.bottom = Math.max(this.bottom, s.bottom);
	this.width += s.width;
	tmpsize[i] = s;
    }
    this.width += (this.rowfactors.length - 1) * 4 // 内容間の隙間
    this.height = this.top + this.bottom;
    this._resize();
    var left = 0;
    for(i = 0; i < this.rowfactors.length; i++) {
	this.rowfactors[i].setPos(left, this.top - tmpsize[i].top);
	left += (4 + tmpsize[i].width);
    }
}

Bredima.style.add(
    ['.row', 'margin: 2px;']
);
/**
 * @class Rowfactorオブジェクトの共通抽象クラス
 * @constructor
 * @extends Bredima.Exp
 * this.rowの設定とrowのparam.order位置への挿入
 */
Bredima.Exp.Rowfactor = function(param) {
    arguments.callee.superClass.apply(this, arguments);
    this.row = param.parent;
}

Bredima.util.inherit(Bredima.Exp.Rowfactor, Bredima.Exp);

Bredima.util.setProperty(Bredima.Exp.Rowfactor, {
    margin: 1
});

/* ====================
 Exp.Rowfactorで定義
 */

/**
 * 要素のheadかtailにカーソルを置く
 * オーバーライドしなければカーソルが置けないrowfactor
 */
Bredima.Exp.Rowfactor.prototype.setCursorTo = function(loc) {
    return false;
}

/**
 * 選択されるべきinput要素を返す
 * オーバーライドしなければ選択できない
 */
Bredima.Exp.Rowfactor.prototype.getIncomingInput = function() {
    return false;
}

/**
 * rowの中の自分の位置を返す
 */
Bredima.Exp.Rowfactor.prototype.getOrder = function() {
    return this.row.orderOf(this);
}

/* ====================
 Expの実装
*/
Bredima.Exp.Rowfactor.prototype.repositionToRoot = function() {
    this._reposition();
    this.row.repositionToRoot();
}


/**
 * @class トークン要素抽象クラス
 * @extends Bredima.Exp.Rowfactor
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Token = function(param) {
    arguments.callee.superClass.apply(this, arguments);
    this.height = Bredima.Token.height[this.row.getLevel()];
    this.addClassName('token_level' + this.row.getLevel());
}

Bredima.util.inherit(Bredima.Token, Bredima.Exp.Rowfactor);

Bredima.Token.width = [0, 10, 7];
Bredima.Token.height = [0, 22,18];

/**
 * （継承先で使用）heightはスタイルシートの指定から変更がないのでwidthのみ変更
 * 
 */
Bredima.Token.prototype._resizeWidth = function() {
    this.obj.style.width = this.width + 'px';
}

/* -----------------
 Expの実装
 */
Bredima.Token.prototype._toLatexContent = function() {
    // シンボルのみなら基本的にコマンドのみ
    return '';
}

Bredima.style.add(
    ['.token', "border: 1px dashed darkturquoise; font-family: 'Courier New', monospace; text-align: center;"],
    ['.token_level1', 'font-size: 16px; height: 22px; line-height: 22px;'],
    ['.token_level2', 'font-size: 12px; height: 18px; line-height: 18px;'],
    ['.token .parts', 'position: static;']
);
Bredima.style.add(
);


/* ===========================
 サブクラス
*/
/**
 * @class mi/moオブジェクト
 * @extends Bredima.Token
 * @param {Object} param オブジェクト構築共通パラメタ
 * @param {Object} sid シンボルID 大抵はMathML名
 */
Bredima.Token.Symbol = function(param, sid) {
    this.__super__(param);
    this.symbolID = (sid) ? sid : ((param.rundown) ? param.rundown[0] : '');
    this.symbol = new Bredima.Parts.Symbol(this.row.container.exp.bredima, this.symbolID, this.row.level, 'fixsize');
    this.MML = (this.symbol.getGroup() == 'operator') ? 'mo' : 'mi';
    this.width = Bredima.Token.height[this.row.level];
    this.obj.appendChild(this.symbol.obj);

    this._resizeWidth();
}

Bredima.util.inherit(Bredima.Token.Symbol, Bredima.Token);

Bredima.Exp.register(Bredima.Token.Symbol, 'token', 'symbol');

/* ------------------
 Expの実装
 */
/**
 * 文字列表記としてシンボルのIDを返す
 * 概要オブジェクトの中身もこの値
 */
Bredima.Token.Symbol.prototype.toString = function() {
    return this.symbolID;
}

Bredima.Token.Symbol.prototype._toMMLContent = function(expand) {
    return this.symbol.toMML(expand);
}

Bredima.Token.Symbol.prototype.toLatex = function() {
    return this.symbol.toLatex();
}

/* ===========================
*/
/**
 * @class sinとか
 * @extends Bredima.Token
 * @param {Object} param オブジェクト構築共通パラメタ
 * @param {Strin} val 表示する文字列
 */
Bredima.Token.String = function(param, val) {
    this.__super__(param);
    this.value = (val) ? val : ((param.rundown) ? param.rundown[0] : '');
    this.tex = this.value;
    this.content = new Bredima.Parts.String(this.value, this.row.getLevel());
    this.width = this.content.width;
    this.obj.appendChild(this.content.getDom());
    
    this._resizeWidth();
}

Bredima.util.inherit(Bredima.Token.String, Bredima.Token);

Bredima.util.setProperty(Bredima.Token.String, {
    MML: 'mi'
});

Bredima.Exp.register(Bredima.Token.String, 'token', 'string');

Bredima.Token.String.prototype.toString = function() {
    return this.value;
}


/**
 * @class inputフォーム管理オブジェクト（mi, mo, mn相当）
 * @param {Object} param オブジェクト構築共通パラメタ
 * @extends Bredima.Token
 * @constructor
 */
Bredima.Token.Input = function(param) {
    this.__super__(param);
    var self = this;
    var exp = this.row.container.exp;

    Bredima.util.addListener(this.obj, 'focus', 
			     function(evt) {
				 exp.setFocus(self);
			     });
    Bredima.util.addListener(this.obj, 'blur',
			     function(evt) {
				 exp.leaveFocus();
			     });
    // ToDo 特殊なinputへの対応
    if(param.value) this.obj.value = param.value;
}

Bredima.util.inherit(Bredima.Token.Input, Bredima.Token);

Bredima.util.setProperty(Bredima.Token.Input, {
			     domName: 'textarea'
			 });

Bredima.Exp.register(Bredima.Token.Input, 'token', 'input');

/* ====================
 Token.Inputで定義
*/
/**
 * 指定した値と選択範囲にinputを初期化
 * @param {Object} arg 初期化内容を内包したオブジェクト value:文字, selction: 選択範囲
 */
Bredima.Token.Input.prototype.init = function(arg) {
    if(typeof(arg) == 'object') {
	this.obj.value = arg.value;
	this.selectRange(arg.selection);
    }
}

/**
 * inputの選択範囲を取得
 * @return Object Bredima.Range型
 */
Bredima.Token.Input.prototype.getSelection = function() {
    if(this.obj.selectionStart != null) {
	start = this.obj.selectionStart;
	end = this.obj.selectionEnd;
    }
    else {
	this.obj.focus();
	var docRange = document.selection.createRange();
	var textRange = document.body.createTextRange();
	textRange.moveToElementText(this.obj);
	
	var range = textRange.duplicate();
	range.setEndPoint('EndToStart', docRange);
	start = range.text.length;
	
	range = textRange.duplicate();
	range.setEndPoint('EndToEnd', docRange);
	end = range.text.length;
    }
    return new Bredima.Range(start, end);
}

/**
 * 選択位置でinputを分割、選択されていた値を返す
 */
Bredima.Token.Input.prototype.split = function(sel) {
    if(!sel) sel = this.getSelection();
    var txt = this.obj.value;
    this.obj.value = txt.substring(0, sel.start);
    var post = new Bredima.Token.Input({parent: this.row, value: txt.substring(sel.end)});
    this.row.add(post, this.getOrder() + 1);
    return {
	selectedValue : txt.substring(sel.start, sel.end),
	addedInput : post
    };
}

/**
 * 自分の前のInput（があれば）と併合
 */
Bredima.Token.Input.prototype.mergeWithPrevInput = function() {
    var order = this.getOrder();
    var prev = (order > 0) ? this.row.getRowfactor(order - 1) : '';
    if(prev && (prev.constructor.classID == 'input')) {
	var txt = prev.toString();
	this.obj.value = txt + this;
	this.row.remove(prev);
	this._setCursor(txt.length);
	return prev;
    }
}

/**
 * input内の特定の領域を選択
 * この後ブラウザを通じてonfocusが発生することに注意
 * @param {Object} range Bredima.Rangeオブジェクト
 */
Bredima.Token.Input.prototype.selectRange = function(range) {
    //ブラウザ依存判別
    if(this.obj.setSelectionRange) {
	this.obj.focus();
	this.obj.setSelectionRange(range.start, range.end);
    }
    else {
	var r2 = this.obj.createTextRange();
	r2.moveStart('character', range.start);
	r2.moveEnd('character', range.end - this.obj.value.length);
	r2.select();
    }
}

/**
 * input内のテキストを全選択
 */
Bredima.Token.Input.prototype.selectAll = function() {
    this.selectRange({start: 0, end: this.obj.value.length});
}

/**
 * selectRangeのエイリアス（Bredima.Rangeではなくて数値でカーソル位置を指定）
 * @param {Number} pos カーソルを置く位置
 * @private
 */
Bredima.Token.Input.prototype._setCursor = function(pos) {
    this.selectRange({start: pos, end: pos});
}

/* ====================
 Exp.Rowfactorの実装
*/

/**
 * input内の先頭か末尾にカーソルを置く
 * この後ブラウザを通じてonfocusが発生することに注意
 * @param {String} loc カーソルを置く位置 head /tail
 */
Bredima.Token.Input.prototype.setCursorTo = function(loc) {
    this._setCursor((loc == 'head') ? 0 : this.obj.value.length);
    return true;
}

Bredima.Token.Input.prototype.getIncomingInput = function() {
    return this;
}

/* ====================
 Expの実装
*/

/**
 * 自分のDOMオブジェクト内の値を文字列表記として返す
 */
Bredima.Token.Input.prototype.toString = function() { return this.obj.value; }

/**
 * 概要オブジェクトを返す
 * 頭にclassIDが不要なので関数をオーバーライド
 */
Bredima.Token.Input.prototype.toRundown = function() {
    return this.obj.value;
}

Bredima.Token.Input.prototype.toMML = function(expand) {
    return (this.obj.value == '') ? '' : Bredima.util.addTag('mi', this.obj.value); // 仮
}

Bredima.Token.Input.prototype.toLatex = function() {
    return this.obj.value;
}

Bredima.Token.Input.prototype._reposition = function() {
    var pos = this.getOrder();
    // どれかが当てはまれば最小化表示をしない
    var visible = (
	(this.row.container.exp.getCurrent() === this) || // フォーカスが自分に乗ってる
	(this.obj.value.length > 0) || // 中身がある
	(this.row.getLength() == 1) || // row内の要素が1つだけ
	// 自分が左端で、かつ右隣が左隣を必要とするもの（累乗）
	// 左端でなければ累乗とかは自分の左隣に作用しているものと仮定　(x+2)^2とか
	((pos == 1) && (this.row.getLength() > pos) && (this.row.getRowfactor(pos + 1).isNeedPrev)));
    if(visible) {
	this.removeClassNameSuffix();
	this.width = Math.floor((this.obj.value.length + 2) * Bredima.Token.width[this.row.level]);
	this._resizeWidth();
    }
    else {
	this.width = 4;
	this.obj.style.width = '';
	this.setClassNameSuffix('hidden');
    }
}

Bredima.style.add(
    ['.input', "border: 1px solid gray; margin: 0; text-align: left; overflow: hidden; resize: none;"],
    ['textarea.hidden', 'border-color: #ddd; background-color: #ddd; width: 4px !important;'],
    ['textarea.hidden:hover', 'border-color: gray;']
);

/**
 * @class数式内のコンテナオブジェクトの抽象クラス
 * @extends Bredima.Exp.Rowfactor
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 * @param {Array} rowLevelMap mrowを初期化する際の表示レベルを示した配列
 * 
 * mrowの初期化
 */
Bredima.Container = function(param) {
    arguments.callee.superClass.apply(this, arguments);
    this.exp = this.row.container.exp;
    this.parts = {};

    // rowdownにその他のプロパティがあるか確認
    if(param.rundown && (param.rundown.length > this.rowCount))
	this.config = param.rundown.splice(0, (param.rundown.length - this.rowCount));
    
    // 指定分だけmrowを用意
    this.rows = new Array;
    var p = {};
    for(var i = 0; i < this.rowCount; i++) {
	p = {
	    parent: this,
	    rundown: (param.rundown) ? param.rundown[i] : undefined,
	    value: (param.value && (i == this.primaryRow)) ? param.value : undefined,
	    level: (this.rowLevels && (this.rowLevels[i] > 0)) ? this.rowLevels[i] : this.row.getLevel()
	};
	this.rows.push(new Bredima.Exp.Row(p));
	this.obj.appendChild(this.rows[i].getDom());
    }
    this.content = this.rows[this.primaryRow];
}

Bredima.util.inherit(Bredima.Container, Bredima.Exp.Rowfactor);

Bredima.util.setProperty(Bredima.Container, {
    // Containerが使う属性
			     rowCount: 1,
			     padding: {top:0, right:0, bottom:0, left:0},
    primaryRow: 0
			 });

/* ===========================
 Bredima.Containerで定義
   -----------------------------
 配列管理
*/

/**
 * 管理しているrowの数を返す
 */
Bredima.Container.prototype.getLength = function() { return this.rows.length; }

/**
 * 指定した位置のrowを返す
 * @param {Number} order 位置
 */
Bredima.Container.prototype.getRow = function(order) {return this.rows[order]; }

/**
 * コンテナオブジェクト内のmrowの順序を返す
 */
Bredima.Container.prototype.orderOf = function(row) {
    for(var i = 0; i < this.rows.length; i++) {
	if(this.rows[i] == row) return i;
    }
    return null;
}

/**
 * mrowの縦方向の位置を返す
 * liftCursorの処理に使用
 * @private
 */
Bredima.Container.prototype._verticalOrderOf = function(row) {
    var order = row.getOrder();
    for(var i = 0; i < this.verticalOrder.length; i++) {
	if(order == this.verticalOrder[i]) return i;
    }
    return null;
}

/* ----------------------
 カーソル移動
*/

/**
 * コンテナオブジェクト内でのmrow間のカーソル横移動
 * Expからの呼び出し専用
 * @param {Object} row mrowオブジェクト
 * @param {String} dir 移動方向 back / fwd
 */
Bredima.Container.prototype.slideCursorFrom = function(row, dir) {
    var order = this.orderOf(row);
    var c = (dir == 'back');
    if((c && order == 0) || (!c && order == this.rows.length - 1)) return false;
    this.rows[order + (c ? -1 : 1)].setCursorToRow(c ? 'tail' : 'head');
    return true;
}

/**
 * コンテナオブジェクト内でのmrow間のカーソル縦移動
 * Expからの呼び出し専用
 * @param {Object} rowfactor rowfactorオブジェクト
 * @param {String} dir 移動方向 up/down
 */
Bredima.Container.prototype.liftCursorFrom = function(rowfactor, dir) {
    if(!this.verticalOrder) return false;
    var order = this._verticalOrderOf(rowfactor.row);
    var c = (dir == 'up');
    if((c && order == 0) || (!c && order == this.rows.length - 1)) return false;
    var row = this.rows[this.verticalOrder[order + (c ? -1 : 1)]];
    var rf2 = row.getRowfactor(Math.floor(row.getLength() * rowfactor.getOrder() / rowfactor.row.getLength()));
    if(!rf2.setCursorTo('head'))
	this.exp.slideCursorFrom(rf2, 'back');
    return true;
}

/* ---------------------------
 その他
*/

/**
 * （継承先で使用）構成パーツを登録
 * @param {Object} parts 登録するパーツの連想配列{登録名: 登録オブジェクト}
 */
Bredima.Container.prototype.add = function(key, parts) {
    this.parts[key] = parts;
    this.obj.appendChild(parts.getDom());
}

/**
 * 概要オブジェクト先頭の特殊属性を返す
 */
Bredima.Container.prototype._toRundownContentParam = function() {
}

/* ====================
 Exp.Rowfactorの実装
*/
/**
 * コンテナオブジェクトのheadかtailにカーソルを置く
 * @param {String} loc カーソルを置く位置 head / tail
 */
Bredima.Container.prototype.setCursorTo = function(loc) {
    this.rows[((loc == 'head') ? 0 : (this.rows.length - 1))].setCursorToRow(loc);
    return true;
}

Bredima.Container.prototype.getIncomingInput = function() {
    return this.content.getRowfactor(0).getIncomingInput();
}

/* ====================
 Expの実装
*/
/**
 * 概要オブジェクトの中身を返す
 */
Bredima.Container.prototype._toRundownContent = function() {
    var ret = new Array;
    var param = this._toRundownContentParam();
    if(param) ret = ret.concat(param);
    for(var i = 0; i < this.rowCount; i++) {
	ret.push(this.rows[i].toRundown());
    }
    return ret;
}

Bredima.Container.prototype._toMMLContent = function(expand) {
    var out = '';
    for(var i = 0; i < this.rows.length; i++)
	out += this.rows[i].toMML(expand);
    return out;
}

Bredima.Container.prototype._toLatexContent = function() {
    var out = '';
    for(var i = 0; i < this.rows.length; i++)
	out += this.rows[i].toLatex();
    return out;
}


/* ------------------------
 レイアウト管理
*/

Bredima.Container.prototype._reposition = function() {
    var ps = new Array;
    for(var i = 0; i < this.rows.length; i++)
	ps.push(this.rows[i].getPreferredSize());
    this._repositionContent((ps.length == 1) ? ps[0] : ps);
}

Bredima.Container.prototype._repositionContent = function(ps) {
    this.top = ps.top + this.padding.top;
    this.bottom = ps.bottom + this.padding.bottom;
    this.width = ps.width + this.padding.left + this.padding.right;

    this._resize();
    this.content.setPos(this.padding.left, this.padding.top);
}

Bredima.Container.prototype.repositionAll = function() {
    for(var i = 0; i < this.rows.length; i++) {
	this.rows[i].repositionAll();
    }
    this._reposition();
}

Bredima.style.add(
    ['.container', 'border: 1px dashed orange;']
);

/**
  @fileoverview
  コンテナオブジェクト（レイアウトスキーマ）

  Bredima.Container.
    Frac
    
 */
/* ===============================
*/
/**
 * @class 分数
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Frac = function(param) {
    this.__super__(param);
    this.add('separator', new Bredima.Parts.Separator());
}

Bredima.util.inherit(Bredima.Container.Frac, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Frac, {
			     rowCount : 2,
			     MML : 'mfrac',
			     tex : 'frac',
			     verticalOrder : [0, 1]
			 });

Bredima.Exp.register(Bredima.Container.Frac, 'container', 'frac');

Bredima.Container.Frac.prototype._repositionContent = function(ps) {
    this.width = Math.max(ps[0].width, ps[1].width);
    this.top = ps[0].height;
    this.bottom = ps[1].height + 1;
    
    this._resize();
    this.rows[0].setPos((this.width - ps[0].width) / 2, 0);
    this.rows[1].setPos((this.width - ps[1].width) / 2, ps[0].height + 1);
    this.parts.separator.setPos(0, ps[0].height);
}

/* ===============================
*/
/**
 * @class 括弧
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 * @param {Number} type 括弧の種類 0: (), 1: {}, 2: []
 */
Bredima.Container.Fenced = function(param, type) {
    this.__super__(param);
    this.type = (this.config) ? this.config : ((type == null) ? 0 : type);
    //var menu = new Bredima.Parts.Popup(this.exp.bredima, 'fenced', this);
    this.add('open', new Bredima.Parts.Image(this.types[this.type].name + '_open.png'));
    this.add('close', new Bredima.Parts.Image(this.types[this.type].name + '_close.png'));
    this.add('menu', new Bredima.Parts.Popup(this.exp.bredima, 'fenced', this));
    this.parts.open.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Fenced, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Fenced, {
    MML: 'mfenced',
    padding: {top:0, right:16, bottom:0, left:8}
});

Bredima.Exp.register(Bredima.Container.Fenced, 'container', 'fenced');

Bredima.Container.Fenced.prototype.types = [
    {name: 'parenthesis', character: ['(', ')'], certified: true},
    {name: 'brace', character: ['{', '}']},
    {name: 'bracket', character: ['[', ']']}
];

/* ------------------
 Containerの実装
*/
Bredima.Container.Fenced.prototype._toRundownContentParam = function() {
    if(this.type != 0) return this.type;
}

/* ------------------
 Expの実装
*/
/**
 * MathML表記を返す
 * 開きタグが特殊なので全て構築
 */
Bredima.Container.Fenced.prototype.toMML = function(expand) {
    var attr = this.types[this.type];
    var out = (!attr.certified) ? ('open="' + attr.character[0] + '" close="' + attr.character[1] + '"') : '';
    return Bredima.util.addTag(this.MML, this.content.toMML(expand), out);
}

/**
 * LaTeX表記を返す
 * 閉じコマンドも必要なので全て構築
 */
Bredima.Container.Fenced.prototype.toLatex = function() {
    var out = '\\left' + this.types[this.type].character[0];
    out += this.content.toLatex('noFence');
    out += '\\right' + this.types[this.type].character[1];
    return out;
}

Bredima.Container.Fenced.prototype._repositionContent = function() {
    Bredima.util.applySuper('_repositionContent', this, arguments);
    this.parts.open.setSize(10, this.height);
    this.parts.close.setSize(10, this.height);
    this.parts.close.setPos(this.width - 16, 0);
    this.parts.menu.setPos(this.width - 10, this.height - 10);
}

/* ------------------
 PopupListenerの実装
*/
/**
 * ポップアップメニューの選択を受信して括弧の種類を変更
 * @privete
 */
Bredima.Container.Fenced.prototype.menuClickPerformed = function(arg) {
    if(this.type != arg) {
	this.exp.startRowRec(this.row);
	this.type = arg;
	this.parts.open.setURI(this.types[this.type].name + '_open.png');
	this.parts.close.setURI(this.types[this.type].name + '_close.png');
	this.exp.endRowRec('nofocus');
    }
}

/* ===============================
*/
/**
 * @class 平方根
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sqrt = function(param) {
    this.__super__(param);
    this.add('radical', new Bredima.Parts.Image('radical.png'));
    this.add('line', new Bredima.Parts.Separator());
    this.parts.radical.setPos(0, 2);
    this.parts.line.setPos(10, 2);
}

Bredima.util.inherit(Bredima.Container.Sqrt, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sqrt, {
			     MML: 'msqrt',
			     tex: 'sqrt',
			     padding: {top: 4, right: 0, bottom: 0, left: 10}
			 });

Bredima.Exp.register(Bredima.Container.Sqrt, 'container', 'sqrt');

Bredima.Container.Sqrt.prototype._repositionContent = function() {
    Bredima.util.applySuper('_repositionContent', this, arguments);
    this.parts.radical.setSize(10, this.height - 2);
    this.parts.line.setSize(this.width - 10, 1);
}

/* ===============================
*/
/**
 * @class 累乗根
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Root = function(param) {
    this.__super__(param);
    this.padding = {top: 4, right: 0, bottom: 0, left: 10}; // repositionで変更するのでインスタンス毎に持つ
    this.add('radical', new Bredima.Parts.Image('radical.png'));
    this.add('line', new Bredima.Parts.Separator());
    this.rows[0].setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Root, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Root, {
			     rowCount: 2,
			     rowLevels: [2, 0],
			     primaryRow: 1,
			     MML: 'mroot',
			     tex: 'sqrt'
			 });

Bredima.Exp.register(Bredima.Container.Root, 'container', 'root');

Bredima.Container.Root.prototype._toMMLContent = function(expand) {
    return this.rows[1].toMML(expand) + this.rows[0].toMML(expand);
}

Bredima.Container.Root.prototype._toLatexContent = function() {
    return '[' + this.rows[0].toLatex('nofence') + ']' + this.rows[1].toLatex();
}

Bredima.Container.Root.prototype._repositionContent = function(ps) {
    var offset = Math.max((ps[0].height - ps[1].top), 4);
    this.top = offset + ps[1].top;
    this.bottom = ps[1].bottom;
    this.width = ps[0].width + 8 + ps[1].width;

    this._resize();
    this.rows[1].setPos(ps[0].width + 8, offset);
    this.parts.radical.setSize(10, ps[1].height + 2);
    this.parts.radical.setPos(ps[0].width - 2, offset - 2);
    this.parts.line.setSize(ps[1].width, 1);
    this.parts.line.setPos(ps[0].width + 8, offset - 2);
}

/* ===============================
*/
/**
 * @class 累乗
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sup = function(param) {
    this.__super__(param);
    this.content.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Sup, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sup, {
			     MML: 'msup',
			     isMMLNeedPrev: true,
			     rowLevels: [2]
			 });

Bredima.Exp.register(Bredima.Container.Sup, 'container', 'sup');

Bredima.Container.Sup.prototype._toMMLContent = function(expand, prev) {
    if(prev == '') prev = '<mo></mo>';
    return prev + this.content.toMML(expand);
}

Bredima.Container.Sup.prototype.toLatex = function() {
    return '^' + this.content.toLatex();
}

Bredima.Container.Sup.prototype._repositionContent = function(ps) {
    this.top = ps.height;
    this.bottom = 0;
    this.width = ps.width;

    this._resize();
}

/* ===============================
*/
/**
 * @class 添字
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sub = function(param) {
    this.__super__(param);
    this.content.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Sub, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sub, {
			     MML: 'msub',
			     isMMLNeedPrev: true,
			     rowLevels: [2]
			 });

Bredima.Exp.register(Bredima.Container.Sub, 'container', 'sub');

Bredima.Container.Sub.prototype._toMMLContent = function(expand, prev) {
    if(prev == '') prev = '<mo></mo>';
    return prev + this.content.toMML(expand);
}

Bredima.Container.Sub.prototype.toLatex = function() {
    return '_' + this.content.toLatex();
}

Bredima.Container.Sub.prototype._repositionContent = function(ps) {
    this.top = 0;
    this.bottom = ps.height;
    this.width = ps.width;

    this._resize();
}


/* ===============================
*/
/**
 * @class オーバーライン
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Over = function(param) {
    this.__super__(param);
    this.add('line', new Bredima.Parts.Separator());
    this.parts.line.setPos(2, 2);
}

Bredima.util.inherit(Bredima.Container.Over, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Over, {
			     MML: 'mover',
			     tex: 'overline',
			     padding: {top: 4, right: 0, bottom: 0, left: 0}
			 });

Bredima.Exp.register(Bredima.Container.Over, 'container', 'over');

Bredima.Container.Over.prototype._toMMLContent = function(expand) {
    return this.content.toMML(expand) +
	Bredima.util.addTag('mo', (expand) ? Bredima.Parts.Symbol.all.OverBar.code : '&OverBar;');
}

Bredima.Container.Over.prototype._repositionContent = function() {
    Bredima.util.applySuper('_repositionContent', this, arguments);
    this.parts.line.setSize(this.width - 4, 1);
}

/* ===============================
*/
/**
 * @class ベクトル
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Vector = function(param) {
    this.__super__(param);
    this.add('head', new Bredima.Parts.Image('arrowhead.png'));
    this.add('shaft', new Bredima.Parts.Image('arrowshaft.png'));
    this.parts.shaft.setPos(2, 2);
}

Bredima.util.inherit(Bredima.Container.Vector, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Vector, {
			     MML: 'mover',
			     tex: 'vec',
			     padding: {top: 10, right: 0, bottom: 0, left: 0}
			 });

Bredima.Exp.register(Bredima.Container.Vector, 'container', 'vector');

Bredima.Container.Vector.prototype._toMMLContent = function(expand) {
    return this.content.toMML(expand) +
	Bredima.util.addTag('mo', (expand) ? Bredima.Parts.Symbol.all.rarr.code : '&RightArrow;');
    //RightArrowはrarrのエイリアス
}

Bredima.Container.Vector.prototype._repositionContent = function() {
    Bredima.util.applySuper('_repositionContent', this, arguments);
    this.parts.shaft.setSize(this.width - 14, 9);
    this.parts.head.setPos(this.width - 12, 2);
}

/* ===============================
*/
/**
 * @class 順列・組合せ・重複組合せ
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 * @param {Number} kind 種類
 *     0: 順列(Permutation), 1: 組合せ(Combination), 2: 重複組合せ(Homogeneous)
 */

Bredima.Container.Permutation = function(param, kind) {
    this.__super__(param);
    this.kind = (this.config) ? this.config : kind;
    this.add('symbol', new Bredima.Parts.String(this.values[this.kind], 1));
    this.add('menu', new Bredima.Parts.Popup(this.exp.bredima, 'permutation', this));
}

Bredima.util.inherit(Bredima.Container.Permutation, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Permutation, {
			     MML: 'mmultiscripts',
			     rowCount: 2,
			     rowLevels: [2, 2]
			 });

Bredima.Exp.register(Bredima.Container.Permutation, 'container', 'permutation');

Bredima.Container.Permutation.prototype.values = ['P', 'C', 'H'];

/* ------------------
 Containerの実装
*/
Bredima.Container.Permutation.prototype._toRundownContentParam = function() {
    return this.kind;
}

/* ------------------
 Expの実装
*/
Bredima.Container.Permutation.prototype._toMMLContent = function(expand) {
    return Bredima.util.addTag('mi', this.values[this.kind]) +
	this.rows[1].toMML(expand) + '<none/>' + "\n" +
	'<mprescripts/>' + "\n" +
	this.rows[0].toMML(expand) + '<none/>' + "\n";
}

Bredima.Container.Permutation.prototype.toLatex = function() {
    return '{{}_' + this.rows[0].toLatex() +
	this.values[this.kind] +
	'_' + this.rows[1].toLatex() + '}';
}

Bredima.Container.Permutation.prototype._repositionContent = function(ps) {
    var symbol = this.parts.symbol.getPreferredSize();
    this.width = symbol.width + ps[0].width + ps[1].width + 8;
    this.top = symbol.top;
    this.bottom = Math.max(ps[0].height, ps[1].height, symbol.bottom);

    this._resize();
    this.parts.symbol.setPos(ps[0].width, 0);
    this.parts.menu.setPos(this.width - 8, this.height - 8);
    this.rows[0].setPos(0, symbol.top);
    this.rows[1].setPos(ps[0].width + symbol.width, symbol.top);
}

/* ------------------
 PopupListenerの実装
*/
Bredima.Container.Permutation.prototype.menuClickPerformed = function(arg) {
    if(this.kind != arg) {
	this.exp.startRowRec(this.row);
	this.kind = arg;
	this.parts.symbol.setString(this.values[this.kind]);
	this.exp.endRowRec();
    }
}

/* ===============================
*/
/**
 * @class 総和
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sum = function(param) {
    this.__super__(param);
    this.add('symbol', new Bredima.Parts.Symbol(this.exp.bredima, 'sum', this.row.level));
}

Bredima.util.inherit(Bredima.Container.Sum, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sum, {
			     MML: 'munderover',
			     rowCount: 2,
			     rowLevels: [2, 2],
			     verticalOrder: [1, 0]
			 });

Bredima.Exp.register(Bredima.Container.Sum, 'container', 'sum');

Bredima.Container.Sum.prototype._toMMLContent = function(expand) {
    return Bredima.util.addTag('mo', this.parts.symbol.toMML(expand)) +
	Bredima.util.applySuper('_toMMLContent', this);
}

Bredima.Container.Sum.prototype.toLatex = function() {
    return this.parts.symbol.toLatex() +
	'_' + this.rows[0].toLatex() +
	'^' + this.rows[1].toLatex();
}

Bredima.Container.Sum.prototype._repositionContent = function(ps) {
    var symbol = this.parts.symbol.getPreferredSize();
    this.width = Math.max((symbol.width + 4), ps[0].width, ps[1].width);
    this.top = symbol.top + ps[1].height;
    this.bottom = symbol.bottom + ps[0].height;

    this._resize();
    this.parts.symbol.setPos((this.width - symbol.width) / 2, ps[1].height);
    this.rows[1].setPos((this.width - ps[1].width) / 2, 0);
    this.rows[0].setPos((this.width - ps[0].width) / 2, ps[1].height + symbol.height);
}

/* ===============================
*/
/**
 * @class 積分
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Integral = function(param) {
    this.__super__(param);
    this.add('symbol', new Bredima.Parts.Symbol(this.exp.bredima, 'mathint', this.row.level));
}

Bredima.util.inherit(Bredima.Container.Integral, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Integral, {
			     MML: 'msubsup',
			     rowCount: 2,
			     rowLevels: [2, 2],
			     verticalOrder: [1, 0]
			 });

Bredima.Exp.register(Bredima.Container.Integral, 'container', 'integral');

Bredima.Container.Integral.prototype._toMMLContent = function(expand) {
    return Bredima.util.addTag('mo', this.parts.symbol.toMML(expand)) + 
	Bredima.util.applySuper('_toMMLContent', this);
}

Bredima.Container.Integral.prototype.toLatex = function() {
    return this.parts.symbol.toLatex() +
	'_' + this.rows[0].toLatex() +
	'^' + this.rows[1].toLatex();
}

Bredima.Container.Integral.prototype._repositionContent = function(ps) {
    var symbol = this.parts.symbol.getPreferredSize();
    var center = (ps[1].bottom + ps[0].top) / 2
    this.width = symbol.width + Math.max(ps[0].width, ps[1].width) + 2;
    this.top = ps[1].top + Math.ceil(center);
    this.bottom = ps[0].bottom + Math.floor(center);

    this._resize();
    this.parts.symbol.setPos(2, this.top - symbol.top);
    this.rows[1].setPos(symbol.width + 2, 0);
    this.rows[0].setPos(symbol.width + 2, this.height - ps[0].height);
}

/* ===============================
*/
/**
 * @class 定積分後の区間
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.IntInterval = function(param) {
    this.__super__(param);
    this.add('open', new Bredima.Parts.Image('bracket_open.png'));
    this.add('close', new Bredima.Parts.Image('bracket_close.png'));
}

Bredima.util.inherit(Bredima.Container.IntInterval, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.IntInterval, {
			     MML: 'msubsup',
			     rowCount: 3,
			     rowLevels: [0, 2, 2],
			     verticalOrder: [2, 1]
			 });

Bredima.Exp.register(Bredima.Container.IntInterval, 'container', 'intinterval');

Bredima.Container.IntInterval.prototype.liftCursorFrom = function(rowfactor, dir) {
    var order = this.orderOf(rowfactor.row);
    if(((order == 1) && (dir == 'up')) || ((order == 2) && (dir == 'down')))
	return Bredima.util.applySuper('liftCursorFrom', this, arguments);
    return false;
}

Bredima.Container.IntInterval.prototype._toMMLContent = function(expand) {
    return Bredima.util.addTag('mfenced', this.content.toMML(expand), 'open="[" close="]"') +
	Bredima.util.applySuper('_toMMLContent', this);
}

Bredima.Container.IntInterval.prototype.toLatex = function() {
    return '\\left[' + this.content.toLatex('nofence') + '\\right]' +
	'_' + this.rows[1].toLatex() +
	'^' + this.rows[2].toLatex();
}

Bredima.Container.IntInterval.prototype._repositionContent = function(ps) {
    var center = (ps[2].bottom + ps[1].top) / 2;
    var top = Math.max(ps[2].top + Math.ceil(center) - ps[0].top, 0);
    this.width = ps[0].width + Math.max(ps[1].width, ps[2].width) + 22;
    this.top = top + ps[0].top;
    this.bottom = Math.max(ps[0].bottom, ps[1].bottom + Math.floor(center));

    this._resize();
    this.parts.open.setSize(10, ps[0].height);
    this.parts.open.setPos(2, top);
    this.parts.close.setSize(10, ps[0].height);
    this.parts.close.setPos(ps[0].width + 12, top);
    this.content.setPos(12, top);
    this.rows[2].setPos(ps[0].width + 22, 0);
    this.rows[1].setPos(ps[0].width + 22, this.height - ps[1].height);
}

/* ===============================
*/
/**
 * @class 極限
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Limit = function(param) {
    this.__super__(param);
    this.add('string', new Bredima.Parts.String('lim', this.row.getLevel()));
}

Bredima.util.inherit(Bredima.Container.Limit, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Limit, {
			     MML: 'munder',
			     tex: 'lim_',
			     rowLevels: [2]
			 });

Bredima.Exp.register(Bredima.Container.Limit, 'container', 'lim');

Bredima.Container.Limit.prototype._toMMLContent = function(expand) {
    return Bredima.util.addTag('mi', 'lim') + this.content.toMML(expand);
}

Bredima.Container.Limit.prototype._repositionContent = function(ps) {
    var string = this.parts.string.getPreferredSize();
    this.width = Math.max(string.width, ps.width);
    this.top = string.top;
    this.bottom = string.bottom + ps.height;

    this._resize();
    this.parts.string.setPos((this.width - string.width) / 2, 0);
    this.content.setPos((this.width - ps.width) / 2, string.height);
}

/**
 * @fileoverview 終了処理
 */

Bredima.style.addStylesheet();

