
/**
 * @class 変更履歴管理
 * @constructor
 */
Bredima.History = function() {
    this.records = new Array;
    this.ptr = 0;
    this.p2 = 0; // 各record内での位置
    this.length = 0;
    this.isEnable = {undo: false, redo: false};
    this.observer = new Bredima.Observer();
}
// 履歴の最大保存個数 同一input内の履歴はまとめて1個
Bredima.History.maxLength = 10;

/**
 * 履歴を追加
 */
Bredima.History.prototype.add = function(record) {
    if(this.p2 > 0) {
	this.records[this.ptr].chop(this.p2);
	this.ptr++;
    }
    this.records[this.ptr++] = record;
    if(this.ptr >= Bredima.History.maxLength) {
	this.records.shift();
	this.ptr--;
    }
    this.length = this.ptr;
    this.p2 = 0;
    this.ackRecordArise();
}

/**
 * historyに追加前だけど履歴が追加されたことをhistoryとその受信者に連絡
 */
Bredima.History.prototype.ackRecordArise = function() {
    this._turn('undo', true);
    this._turn('redo', false);
    this.observer.notify('HistoryChange');
}

/**
 * Undo
 */
Bredima.History.prototype.undo = function() {
    if(this.p2 == 0) {
	if(this.ptr == 0) return false;
	this.p2 = this.records[--this.ptr].getLength() - 1;
    }
    this.records[this.ptr].revert(--this.p2);
    this._turn('redo', true);
    if((this.p2 == 0) && (this.ptr == 0)) this._turn('undo', false);
    this.observer.notify('HistoryChange');
    return true;
}

/**
 * Redo
 */
Bredima.History.prototype.redo = function() {
    if(this.ptr == this.length) return false;
    this.records[this.ptr].revert(++this.p2);
    this._turn('undo', true);
    if(this.p2 == this.records[this.ptr].getLength() - 1) {
	this.ptr++;
	this.p2 = 0;
	if(this.ptr == this.length) this._turn('redo', false);
    }
    this.observer.notify('HistoryChange');
    return true;
}

/**
 * Undo/Redoの実行可能属性を切り替え
 *@private
 */
Bredima.History.prototype._turn = function(name, b) {
    if(this.isEnable[name] != b) {
	this.observer.notify('HistoryEnableChange', (b ? 'enable' : 'disable') + '_' + name);
	this.isEnable[name] = b;
    }
}

/* ========================
*/
/**
 * @class rowfactor増減についての記録の実体
 * @constructor
 * @param {Object} row 状態を記録するrow
 */
Bredima.History.RowRecord = function(row) {
    this.row = row;
    this.records = new Array;
}

/**
 * input要素についての履歴ではないのでfalseを返す
 */
Bredima.History.RowRecord.prototype.isInput = function() { return false; }

/**
 * 変更を記録
 * オブジェクト変更操作の前と後で1回ずつ呼ぶ
 */
Bredima.History.RowRecord.prototype.commit = function(opt) {
    var current = this.row.container.exp.getCurrent();
    this.records.push({
			  rundown: this.row.toRundown(),
			  position: new Bredima.History.TreePosition(current),
			  selection: ((opt == 'selected') ? '' : current.getSelection())
		      });
    return true;
}

/**
 * 記録を終了
 */
Bredima.History.RowRecord.prototype.close = function() {
    this.position = new Bredima.History.TreePosition(this.row);
    this.row = '';
    return true;
}

/**
 * 記録している状態へ戻す
 */
Bredima.History.RowRecord.prototype.revert = function(order) {
    var row = this.position.trace();
    var record = this.records[order];
    row.initFromRundown(record.rundown);

    // 再配置
    row.repositionAll();
    // 選択でrootへの再配置を兼ねる
    record.position.trace().selectRange(record.selection);
}

/**
 * 記録している状態の個数を返す
 * 通常は2
 */
Bredima.History.RowRecord.prototype.getLength = function() {
    return this.records.length;
}

/* ========================
*/
/**
 * @class input要素内での文字列操作の記録
 * @constructor
 * @param {Object} input 状態を記録するinput
 */
Bredima.History.InputRecord = function(input) {
    this.input = input;
    this.records = new Array;
    this.cache();
    this.records.push({value: this.value, selection: this.selection});
    this.isTemporary = false;
    this.isCached = false;
}

/**
 * input要素についての履歴なのでtrueを返す
 */
Bredima.History.InputRecord.prototype.isInput = function() { return true; }

/**
 * 変更が行われた判定するために現在の状態を一時記録
 */
Bredima.History.InputRecord.prototype.cache = function() {
    if(!this.isCached) {
	this.isCached = true;
	this.value = this.input.toString();
	this.selection = this.input.getSelection();
    }
}

/**
 * cacheと比較して変更されていれば記録
 * @param {String} type 記録の種類
 * temporary: このcommitで変更が記録された場合その記録は1つ前の履歴を上書きする可能性がある
 * noselection: commit時にselectionを取得しない focusがない場所のcommit時に使用
 */
Bredima.History.InputRecord.prototype.commit = function(type) {
    this.isCached = false;
    var sel = (type == 'noselection') ? this.selection : this.input.getSelection();
    if(this.value != this.input.toString()) {
	if((type == 'temporary') && (this.isTemporary)){
	    this.records.pop();
	}
	this.records.push({ value: this.input.toString(), selection: sel });
	this.isTemporary = (type == 'temporary');
	return true;
    }
    else if(!this.selection.equals(sel)) {
	this.isTemporary = false;
    }
    return false;
}

/**
 * 記録を終了
 */
Bredima.History.InputRecord.prototype.close = function() {
    this.position = new Bredima.History.TreePosition(this.input);
    this.input = '';
    return ((this.records.length > 1) &&
	    !((this.records.length == 2) && (this.records[0].value == this.records[1].value)));
}

/**
 * 記録している状態へ戻す
 */
Bredima.History.InputRecord.prototype.revert = function(order) {
    var input = this.position.trace();
    input.init(this.records[order]);
}

/**
 * Undo後に変更が合った場合に不要になる部分を切り捨てる
 */
Bredima.History.InputRecord.prototype.chop = function(order) {
    this.records.splice(order + 1, this.records.length);
}

/**
 * 記録している状態の個数を返す
 */
Bredima.History.InputRecord.prototype.getLength = function() {
    return this.records.length;
}

/* ========================
*/
/**
 * @class 数式ツリー内での場所の記録
 * @constructor
 * @param {Object} row 記録を行う要素
 */
Bredima.History.TreePosition = function(elem) {
    this.tree = new Array;

    if(elem.constructor.classID == 'row') {
	this.tree.push(elem.getOrder());
	elem = elem.container;
    }
    while(elem.constructor.classID != 'exproot') {
	this.tree.push(elem.getOrder());
	this.tree.push(elem.row.getOrder());
	elem = elem.row.container;
    }
    this.root = elem;
}

/**
 * 記録していた要素を返す
 */
Bredima.History.TreePosition.prototype.trace = function() {
    var point = this.root;
    var i;
    for(i = this.tree.length - 1; i > 0; i-=2) {
	point = point.getRow(this.tree[i]);
	point = point.getRowfactor(this.tree[i-1]);
    }
    if(i == 0) point = point.getRow(this.tree[i]);
    return point;
}
