/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 - 2007 Phex Development Group
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  --- SVN Information ---
 *  $Id: BanHostBatch.java 3859 2007-07-01 20:15:19Z gregork $
 */
package phex.security;

import java.util.Stack;

import phex.common.ExpiryDate;
import phex.common.RunnerQueueWorker;
import phex.common.ThreadTracking;
import phex.common.address.DestAddress;
import phex.utils.Localizer;
import phex.utils.NLogger;
import phex.utils.NLoggerNames;

/**
 * The ban host batch is responsible to ban larger amount of addresses in the
 * background.
 */
public class BanHostBatch extends RunnerQueueWorker
{
    private static Object lock = new Object();

    private static BanHostBatch instance;

    private Thread runnerThread;

    private Stack<BanHostHolder> rules;

    private BanHostBatch()
    {
        rules = new Stack<BanHostHolder>();
    }

    private static void init()
    {
        synchronized (lock)
        {
            if (instance == null)
            {
                NLogger.debug(BanHostBatch.class, "Creating Instance");
                instance = new BanHostBatch();
            }
            if (instance.runnerThread == null)
            {
                instance.createRunner();
            }
        }
    }

    private synchronized void createRunner()
    {
        NLogger.debug(BanHostBatch.class, "Creating RunnerThread");
        runnerThread = new Thread(ThreadTracking.rootThreadGroup,
                new BatchWorker());
        runnerThread.setPriority(Thread.NORM_PRIORITY);
        runnerThread.setDaemon(true);
        runnerThread.start();
    }

    public static void addDestAddress(DestAddress address, ExpiryDate expDate)
    {
        synchronized (lock)
        {
            init();
            instance.rules.add(new BanHostHolder(address, expDate));
        }
        synchronized (instance)
        {
            instance.notify();
        }
    }

    private class BatchWorker implements Runnable
    {
        public void run()
        {
            PhexSecurityManager securityMgr = PhexSecurityManager.getInstance();
            try
            {
                while (true)
                {
                    BanHostHolder next = rules.pop();
                    AccessType access = securityMgr
                            .controlHostAddressAccess( next.address );
                    // only add if not already added through earlier batch.
                    if ( access == AccessType.ACCESS_GRANTED)
                    {
                        securityMgr.createIPAccessRule(
                            Localizer.getString( "UserBanned" ),
                            next.address.getIpAddress().getHostIP(), (byte)32,
                                false, next.expDate, true);
                    }

                    synchronized (instance)
                    {
                        if (rules.isEmpty())
                        {
                            instance.wait(30 * 1000);
                        }
                        if (!rules.isEmpty())
                        {
                            continue;
                        }
                        // rules are still empty... stop batch process...
                        runnerThread = null;
                        NLogger.debug(BanHostBatch.class, "Releasing RunnerThread");
                        break;
                    }
                }
            } catch (Throwable th)
            {
                runnerThread = null;
                NLogger.error(NLoggerNames.GLOBAL, th, th);
            }
            // safty check
            synchronized (lock)
            {
                if (!rules.isEmpty())
                {// oups... somebody is left we need to restart..
                    createRunner();
                } else
                {
                    instance = null;
                }
            }
        }
    }

    private static class BanHostHolder
    {
        private DestAddress address;

        private ExpiryDate expDate;

        public BanHostHolder(DestAddress address, ExpiryDate expDate)
        {
            this.address = address;
            this.expDate = expDate;
        }
    }
}
