/*
 * $HeadURL: https://svn.apache.org/repos/asf/jakarta/httpcomponents/httpclient/tags/4.0-alpha2/module-client/src/main/java/org/apache/http/impl/client/DefaultRedirectHandler.java $
 * $Revision: 589630 $
 * $Date: 2007-10-29 14:57:15 +0100 (Mon, 29 Oct 2007) $
 *
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.apache.http.impl.client;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.util.HashSet;
import java.util.Set;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.ProtocolException;
import org.apache.http.client.CircularRedirectException;
import org.apache.http.client.RedirectHandler;
import org.apache.http.client.params.ClientPNames;
import org.apache.http.client.utils.URLUtils;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.ExecutionContext;
import org.apache.http.protocol.HttpContext;


/**
 * Default implementation of a redirect handler.
 * 
 * @author <a href="mailto:oleg at ural.ru">Oleg Kalnichevski</a>
 *
 *
 * <!-- empty lines to avoid svn diff problems -->
 * @version $Revision: 589630 $
 *
 * @since 4.0
 */
public class DefaultRedirectHandler implements RedirectHandler {

    private static final Log LOG = LogFactory.getLog(DefaultRedirectHandler.class);
    
    private static final String REDIRECT_LOCATIONS = "http.protocol.redirect-locations";

    public DefaultRedirectHandler() {
        super();
    }
    
    public boolean isRedirectRequested(
            final HttpResponse response,
            final HttpContext context) {
        if (response == null) {
            throw new IllegalArgumentException("HTTP response may not be null");
        }
        int statusCode = response.getStatusLine().getStatusCode();
        switch (statusCode) {
        case HttpStatus.SC_MOVED_TEMPORARILY:
        case HttpStatus.SC_MOVED_PERMANENTLY:
        case HttpStatus.SC_SEE_OTHER:
        case HttpStatus.SC_TEMPORARY_REDIRECT:
            return true;
        default:
            return false;
        } //end of switch
    }
 
    public URI getLocationURI(
            final HttpResponse response, 
            final HttpContext context) throws ProtocolException {
        if (response == null) {
            throw new IllegalArgumentException("HTTP response may not be null");
        }
        //get the location header to find out where to redirect to
        Header locationHeader = response.getFirstHeader("location");
        if (locationHeader == null) {
            // got a redirect response, but no location header
            throw new ProtocolException(
                    "Received redirect response " + response.getStatusLine()
                    + " but no location header");
        }
        String location = locationHeader.getValue();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Redirect requested to location '" + location + "'");
        }

        // URI(String uri) expects an encoded uri
//        try {
//            location = encodeLocation(location);
//        } catch (UnsupportedEncodingException e) {
//            throw new ProtocolException("Invalid redirect URI: " + location, e);
//        }

        URI uri;
        try {
            uri = new URI(location);            
        } catch (URISyntaxException ex) {
            throw new ProtocolException("Invalid redirect URI: " + location, ex);
        }

        HttpParams params = response.getParams();
        // rfc2616 demands the location value be a complete URI
        // Location       = "Location" ":" absoluteURI
        if (!uri.isAbsolute()) {
            if (params.isParameterTrue(ClientPNames.REJECT_RELATIVE_REDIRECT)) {
                throw new ProtocolException("Relative redirect location '" 
                        + uri + "' not allowed");
            }
            // Adjust location URI
            HttpHost target = (HttpHost) context.getAttribute(
                    ExecutionContext.HTTP_TARGET_HOST);
            if (target == null) {
                throw new IllegalStateException("Target host not available " +
                        "in the HTTP context");
            }
            
            HttpRequest request = (HttpRequest) context.getAttribute(
                    ExecutionContext.HTTP_REQUEST);
            
            try {
                URI requestURI = new URI(request.getRequestLine().getUri());
                URI absoluteRequestURI = URLUtils.toURI(
                        target.getSchemeName(),
                        null,
                        target.getHostName(),
                        target.getPort(),
                        requestURI.getRawPath(),
                        requestURI.getRawQuery(),
                        requestURI.getRawFragment());
                uri = absoluteRequestURI.resolve(uri); 
            } catch (URISyntaxException ex) {
                throw new ProtocolException(ex.getMessage(), ex);
            }
        }
        
        if (params.isParameterFalse(ClientPNames.ALLOW_CIRCULAR_REDIRECTS)) {
            
            Set redirectLocations = (Set) context.getAttribute(REDIRECT_LOCATIONS);
            
            if (redirectLocations == null) {
                redirectLocations = new HashSet();
                context.setAttribute(REDIRECT_LOCATIONS, redirectLocations);
            }
            
            URI redirectURI;
            if (uri.getQuery() != null || uri.getFragment() != null) {
                try {
                    redirectURI = URLUtils.toURI(
                            uri.getScheme(),
                            null,
                            uri.getHost(),
                            uri.getPort(),
                            uri.getRawPath(),
                            null,
                            null);
                } catch (URISyntaxException ex) {
                    throw new ProtocolException(ex.getMessage(), ex);
                }
            } else {
                redirectURI = uri;
            }
            
            if (redirectLocations.contains(redirectURI)) {
                throw new CircularRedirectException("Circular redirect to '" +
                        redirectURI + "'");
            } else {
                redirectLocations.add(redirectURI);
            }
        }
        
        return uri;
    }

    private String encodeLocation(String location) throws UnsupportedEncodingException {
        if(location == null) {
            return null;
        }
        int queryStringStart = location.indexOf("?");        
        if(queryStringStart > 0) {
            String fragment = "";
            int fragmentStart = location.indexOf("#");
            if(fragmentStart < 0) {
                fragmentStart = location.length();
            } else {
                fragment = location.substring(fragmentStart + 1);
            }
            String queryString = location.substring(queryStringStart + 1, fragmentStart);    
            StringTokenizer st = new StringTokenizer(queryString, "&", true);
            StringBuilder newQuery = new StringBuilder();
            while(st.hasMoreElements()) {
                String nameValuePair = st.nextToken();
                if(nameValuePair.equals("&")) {
                    newQuery.append("&");
                } else {
                    StringTokenizer st2 = new StringTokenizer(nameValuePair, "=", true);
                    while(st2.hasMoreElements()) {
                        String s = st2.nextToken();
                        if(s.equals("=")) {
                            newQuery.append("=");
                        } else {
                            newQuery.append(URLEncoder.encode(s, "ISO-8859-1"));
                        }
                    }
                }
            }
            location = location.substring(0, queryStringStart);
            location += "?";
            location += newQuery.toString();
            if(!fragment.equals("")) {
                location += "#";
                location += fragment;
            }
        }
        return location;
    }

}
